<?php
/*
Plugin Name: ePN Affiliate Banner
Plugin URI: https://ebay.buzin.biz
Description: Customize, search and show eBay Partner Network affiliate banners appropriate to each blog post content and earn money
Version: 1.1.1
Author: Vitalii Buzin
Author URI: https://buzin.biz
Text Domain: epn-affiliate-banner
Domain Path: /languages
License: GNU General Public License v3.0
License URI: http://www.gnu.org/licenses/gpl-3.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! function_exists( 'add_action' ) ) {
	exit;
}

define( 'EPNAB_VENDOR', 'Buzin' );
define( 'EPNAB_PRODUCT', EPNAB_VENDOR . '_epnab' );
define( 'EPNAB_VERSION', '1.1.1' );
define( 'EPNAB_DIR_PLUGIN', plugin_dir_path( __FILE__ ) );
define( 'EPNAB_DIR_CLASSES', EPNAB_DIR_PLUGIN . 'classes' . DIRECTORY_SEPARATOR );
define( 'EPNAB_DIR_TEMPLATES', EPNAB_DIR_PLUGIN . 'templates' . DIRECTORY_SEPARATOR );
define( 'EPNAB_DIR_VIEWS', EPNAB_DIR_PLUGIN . 'views' . DIRECTORY_SEPARATOR );
define( 'EPNAB_PATH_CSS', '/assets/css/' );
define( 'EPNAB_PATH_JS', '/assets/js/' );
define( 'EPNAB_API_HOST', 'https://api.ebay.buzin.biz' );
define( 'EPNAB_API_ENDPOINT_FIND', '/find' );

require_once EPNAB_DIR_CLASSES . "Epnab_FileIterator.php";
require_once EPNAB_DIR_CLASSES . "Epnab_Keywords.php";
require_once EPNAB_DIR_CLASSES . "Epnab_Viewer.php";

/**
 * @return void
 */
function epnab_init() {
	load_plugin_textdomain( 'wpdocs_textdomain', false, dirname( plugin_basename( __FILE__ ) ) . '/i18n' );
}

/**
 * @return void
 */
function epnab_scripts() {
	// Global CSS
	wp_register_style( 'epnab-styles', plugin_dir_url( __FILE__ ) . EPNAB_PATH_CSS . 'styles.css', [], EPNAB_VERSION );
	wp_enqueue_style( 'epnab-styles' );
	// Current template CSS
	$template    = get_option( 'epnab_template' );
	$pathinfo    = pathinfo( $template );
	$current_css = basename( $template, '.' . $pathinfo['extension'] ) . '.css';
	wp_register_style( 'epnab-current-style', plugin_dir_url( __FILE__ ) . EPNAB_PATH_CSS . $current_css, [], EPNAB_VERSION );
	wp_enqueue_style( 'epnab-current-style' );
	// Customized CSS
	if ( $button_color = get_option( 'epnab_button_color' ) ) {
		wp_add_inline_style(
			'epnab-current-style',
			"div.epnab p a {background-color: " . esc_html( $button_color ) . "}"
		);
	}
	if ( $button_hover_color = get_option( 'epnab_button_hover_color' ) ) {
		wp_add_inline_style(
			'epnab-current-style',
			"div.epnab p a:hover {background-color: " . esc_html( $button_hover_color ) . "}"
		);
	}
	if ( $button_text_color = get_option( 'epnab_button_text_color' ) ) {
		wp_add_inline_style(
			'epnab-current-style',
			"div.epnab p a {color: " . esc_html( $button_text_color ) . "}"
		);
	}
	if ( $button_text_hover_color = get_option( 'epnab_button_text_hover_color' ) ) {
		wp_add_inline_style(
			'epnab-current-style',
			"div.epnab p a:hover {color: " . esc_html( $button_text_hover_color ) . "}"
		);
	}
	if ( $button_border_radius = get_option( 'epnab_button_border_radius' ) ) {
		wp_add_inline_style(
			'epnab-current-style',
			"div.epnab p a, 
			div.epnab p a:hover {border-radius: " . esc_html( $button_border_radius ) . "px}"
		);
	}
}

/**
 * @param $content
 *
 * @return string
 */
function epnab_content( $content ) {
	if ( is_single() ) {
		wp_enqueue_script(
			'epnab_loader',
			plugins_url( EPNAB_PATH_JS . 'loader.js?v=2.2.1', __FILE__ ),
			array( 'jquery' ), EPNAB_VERSION,
			array(
				'in_footer' => true,
				'strategy'  => 'async'
			)
		);

		wp_localize_script(
			'epnab_loader',
			'epnab_data',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'epnab_nonce' ),
				'post_id'  => get_the_ID()
			)
		);

		$position = get_option( 'epnab_position' );
		if ( $position == 'before' ) {
			$content = '<div id="epnab-container"></div>' . $content;
		}
		if ( $position == 'after' ) {
			$content = $content . '<div id="epnab-container"></div>';
		}
	}

	return $content;
}

/**
 * @return void
 */
function epnab_load() {
	check_ajax_referer( 'epnab_nonce' );
	if ( isset( $_POST['post_id'] ) ) {
		if ( $post_id = sanitize_text_field( wp_unslash( $_POST['post_id'] ) ) ) {
			$view = new Epnab_Viewer();

			// Extract keywords
			$extractor = new Epnab_Keywords();
			$extractor->setStopwords( get_option( 'epnab_stopwords' ), ',' );
			if ( get_option( 'epnab_keywords_title' ) ) {
				$extractor->extract( get_the_title( $post_id ) );
			}
			if ( get_option( 'epnab_keywords_tags' ) ) {
				$tags = get_the_tags( $post_id );
				foreach ( $tags as $tag ) {
					$extractor->extract( $tag->name );
				}
			}
			if ( get_option( 'epnab_keywords_content' ) ) {
				$extractor->extract( wp_strip_all_tags( get_the_content( null, false, $post_id ) ) );
			}
			if ( get_option( 'epnab_keywords_custom' ) ) {
				$extractor->extract( get_option( 'epnab_keywords' ) );
			}

			// Prepare keywords using different strategies
			$keywords          = '';
			$keywords_strategy = get_option( 'epnab_keywords_strategy' );
			$keywords_count    = get_option( 'epnab_keywords_count' );
			if ( $keywords_strategy == 'popular' ) {
				$top      = $extractor->getTop( 1 );
				$keywords = array_pop( $top );
			}
			if ( $keywords_strategy == 'any' ) {
				$top      = $extractor->getTop( $keywords_count );
				$keywords = '(' . implode( ',', $top ) . ')';
			}
			if ( $keywords_strategy == 'random' ) {
				$top      = $extractor->getTop( $keywords_count );
				$keywords = $top[ array_rand( $top ) ];
			}

			// Setup API call parameters
			$license_key      = get_option( 'epnab_license_key' );
			$ebay_campaign_id = get_option( 'epnab_ebay_campaign_id' );
			$min_price        = get_option( 'epnab_min_price' );
			$max_price        = get_option( 'epnab_max_price' );
			$condition        = epnab_get_condition();
			$type             = epnab_get_type();
			$free_shipping    = get_option( 'epnab_free_shipping' );
			$sort_order       = get_option( 'epnab_sort_order' );
			$page_size        = get_option( 'epnab_page_size' );

			// Call API
			$body     = '{
			"license_key": "' . $license_key . '",
            "ebay_campaign_id": "' . $ebay_campaign_id . '"' .
			            ( $keywords ? ',"keywords": "' . $keywords . '"' : '' ) .
			            ( $min_price ? ',"min_price": "' . $min_price . '"' : '' ) .
			            ( $max_price ? ',"max_price": "' . $max_price . '"' : '' ) .
			            ( count( $condition ) ? ',"condition": "' . implode( ',', $condition ) . '"' : '' ) .
			            ( count( $type ) ? ',"type": "' . implode( ',', $type ) . '"' : '' ) .
			            ( $free_shipping ? ',"free_shipping": "' . $free_shipping . '"' : '' ) .
			            ( $sort_order ? ',"sort_order": "' . $sort_order . '"' : '' ) .
			            ( $page_size ? ',"page_size": "' . $page_size . '"' : '' ) .
			            '}';
			$response = wp_remote_post( EPNAB_API_HOST . EPNAB_API_ENDPOINT_FIND, [
				'body'        => $body,
				'timeout'     => '10',
				'redirection' => '5',
				'httpversion' => '1.0',
				'blocking'    => true,
				'headers'     => [],
				'cookies'     => []
			] );
			$items    = json_decode( wp_remote_retrieve_body( $response ) );
			if ( $items ) {
				foreach ( $items as $item ) {
					if ( $titles = $view->getData( $item->title ) ) {
						if ( in_array( $item->title, $titles ) ) {
							// Skip duplicates
							continue;
						}
					}
					$view->setData( 'title', $item->title );
					$view->setData( 'link', $item->link . '&campid=' . $ebay_campaign_id );
					$view->setData( 'image_src', $item->image );
				}
			}

			// Show search result
			if ( $view->hasData() ) {
				try {
					$view->setData( 'carousel', get_option( 'epnab_carousel' ) );
					$template = get_option( 'epnab_template' );
					$view->view( EPNAB_DIR_TEMPLATES . $template );
				} catch ( Exception $exception ) {
				}
			}
		}
	}
	wp_die();
}

function epnab_get_condition() {
	$condition = [];
	if ( get_option( 'epnab_condition_new' ) ) {
		$condition[] = 'new';
	}
	if ( get_option( 'epnab_condition_used' ) ) {
		$condition[] = 'used';
	}

	return $condition;
}

function epnab_get_type() {
	$type = [];
	if ( get_option( 'epnab_type_auction' ) ) {
		$type[] = 'auction';
	}
	if ( get_option( 'epnab_type_classified' ) ) {
		$type[] = 'classified';
	}
	if ( get_option( 'epnab_type_fixed' ) ) {
		$type[] = 'fixed';
	}
	if ( get_option( 'epnab_type_store' ) ) {
		$type[] = 'store';
	}

	return $type;
}

add_action( 'init', 'epnab_init' );
add_action( 'wp_enqueue_scripts', 'epnab_scripts' );
add_action( 'the_content', 'epnab_content' );
add_action( 'wp_ajax_epnab_load', 'epnab_load' );
add_action( 'wp_ajax_nopriv_epnab_load', 'epnab_load' );

if ( is_admin() ) {
	require_once( EPNAB_DIR_CLASSES . 'Epnab_Admin.php' );
	add_action( 'init', array( 'Epnab_Admin', 'init' ) );
}
