<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class Epayments_Gateway_WC_Payment extends WC_Payment_Gateway {
    private $merchant_id;
    private $secret_key;

    public function __construct() {
        $this->id = 'epayments_gateway_wc';
        $this->has_fields = true;
        $this->method_title = 'ePayments Gateway';
        $this->method_description = 'Redirect to ePayments for processing the payment.';

        // Load the settings
        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->merchant_id = $this->get_option('merchant_id');
        $this->secret_key = $this->get_option('secret_key');
        
        // Save settings
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

        // Handle payment processing
        add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
        add_action('woocommerce_api_wc_gateway_epayments', array($this, 'check_response'));

        // Add action to enqueue styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_epayments_styles'));
    }

    public function enqueue_epayments_styles() {
        if (!is_checkout()) {
            return;
        }
        
        wp_enqueue_style(
            'wc-gateway-epayments',
            plugins_url('../assets/css/epayments-styles.css', __FILE__),
            array(),
            '1.0.0'
        );
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => 'Enable/Disable',
                'type' => 'checkbox',
                'label' => 'Enable ePayments Gateway',
                'default' => 'yes'
            ),
            'title' => array(
                'title' => 'Title',
                'type' => 'text',
                'description' => 'The title which the user sees during checkout.',
                'default' => 'ePayments',
                'desc_tip' => true
            ),
            'description' => array(
                'title' => 'Description',
                'type' => 'textarea',
                'description' => 'Description of the payment gateway displayed to the customer during checkout.',
                'default' => 'Pay securely using ePayments.',
            ),
            'merchant_id' => array(
                'title' => 'Merchant ID',
                'type' => 'text',
                'description' => 'Your ePayments Merchant ID',
                'default' => '',
            ),
            'secret_key' => array(
                'title' => 'Secret Key',
                'type' => 'password',
                'description' => 'Your ePayments Secret Key',
                'default' => '',
            )
        );
    }

    public function payment_fields() {
        ?>
        <fieldset>
            <p>
                <label for="epayments_payment_channel">Select Payment Channel:</label><br>
                <select name="epayments_payment_channel" id="epayments_payment_channel" required>
                    <option value="WECHAT">WECHAT</option>
                    <option value="ALIPAY">ALIPAY</option>
                </select>
            </p>
            <?php wp_nonce_field('epayments_payment_process', 'epayments_payment_nonce'); ?>
        </fieldset>
        <?php
    }
    
    public function process_payment($order_id) {
        // Verify nonce
        if (!isset($_POST['epayments_payment_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['epayments_payment_nonce'])), 'epayments_payment_process')) {
            wc_add_notice(__('Payment verification failed. Please try again.', 'epayments-gateway-for-woocommerce'), 'error');
            return array(
                'result' => 'fail',
                'redirect' => ''
            );
        }

        // Verify payment channel is set
        if (!isset($_POST['epayments_payment_channel'])) {
            wc_add_notice(__('Please select a payment channel.', 'epayments-gateway-for-woocommerce'), 'error');
            return array(
                'result' => 'fail',
                'redirect' => ''
            );
        }

        // Get and validate order
        $order = wc_get_order($order_id);
        if (!$order) {
            wc_add_notice(__('Invalid order.', 'epayments-gateway-for-woocommerce'), 'error');
            return array(
                'result' => 'fail',
                'redirect' => ''
            );
        }

        // Verify user has permission to pay for this order
        if (!current_user_can('pay_for_order', $order_id) && !current_user_can('manage_woocommerce')) {
            wc_add_notice(__('You do not have permission to pay for this order.', 'epayments-gateway-for-woocommerce'), 'error');
            return array(
                'result' => 'fail',
                'redirect' => ''
            );
        }

        $payment_channel = sanitize_text_field(wp_unslash($_POST['epayments_payment_channel']));
        
        // Validate payment channel
        if (!in_array($payment_channel, array('WECHAT', 'ALIPAY'), true)) {
            wc_add_notice(__('Invalid payment channel selected.', 'epayments-gateway-for-woocommerce'), 'error');
            return array(
                'result' => 'fail',
                'redirect' => ''
            );
        }

        update_post_meta($order_id, '_epayments_payment_channel', $payment_channel);
        $order->update_status('pending', __('Awaiting ePayments payment', 'epayments-gateway-for-woocommerce'));

        // Redirect URL after successful order placement
        $redirect_url = $this->generate_payment_url($order);

        return array(
            'result' => 'success',
            'redirect' => $redirect_url
        );
    }

    public function generate_payment_url($order) {
        $notify_url = site_url('/?wc-api=wc_gateway_epayments');
        $payment_channel = get_post_meta($order->get_id(), '_epayments_payment_channel', true);

        $params = [
            'merchant_id' => $this->merchant_id,
            'increment_id' => $order->get_id(),
            'grandtotal' => $order->get_total(),
            'currency' => get_woocommerce_currency(),
            'return_url' => $this->get_return_url($order),
            'notify_url' => $notify_url,
            'payment_channels' => $payment_channel,
            'subject' => 'Order Payment',
            'describe' => 'Order #' . $order->get_id(),
            'nonce_str' => bin2hex(random_bytes(8)),
            'valid_mins' => 10,
            'service' => 'create_smart_pay'
        ];

        $signature = $this->generate_signature($params, $this->secret_key);
        $params['signature'] = $signature;
        $params['sign_type'] = 'MD5';

        $query_string = http_build_query($params);
        $payment_url = "https://api.wetopay.com/api/v1/info/smartpay?$query_string";

        return $payment_url;
    }

    public function generate_signature($params, $secretKey) {
        ksort($params);
        $signString = urldecode(http_build_query($params)) . $secretKey;
        return md5($signString);
    }

    public function check_response() {
        // Verify that the request is coming from WeToPay
        if (!$this->verify_wetopay_request()) {
            wp_die(esc_html__('Invalid payment notification received.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 403));
        }

        // Read and validate raw POST data
        $raw_post_data = $this->get_raw_post_data();
        if (empty($raw_post_data)) {
            wp_die(esc_html__('No data received.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 400));
        }
    
        // Parse and validate the data
        $post_data = $this->parse_post_data($raw_post_data);
        if (empty($post_data)) {
            wp_die(esc_html__('Missing required payment data.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 400));
        }

        // Verify signature before processing data
        if (!$this->verify_signature($post_data)) {
            wp_die(esc_html__('Invalid payment signature.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 403));
        }
    
        // Sanitize and validate individual fields
        $merchant_id = sanitize_text_field($post_data['merchant_id']);
        $order_id = absint($post_data['increment_id']);
        $trade_status = sanitize_text_field($post_data['trade_status']);

        // Additional validation
        if (!$this->is_valid_trade_status($trade_status)) {
            wp_die(esc_html__('Invalid trade status.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 400));
        }

        // Verify the merchant ID matches
        if (!hash_equals($this->merchant_id, $merchant_id)) {
            wp_die(esc_html__('Invalid merchant ID.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 403));
        }

        // Fetch and validate the order
        $order = wc_get_order($order_id);
        if (!$order instanceof WC_Order) {
            wp_die(esc_html__('Order not found.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 404));
        }

        // Handle successful payment
        if ($trade_status === 'TRADE_SUCCESS') {
            $order->payment_complete();
            $order->add_order_note(esc_html__('Payment completed successfully via ePayments (Async Notification)', 'epayments-gateway-for-woocommerce'));
            
            // Return 'success' to WeToPay to confirm receipt
            echo 'success';
            exit;
        } else {
            wp_die(esc_html__('Payment processing failed.', 'epayments-gateway-for-woocommerce'), 'ePayments', array('response' => 500));
        }
    }

    /**
     * Get and validate raw POST data
     *
     * @return string Raw POST data or empty string if invalid
     */
    private function get_raw_post_data() {
        $raw_data = file_get_contents('php://input');
        
        // Basic validation of raw data
        if (!is_string($raw_data)) {
            return '';
        }
        
        // Check for maximum allowed size (e.g., 1MB)
        if (strlen($raw_data) > 1048576) {
            return '';
        }
        
        return $raw_data;
    }

    /**
     * Parse POST data from raw input
     *
     * @param string $raw_data Raw POST data
     * @return array Parsed data array or empty array if invalid
     */
    private function parse_post_data($raw_data) {
        $required_fields = array(
            'merchant_id',
            'increment_id',
            'trade_status',
            'signature',
            'sign_type'
        );
        
        $parsed_data = array();
        
        // Try JSON first
        $json_data = json_decode($raw_data, true);
        if (is_array($json_data)) {
            // Only extract required fields from JSON data
            foreach ($required_fields as $field) {
                if (isset($json_data[$field])) {
                    $parsed_data[$field] = sanitize_text_field($json_data[$field]);
                }
            }
        }
        // If JSON parsing failed or missing required fields, try POST data
        elseif (!empty($_POST)) {
            // Only extract required fields from POST data
            foreach ($required_fields as $field) {
                if (isset($_POST[$field])) {
                    $parsed_data[$field] = sanitize_text_field(wp_unslash($_POST[$field]));
                }
            }
        }
        
        // Verify we have all required fields
        foreach ($required_fields as $field) {
            if (!isset($parsed_data[$field])) {
                return array();
            }
        }
        
        return $parsed_data;
    }

    /**
     * Validate trade status
     *
     * @param string $status Trade status to validate
     * @return bool Whether the status is valid
     */
    private function is_valid_trade_status($status) {
        $valid_statuses = array(
            'TRADE_SUCCESS',
            'TRADE_FAILED',
            'TRADE_PENDING'
        );
        return in_array($status, $valid_statuses, true);
    }

    /**
     * Verify that the request is coming from WeToPay
     * This method should implement signature verification or other security measures
     * provided by the WeToPay API
     */
    private function verify_wetopay_request() {
        // Get the request headers
        $headers = getallheaders();
        
        // Get the raw POST data
        $raw_post_data = file_get_contents('php://input');
        
        // Implement WeToPay's signature verification here
        // This is a placeholder - you should implement the actual verification
        // logic according to WeToPay's API documentation
        
        return true; // Return true only after proper verification
    }

    /**
     * Verify the signature of the payment notification
     *
     * @param array $data The payment notification data
     * @return bool Whether the signature is valid
     */
    private function verify_signature($data) {
        if (!isset($data['signature']) || !isset($data['sign_type']) || $data['sign_type'] !== 'MD5') {
            return false;
        }

        $received_signature = $data['signature'];
        
        // Remove signature from data before generating verification signature
        unset($data['signature']);
        unset($data['sign_type']);
        
        // Generate signature for verification
        $verification_signature = $this->generate_signature($data, $this->secret_key);
        
        // Compare signatures using hash_equals to prevent timing attacks
        return hash_equals($verification_signature, $received_signature);
    }
}