<?php
/**
 * WP List Table Admin Class.
 *
 * @since 1.5.0
 *
 * @package Envira_Gallery
 * @author  Envira Team
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Table Class
 *
 * @since 1.5.0
 */
class Envira_Gallery_Table_Admin {

	/**
	 * Holds the class object.
	 *
	 * @since 1.5.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Path to the file.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	public $file = __FILE__;

	/**
	 * Holds the base class object.
	 *
	 * @since 1.5.0
	 *
	 * @var object
	 */
	public $base;

	/**
	 * Holds the metabox class object.
	 *
	 * @since 1.5.0
	 *
	 * @var object
	 */
	public $metabox;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the base class object.
		$this->base = Envira_Gallery_Lite::get_instance();

		// Load the metabox class object.
		$this->metabox = Envira_Gallery_Metaboxes::get_instance();

		// Load CSS and JS.
		add_action( 'admin_enqueue_scripts', [ $this, 'styles' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'scripts' ] );

		// Append data to various admin columns.
		add_filter( 'manage_edit-envira_columns', [ &$this, 'envira_columns' ] );
		add_action( 'manage_envira_posts_custom_column', [ &$this, 'envira_custom_columns' ], 10, 2 );

		// Add custom empty state for galleries list.
		add_action( 'admin_footer', [ $this, 'display_empty_state' ] );
	}

	/**
	 * Loads styles for all Envira-based WP_List_Table Screens.
	 *
	 * @since 1.5.0
	 *
	 * @return void Return early if not on the proper screen.
	 */
	public function styles() {

		// Get current screen.
		$screen = get_current_screen();

		// Bail if we're not on the Envira Post Type screen.
		if ( 'envira' !== $screen->post_type ) {
			return;
		}

		// Bail if we're not on a WP_List_Table.
		if ( 'edit' !== $screen->base ) {
			return;
		}

		// Load necessary admin styles.
		wp_register_style( $this->base->plugin_slug . '-table-style', plugins_url( 'assets/css/table.css', $this->base->file ), [], $this->base->version );
		wp_enqueue_style( $this->base->plugin_slug . '-table-style' );

		// Fire a hook to load in custom admin styles.
		do_action( 'envira_gallery_table_styles' );
	}

	/**
	 * Loads scripts for all Envira-based Administration Screens.
	 *
	 * @since 1.5.0
	 *
	 * @return void Return early if not on the proper screen.
	 */
	public function scripts() {

		// Get current screen.
		$screen = get_current_screen();

		// Bail if we're not on the Envira Post Type screen.
		if ( 'envira' !== $screen->post_type ) {
			return;
		}

		// Bail if we're not on a WP_List_Table.
		if ( 'edit' !== $screen->base ) {
			return;
		}

		// Gallery / Album Selection.
		// Just register and localize the script; if a third party Addon wants to use this, they can use both functions.
		wp_register_script( $this->base->plugin_slug . '-gallery-select-script', plugins_url( 'assets/js/gallery-select.js', $this->base->file ), [ 'jquery' ], $this->base->version, true );
		wp_localize_script(
			$this->base->plugin_slug . '-gallery-select-script',
			'envira_gallery_select',
			[
				'get_galleries_nonce' => wp_create_nonce( 'envira-gallery-editor-get-galleries' ),
				'modal_title'         => __( 'Insert', 'envira-gallery-lite' ),
				'insert_button_label' => __( 'Insert', 'envira-gallery-lite' ),
			]
		);

		wp_register_script( $this->base->plugin_slug . '-table-script', plugins_url( 'assets/js/min/table-min.js', $this->base->file ), [ 'jquery' ], $this->base->version, true );
		wp_enqueue_script( $this->base->plugin_slug . '-table-script' );
		// Fire a hook to load in custom admin scripts.
		do_action( 'envira_gallery_admin_scripts' );
	}

	/**
	 * Customize the post columns for the Envira post type.
	 *
	 * @since 1.0.0
	 *
	 * @param array $columns  The default columns.
	 * @return array $columns Amended columns.
	 */
	public function envira_columns( $columns ) {

		// Add additional columns we want to display.
		$envira_columns = [
			'cb'        => '<input type="checkbox" />',
			'image'     => '',
			'title'     => __( 'Title', 'envira-gallery-lite' ),
			'shortcode' => __( 'Shortcode', 'envira-gallery-lite' ),
			'posts'     => __( 'Posts', 'envira-gallery-lite' ),
			'modified'  => __( 'Last Modified', 'envira-gallery-lite' ),
			'date'      => __( 'Date', 'envira-gallery-lite' ),
		];

		// Allow filtering of columns.
		$envira_columns = apply_filters( 'envira_gallery_table_columns', $envira_columns, $columns );

		// Return merged column set.  This allows plugins to output their columns (e.g. Yoast SEO),
		// and column management plugins, such as Admin Columns, should play nicely.
		return array_merge( $envira_columns, $columns );
	}

	/**
	 * Add data to the custom columns added to the Envira post type.
	 *
	 * @since 1.0.0
	 *
	 * @global object $post  The current post object.
	 * @param string $column The name of the custom column.
	 * @param int    $post_id   The current post ID.
	 */
	public function envira_custom_columns( $column, $post_id ) {

		global $post;
		$post_id = absint( $post_id );

		switch ( $column ) {
			/**
			* Image.
			*/
			case 'image':
				// Get Gallery Images.
				$gallery_data = get_post_meta( $post_id, '_eg_gallery_data', true );
				if ( ! empty( $gallery_data['gallery'] ) && is_array( $gallery_data['gallery'] ) ) {
					// Display the first image.
					$images = $gallery_data['gallery'];
					$image  = reset( $images );
					$key    = key( $images );
					if ( is_numeric( $key ) ) {
						$thumb = wp_get_attachment_image_src( $key );
						if ( false === $thumb ) {
							// Image was deleted while this plugin was disabled.
							$thumb = [ '' ];
						}
					} else {
						$thumb = [ $image['src'] ];
					}

					echo '<img src="' . esc_url( $thumb[0] ) . '" width="75" /><br />';
					/* translators: %s: image count */
					printf( esc_html( _n( '%d Image', '%d Images', count( $gallery_data['gallery'] ), 'envira-gallery-lite' ) ), count( $gallery_data['gallery'] ) );
				}
				break;

			/**
			* Shortcode
			*/
			case 'shortcode':
				echo '
                <div class="envira-code">
                    <code id="envira_shortcode_' . intval( $post_id ) . '">[envira-gallery id="' . intval( $post_id ) . '"]</code>
                    <a href="#" title="' . esc_html__( 'Copy Shortcode to Clipboard', 'envira-gallery-lite' ) . '" data-clipboard-target="#envira_shortcode_' . intval( $post_id ) . '" class="dashicons dashicons-clipboard envira-clipboard">
                        <span>' . esc_html__( 'Copy to Clipboard', 'envira-gallery-lite' ) . '</span>
                    </a>
                </div>';

				// Hidden fields are for Quick Edit
				// class is used by assets/js/admin.js to remove these fields when a search is about to be submitted, so we dont' get long URLs.
				echo '<input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[columns]" value="' . esc_attr( $this->metabox->get_config( 'columns' ) ) . '" />
                <input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[gallery_theme]" value="' . esc_attr( $this->metabox->get_config( 'gallery_theme' ) ) . '" />
                <input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[gutter]" value="' . esc_attr( $this->metabox->get_config( 'gutter' ) ) . '" />
                <input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[margin]" value="' . esc_attr( $this->metabox->get_config( 'margin' ) ) . '" />
                <input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[crop_width]" value="' . esc_attr( $this->metabox->get_config( 'crop_width' ) ) . '" />
                <input class="envira-quick-edit" type="hidden" name="_envira_gallery_' . intval( $post_id ) . '[crop_height]" value="' . esc_attr( $this->metabox->get_config( 'crop_height' ) ) . '" />';
				break;

			/**
			* Posts
			*/
			case 'posts':
				$posts = get_post_meta( $post_id, '_eg_in_posts', true );
				if ( is_array( $posts ) ) {
					foreach ( $posts as $in_post_id ) {
						echo '<a href="' . esc_url( get_permalink( $in_post_id ) ) . '" target="_blank">' . esc_html( get_the_title( $in_post_id ) ) . '</a><br />';
					}
				}
				break;

			/**
			* Last Modified
			*/
			case 'modified':
				the_modified_date();
				break;
		}
	}

	/**
	 * Adds Envira fields to the quick editing and bulk editing screens
	 *
	 * @since 1.3.1
	 *
	 * @param string $column_name Column Name.
	 * @param string $post_type Post Type.
	 * @return void
	 */
	public function quick_edit_custom_box( $column_name, $post_type ) {

		// Check post type is Envira.
		if ( 'envira' !== $post_type ) {
			return;
		}

		// Get metabox instance.
		$this->metabox = Envira_Gallery_Metaboxes::get_instance();

		// Depending on the column we're on, output some additional fields.
		switch ( $column_name ) {
			case 'shortcode':
				?>
				<fieldset class="inline-edit-col-left inline-edit-envira-gallery">
					<div class="inline-edit-col inline-edit-<?php echo esc_attr( $column_name ); ?>">
						<label class="inline-edit-group">
							<span class="title"><?php esc_html_e( 'Number of Columns', 'envira-gallery-lite' ); ?></span>
							<select name="_envira_gallery[columns]">
								<?php foreach ( (array) $this->metabox->get_columns() as $i => $data ) : ?>
									<option value="<?php echo esc_attr( $data['value'] ); ?>"><?php echo esc_html( $data['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
						</label>

						<label class="inline-edit-group">
							<span class="title"><?php esc_html_e( 'Gallery Theme', 'envira-gallery-lite' ); ?></span>
							<select name="_envira_gallery[gallery_theme]">
								<?php foreach ( (array) $this->metabox->get_gallery_themes() as $i => $data ) : ?>
									<option value="<?php echo esc_attr( $data['value'] ); ?>"><?php echo esc_html( $data['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
						</label>
					</div>
				</fieldset>
				<?php
				break;
		}

		wp_nonce_field( 'envira-gallery', 'envira-gallery' );
	}

	/**
	 * Adds Envira fields to the  bulk editing screens
	 *
	 * @since 1.3.1
	 *
	 * @param string $column_name Column Name.
	 * @param string $post_type Post Type.
	 * @return void
	 */
	public function bulk_edit_custom_box( $column_name, $post_type ) {

		// Check post type is Envira.
		if ( 'envira' !== $post_type ) {
			return;
		}

		// Only apply to shortcode column.
		if ( 'shortcode' !== $column_name ) {
			return;
		}

		// Get metabox instance.
		$this->metabox = Envira_Gallery_Metaboxes::get_instance();

		switch ( $column_name ) {
			case 'shortcode':
				?>
				<fieldset class="inline-edit-col-left inline-edit-envira-gallery">
					<div class="inline-edit-col inline-edit-<?php echo esc_attr( $column_name ); ?>">
						<label class="inline-edit-group">
							<span class="title"><?php esc_html_e( 'Number of Columns', 'envira-gallery-lite' ); ?></span>
							<select name="_envira_gallery[columns]">
								<option value="-1" selected><?php esc_html_e( '— No Change —', 'envira-gallery-lite' ); ?></option>

								<?php foreach ( (array) $this->metabox->get_columns() as $i => $data ) : ?>
									<option value="<?php echo esc_attr( $data['value'] ); ?>"><?php echo esc_html( $data['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
						</label>

						<label class="inline-edit-group">
							<span class="title"><?php esc_html_e( 'Gallery Theme', 'envira-gallery-lite' ); ?></span>
							<select name="_envira_gallery[gallery_theme]">
								<option value="-1" selected><?php esc_html_e( '— No Change —', 'envira-gallery-lite' ); ?></option>

								<?php foreach ( (array) $this->metabox->get_gallery_themes() as $i => $data ) : ?>
									<option value="<?php echo esc_attr( $data['value'] ); ?>"><?php echo esc_html( $data['name'] ); ?></option>
								<?php endforeach; ?>
							</select>
						</label>

					</div>
				</fieldset>
				<?php
				break;
		}

		wp_nonce_field( 'envira-gallery', 'envira-gallery' );
	}

	/**
	 * Called every time a WordPress Post is updated
	 *
	 * Checks to see if the request came from submitting the Bulk Editor form,
	 * and if so applies the updates.  This is because there is no direct action
	 * or filter fired for bulk saving
	 *
	 * @since 1.3.1
	 *
	 * @param int $post_ID Post ID.
	 */
	public function bulk_edit_save( $post_ID ) {

		// Check we are performing a Bulk Edit.
		if ( ! isset( $_REQUEST['bulk_edit'] ) ) {
			return;
		}

		// Bail out if we fail a security check.
		if ( ! isset( $_REQUEST['envira-gallery'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_REQUEST['envira-gallery'] ) ), 'envira-gallery' ) ) {
			return;
		}

		// Check Post IDs have been submitted.
		$post_ids = ( ! empty( $_REQUEST['post'] ) ) ? wp_unslash( $_REQUEST['post'] ) : array(); // @codingStandardsIgnoreLine
		if ( empty( $post_ids ) || ! is_array( $post_ids ) ) {
			return;
		}

		// Get metabox instance.
		$this->metabox = Envira_Gallery_Metaboxes::get_instance();

		// Iterate through post IDs, updating settings.
		foreach ( $post_ids as $post_id ) {
			// Get settings.
			$settings = get_post_meta( $post_id, '_eg_gallery_data', true );
			if ( empty( $settings ) ) {
				continue;
			}

			// Update Settings, if they have values.
			if ( ! empty( $_REQUEST['_envira_gallery']['columns'] ) && -1 !== sanitize_text_field( wp_unslash( $_REQUEST['_envira_gallery']['columns'] ) ) ) {
				$settings['config']['columns'] = preg_replace( '#[^a-z0-9-_]#', '', sanitize_text_field( wp_unslash( $_REQUEST['_envira_gallery']['columns'] ) ) );
			}
			if ( ! empty( $_REQUEST['_envira_gallery']['gallery_theme'] ) && -1 !== sanitize_text_field( wp_unslash( $_REQUEST['_envira_gallery']['gallery_theme'] ) ) ) {
				$settings['config']['gallery_theme'] = preg_replace( '#[^a-z0-9-_]#', '', sanitize_text_field( wp_unslash( $_REQUEST['_envira_gallery']['gallery_theme'] ) ) );
			}
			if ( ! empty( $_REQUEST['_envira_gallery']['gutter'] ) ) {
				$settings['config']['gutter'] = absint( $_REQUEST['_envira_gallery']['gutter'] );
			}
			if ( ! empty( $_REQUEST['_envira_gallery']['margin'] ) ) {
				$settings['config']['margin'] = absint( $_REQUEST['_envira_gallery']['margin'] );
			}
			if ( ! empty( $_REQUEST['_envira_gallery']['crop_width'] ) ) {
				$settings['config']['crop_width'] = absint( $_REQUEST['_envira_gallery']['crop_width'] );
			}
			if ( ! empty( $_REQUEST['_envira_gallery']['crop_height'] ) ) {
				$settings['config']['crop_height'] = absint( $_REQUEST['_envira_gallery']['crop_height'] );
			}

			// Provide a filter to override settings.
			$settings = apply_filters( 'envira_gallery_bulk_edit_save_settings', $settings, $post_id );

			// Update the post meta.
			update_post_meta( $post_id, '_eg_gallery_data', $settings );

			// Finally, flush all gallery caches to ensure everything is up to date.
			$this->metabox->flush_gallery_caches( $post_id, $settings['config']['slug'] );

		}
	}

	/**
	 * Display custom empty state when no galleries are found.
	 *
	 * @since 1.8.15
	 *
	 * @return void
	 */
	public function display_empty_state() {
		// Get current screen.
		$screen = get_current_screen();

		// Bail if we're not on the Envira Post Type screen.
		if ( 'envira' !== $screen->post_type ) {
			return;
		}

		// Bail if we're not on a WP_List_Table.
		if ( 'edit' !== $screen->base ) {
			return;
		}

		// Check if there are any galleries.
		$galleries = get_posts(
			[
				'post_type'        => 'envira',
				'post_status'      => 'any',
				'posts_per_page'   => 1,
				'fields'           => 'ids',
				'suppress_filters' => true,
			]
		);

		// If galleries exist, don't show empty state.
		if ( ! empty( $galleries ) ) {
			return;
		}

		// Check if we're on the main galleries list page (not filtered or searched).
		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Nonce is not required for display-only checks.
		$search      = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
		$post_status = isset( $_GET['post_status'] ) ? sanitize_text_field( wp_unslash( $_GET['post_status'] ) ) : '';
		$month_value = isset( $_GET['m'] ) ? sanitize_text_field( wp_unslash( $_GET['m'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		if ( '' !== $search || '' !== $post_status || '' !== $month_value ) {
			return;
		}

		// Output empty state in a hidden container, then move it to wpbody-content via JavaScript.
		// The JavaScript is handled in assets/js/admin.js.
		ob_start();
		include ENVIRA_LITE_DIR . 'includes/admin/partials/empty-state.php';
		$empty_state_content = ob_get_clean();

		printf(
			'<div id="envira-empty-state-container" style="display: none;">%s</div>',
			$empty_state_content // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Template partial handles escaping.
		);
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.5.0
	 *
	 * @return object The Envira_Gallery_Table_Admin object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Envira_Gallery_Table_Admin ) ) {
			self::$instance = new Envira_Gallery_Table_Admin();
		}

		return self::$instance;
	}
}

// Load the table admin class.
$envira_gallery_table_admin = Envira_Gallery_Table_Admin::get_instance();
