<?php
/*
Plugin Name: Enrollment Email Notifications for TutorLMS
Plugin URI: https://wordpress.org/plugins/enrollment-email-notifications/
Description: Sends email notifications to students and admin on course enrollment for TutorLMS.
Version: 1.0.0
Author: Code X Designs
Author URI: https://www.codexdesigns.net/
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: enrollment-email-notifications-for-tutorlms
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/*
|--------------------------------------------------------------------------
| Settings Page
|--------------------------------------------------------------------------
*/
add_action( 'admin_menu', function () {
    add_options_page(
        esc_html__( 'Enrollment Email Settings', 'enrollment-email-notifications-for-tutorlms' ),
        esc_html__( 'Enrollment Emails', 'enrollment-email-notifications-for-tutorlms' ),
        'manage_options',
        'enrollment-email-settings',
        'een_settings_page'
    );
} );

function een_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Enrollment Email Settings', 'enrollment-email-notifications-for-tutorlms' ); ?></h1>

        <form method="post" action="options.php">
            <?php
            settings_fields( 'een_settings_group' );
            do_settings_sections( 'een_settings_group' );
            ?>

            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e( 'From Name', 'enrollment-email-notifications-for-tutorlms' ); ?></th>
                    <td>
                        <input type="text" name="een_from_name"
                            value="<?php echo esc_attr( get_option( 'een_from_name', get_bloginfo( 'name' ) ) ); ?>"
                            class="regular-text">
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'From Email', 'enrollment-email-notifications-for-tutorlms' ); ?></th>
                    <td>
                        <input type="email" name="een_from_email"
                            value="<?php echo esc_attr( get_option( 'een_from_email', get_option( 'admin_email' ) ) ); ?>"
                            class="regular-text">
                        <p class="description">
                            <?php esc_html_e(
                                'If WP Mail SMTP is configured, the SMTP sender email will override this value.',
                                'enrollment-email-notifications-for-tutorlms'
                            ); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

add_action( 'admin_init', function () {
    register_setting( 'een_settings_group', 'een_from_name', array(
        'sanitize_callback' => 'sanitize_text_field',
    ) );

    register_setting( 'een_settings_group', 'een_from_email', array(
        'sanitize_callback' => 'sanitize_email',
    ) );
} );

/*
|--------------------------------------------------------------------------
| Override From Email + Name
|--------------------------------------------------------------------------
*/
add_filter( 'wp_mail_from_name', function () {
    return get_option( 'een_from_name', get_bloginfo( 'name' ) );
} );

add_filter( 'wp_mail_from', function () {
    return get_option( 'een_from_email', get_option( 'admin_email' ) );
} );

/*
|--------------------------------------------------------------------------
| TutorLMS Enrollment Email Trigger
|--------------------------------------------------------------------------
*/
add_action( 'tutor_after_enrolled', function ( $course_id, $user_id ) {

    $user_info    = get_userdata( $user_id );
    $course_title = get_the_title( $course_id );
    $course_link  = get_permalink( $course_id );
    $login_url    = wp_login_url();

    $student_subject = sprintf(
        esc_html__( 'You are enrolled in "%s"', 'enrollment-email-notifications-for-tutorlms' ),
        $course_title
    );

    $student_body  = sprintf(
        esc_html__( "Hi %s,\n\n", 'enrollment-email-notifications-for-tutorlms' ),
        $user_info->display_name
    );

    $student_body .= sprintf(
        esc_html__( "You have successfully enrolled in \"%s\".\n\n", 'enrollment-email-notifications-for-tutorlms' ),
        $course_title
    );

    $student_body .= sprintf(
        esc_html__( "Login URL: %s\n", 'enrollment-email-notifications-for-tutorlms' ),
        $login_url
    );

    $student_body .= sprintf(
        esc_html__( "Username: %s\n\n", 'enrollment-email-notifications-for-tutorlms' ),
        $user_info->user_email
    );

    $student_body .= esc_html__( "Thank you.\n", 'enrollment-email-notifications-for-tutorlms' );

    wp_mail( $user_info->user_email, $student_subject, $student_body );

    $admin_subject = sprintf(
        esc_html__( 'New Enrollment: %s', 'enrollment-email-notifications-for-tutorlms' ),
        $course_title
    );

    $admin_body  = esc_html__( "A new student has enrolled.\n\n", 'enrollment-email-notifications-for-tutorlms' );
    $admin_body .= sprintf( esc_html__( "Name: %s\n", 'enrollment-email-notifications-for-tutorlms' ), $user_info->display_name );
    $admin_body .= sprintf( esc_html__( "Email: %s\n", 'enrollment-email-notifications-for-tutorlms' ), $user_info->user_email );
    $admin_body .= sprintf( esc_html__( "Course: %s\n", 'enrollment-email-notifications-for-tutorlms' ), $course_title );
    $admin_body .= sprintf( esc_html__( "Course Link: %s\n", 'enrollment-email-notifications-for-tutorlms' ), $course_link );

    wp_mail( get_option( 'admin_email' ), $admin_subject, $admin_body );

}, 10, 2 );

/*
|--------------------------------------------------------------------------
| TutorLMS Notices
|--------------------------------------------------------------------------
*/
add_action( 'admin_notices', function () {

    include_once ABSPATH . 'wp-admin/includes/plugin.php';

    if ( ! file_exists( WP_PLUGIN_DIR . '/tutor/tutor.php' ) ) {
        echo '<div class="notice notice-error"><p><strong>' .
            esc_html__( 'TutorLMS is not installed.', 'enrollment-email-notifications-for-tutorlms' ) .
            '</strong></p></div>';
        return;
    }

    if ( ! is_plugin_active( 'tutor/tutor.php' ) ) {
        echo '<div class="notice notice-error"><p><strong>' .
            esc_html__( 'TutorLMS is installed but not active.', 'enrollment-email-notifications-for-tutorlms' ) .
            '</strong></p></div>';
    }
} );
