<?php
/**
 * Plugin Name: Engineo
 * Description: All-engine optimization plugin (SEO + GEO + AEO) - Simplified version with core features
 * Version: 1.0.0
 * Author: Naveen Rajan
 * Author URI: https://github.com/NaviRocker
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: engineo
 * Domain Path: /languages
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Requires PHP: 8.0
 *
 * @package Engineo
 * 
 * Production Ready: This plugin is optimized for production use with:
 * - Comprehensive security measures (nonce verification, input sanitization, output escaping)
 * - Performance optimizations (caching, efficient database queries)
 * - WordPress coding standards compliance
 * - Clean, maintainable code structure
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('ENGINEO_VERSION', '1.0.0');
define('ENGINEO_PLUGIN_FILE', __FILE__);
define('ENGINEO_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ENGINEO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ENGINEO_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main Engineo class - Simplified Version
 */
class Engineo {

    /**
     * Plugin instance
     */
    private static $instance = null;

    /**
     * Get plugin instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('init', [$this, 'init']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('add_meta_boxes', [$this, 'add_meta_boxes']);
        add_action('save_post', [$this, 'save_meta_data']);
        add_action('wp_head', [$this, 'output_meta_tags'], 1);
        add_action('wp_head', [$this, 'output_schema_markup'], 2);
        add_action('template_redirect', [$this, 'handle_requests'], 1);
        add_filter('query_vars', [$this, 'add_query_vars']);
        add_filter('document_title_parts', [$this, 'filter_document_title'], 10, 1);
        add_action('template_redirect', [$this, 'handle_redirects'], 1);
        add_action('init', [$this, 'prevent_elementor_conflicts'], 1);
        add_action('parse_request', [$this, 'early_request_handler'], 1);
        add_action('init', [$this, 'handle_direct_requests'], 1);
        add_action('wp_loaded', [$this, 'handle_direct_requests'], 1);
        add_action('template_redirect', [$this, 'handle_direct_requests'], 0);
        
        // Ensure WordPress sitemaps are enabled
        add_action('init', [$this, 'enable_wordpress_sitemaps']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check minimum requirements
        if (!$this->check_requirements()) {
            return;
        }

        // Add rewrite rules
        $this->add_rewrite_rules();
    }

    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        // Enqueue admin scripts
        if (is_admin()) {
            wp_enqueue_script(
                'engineo-admin-script',
                ENGINEO_PLUGIN_URL . 'js/admin.js',
                ['jquery'],
                ENGINEO_VERSION,
                true
            );

            // Localize script with AJAX URL and nonce
            wp_localize_script('engineo-admin-script', 'engineo_ajax', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('engineo_ajax_nonce'),
            ]);
        }
    }

    /**
     * Check minimum requirements
     */
    private function check_requirements() {
        // Check WordPress version
        if (version_compare(get_bloginfo('version'), '6.0', '<')) {
            add_action('admin_notices', [$this, 'wp_version_notice']);
            return false;
        }

        // Check PHP version
        if (version_compare(PHP_VERSION, '8.0', '<')) {
            add_action('admin_notices', [$this, 'php_version_notice']);
            return false;
        }

        return true;
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Main menu
        add_menu_page(
            __('Engineo', 'engineo'),
            __('Engineo', 'engineo'),
            'manage_options',
            'engineo',
            [$this, 'admin_dashboard'],
            'dashicons-performance',
            30
        );

        // Submenu pages
        add_submenu_page(
            'engineo',
            __('Dashboard', 'engineo'),
            __('Dashboard', 'engineo'),
            'manage_options',
            'engineo',
            [$this, 'admin_dashboard']
        );

        add_submenu_page(
            'engineo',
            __('SEO Settings', 'engineo'),
            __('SEO Settings', 'engineo'),
            'manage_options',
            'engineo-seo',
            [$this, 'seo_settings_page']
        );

        add_submenu_page(
            'engineo',
            __('XML Sitemap', 'engineo'),
            __('XML Sitemap', 'engineo'),
            'manage_options',
            'engineo-sitemap',
            [$this, 'sitemap_page']
        );

        add_submenu_page(
            'engineo',
            __('Robots.txt', 'engineo'),
            __('Robots.txt', 'engineo'),
            'manage_options',
            'engineo-robots',
            [$this, 'robots_page']
        );

        add_submenu_page(
            'engineo',
            __('llms.txt', 'engineo'),
            __('llms.txt', 'engineo'),
            'manage_options',
            'engineo-llms',
            [$this, 'llms_page']
        );

        add_submenu_page(
            'engineo',
            __('Redirects', 'engineo'),
            __('Redirects', 'engineo'),
            'manage_options',
            'engineo-redirects',
            [$this, 'redirects_page']
        );

        add_submenu_page(
            'engineo',
            __('Health Check', 'engineo'),
            __('Health Check', 'engineo'),
            'manage_options',
            'engineo-health',
            [$this, 'health_check_page']
        );
    }

    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        $post_types = get_post_types(['public' => true], 'names');
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'engineo-seo',
                __('Engineo SEO', 'engineo'),
                [$this, 'meta_box_callback'],
                $post_type,
                'normal',
                'high'
            );
        }
    }

    /**
     * Meta box callback
     */
    public function meta_box_callback($post) {
        wp_nonce_field('engineo_meta_box', 'engineo_meta_box_nonce');
        
        $meta_data = $this->get_meta_data($post->ID);
        
        ?>
        <div class="engineo-meta-editor">
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="engineo_meta_title"><?php esc_html_e('Meta Title', 'engineo'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="engineo_meta_title" name="engineo_meta_title" 
                               value="<?php echo esc_attr($meta_data['meta_title'] ?? ''); ?>" 
                               maxlength="60" class="regular-text" />
                        <p class="description"><?php esc_html_e('Recommended: 50-60 characters', 'engineo'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="engineo_meta_description"><?php esc_html_e('Meta Description', 'engineo'); ?></label>
                    </th>
                    <td>
                        <textarea id="engineo_meta_description" name="engineo_meta_description" 
                                  maxlength="160" class="large-text" rows="3"><?php echo esc_textarea($meta_data['meta_description'] ?? ''); ?></textarea>
                        <p class="description"><?php esc_html_e('Recommended: 150-160 characters', 'engineo'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="engineo_canonical_url"><?php esc_html_e('Canonical URL', 'engineo'); ?></label>
                    </th>
                    <td>
                        <input type="url" id="engineo_canonical_url" name="engineo_canonical_url" 
                               value="<?php echo esc_attr($meta_data['canonical_url'] ?? ''); ?>" 
                               class="regular-text" placeholder="<?php echo esc_attr(get_permalink($post->ID)); ?>" />
                        <p class="description"><?php esc_html_e('Leave empty to use default URL', 'engineo'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="engineo_schema_type"><?php esc_html_e('Schema Type', 'engineo'); ?></label>
                    </th>
                    <td>
                        <select id="engineo_schema_type" name="engineo_schema_type" class="regular-text">
                            <option value=""><?php esc_html_e('None', 'engineo'); ?></option>
                            <option value="Article" <?php selected($this->get_schema_type($meta_data), 'Article'); ?>><?php esc_html_e('Article', 'engineo'); ?></option>
                            <option value="BlogPosting" <?php selected($this->get_schema_type($meta_data), 'BlogPosting'); ?>><?php esc_html_e('Blog Posting', 'engineo'); ?></option>
                            <option value="Product" <?php selected($this->get_schema_type($meta_data), 'Product'); ?>><?php esc_html_e('Product', 'engineo'); ?></option>
                            <option value="LocalBusiness" <?php selected($this->get_schema_type($meta_data), 'LocalBusiness'); ?>><?php esc_html_e('Local Business', 'engineo'); ?></option>
                            <option value="FAQ" <?php selected($this->get_schema_type($meta_data), 'FAQ'); ?>><?php esc_html_e('FAQ', 'engineo'); ?></option>
                            <option value="HowTo" <?php selected($this->get_schema_type($meta_data), 'HowTo'); ?>><?php esc_html_e('How To', 'engineo'); ?></option>
                        </select>
                        <p class="description"><?php esc_html_e('Select the appropriate schema type for this content', 'engineo'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }

    /**
     * Save meta data
     */
    public function save_meta_data($post_id) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check nonce
        if (!isset($_POST['engineo_meta_box_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_meta_box_nonce'])), 'engineo_meta_box')) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save meta data
        $meta_title = sanitize_text_field(wp_unslash($_POST['engineo_meta_title'] ?? ''));
        $meta_description = sanitize_textarea_field(wp_unslash($_POST['engineo_meta_description'] ?? ''));
        $canonical_url = esc_url_raw(wp_unslash($_POST['engineo_canonical_url'] ?? ''));
        $schema_type = sanitize_text_field(wp_unslash($_POST['engineo_schema_type'] ?? ''));

        // Generate schema data if type is selected
        $schema_data = '';
        if ($schema_type) {
            $schema_data = $this->generate_schema_data($post_id, $schema_type);
        }

        // Save to post meta
        update_post_meta($post_id, '_engineo_meta_title', $meta_title);
        update_post_meta($post_id, '_engineo_meta_description', $meta_description);
        update_post_meta($post_id, '_engineo_canonical_url', $canonical_url);
        update_post_meta($post_id, '_engineo_schema_type', $schema_type);
        update_post_meta($post_id, '_engineo_schema_data', $schema_data);
        
        // Clear health check cache when meta data is updated
        delete_transient('engineo_health_check_issues');
        wp_cache_delete('engineo_missing_meta_query_results', 'engineo');
    }

    /**
     * Get meta data
     */
    private function get_meta_data($post_id) {
        return [
            'meta_title' => get_post_meta($post_id, '_engineo_meta_title', true),
            'meta_description' => get_post_meta($post_id, '_engineo_meta_description', true),
            'canonical_url' => get_post_meta($post_id, '_engineo_canonical_url', true),
            'schema_type' => get_post_meta($post_id, '_engineo_schema_type', true),
            'schema_data' => get_post_meta($post_id, '_engineo_schema_data', true),
        ];
    }

    /**
     * Get schema type from meta data
     */
    private function get_schema_type($meta_data) {
        return $meta_data['schema_type'] ?? '';
    }

    /**
     * Generate schema data
     */
    private function generate_schema_data($post_id, $schema_type) {
        $post = get_post($post_id);
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => $schema_type,
            'headline' => get_the_title($post_id),
            'description' => get_the_excerpt($post_id),
            'url' => get_permalink($post_id),
            'datePublished' => get_the_date('c', $post_id),
            'dateModified' => get_the_modified_date('c', $post_id),
        ];

        // Add author
        $author = get_the_author_meta('display_name', $post->post_author);
        if ($author) {
            $schema['author'] = [
                '@type' => 'Person',
                'name' => $author,
            ];
        }

        // Add image
        $image = get_the_post_thumbnail_url($post_id, 'large');
        if ($image) {
            $schema['image'] = $image;
        }

        // Add specific schema data based on type
        if ($schema_type === 'LocalBusiness') {
            $business_settings = get_option('engineo_business_settings', []);
            if (!empty($business_settings)) {
                $schema['name'] = $business_settings['name'] ?? get_bloginfo('name');
                $schema['description'] = $business_settings['description'] ?? get_bloginfo('description');
                
                if (!empty($business_settings['address'])) {
                    $schema['address'] = [
                        '@type' => 'PostalAddress',
                        'streetAddress' => $business_settings['address'],
                        'addressLocality' => $business_settings['city'] ?? '',
                        'addressRegion' => $business_settings['state'] ?? '',
                        'postalCode' => $business_settings['zip'] ?? '',
                        'addressCountry' => $business_settings['country'] ?? '',
                    ];
                }
                
                if (!empty($business_settings['phone'])) {
                    $schema['telephone'] = $business_settings['phone'];
                }
            }
        }

        return wp_json_encode($schema);
    }

    /**
     * Output meta tags
     */
    public function output_meta_tags() {
        if (is_admin()) {
            return;
        }

        $post_id = get_the_ID();
        if (!$post_id) {
            return;
        }

        $meta_data = $this->get_meta_data($post_id);
        
        // Output meta description
        if (!empty($meta_data['meta_description'])) {
            echo '<meta name="description" content="' . esc_attr($meta_data['meta_description']) . '">' . "\n";
        }

        // Output canonical URL
        if (!empty($meta_data['canonical_url'])) {
            echo '<link rel="canonical" href="' . esc_url($meta_data['canonical_url']) . '">' . "\n";
        } else {
            // Default canonical URL
            echo '<link rel="canonical" href="' . esc_url(get_permalink($post_id)) . '">' . "\n";
        }

        // Output OpenGraph tags
        $this->output_opengraph_tags($meta_data);
    }

    /**
     * Output OpenGraph tags
     */
    private function output_opengraph_tags($meta_data) {
        $og_title = !empty($meta_data['meta_title']) ? $meta_data['meta_title'] : get_the_title();
        $og_description = !empty($meta_data['meta_description']) ? $meta_data['meta_description'] : get_the_excerpt();
        $og_url = !empty($meta_data['canonical_url']) ? $meta_data['canonical_url'] : get_permalink();
        $og_image = get_the_post_thumbnail_url(null, 'large');

        echo '<meta property="og:title" content="' . esc_attr($og_title) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($og_description) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url($og_url) . '">' . "\n";
        echo '<meta property="og:type" content="article">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '">' . "\n";
        
        if ($og_image) {
            echo '<meta property="og:image" content="' . esc_url($og_image) . '">' . "\n";
        }
    }

    /**
     * Output schema markup
     */
    public function output_schema_markup() {
        if (is_admin()) {
            return;
        }

        $post_id = get_the_ID();
        if (!$post_id) {
            return;
        }

        $meta_data = $this->get_meta_data($post_id);
        if (empty($meta_data['schema_data'])) {
            return;
        }

        $schema_data = json_decode($meta_data['schema_data'], true);
        if (!$schema_data) {
            return;
        }

        echo '<script type="application/ld+json">' . "\n";
        echo wp_json_encode($schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
        echo '</script>' . "\n";
    }

    /**
     * Filter document title
     */
    public function filter_document_title($title_parts) {
        $post_id = get_the_ID();
        if (!$post_id) {
            return $title_parts;
        }

        $meta_data = $this->get_meta_data($post_id);
        if (!empty($meta_data['meta_title'])) {
            $title_parts['title'] = $meta_data['meta_title'];
        }

        return $title_parts;
    }

    /**
     * Enable WordPress sitemaps
     */
    public function enable_wordpress_sitemaps() {
        // Ensure WordPress sitemaps are enabled (WordPress 5.5+)
        if (function_exists('wp_sitemaps_get_server')) {
            // WordPress sitemaps are already enabled
            return;
        }
        
        // For older WordPress versions or if sitemaps are disabled
        add_filter('wp_sitemaps_enabled', '__return_true');
    }

    /**
     * Add rewrite rules
     */
    private function add_rewrite_rules() {
        add_rewrite_rule('^sitemap\.xml$', 'index.php?engineo_sitemap=1', 'top');
        add_rewrite_rule('^robots\.txt$', 'index.php?engineo_robots=1', 'top');
        add_rewrite_rule('^llms\.txt$', 'index.php?engineo_llms=1', 'top');
    }

    /**
     * Add query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'engineo_sitemap';
        $vars[] = 'engineo_robots';
        $vars[] = 'engineo_llms';
        return $vars;
    }

    /**
     * Handle redirects
     */
    public function handle_redirects() {
        $redirects = get_option('engineo_redirects', []);
        $current_url = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        
        foreach ($redirects as $redirect) {
            if ($redirect['from'] === $current_url) {
                wp_redirect($redirect['to'], $redirect['type']);
                exit;
            }
        }
    }

    /**
     * Handle requests
     */
    public function handle_requests() {
        // Check for our custom endpoints early, before Elementor processes anything
        if (get_query_var('engineo_sitemap')) {
            $this->output_sitemap();
            exit;
        }
        
        if (get_query_var('engineo_robots')) {
            $this->output_robots_txt();
            exit;
        }
        
        if (get_query_var('engineo_llms')) {
            $this->output_llms_txt();
            exit;
        }
    }

    /**
     * Output XML sitemap
     */
    private function output_sitemap() {
        header('Content-Type: application/xml; charset=utf-8');
        
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        // Add homepage
        echo '<url>' . "\n";
        echo '<loc>' . esc_url(home_url('/')) . '</loc>' . "\n";
        echo '<lastmod>' . esc_html(gmdate('c')) . '</lastmod>' . "\n";
        echo '<priority>1.0</priority>' . "\n";
        echo '<changefreq>daily</changefreq>' . "\n";
        echo '</url>' . "\n";

        // Add posts and pages
        $posts = get_posts([
            'numberposts' => -1,
            'post_status' => 'publish',
            'post_type' => ['post', 'page'],
        ]);

        foreach ($posts as $post) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(get_permalink($post->ID)) . '</loc>' . "\n";
            echo '<lastmod>' . esc_html(get_the_modified_date('c', $post->ID)) . '</lastmod>' . "\n";
            echo '<priority>0.8</priority>' . "\n";
            echo '<changefreq>weekly</changefreq>' . "\n";
            echo '</url>' . "\n";
        }

        // Add WordPress sitemap reference
        if (function_exists('wp_sitemaps_get_server')) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(home_url('/wp-sitemap.xml')) . '</loc>' . "\n";
            echo '<lastmod>' . esc_html(gmdate('c')) . '</lastmod>' . "\n";
            echo '<priority>0.9</priority>' . "\n";
            echo '<changefreq>daily</changefreq>' . "\n";
            echo '</url>' . "\n";
        }

        echo '</urlset>';
    }

    /**
     * Output robots.txt
     */
    private function output_robots_txt() {
        header('Content-Type: text/plain; charset=utf-8');
        
        $robots_content = get_option('engineo_robots_txt', '');
        $existing_robots = $this->get_existing_robots_txt();
        
        // If no custom content, check for existing robots.txt
        if (empty($robots_content)) {
            if (!empty($existing_robots)) {
                // Use existing robots.txt content
                echo wp_kses_post($existing_robots);
            } else {
                // Default robots.txt content
                echo "User-agent: *\n";
                echo "Disallow: /wp-admin/\n";
                echo "Allow: " . esc_url(admin_url('admin-ajax.php')) . "\n";
                echo "Disallow: /wp-includes/\n";
                echo "Disallow: /wp-content/plugins/\n";
                echo "Disallow: /wp-content/themes/\n";
                echo "Disallow: /wp-content/cache/\n";
                echo "Allow: /wp-content/uploads/\n";
                echo "Sitemap: " . esc_url(home_url('/sitemap.xml')) . "\n";
                echo "Sitemap: " . esc_url(home_url('/wp-sitemap.xml')) . "\n";
            }
        } else {
            echo wp_kses_post($robots_content);
        }
    }

    /**
     * Get existing robots.txt content
     */
    private function get_existing_robots_txt() {
        // Use ABSPATH instead of get_home_path() for frontend compatibility
        $robots_file = ABSPATH . 'robots.txt';
        $existing_content = '';
        
        if (file_exists($robots_file)) {
            $existing_content = file_get_contents($robots_file);
        }
        
        return $existing_content;
    }

    /**
     * Handle direct requests for our custom files
     */
    public function handle_direct_requests() {
        // Prevent multiple executions
        static $handled = false;
        if ($handled) {
            return;
        }
        
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        
        // Check for our custom files directly
        if (strpos($request_uri, '/llms.txt') !== false) {
            $handled = true;
            $this->output_llms_txt();
            exit;
        }
        
        if (strpos($request_uri, '/robots.txt') !== false) {
            $handled = true;
            $this->output_robots_txt();
            exit;
        }
        
        if (strpos($request_uri, '/sitemap.xml') !== false) {
            $handled = true;
            $this->output_sitemap();
            exit;
        }
    }

    /**
     * Early request handler to catch our endpoints before Elementor
     */
    public function early_request_handler($wp) {
        // Check for our custom endpoints in the request
        if (isset($wp->query_vars['engineo_llms']) || 
            isset($wp->query_vars['engineo_robots']) || 
            isset($wp->query_vars['engineo_sitemap'])) {
            
            // Completely disable Elementor for our endpoints
            if (class_exists('Elementor\Plugin')) {
                // Remove all Elementor actions and filters
                remove_all_actions('template_redirect');
                remove_all_actions('wp_head');
                remove_all_actions('wp_footer');
                remove_all_actions('elementor/loaded');
                remove_all_actions('elementor/init');
                
                // Disable Elementor's document system
                add_filter('elementor/documents/get', '__return_false');
                add_filter('elementor/frontend/builder_content/before', '__return_false');
            }
        }
    }

    /**
     * Prevent Elementor conflicts
     */
    public function prevent_elementor_conflicts() {
        // Check if this is one of our custom endpoints
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters for custom endpoints, not form data
        if (isset($_GET['engineo_llms']) || isset($_GET['engineo_robots']) || isset($_GET['engineo_sitemap'])) {
            // Remove Elementor's template_redirect action completely
            if (class_exists('Elementor\Plugin')) {
                remove_action('template_redirect', ['Elementor\Plugin', 'init']);
                remove_action('template_redirect', ['Elementor\Core\Base\Document', 'init']);
                
                // Remove Elementor's document actions
                remove_all_actions('elementor/documents/register_controls');
                remove_all_actions('elementor/documents/after_save');
                remove_all_actions('elementor/documents/before_save');
            }
        }
    }

    /**
     * Output llms.txt
     */
    private function output_llms_txt() {
        // Completely bypass WordPress and Elementor
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        // Set proper headers
        header('Content-Type: text/plain; charset=utf-8');
        header('X-Robots-Tag: noindex, nofollow');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('HTTP/1.1 200 OK');
        
        // Get content without WordPress functions that might trigger Elementor
        $llms_content = get_option('engineo_llms_txt', '');
        
        // If content is too short or just whitespace, use default
        if (empty($llms_content) || strlen(trim($llms_content)) < 10) {
            // Default llms.txt content
            echo "# llms.txt - AI Engine Optimization\n";
            echo "# This file provides information about this website for AI engines\n\n";
            echo "Website: " . esc_url(home_url()) . "\n";
            echo "Name: " . esc_html(get_bloginfo('name')) . "\n";
            echo "Description: " . esc_html(get_bloginfo('description')) . "\n";
            echo "Language: " . esc_html(get_locale()) . "\n";
            echo "Last Updated: " . esc_html(gmdate('Y-m-d')) . "\n\n";
            echo "# Content Guidelines\n";
            echo "This website contains original content that may be used for AI training purposes.\n";
            echo "Please respect our content and attribution guidelines.\n\n";
            echo "# Contact\n";
            echo "For questions about AI training data usage, please contact us.\n";
        } else {
            echo wp_kses_post($llms_content);
        }
        
        exit;
    }

    /**
     * Admin dashboard
     */
    public function admin_dashboard() {
        // Handle flush rewrite rules
        if (isset($_GET['flush_rules']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'] ?? '')), 'engineo_flush_rules')) {
            $this->add_rewrite_rules();
            flush_rewrite_rules();
            echo '<div class="notice notice-success"><p>' . esc_html__('Rewrite rules flushed successfully!', 'engineo') . '</p></div>';
        }
        
        // Handle cache refresh
        if (isset($_GET['refresh_cache']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'] ?? '')), 'engineo_refresh_cache')) {
            delete_transient('engineo_health_check_issues');
            wp_cache_delete('engineo_missing_meta_query_results', 'engineo');
            echo '<div class="notice notice-success"><p>' . esc_html__('Health check cache refreshed successfully!', 'engineo') . '</p></div>';
        }
        
        $health_issues = $this->get_health_check_issues();
        $total_posts = wp_count_posts()->publish + wp_count_posts('page')->publish;
        
        // Calculate health score based on actual posts with missing meta data
        global $wpdb;
        $query_cache_key = 'engineo_missing_meta_query_results';
        $missing_meta_results = wp_cache_get($query_cache_key, 'engineo');
        
        if (false === $missing_meta_results) {
            // Get fresh data if not cached
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Cached query with proper preparation
            $missing_meta_results = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p.ID,
                    CASE 
                        WHEN mt.meta_value IS NULL AND md.meta_value IS NULL THEN 'both'
                        WHEN mt.meta_value IS NULL THEN 'title'
                        WHEN md.meta_value IS NULL THEN 'description'
                    END as missing_type
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} mt ON p.ID = mt.post_id AND mt.meta_key = %s
                LEFT JOIN {$wpdb->postmeta} md ON p.ID = md.post_id AND md.meta_key = %s
                WHERE p.post_status = 'publish' 
                AND p.post_type IN ('post', 'page')
                AND (mt.meta_value IS NULL OR md.meta_value IS NULL)
                LIMIT 1000
            ", '_engineo_meta_title', '_engineo_meta_description'));
        }
        
        $posts_with_issues = count($missing_meta_results);
        $optimized_posts = $total_posts - $posts_with_issues;
        $health_score = $total_posts > 0 ? round(($optimized_posts / $total_posts) * 100) : 100;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Engineo Dashboard', 'engineo'); ?></h1>
            
            <div class="notice notice-success">
                <p><strong><?php esc_html_e('Welcome to Engineo!', 'engineo'); ?></strong> <?php esc_html_e('Your all-engine optimization plugin is ready to use.', 'engineo'); ?></p>
            </div>

            <?php if (!empty($health_issues)): ?>
            <div class="notice notice-warning">
                <p><strong><?php esc_html_e('SEO Health Check:', 'engineo'); ?></strong></p>
                <ul>
                    <?php foreach ($health_issues as $issue): ?>
                    <li><?php echo esc_html($issue); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p><a href="<?php echo esc_url(admin_url('admin.php?page=engineo-health')); ?>" class="button button-primary"><?php esc_html_e('View Full Health Check', 'engineo'); ?></a></p>
            </div>
            <?php else: ?>
            <div class="notice notice-success">
                <p><strong>✅ <?php esc_html_e('SEO Health Check:', 'engineo'); ?></strong> <?php esc_html_e('No errors found! Your site is well optimized.', 'engineo'); ?></p>
                <p><a href="<?php echo esc_url(admin_url('admin.php?page=engineo-health')); ?>" class="button button-secondary"><?php esc_html_e('View Full Health Check', 'engineo'); ?></a></p>
            </div>
            <?php endif; ?>

            <div class="card">
                <h2><?php esc_html_e('Quick Stats', 'engineo'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e('Total Posts', 'engineo'); ?></th>
                        <td><?php echo esc_html(wp_count_posts()->publish); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Total Pages', 'engineo'); ?></th>
                        <td><?php echo esc_html(wp_count_posts('page')->publish); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Active Redirects', 'engineo'); ?></th>
                        <td><?php echo esc_html(count(get_option('engineo_redirects', []))); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Plugin Version', 'engineo'); ?></th>
                        <td><?php echo esc_html(ENGINEO_VERSION); ?></td>
                    </tr>
                </table>
            </div>

            <div class="card">
                <h2><?php esc_html_e('Available URLs', 'engineo'); ?></h2>
                <ul>
                    <li><strong><?php esc_html_e('XML Sitemap:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank"><?php echo esc_url(home_url('/sitemap.xml')); ?></a></li>
                    <li><strong><?php esc_html_e('Robots.txt:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/robots.txt')); ?>" target="_blank"><?php echo esc_url(home_url('/robots.txt')); ?></a></li>
                    <li><strong><?php esc_html_e('llms.txt:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/llms.txt')); ?>" target="_blank"><?php echo esc_url(home_url('/llms.txt')); ?></a></li>
                </ul>
                
                <p>
                    <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=engineo&flush_rules=1'), 'engineo_flush_rules')); ?>" class="button button-secondary"><?php esc_html_e('Flush Rewrite Rules', 'engineo'); ?></a>
                    <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=engineo&refresh_cache=1'), 'engineo_refresh_cache')); ?>" class="button button-secondary"><?php esc_html_e('Refresh Health Check Cache', 'engineo'); ?></a>
                </p>
            </div>

            <div class="card">
                <h2><?php esc_html_e('Getting Started', 'engineo'); ?></h2>
                <ol>
                    <li><?php esc_html_e('Edit your posts and pages to add SEO meta data', 'engineo'); ?></li>
                    <li><?php esc_html_e('Configure your robots.txt file', 'engineo'); ?></li>
                    <li><?php esc_html_e('Customize your llms.txt for AI engines', 'engineo'); ?></li>
                    <li><?php esc_html_e('Set up Local Business schema if applicable', 'engineo'); ?></li>
                    <li><?php esc_html_e('Add redirects for old URLs', 'engineo'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }

    /**
     * SEO settings page
     */
    public function seo_settings_page() {
        if (isset($_POST['submit']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_seo_nonce'] ?? '')), 'engineo_seo_settings')) {
            $this->save_seo_settings();
        }

        $business_settings = get_option('engineo_business_settings', []);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('SEO Settings', 'engineo'); ?></h1>
            
            <form method="post" action="">
                <?php wp_nonce_field('engineo_seo_settings', 'engineo_seo_nonce'); ?>
                
                <h2><?php esc_html_e('Local Business Settings', 'engineo'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Business Name', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_name" value="<?php echo esc_attr($business_settings['name'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Business Description', 'engineo'); ?></th>
                        <td>
                            <textarea name="business_description" class="large-text" rows="3"><?php echo esc_textarea($business_settings['description'] ?? ''); ?></textarea>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Address', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_address" value="<?php echo esc_attr($business_settings['address'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('City', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_city" value="<?php echo esc_attr($business_settings['city'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('State', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_state" value="<?php echo esc_attr($business_settings['state'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('ZIP Code', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_zip" value="<?php echo esc_attr($business_settings['zip'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Country', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_country" value="<?php echo esc_attr($business_settings['country'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Phone', 'engineo'); ?></th>
                        <td>
                            <input type="text" name="business_phone" value="<?php echo esc_attr($business_settings['phone'] ?? ''); ?>" class="regular-text" />
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Save SEO settings
     */
    private function save_seo_settings() {
        if (!isset($_POST['engineo_seo_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_seo_nonce'])), 'engineo_seo_settings')) {
            return;
        }

        $business_settings = [
            'name' => sanitize_text_field(wp_unslash($_POST['business_name'] ?? '')),
            'description' => sanitize_textarea_field(wp_unslash($_POST['business_description'] ?? '')),
            'address' => sanitize_text_field(wp_unslash($_POST['business_address'] ?? '')),
            'city' => sanitize_text_field(wp_unslash($_POST['business_city'] ?? '')),
            'state' => sanitize_text_field(wp_unslash($_POST['business_state'] ?? '')),
            'zip' => sanitize_text_field(wp_unslash($_POST['business_zip'] ?? '')),
            'country' => sanitize_text_field(wp_unslash($_POST['business_country'] ?? '')),
            'phone' => sanitize_text_field(wp_unslash($_POST['business_phone'] ?? '')),
        ];

        update_option('engineo_business_settings', $business_settings);
        add_settings_error('engineo_seo', 'settings_updated', __('Settings saved successfully.', 'engineo'), 'updated');
    }

    /**
     * Sitemap page
     */
    public function sitemap_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('XML Sitemap', 'engineo'); ?></h1>
            
            <div class="card">
                <h2><?php esc_html_e('Custom Sitemap', 'engineo'); ?></h2>
                <p><?php esc_html_e('Your custom XML sitemap is automatically generated and available at:', 'engineo'); ?></p>
                <p><a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank" class="button button-primary"><?php esc_html_e('View Custom Sitemap', 'engineo'); ?></a></p>
                <p><em><?php esc_html_e('This sitemap includes all your posts and pages with optimized structure.', 'engineo'); ?></em></p>
            </div>
            
            <div class="card">
                <h2><?php esc_html_e('WordPress Sitemap', 'engineo'); ?></h2>
                <p><?php esc_html_e('WordPress also generates its own sitemap automatically:', 'engineo'); ?></p>
                <p><a href="<?php echo esc_url(home_url('/wp-sitemap.xml')); ?>" target="_blank" class="button button-secondary"><?php esc_html_e('View WordPress Sitemap', 'engineo'); ?></a></p>
                <p><em><?php esc_html_e('This is the default WordPress sitemap that includes all content types.', 'engineo'); ?></em></p>
            </div>
            
            <div class="card">
                <h2><?php esc_html_e('Sitemap Index', 'engineo'); ?></h2>
                <p><?php esc_html_e('Both sitemaps are automatically included in your robots.txt file:', 'engineo'); ?></p>
                <ul>
                    <li><strong><?php esc_html_e('Custom Sitemap:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank"><?php echo esc_url(home_url('/sitemap.xml')); ?></a></li>
                    <li><strong><?php esc_html_e('WordPress Sitemap:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/wp-sitemap.xml')); ?>" target="_blank"><?php echo esc_url(home_url('/wp-sitemap.xml')); ?></a></li>
                </ul>
                <p><?php esc_html_e('Submit both URLs to Google Search Console and other search engines for comprehensive coverage.', 'engineo'); ?></p>
            </div>
        </div>
        <?php
    }

    /**
     * Robots page
     */
    public function robots_page() {
        if (isset($_POST['submit']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_robots_nonce'] ?? '')), 'engineo_robots_settings')) {
            $this->save_robots_txt();
        }

        $robots_content = get_option('engineo_robots_txt', '');
        $existing_robots = $this->get_existing_robots_txt();
        
        // If no custom content saved, show what would actually be output
        if (empty($robots_content)) {
            if (!empty($existing_robots)) {
                // Use existing robots.txt content
                $robots_content = $existing_robots;
            } else {
                // Show default robots.txt content that would be output
                $robots_content = "User-agent: *\n";
                $robots_content .= "Disallow: /wp-admin/\n";
                $robots_content .= "Allow: " . admin_url('admin-ajax.php') . "\n";
                $robots_content .= "Disallow: /wp-includes/\n";
                $robots_content .= "Disallow: /wp-content/plugins/\n";
                $robots_content .= "Disallow: /wp-content/themes/\n";
                $robots_content .= "Disallow: /wp-content/cache/\n";
                $robots_content .= "Allow: /wp-content/uploads/\n";
                $robots_content .= "Sitemap: " . home_url('/sitemap.xml') . "\n";
                $robots_content .= "Sitemap: " . home_url('/wp-sitemap.xml') . "\n";
            }
        }
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Robots.txt Editor', 'engineo'); ?></h1>
            
            <?php if (!empty($existing_robots) && empty(get_option('engineo_robots_txt', ''))): ?>
            <div class="notice notice-info">
                <p><strong><?php esc_html_e('Existing robots.txt detected!', 'engineo'); ?></strong> <?php esc_html_e('The content below is from your existing robots.txt file. You can edit it or leave empty to use default settings.', 'engineo'); ?></p>
            </div>
            <?php endif; ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('engineo_robots_settings', 'engineo_robots_nonce'); ?>
                
                <p><?php esc_html_e('Edit your robots.txt file below. Leave empty to use default settings.', 'engineo'); ?></p>
                <textarea name="robots_content" rows="20" cols="80" class="large-text code" placeholder="User-agent: *
Disallow: /wp-admin/
Allow: <?php echo esc_url(admin_url('admin-ajax.php')); ?>
Sitemap: <?php echo esc_url(home_url('/sitemap.xml')); ?>"><?php echo esc_textarea($robots_content); ?></textarea>
                
                <?php submit_button(); ?>
            </form>
            
            <div class="card">
                <h2><?php esc_html_e('Current robots.txt', 'engineo'); ?></h2>
                <p><?php esc_html_e('Your robots.txt is available at:', 'engineo'); ?> <a href="<?php echo esc_url(home_url('/robots.txt')); ?>" target="_blank"><?php echo esc_url(home_url('/robots.txt')); ?></a></p>
                
                <?php if (!empty($existing_robots)): ?>
                <h3><?php esc_html_e('Existing robots.txt content:', 'engineo'); ?></h3>
                <pre style="background: #f1f1f1; padding: 15px; border: 1px solid #ddd; overflow-x: auto;"><?php echo esc_html($existing_robots); ?></pre>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Save robots.txt
     */
    private function save_robots_txt() {
        if (!isset($_POST['engineo_robots_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_robots_nonce'])), 'engineo_robots_settings')) {
            return;
        }

        $robots_content = sanitize_textarea_field(wp_unslash($_POST['robots_content'] ?? ''));
        update_option('engineo_robots_txt', $robots_content);
        add_settings_error('engineo_robots', 'settings_updated', __('Robots.txt saved successfully.', 'engineo'), 'updated');
    }

    /**
     * LLMS page
     */
    public function llms_page() {
        if (isset($_POST['submit']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_llms_nonce'] ?? '')), 'engineo_llms_settings')) {
            $this->save_llms_txt();
        }
        
        if (isset($_POST['reset_to_default']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_llms_nonce'] ?? '')), 'engineo_llms_settings')) {
            $this->reset_llms_txt_to_default();
        }

        $llms_content = get_option('engineo_llms_txt', '');
        
        // Debug info (only for administrators in development)
        if (current_user_can('manage_options') && defined('WP_DEBUG') && WP_DEBUG) {
            $debug_info = empty($llms_content) ? 'EMPTY' : 'HAS CONTENT (' . strlen($llms_content) . ' chars)';
            if (!empty($llms_content)) {
                $debug_info .= '<br><strong>Content preview:</strong> "' . esc_html(substr($llms_content, 0, 100)) . '"';
            }
            echo '<div class="notice notice-info"><p><strong>Debug:</strong> ' . wp_kses_post($debug_info) . '</p></div>';
        }
        
        // If no custom content or just whitespace, show default content in editor
        if (empty($llms_content) || strlen(trim($llms_content)) < 10) {
            $llms_content = "# llms.txt - AI Engine Optimization\n";
            $llms_content .= "# This file provides information about this website for AI engines\n\n";
            $llms_content .= "Website: " . home_url() . "\n";
            $llms_content .= "Name: " . get_bloginfo('name') . "\n";
            $llms_content .= "Description: " . get_bloginfo('description') . "\n";
            $llms_content .= "Language: " . get_locale() . "\n";
            $llms_content .= "Last Updated: " . gmdate('Y-m-d') . "\n\n";
            $llms_content .= "# Content Guidelines\n";
            $llms_content .= "This website contains original content that may be used for AI training purposes.\n";
            $llms_content .= "Please respect our content and attribution guidelines.\n\n";
            $llms_content .= "# Contact\n";
            $llms_content .= "For questions about AI training data usage, please contact us.\n";
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('llms.txt Editor', 'engineo'); ?></h1>
            
            <?php settings_errors('engineo_llms'); ?>
            
            <div class="notice notice-info">
                <p><strong><?php esc_html_e('About llms.txt:', 'engineo'); ?></strong> <?php esc_html_e('This is a unique feature for AI Engine Optimization (AEO). It provides information about your website for AI engines and language models.', 'engineo'); ?></p>
            </div>
            
            <form method="post" action="">
                <?php wp_nonce_field('engineo_llms_settings', 'engineo_llms_nonce'); ?>
                
                <p><?php esc_html_e('Edit your llms.txt file below. The content above is the default template - you can modify it as needed.', 'engineo'); ?></p>
                <textarea name="llms_content" rows="20" cols="80" class="large-text code"><?php echo esc_textarea($llms_content); ?></textarea>
                
                <p class="submit">
                    <?php submit_button(__('Save llms.txt', 'engineo'), 'primary', 'submit', false); ?>
                    <?php submit_button(__('Reset to Default', 'engineo'), 'secondary', 'reset_to_default', false, ['onclick' => 'return confirm("' . esc_js(__('Are you sure you want to reset to default content?', 'engineo')) . '");']); ?>
                </p>
            </form>
            
            <div class="card">
                <h2><?php esc_html_e('Current llms.txt', 'engineo'); ?></h2>
                <p><?php esc_html_e('Your llms.txt is available at:', 'engineo'); ?> <a href="<?php echo esc_url(home_url('/llms.txt')); ?>" target="_blank"><?php echo esc_url(home_url('/llms.txt')); ?></a></p>
                
                <h3><?php esc_html_e('Preview:', 'engineo'); ?></h3>
                <pre style="background: #f1f1f1; padding: 15px; border: 1px solid #ddd; overflow-x: auto; max-height: 300px;"><?php echo esc_html($llms_content); ?></pre>
            </div>
        </div>
        <?php
    }

    /**
     * Save llms.txt
     */
    private function save_llms_txt() {
        if (!isset($_POST['engineo_llms_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_llms_nonce'])), 'engineo_llms_settings')) {
            add_settings_error('engineo_llms', 'nonce_error', __('Security check failed. Please try again.', 'engineo'), 'error');
            return;
        }

        $llms_content = sanitize_textarea_field(wp_unslash($_POST['llms_content'] ?? ''));
        $result = update_option('engineo_llms_txt', $llms_content);
        
        if ($result) {
            add_settings_error('engineo_llms', 'settings_updated', __('llms.txt saved successfully.', 'engineo'), 'updated');
        } else {
            add_settings_error('engineo_llms', 'settings_error', __('Failed to save llms.txt. Please try again.', 'engineo'), 'error');
        }
    }

    /**
     * Reset llms.txt to default
     */
    private function reset_llms_txt_to_default() {
        if (!isset($_POST['engineo_llms_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_llms_nonce'])), 'engineo_llms_settings')) {
            add_settings_error('engineo_llms', 'nonce_error', __('Security check failed. Please try again.', 'engineo'), 'error');
            return;
        }

        // Delete the option to reset to default
        delete_option('engineo_llms_txt');
        add_settings_error('engineo_llms', 'settings_reset', __('llms.txt reset to default successfully.', 'engineo'), 'updated');
    }

    /**
     * WordPress version notice
     */
    public function wp_version_notice() {
        echo '<div class="notice notice-error"><p>';
        printf(
            /* translators: %s: WordPress version */
            esc_html__('Engineo requires WordPress 6.0 or higher. You are running version %s.', 'engineo'),
            esc_html(get_bloginfo('version'))
        );
        echo '</p></div>';
    }

    /**
     * PHP version notice
     */
    public function php_version_notice() {
        echo '<div class="notice notice-error"><p>';
        printf(
            /* translators: %s: PHP version */
            esc_html__('Engineo requires PHP 8.0 or higher. You are running version %s.', 'engineo'),
            esc_html(PHP_VERSION)
        );
        echo '</p></div>';
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Add rewrite rules
        $this->add_rewrite_rules();
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Set default options
        add_option('engineo_version', ENGINEO_VERSION);
    }

    /**
     * Get health check issues
     */
    private function get_health_check_issues() {
        // Check for cached results first
        $cache_key = 'engineo_health_check_issues';
        $cached_issues = get_transient($cache_key);
        
        if ($cached_issues !== false && is_array($cached_issues)) {
            return $cached_issues;
        }
        
        $issues = [];
        $total_posts = wp_count_posts()->publish + wp_count_posts('page')->publish;
        
        // Check for missing meta titles and descriptions using a single optimized query
        global $wpdb;
        
        // Check for cached query results first
        $query_cache_key = 'engineo_missing_meta_query_results';
        $missing_meta_results = wp_cache_get($query_cache_key, 'engineo');
        
        if (false === $missing_meta_results) {
            // Use direct SQL query for better performance - this is necessary for complex meta queries
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Complex meta query optimization with caching
            $missing_meta_results = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p.ID,
                    CASE 
                        WHEN mt.meta_value IS NULL AND md.meta_value IS NULL THEN 'both'
                        WHEN mt.meta_value IS NULL THEN 'title'
                        WHEN md.meta_value IS NULL THEN 'description'
                    END as missing_type
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} mt ON p.ID = mt.post_id AND mt.meta_key = %s
                LEFT JOIN {$wpdb->postmeta} md ON p.ID = md.post_id AND md.meta_key = %s
                WHERE p.post_status = 'publish' 
                AND p.post_type IN ('post', 'page')
                AND (mt.meta_value IS NULL OR md.meta_value IS NULL)
                LIMIT 1000
            ", '_engineo_meta_title', '_engineo_meta_description'));
            
            // Cache the query results for 30 minutes
            wp_cache_set($query_cache_key, $missing_meta_results, 'engineo', 30 * MINUTE_IN_SECONDS);
        }
        
        // Count missing titles and descriptions
        $missing_titles = 0;
        $missing_descriptions = 0;
        
        foreach ($missing_meta_results as $result) {
            if ($result->missing_type === 'title') {
                $missing_titles++;
            } elseif ($result->missing_type === 'description') {
                $missing_descriptions++;
            } elseif ($result->missing_type === 'both') {
                $missing_titles++;
                $missing_descriptions++;
            }
        }
        
        if ($missing_titles > 0) {
            $issues[] = sprintf(
                /* translators: %d: number of posts/pages */
                __('%d posts/pages missing meta titles', 'engineo'), 
                $missing_titles
            );
        }
        
        if ($missing_descriptions > 0) {
            $issues[] = sprintf(
                /* translators: %d: number of posts/pages */
                __('%d posts/pages missing meta descriptions', 'engineo'), 
                $missing_descriptions
            );
        }
        
        // Check for empty site title
        $site_title = get_bloginfo('name');
        if (empty($site_title)) {
            $issues[] = __('Site title is empty', 'engineo');
        }
        
        // Check for empty site description
        $site_description = get_bloginfo('description');
        if (empty($site_description)) {
            $issues[] = __('Site tagline/description is empty', 'engineo');
        }
        
        // Check if robots.txt is accessible
        $robots_url = home_url('/robots.txt');
        $robots_response = wp_remote_get($robots_url);
        $robots_code = wp_remote_retrieve_response_code($robots_response);
        if (is_wp_error($robots_response) || $robots_code !== 200) {
            $issues[] = __('Robots.txt is not accessible', 'engineo');
        }
        
        // Check if sitemap.xml is accessible
        $sitemap_url = home_url('/sitemap.xml');
        $sitemap_response = wp_remote_get($sitemap_url);
        if (is_wp_error($sitemap_response) || wp_remote_retrieve_response_code($sitemap_response) !== 200) {
            $issues[] = __('XML Sitemap is not accessible', 'engineo');
        }
        
        // Cache the results for 2 hours to improve performance
        set_transient($cache_key, $issues, 2 * HOUR_IN_SECONDS);
        
        return $issues;
    }

    /**
     * Redirects page
     */
    public function redirects_page() {
        if (isset($_POST['add_redirect']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_redirects_nonce'] ?? '')), 'engineo_redirects')) {
            $this->add_redirect();
        }
        
        if (isset($_POST['delete_redirect']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_redirects_nonce'] ?? '')), 'engineo_redirects')) {
            $this->delete_redirect();
        }

        $redirects = get_option('engineo_redirects', []);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('301/302 Redirect Manager', 'engineo'); ?></h1>
            
            <div class="card">
                <h2><?php esc_html_e('Add New Redirect', 'engineo'); ?></h2>
                <form method="post" action="">
                    <?php wp_nonce_field('engineo_redirects', 'engineo_redirects_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('From URL', 'engineo'); ?></th>
                            <td>
                                <input type="text" name="redirect_from" class="regular-text" placeholder="/old-page/" required />
                                <p class="description"><?php esc_html_e('The old URL (relative path)', 'engineo'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('To URL', 'engineo'); ?></th>
                            <td>
                                <input type="url" name="redirect_to" class="regular-text" placeholder="https://example.com/new-page/" required />
                                <p class="description"><?php esc_html_e('The new URL (full URL)', 'engineo'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Redirect Type', 'engineo'); ?></th>
                            <td>
                                <select name="redirect_type">
                                    <option value="301"><?php esc_html_e('301 - Permanent Redirect', 'engineo'); ?></option>
                                    <option value="302"><?php esc_html_e('302 - Temporary Redirect', 'engineo'); ?></option>
                                </select>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(__('Add Redirect', 'engineo'), 'primary', 'add_redirect'); ?>
                </form>
            </div>

            <div class="card">
                <h2><?php esc_html_e('Active Redirects', 'engineo'); ?></h2>
                <?php if (empty($redirects)): ?>
                    <p><?php esc_html_e('No redirects configured yet.', 'engineo'); ?></p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('From URL', 'engineo'); ?></th>
                                <th><?php esc_html_e('To URL', 'engineo'); ?></th>
                                <th><?php esc_html_e('Type', 'engineo'); ?></th>
                                <th><?php esc_html_e('Actions', 'engineo'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($redirects as $index => $redirect): ?>
                            <tr>
                                <td><?php echo esc_html($redirect['from']); ?></td>
                                <td><?php echo esc_html($redirect['to']); ?></td>
                                <td><?php echo esc_html($redirect['type']); ?></td>
                                <td>
                                    <form method="post" style="display: inline;">
                                        <?php wp_nonce_field('engineo_redirects', 'engineo_redirects_nonce'); ?>
                                        <input type="hidden" name="redirect_index" value="<?php echo esc_attr($index); ?>" />
                                        <input type="submit" name="delete_redirect" value="<?php esc_attr_e('Delete', 'engineo'); ?>" class="button button-small" onclick="return confirm('<?php esc_attr_e('Are you sure?', 'engineo'); ?>');" />
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Add redirect
     */
    private function add_redirect() {
        if (!isset($_POST['engineo_redirects_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_redirects_nonce'])), 'engineo_redirects')) {
            return;
        }

        $redirects = get_option('engineo_redirects', []);
        
        $redirects[] = [
            'from' => sanitize_text_field(wp_unslash($_POST['redirect_from'] ?? '')),
            'to' => esc_url_raw(wp_unslash($_POST['redirect_to'] ?? '')),
            'type' => intval($_POST['redirect_type'] ?? 301)
        ];
        
        update_option('engineo_redirects', $redirects);
        add_settings_error('engineo_redirects', 'redirect_added', __('Redirect added successfully.', 'engineo'), 'updated');
    }

    /**
     * Delete redirect
     */
    private function delete_redirect() {
        if (!isset($_POST['engineo_redirects_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['engineo_redirects_nonce'])), 'engineo_redirects')) {
            return;
        }

        $redirects = get_option('engineo_redirects', []);
        $index = intval($_POST['redirect_index'] ?? 0);
        
        if (isset($redirects[$index])) {
            unset($redirects[$index]);
            $redirects = array_values($redirects); // Re-index array
            update_option('engineo_redirects', $redirects);
            add_settings_error('engineo_redirects', 'redirect_deleted', __('Redirect deleted successfully.', 'engineo'), 'updated');
        }
    }

    /**
     * Health check page
     */
    public function health_check_page() {
        $health_issues = $this->get_health_check_issues();
        $total_posts = wp_count_posts()->publish + wp_count_posts('page')->publish;
        
        // Calculate health score based on actual posts with missing meta data
        global $wpdb;
        $query_cache_key = 'engineo_missing_meta_query_results';
        $missing_meta_results = wp_cache_get($query_cache_key, 'engineo');
        
        if (false === $missing_meta_results) {
            // Get fresh data if not cached
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Cached query with proper preparation
            $missing_meta_results = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p.ID,
                    CASE 
                        WHEN mt.meta_value IS NULL AND md.meta_value IS NULL THEN 'both'
                        WHEN mt.meta_value IS NULL THEN 'title'
                        WHEN md.meta_value IS NULL THEN 'description'
                    END as missing_type
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} mt ON p.ID = mt.post_id AND mt.meta_key = %s
                LEFT JOIN {$wpdb->postmeta} md ON p.ID = md.post_id AND md.meta_key = %s
                WHERE p.post_status = 'publish' 
                AND p.post_type IN ('post', 'page')
                AND (mt.meta_value IS NULL OR md.meta_value IS NULL)
                LIMIT 1000
            ", '_engineo_meta_title', '_engineo_meta_description'));
        }
        
        $posts_with_issues = count($missing_meta_results);
        $optimized_posts = $total_posts - $posts_with_issues;
        $health_score = $total_posts > 0 ? round(($optimized_posts / $total_posts) * 100) : 100;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('SEO Health Check', 'engineo'); ?></h1>
            
            <div class="card">
                <h2><?php esc_html_e('Health Score', 'engineo'); ?></h2>
                <div style="text-align: center; margin: 20px 0;">
                    <div style="font-size: 48px; font-weight: bold; color: <?php echo $health_score >= 80 ? '#00a32a' : ($health_score >= 60 ? '#dba617' : '#d63638'); ?>;">
                        <?php echo esc_html($health_score); ?>%
                    </div>
                    <p><?php esc_html_e('Overall SEO Health', 'engineo'); ?></p>
                    
                    <div style="margin-top: 20px; text-align: left; background: #f9f9f9; padding: 15px; border-radius: 5px;">
                        <h4><?php esc_html_e('How is the Health Score calculated?', 'engineo'); ?></h4>
                        <p><?php esc_html_e('The Health Score is based on the percentage of your content that meets SEO best practices:', 'engineo'); ?></p>
                        <ul>
                            <li><strong><?php esc_html_e('Total Posts/Pages:', 'engineo'); ?></strong> <?php echo esc_html($total_posts); ?></li>
                            <li><strong><?php esc_html_e('Posts with Issues:', 'engineo'); ?></strong> <?php echo esc_html($posts_with_issues); ?></li>
                            <li><strong><?php esc_html_e('Optimized Content:', 'engineo'); ?></strong> <?php echo esc_html($optimized_posts); ?></li>
                            <li><strong><?php esc_html_e('Issue Types Found:', 'engineo'); ?></strong> <?php echo esc_html(count($health_issues)); ?></li>
                        </ul>
                        <p><strong><?php esc_html_e('Formula:', 'engineo'); ?></strong> <?php esc_html_e('(Optimized Content ÷ Total Content) × 100 = Health Score', 'engineo'); ?></p>
                        
                        <div style="margin-top: 15px;">
                            <span style="color: #00a32a; font-weight: bold;">🟢 80-100%:</span> <?php esc_html_e('Excellent SEO health', 'engineo'); ?><br>
                            <span style="color: #dba617; font-weight: bold;">🟡 60-79%:</span> <?php esc_html_e('Good SEO health', 'engineo'); ?><br>
                            <span style="color: #d63638; font-weight: bold;">🔴 0-59%:</span> <?php esc_html_e('Needs improvement', 'engineo'); ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <h2><?php esc_html_e('Issues Found', 'engineo'); ?></h2>
                
                <?php if (empty($health_issues)): ?>
                    <div class="notice notice-success" style="padding: 20px; border-left: 4px solid #00a32a;">
                        <h3 style="margin-top: 0; color: #00a32a;">🎉 <?php esc_html_e('Excellent! No Issues Found', 'engineo'); ?></h3>
                        <p style="font-size: 16px; margin-bottom: 15px;"><strong><?php esc_html_e('Your website is perfectly optimized!', 'engineo'); ?></strong></p>
                        <p><?php esc_html_e('All your content meets SEO best practices:', 'engineo'); ?></p>
                        <ul style="margin: 10px 0;">
                            <li>✅ <?php esc_html_e('All posts and pages have meta titles', 'engineo'); ?></li>
                            <li>✅ <?php esc_html_e('All posts and pages have meta descriptions', 'engineo'); ?></li>
                            <li>✅ <?php esc_html_e('Site title and description are configured', 'engineo'); ?></li>
                            <li>✅ <?php esc_html_e('Robots.txt is accessible', 'engineo'); ?></li>
                            <li>✅ <?php esc_html_e('XML Sitemap is accessible', 'engineo'); ?></li>
                        </ul>
                        <p style="margin-top: 15px; font-style: italic; color: #666;">
                            <?php esc_html_e('Keep up the great work! Your site is ready for search engines and AI engines.', 'engineo'); ?>
                        </p>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 15px; margin: 20px 0; border-radius: 5px; border-left: 4px solid #17a2b8;">
                        <h4 style="margin-top: 0; color: #17a2b8;">ℹ️ <?php esc_html_e('What Issues Are Monitored?', 'engineo'); ?></h4>
                        <p><?php esc_html_e('This section monitors the following SEO and technical issues:', 'engineo'); ?></p>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-top: 10px;">
                            <div>
                                <h5 style="margin: 0 0 8px 0; color: #495057;"><?php esc_html_e('Content Issues', 'engineo'); ?></h5>
                                <ul style="margin: 0; padding-left: 20px; font-size: 14px;">
                                    <li><?php esc_html_e('Missing meta titles', 'engineo'); ?></li>
                                    <li><?php esc_html_e('Missing meta descriptions', 'engineo'); ?></li>
                                    <li><?php esc_html_e('Empty site title', 'engineo'); ?></li>
                                    <li><?php esc_html_e('Empty site description', 'engineo'); ?></li>
                                </ul>
                            </div>
                            <div>
                                <h5 style="margin: 0 0 8px 0; color: #495057;"><?php esc_html_e('Technical Issues', 'engineo'); ?></h5>
                                <ul style="margin: 0; padding-left: 20px; font-size: 14px;">
                                    <li><?php esc_html_e('Robots.txt not accessible', 'engineo'); ?></li>
                                    <li><?php esc_html_e('XML Sitemap not accessible', 'engineo'); ?></li>
                                    <li><?php esc_html_e('Missing redirects', 'engineo'); ?></li>
                                    <li><?php esc_html_e('Schema markup issues', 'engineo'); ?></li>
                                </ul>
                            </div>
                        </div>
                        <p style="margin: 15px 0 0 0; font-size: 14px; color: #6c757d;">
                            <em><?php esc_html_e('Issues are automatically detected and will appear here when found. Regular monitoring helps maintain optimal SEO performance.', 'engineo'); ?></em>
                        </p>
                    </div>
                <?php else: ?>
                    <div class="notice notice-warning">
                        <p><strong><?php esc_html_e('Issues to fix:', 'engineo'); ?></strong></p>
                        <ul>
                            <?php foreach ($health_issues as $issue): ?>
                            <li><?php echo esc_html($issue); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div style="background: #fff3cd; padding: 15px; margin: 20px 0; border-radius: 5px; border-left: 4px solid #ffc107;">
                        <h4 style="margin-top: 0; color: #856404;">💡 <?php esc_html_e('How to Fix These Issues', 'engineo'); ?></h4>
                        <p style="margin-bottom: 10px;"><?php esc_html_e('Here are some quick solutions for common issues:', 'engineo'); ?></p>
                        <ul style="margin: 0; padding-left: 20px; font-size: 14px;">
                            <li><strong><?php esc_html_e('Missing meta titles/descriptions:', 'engineo'); ?></strong> <?php esc_html_e('Edit your posts and pages to add SEO meta data using the Engineo meta boxes.', 'engineo'); ?></li>
                            <li><strong><?php esc_html_e('Empty site title/description:', 'engineo'); ?></strong> <?php esc_html_e('Go to Settings → General and fill in your site title and tagline.', 'engineo'); ?></li>
                            <li><strong><?php esc_html_e('Robots.txt/Sitemap issues:', 'engineo'); ?></strong> <?php esc_html_e('Check the Robots.txt and XML Sitemap pages in this plugin to ensure they are working correctly.', 'engineo'); ?></li>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>

            <div class="card">
                <h2><?php esc_html_e('AI Readiness - Rich Snippet Preview', 'engineo'); ?></h2>
                <p><?php esc_html_e('Your content is optimized for AI engines with:', 'engineo'); ?></p>
                <ul>
                    <li>✅ <strong><?php esc_html_e('llms.txt', 'engineo'); ?></strong> - <?php esc_html_e('AI engine optimization file', 'engineo'); ?></li>
                    <li>✅ <strong><?php esc_html_e('Structured Data', 'engineo'); ?></strong> - <?php esc_html_e('Schema markup for better understanding', 'engineo'); ?></li>
                    <li>✅ <strong><?php esc_html_e('Clean URLs', 'engineo'); ?></strong> - <?php esc_html_e('SEO-friendly permalinks', 'engineo'); ?></li>
                    <li>✅ <strong><?php esc_html_e('Meta Descriptions', 'engineo'); ?></strong> - <?php esc_html_e('Content summaries for AI', 'engineo'); ?></li>
                </ul>
                
                <h3><?php esc_html_e('AI Engine URLs', 'engineo'); ?></h3>
                <ul>
                    <li><strong><?php esc_html_e('llms.txt:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/llms.txt')); ?>" target="_blank"><?php echo esc_url(home_url('/llms.txt')); ?></a></li>
                    <li><strong><?php esc_html_e('XML Sitemap:', 'engineo'); ?></strong> <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank"><?php echo esc_url(home_url('/sitemap.xml')); ?></a></li>
                </ul>
            </div>
        </div>
        <?php
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

// Initialize the plugin
function engineo() {
    return Engineo::instance();
}

// Start the plugin
engineo();
