<?php
/**
 * EngageBay LearnDash Settings Section.
 *
 * @since 1.0.0
 * @package EngageBayLearnDash/includes/settings
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ( class_exists( 'LearnDash_Settings_Section' ) ) && ( ! class_exists( 'EngageBay_LearnDash_Settings_Section' ) ) ) {

	/**
	 * Class EngageBay Settings Section
	 *
	 * @since 1.0.0
	 */
	class EngageBay_LearnDash_Settings_Section extends LearnDash_Settings_Section {

		/**
		 * Protected constructor for class
		 *
		 * @since 3.6.0
		 */
		protected function __construct() {
			$this->settings_page_id = 'learndash_lms_engagebay';

			// This is the 'option_name' key used in the wp_options table.
			$this->setting_option_key = 'learndash_settings_engagebay';

			// This is the HTML form field prefix used.
			$this->setting_field_prefix = 'learndash_settings_engagebay';

			// Used within the Settings API to uniquely identify this section.
			$this->settings_section_key = 'settings_learndash_engagebay';

			// Section label/header.
			$this->settings_section_label = esc_html__( 'EngageBay Settings', 'engagebay-add-on-for-learndash' );

			parent::__construct();
		}

		/**
		 * Initialize the metabox settings values.
		 *
		 * @since 1.0.0
		 */
		public function load_settings_values() {
			parent::load_settings_values();

			if ( ! isset( $this->setting_option_values['rest_api_key'] ) ) {
				$this->setting_option_values['rest_api_key'] = '';
			}
		}

		/**
		 * Initialize the metabox settings fields.
		 *
		 * @since 1.0.0
		 */
		public function load_settings_fields() {
			$this->setting_option_fields = array();

			$this->setting_option_fields['rest_api_key'] = array(
				'name'        => 'rest_api_key',
				'label'       => esc_html__( 'REST API Key', 'engagebay-add-on-for-learndash' ),
				'type'        => 'password',
				'help_text'   => esc_html__( 'You can find this in your EngageBay account under Account Settings → API & Tracking Code', 'engagebay-add-on-for-learndash' ),
				'value'       => $this->setting_option_values['rest_api_key'],
				'placeholder' => esc_html__( 'Enter your REST API Key', 'engagebay-add-on-for-learndash' ),
			);

			$this->setting_option_fields = apply_filters( 'learndash_settings_fields', $this->setting_option_fields, $this->settings_section_key );

			parent::load_settings_fields();
		}

		public function settings_section_fields_validate( $post_fields = array() ) {

			if ( ! empty( $post_fields['rest_api_key'] ) ) {
				$api_key = sanitize_text_field( $post_fields['rest_api_key'] );

				if ( ! $this->is_valid_rest_api_key( $api_key ) ) {
					add_settings_error( 'learndash_settings_engagebay', 'invalid_api_key', __( 'Invalid EngageBay API Key. Please check and try again.', 'engagebay-add-on-for-learndash' ) );
					// Prevent saving the invalid key.
					unset( $post_fields['rest_api_key'] );
				} else {
//					if (!EngageBay_LearnDash_Settings_Section::triggered_install_event()) {
//						update_option('learndash_engaegbay_installed', true);
//						EngageBay_LearnDash_Api::trigger_event('plugin.installed', [
//							'site_url' => site_url()
//						]);
//					}
					update_option( 'learndash_engaegbay_installed', true );
					EngageBay_LearnDash_Api::trigger_event( 'plugin.installed', [
						'rest_api_key' => $post_fields['rest_api_key'],
						'site_url' => site_url()
					] );
				}
			} else {
				add_settings_error( 'learndash_settings_engagebay', 'blank_api_key', __( 'REST API Key is required', 'engagebay-add-on-for-learndash' ) );
			}

			return $post_fields;
		}

		public static function rest_api_key() {
			$settings = get_option( 'learndash_settings_engagebay' );
			if ( $settings && is_array( $settings ) && array_key_exists( 'rest_api_key', $settings ) ) {
				return $settings['rest_api_key'];
			}

			return false;
		}

		public static function triggered_install_event() {
			return get_option( 'learndash_engaegbay_installed' );
		}

		public function is_valid_rest_api_key( $rest_api_key ) {
			$engagebay_api = new EngageBay_LearnDash_Api();
			if ( $engagebay_api->validate_rest_api_key( $rest_api_key ) ) {
				return true;
			}

			return false;
		}

		// End of functions.
	}
}
add_action(
	'learndash_settings_sections_init',
	function () {
		EngageBay_LearnDash_Settings_Section::add_section_instance();
	}
);
