<?php
/**
 * EngageBay LearnDash Preferences Section.
 *
 * @since 1.0.0
 * @package EngageBayLearnDash/includes/settings
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ( class_exists( 'LearnDash_Settings_Section' ) ) && ( ! class_exists( 'EngageBay_LearnDash_Preferences_Section' ) ) ) {

	/**
	 * Class EngageBay Preferences Section
	 *
	 * @since 1.0.0
	 */
	class EngageBay_LearnDash_Preferences_Section extends LearnDash_Settings_Section {

		public const USER_CREATED = 'user.created';
		public const USER_UPDATED = 'user.updated';
		public const NEW_ENROLMENT = 'course.enrolled';
		public const COURSE_COMPLETED = 'course.completed';
		public const LESSON_COMPLETED = 'lesson.completed';
		public const TOPIC_COMPLETED = 'topic.completed';
		public const QUIZ_COMPLETED = 'quiz.completed';
		public const ESSAY_SUBMITTED = 'essay.submitted';
		public const COURSE_PURCHASED = 'course.purchased';

		/**
		 * Protected constructor for class
		 *
		 * @since 3.6.0
		 */
		protected function __construct() {

			if ( ! EngageBay_LearnDash_Settings_Section::rest_api_key() ) {
				return;
			}

			$this->settings_page_id = 'learndash_lms_engagebay';

			// This is the 'option_name' key used in the wp_options table.
			$this->setting_option_key = 'learndash_engagebay_preferences';

			// This is the HTML form field prefix used.
			$this->setting_field_prefix = 'learndash_engagebay_preferences';

			// Add EngageBay Preferences Section.
			$this->settings_section_label = esc_html__( 'Preferences', 'engagebay-add-on-for-learndash' );

			parent::__construct();
		}

//		public function add_meta_boxes( $settings_screen_id = '' ) {
//			parent::add_meta_boxes( $settings_screen_id );
//
//			if ( $settings_screen_id === $this->settings_screen_id ) {
//				add_meta_box(
//					'testing-meta-id',
//					'Email Marketing',
//					array( $this, 'show_meta_box' ),
//					$this->settings_screen_id,
//					'',
//					$this->metabox_priority
//				);
//
//				add_filter( 'postbox_classes_' . $this->settings_screen_id . '_' . 'testing-meta-id', array(
//					$this,
//					'add_meta_box_classes'
//				), 30, 1 );
//			}
//		}
//
//		public function render_email_marketing_meta_box() {
//			echo '<p><a href="https://example.com" target="_blank" class="button button-primary">Go to Email Marketing Settings</a></p>';
//		}

		/**
		 * Initialize the metabox settings fields.
		 *
		 * @since 1.0.0
		 */
		public function load_settings_fields() {
			if ( ! EngageBay_LearnDash_Settings_Section::rest_api_key() ) {
				return;
			}

			$this->setting_option_fields = array();

			// EngageBay Preferences.
			$this->setting_option_fields['engagebay_preferences'] = array(
				'name'             => 'engagebay_preferences',
				'type'             => 'custom',
				'label'            => '',
				'display_callback' => array( $this, 'engagebay_preferences_callback' ),
			);

			parent::load_settings_fields();
		}

		/**
		 * Render EngageBay Preferences section.
		 */
		public function engagebay_preferences_callback() {
			$options = $this->get_event_options();

			$contact_fields = EngageBay_LearnDash_Api::get_fields( 'contact' );
			$deal_fields    = EngageBay_LearnDash_Api::get_fields( 'deal' );

			$label_width = null;
			echo '<div class="engagebay-ld-preferences-wrapper">';
			foreach ( $options as $group_key => $group ) {
				echo '<div class="eb-setting-row eb-setting-separator">';
				echo '<div class="engagebay-ld-preferences-actions"><strong>' . esc_html( $group['label'] ) . '</strong></div>';
				echo '<div class="engagebay-ld-preferences-action-group">';
				foreach ( $group['values'] as $key => $label ) {
					if ( $key == 'update_contact_field' || $key == 'create_tag' || $key == 'create_task' || $key == 'create_contact_and_deal' ) {
						switch ( $group_key ) {
							case self::NEW_ENROLMENT:
							case self::COURSE_COMPLETED:
							case self::QUIZ_COMPLETED:
							case self::LESSON_COMPLETED:
								$flex_style = 'eb-setting-form-check-inline';
								break;
							default:
								$flex_style = '';
						}
					} else if ( $key === 'update_deal_field' && $group_key === self::ESSAY_SUBMITTED ) {
						$flex_style = 'eb-setting-form-check-inline';
					} else {
						$flex_style = '';
					}

					switch ($group_key) {
						case self::COURSE_COMPLETED:
						case self::QUIZ_COMPLETED:
						case self::ESSAY_SUBMITTED:
							if ($key === 'update_contact_field' || $key === 'update_deal_field') {
								$label_width = 'width-30';
							} else {
								$label_width = null;
							}
							break;
					}

					echo '<div class="' . esc_attr( $flex_style ) . '">';
					$checked = checked( $this->is_preference_enabled( $group_key, $key ), true, false );
					if ( $group_key === self::NEW_ENROLMENT && $key === 'create_contact_and_deal' ) {
						$checked = checked( true, true, false );
					}
					echo '<label class="' . esc_attr( $label_width ) . '"><input id="' . esc_attr( str_replace( '.', '_', $group_key ) . '_' . $key ) . '" type="checkbox" name="' . esc_attr( $this->setting_field_prefix ) . '[engagebay_preferences][' . esc_attr( $group_key ) . '][]" value="' . esc_attr( $key ) . '" ' . esc_attr( $checked ) . '> ' . esc_html( $label ) . '</label>';
					$formatted_key = str_replace( '.', '_', $group_key );
					if ( $key == 'create_note' ) {
						switch ( $group_key ) {
							case self::NEW_ENROLMENT:
								$value = $this->get_setting_value( 'course_enrolled_note' );
								echo '<textarea name="learndash_engagebay_preferences[engagebay_preferences][course_enrolled_note]" style="margin-top: 4px;" id="note_' . esc_attr( $formatted_key . '_' . $key ) . '" placeholder="{course_title} is enrolled">';
								if ( $value ) {
									echo esc_html( $value );
								}
								echo '</textarea>';
								break;
							case self::COURSE_COMPLETED:
								$value = $this->get_setting_value( 'course_completed_note' );
								echo '<textarea name="learndash_engagebay_preferences[engagebay_preferences][course_completed_note]" style="margin-top: 4px" id="note_' . esc_attr( $formatted_key . '_' . $key ) . '" placeholder="{course_title} is completed">';
								if ( $value ) {
									echo esc_html( $value );
								}
								echo '</textarea>';
								break;
							case self::TOPIC_COMPLETED:
								$value = $this->get_setting_value( 'topic_completed_note' );
								echo '<textarea name="learndash_engagebay_preferences[engagebay_preferences][topic_completed_note]" style="margin-top: 4px" id="note_' . esc_attr( $formatted_key . '_' . $key ) . '" placeholder="{topic_name} is completed">';
								if ( $value ) {
									echo esc_html( $value );
								}
								echo '</textarea>';
								break;
						}
					} else if ( $key == 'update_contact_field' ) {
						switch ( $group_key ) {
							case self::COURSE_COMPLETED:
								$value      = $this->get_setting_value( 'course_completed_contact_field' );
								$name       = "learndash_engagebay_preferences[engagebay_preferences][course_completed_contact_field]";
								$input_name = "learndash_engagebay_preferences[engagebay_preferences][course_completed_contact_field_value]";;
								$hint_name         = 'Supports placeholders: {course_title}';
								$input_placeholder = 'Course {course_title} completed';
								$input_value       = $this->get_setting_value( 'course_completed_contact_field_value' );
								break;
							case self::QUIZ_COMPLETED:
								$value             = $this->get_setting_value( 'quiz_completed_contact_field' );
								$name              = "learndash_engagebay_preferences[engagebay_preferences][quiz_completed_contact_field]";
								$input_name        = "learndash_engagebay_preferences[engagebay_preferences][quiz_completed_contact_field_value]";
								$hint_name         = 'Supports placeholders: {course_title},{lesson_title},{quiz_title},{quiz_score},{quiz_pass},{quiz_percentage},{quiz_started},{quiz_completed},{quiz_url}';
								$input_placeholder = 'Quiz {quiz_title} completed';
								$input_value       = $this->get_setting_value( 'quiz_completed_contact_field_value' );
								break;
							default:
								$value             = null;
								$input_name        = '';
								$input_value       = '';
								$name              = '';
								$hint_name         = '';
								$input_placeholder = '';
						}
						echo '<div class="eb-setting-form-check-fields contact_select_' . esc_attr( $formatted_key . '_' . $key ) . '">';
						echo '<select class="eb-setting-form-select" name=' . esc_attr( $name ) . ' id="contact_deal_' . esc_attr( $formatted_key . '_' . $key ) . '">';
						echo '<option>Select</option>';
						foreach ( $contact_fields as $field ) {
							$field_value = $field['id'] ?? $field['name'];
							$selected    = $value == $field_value ? 'selected' : null;
							echo '<option ' . esc_attr( $selected ) . ' value="' . esc_attr( $field_value ) . '">' . esc_html( $field['label'] ) . '</option>';
						}
						echo '</select> <span class="eb-setting-as-text">as</span>';
						echo '<div class="eb-setting-support-group">';
						echo '<input class="eb-setting-form-control engagebay-ld-input-text" type="text" name="' . esc_attr( $input_name ) . '" placeholder="' . esc_attr( $input_placeholder ) . '" value="' . esc_attr( $input_value ) . '">';
						echo '<small class="engagebay-ld-hint" id="contact_deal_' . esc_attr( $formatted_key . '_' . $key ) . '_hint">' . esc_html( $hint_name ) . '</small>';
						echo '</div>';
						echo '</div>';
					} else if ( $key === 'update_deal_field' && $group_key === self::ESSAY_SUBMITTED ) {
						$name              = 'learndash_engagebay_preferences[engagebay_preferences][essay_submitted_deal_field]';
						$input_name        = 'learndash_engagebay_preferences[engagebay_preferences][essay_submitted_deal_field_value]';
						$hint_name         = 'Supported placeholders: {course_title},{course_url},{essay_title},{essay_content},{essay_url}';
						$input_placeholder = 'Essay {essay_title} submitted';
						echo '<div class="eb-setting-form-check-fields contact_select_' . esc_attr( $formatted_key . '_' . $key ) . '">';
						echo '<select class="eb-setting-form-select" name=' . esc_attr( $name ) . ' id="contact_deal_' . esc_attr( $formatted_key . '_' . $key ) . '">';
						echo '<option>Select</option>';
						$value       = $this->get_setting_value( 'essay_submitted_deal_field' );
						$input_value = $this->get_setting_value( 'essay_submitted_deal_field_value' );
						foreach ( $deal_fields as $field ) {
							$field_value = $field['id'] ?? $field['name'];
							$selected    = $value == $field_value ? 'selected' : null;
							echo '<option ' . esc_attr( $selected ) . ' value="' . esc_attr( $field_value ) . '">' . esc_html( $field['label'] ) . '</option>';
						}
						echo '</select> <span class="eb-setting-as-text">as</span>';
						echo '<div class="eb-setting-support-group">';
						echo '<input class="eb-setting-form-control engagebay-ld-input-text" type="text" name="' . esc_attr( $input_name ) . '" placeholder="' . esc_attr( $input_placeholder ) . '" value="' . esc_attr( $input_value ) . '">';
						echo '<small class="engagebay-ld-hint" id="contact_deal_' . esc_attr( $formatted_key . '_' . $key ) . '_hint">' . esc_html( $hint_name ) . '</small>';
						echo '</div>';
						echo '</div>';
					} else if ( $key === 'create_task' ) {
						echo '<small class="engagebay-ld-hint" id="create_task_' . esc_attr( $formatted_key . '_' . $key ) . '_hint">(Task name will be {lesson_name}_{email})</small>';
					} else if ( $key === 'create_tag' ) {
						echo '<small class="engagebay-ld-hint" id="create_tag_' . esc_attr( $formatted_key . '_' . $key ) . '_hint">(Tag will be course_completed_{course_title})</small>';
					}
					echo '</div>';
				}
				echo '</div>';
				echo '</div>';
			}
			echo '</div>';
		}

		private function get_event_options(): array {
			return array(
				self::NEW_ENROLMENT    => array(
					'label'  => __( 'New Enrollment', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'create_contact_and_deal' => __( 'Create a contact and deal', 'engagebay-add-on-for-learndash' ),
						'create_note'             => __( 'Create a course note', 'engagebay-add-on-for-learndash' )
					),
				),
				self::COURSE_COMPLETED => array(
					'label'  => __( 'Course Completed', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'create_tag'           => __( 'Create a tag', 'engagebay-add-on-for-learndash' ),
						'update_contact_field' => __( 'Update contact field', 'engagebay-add-on-for-learndash' ),
						'create_note'          => __( 'Create a course note', 'engagebay-add-on-for-learndash' ),
					),
				),
				self::LESSON_COMPLETED => array(
					'label'  => __( 'Lesson Completed', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'create_task' => __( 'Create a task', 'engagebay-add-on-for-learndash' )
					),
				),
				self::TOPIC_COMPLETED  => array(
					'label'  => __( 'Topic Completed', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'create_note' => __( 'Create a topic note', 'engagebay-add-on-for-learndash' )
					),
				),
				self::QUIZ_COMPLETED   => array(
					'label'  => __( 'Quiz Completed', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'update_contact_field' => __( 'Update contact field', 'engagebay-add-on-for-learndash' )
					),
				),
				self::ESSAY_SUBMITTED  => array(
					'label'  => __( 'Essay Submitted', 'engagebay-add-on-for-learndash' ),
					'values' => array(
						'update_deal_field' => __( 'Update deal field', 'engagebay-add-on-for-learndash' ),
					),
				),
			);
		}

		/**
		 * Sanitize and validate EngageBay preferences.
		 *
		 * @param array $post_fields The settings to sanitize.
		 *
		 * @return array
		 */
		public function settings_section_fields_validate( $post_fields = array() ) {
			if ( isset( $post_fields['engagebay_preferences'] ) && is_array( $post_fields['engagebay_preferences'] ) ) {
				$sanitized_preferences = array();
				foreach ( $post_fields['engagebay_preferences'] as $group_key => $group ) {
					if ( is_array( $group ) ) {
						// Sanitize each checkbox value
						$sanitized_group                     = array_map( 'sanitize_text_field', $group );
						$sanitized_preferences[ $group_key ] = array_values( $sanitized_group );
					} else {
						$sanitized_preferences[ $group_key ] = $group;
					}
				}
				$post_fields['engagebay_preferences'] = $sanitized_preferences;
			} else {
				$post_fields['engagebay_preferences'] = array();
			}

			EngageBay_LearnDash_Api::trigger_event( 'preferences.updated', $post_fields['engagebay_preferences'] );

			return $post_fields;
		}

		public static function is_preference_enabled( $event, $key ) {
			$preferences = get_option( 'learndash_engagebay_preferences' );
			if ( $preferences && is_array( $preferences ) ) {
				if ( isset( $preferences['engagebay_preferences'][ $event ] ) ) {
					return in_array( $key, $preferences['engagebay_preferences'][ $event ] );
				}
			}

			// Default to checked ONLY if no preferences exist yet
			if ( ! $preferences && $event === self::NEW_ENROLMENT && $key === 'create_contact' ) {
				return true;
			}

			return false;
		}

		public static function get_actions_for_event( $event ) {
			$preferences = get_option( 'learndash_engagebay_preferences' );
			if ( $preferences && is_array( $preferences ) && isset( $preferences['engagebay_preferences'][ $event ] ) ) {
				return $preferences['engagebay_preferences'][ $event ];
			}

			return [];
		}

		public static function get_setting_value( $setting_name ) {
			$preferences = get_option( 'learndash_engagebay_preferences' );
			if ( $preferences && is_array( $preferences ) && isset( $preferences['engagebay_preferences'][ $setting_name ] ) ) {
				return $preferences['engagebay_preferences'][ $setting_name ];
			}

			return null;
		}
	}
}
add_action(
	'learndash_settings_sections_init',
	function () {
		EngageBay_LearnDash_Preferences_Section::add_section_instance();
	}
);
