<?php
/**
 * The core plugin class.
 *
 * This is used to define admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    EngageBayLearnDash
 * @subpackage EngageBayLearnDash/includes
 * @author     EngageBay <support@engagebay.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit( 'You are not allowed to access this file directly.' );
}

class EngageBay_LearnDash {
	private static $instance;
	private EngageBay_LearnDash_Admin $engagebay_learndash_admin;
	private EngageBay_LearnDash_User $engagebay_learndash_user;
	private EngageBay_LearnDash_Course $engagebay_learndash_course;
	private EngageBay_LearnDash_Quiz $engage_bay_learn_dash_quiz;
	private EngageBay_LearnDash_Group $engagebay_learndash_group;
	private EngageBay_LearnDash_Rest_Api $engagebay_learndash_rest_api;
	private EngageBay_LearnDash_Payment_Gateway $engagebay_learndash_payment_gateway;

	public function __construct() {
		$this->engagebay_learndash_admin    = new EngageBay_LearnDash_Admin();
		$this->engagebay_learndash_user     = new EngageBay_LearnDash_User();
		$this->engagebay_learndash_course   = new EngageBay_LearnDash_Course();
		$this->engage_bay_learn_dash_quiz   = new EngageBay_LearnDash_Quiz();
		$this->engagebay_learndash_group    = new EngageBay_LearnDash_Group();
		$this->engagebay_learndash_rest_api = new EngageBay_LearnDash_Rest_Api();
		$this->engagebay_learndash_payment_gateway = new EngageBay_LearnDash_Payment_Gateway();
		$this->define_admin_hooks();
		$this->define_public_hooks();
	}

	/**
	 * Main EngageBayLearnDash Instance
	 *
	 * Ensures only one instance of EngageBayLearnDash is loaded or can be loaded.
	 *
	 * @return EngageBay_LearnDash - Main instance
	 * @since 1.0.0
	 * @static
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function define_admin_hooks() {
		add_action( 'admin_enqueue_scripts', array($this->engagebay_learndash_admin, 'engagebay_learndash_enqueue_admin_styles' ));
		add_action( 'admin_enqueue_scripts', array($this->engagebay_learndash_admin, 'engagebay_learndash_enqueue_admin_scripts' ));
	}

	public function define_public_hooks() {
		/**
		 * Triggers when:
		 *
		 * A user registers via the WordPress registration form (wp-login.php?action=register).
		 * An admin manually adds a new user in the WordPress dashboard.
		 * A plugin creates a new user programmatically, such as WooCommerce, LearnDash, or a membership plugin.
		 * A third-party service (like an API or SSO) creates a new user in WordPress.
		 */
		add_action( 'user_register', array( $this->engagebay_learndash_user, 'on_user_registration' ) );
		add_action('profile_update', array( $this->engagebay_learndash_user, 'on_user_updated' ), 10, 2);
		add_action( 'learndash_update_course_access', array(
			$this->engagebay_learndash_course,
			'on_courses_updated'
		), 10, 4 );

		/**
		 * Triggers when:
		 *
		 * A user completes all required lessons, topics, and quizzes in a course.
		 * The course is marked as complete manually by an admin.
		 * A custom script or API marks the course as completed for a user
		 */
		add_action( 'learndash_course_completed', array(
			$this->engagebay_learndash_course,
			'on_course_completed'
		), 10, 2 );

		/**
		 * Triggers when:
		 *
		 * A user marks a lesson as complete.
		 * The last topic inside a lesson is completed, and LearnDash auto-completes the lesson.
		 * An admin manually marks a lesson as complete for a user.
		 * A custom script or API call marks the lesson as complete.
		 */
		add_action( 'learndash_lesson_completed', array(
			$this->engagebay_learndash_course,
			'on_lesson_completed'
		), 10, 1 );

		/**
		 * Triggers when:
		 *
		 * A user clicks "Mark Complete" on a topic.
		 * The last quiz inside a topic is completed, and LearnDash auto-completes the topic.
		 * An admin manually marks a topic as complete for a user.
		 * A custom script or API call marks the topic as complete.
		 */
		add_action( 'learndash_topic_completed', array(
			$this->engagebay_learndash_course,
			'on_topic_completed'
		), 10, 4 );


		add_action( 'learndash_quiz_completed', array(
			$this->engage_bay_learn_dash_quiz,
			'on_quiz_completed'
		), 10, 2 );

		add_action( 'learndash_new_essay_submitted', array(
			$this->engage_bay_learn_dash_quiz,
			'on_essay_submitted'
		), 10, 2 );

		/**
		 * Triggers when:
		 *
		 * An admin manually adds a user to a group.
		 * A membership plugin assigns a user to a group.
		 * A bulk import or API action assigns users.
		 */
		add_action( 'ld_added_group_access', array(
			$this->engagebay_learndash_group,
			'on_user_enroll_to_group'
		), 10, 2 );

		/**
		 * Triggers when:
		 *
		 * Admin manually removes a user from a LearnDash group in the WordPress dashboard.
		 * A membership plugin revokes group access (e.g., MemberPress, Paid Memberships Pro).
		 * A WooCommerce subscription expires or is canceled (if group access was granted via WooCommerce).
		 * A user is removed from a group via API or bulk user update.
		 * A scheduled event removes the user (e.g., time-based access expiration)
		 */
		add_action( 'ld_removed_group_access', array(
			$this->engagebay_learndash_group,
			'on_user_removed_from_group'
		), 10, 2 );

		add_action( 'rest_api_init', array(
			$this->engagebay_learndash_rest_api,
			'learndash_engagebay_register_routes'
		), 10, 0 );

		add_action('learndash_payment_gateway_event_processed', array(
			$this->engagebay_learndash_payment_gateway,
			'payment_processed',
		), 10, 3);
	}
}
