<?php
/**
 * EngageBay APIs
 *
 * @link       https://engagebay.com
 * @since      1.0.0
 *
 * @package    EngageBayLearnDash
 * @subpackage EngageBayLearnDash/includes/api
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit( 'You are not allowed to access this file directly.' );
}

class EngageBay_LearnDash_Api {

	private const BASE_API_URL = 'https://app.engagebay.com/dev/api/panel';

	public function validate_rest_api_key( $rest_api_key ) {
		$url     = self::BASE_API_URL . '/domain/info';
		$headers = [
			'Authorization' => $rest_api_key,
			'ebwhitelist'   => true,
			'Content-Type'  => 'application/json',
			'Accept'        => 'application/json'
		];

		$response      = wp_remote_get( $url, [ 'headers' => $headers ] );
		$response_code = wp_remote_retrieve_response_code( $response );

		if ( $response_code === 200 ) {
			$result = wp_remote_retrieve_body( $response );

			$default_preferences = [
				'engagebay_preferences' => [
					'course.enrolled' => [
						'create_contact_and_deal'
					]
				]
			];

			update_option( 'learndash_engagebay_preferences', $default_preferences );

			return json_decode( $result, true );
		}

		return false;
	}

	public static function trigger_event( $event, $payload, $name = null ) {

		$actions = EngageBay_LearnDash_Preferences_Section::get_actions_for_event( $event );

		$url = self::BASE_API_URL . '/learndash/event';

		$rest_api_key = EngageBay_LearnDash_Settings_Section::rest_api_key();
		if ( ! $rest_api_key && isset( $payload['rest_api_key'] ) ) {
			$rest_api_key = $payload['rest_api_key'];
		}

		$headers = [
			'Authorization' => $rest_api_key,
			'ebwhitelist'   => true,
			'Content-Type'  => 'application/json',
			'Accept'        => 'application/json'
		];

		if ( $event === EngageBay_LearnDash_Preferences_Section::USER_CREATED ) {
			$actions      = [ 'create_contact' ];
		} else if ( $event === EngageBay_LearnDash_Preferences_Section::USER_UPDATED ) {
			$actions      = [ 'update_contact' ];
		} else if ( $event === EngageBay_LearnDash_Preferences_Section::COURSE_COMPLETED
		            || $event === EngageBay_LearnDash_Preferences_Section::LESSON_COMPLETED
		            || $event === EngageBay_LearnDash_Preferences_Section::TOPIC_COMPLETED
		            || $event === EngageBay_LearnDash_Preferences_Section::QUIZ_COMPLETED
		            || $event === EngageBay_LearnDash_Preferences_Section::COURSE_PURCHASED ) {
			$actions = [ ...$actions, 'update_deal' ];
		}

		if ( $event === EngageBay_LearnDash_Preferences_Section::NEW_ENROLMENT ) {
			$actions = array_unique( [ ...$actions, 'create_contact_and_deal' ] );
		}

		$additionalInfo = self::prepare_additional_info( $event, $actions, $payload, $name );

		$current_user = wp_get_current_user();
		if (isset($payload['user_id']) && get_user($payload['user_id'])) {
			$current_user = get_user($payload['user_id']);
		}

		$payload = array_merge(
			$payload,
			[
				'user' => [
					'id'         => $current_user->ID,
					'first_name' => $current_user->first_name,
					'last_name'  => $current_user->last_name,
					'email'      => $current_user->user_email,
					'roles'      => $current_user->roles
				],
			] );

		$payload = array_merge(
			[ 'event' => $event ],
			$payload,
			[
				'actions'               => $actions,
				'additionalInformation' => $additionalInfo
			]
		);

		$response = wp_remote_post( $url, [
			'headers' => $headers,
			'body'    => json_encode( $payload )
		] );

//		$response_code = wp_remote_retrieve_response_code( $response );
		$result        = wp_remote_retrieve_body( $response );
//		if ( $response_code !== 200 ) {
//			file_put_contents( "error.json", $result, FILE_APPEND );
//		}
//
//		file_put_contents( "response.json", json_encode( [
//			'url'         => $url,
//			'payload'     => $payload,
//			'headers'     => $headers,
//			'status_code' => $response_code,
//			'result'      => $result
//		], JSON_PRETTY_PRINT ), FILE_APPEND );
	}

	public static function prepare_additional_info( string $event, array $actions, array $payload, ?string $name = null ): ?array {
		$additional_info = null;
		switch ( $event ) {
			case EngageBay_LearnDash_Preferences_Section::NEW_ENROLMENT:
				if ( in_array( 'create_note', $actions ) ) {
					$note = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'course_enrolled_note' );
					if ( $name ) {
						$note = str_replace( '{course_title}', $name, $note );
					}
					$additional_info['note'] = html_entity_decode( $note );
				}
				break;
			case EngageBay_LearnDash_Preferences_Section::COURSE_COMPLETED:
				if ( in_array( 'create_note', $actions ) ) {
					$note = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'course_completed_note' );
					if ( $name ) {
						$note = str_replace( '{course_title}', $name, $note );
					}
					$additional_info['note'] = html_entity_decode( $note );;
				}
				if ( in_array( 'update_contact_field', $actions ) ) {
					$additional_info['contact_field']       = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'course_completed_contact_field' );
					$additional_info['contact_field_value'] = self::get_field_value_from_payload( $payload, EngageBay_LearnDash_Preferences_Section::get_setting_value( 'course_completed_contact_field_value' ) );
				}
				break;
			case EngageBay_LearnDash_Preferences_Section::TOPIC_COMPLETED:
				if ( in_array( 'create_note', $actions ) ) {
					$note = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'topic_completed_note' );
					if ( $name ) {
						$note = str_replace( '{topic_title}', $name, $note );
						$note = str_replace( '{topic_name}', $name, $note );
					}
					$additional_info['note'] = html_entity_decode( $note );;
				}
				break;
			case EngageBay_LearnDash_Preferences_Section::QUIZ_COMPLETED:
				if ( in_array( 'update_contact_field', $actions ) ) {
					$additional_info['contact_field']       = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'quiz_completed_contact_field' );
					$additional_info['contact_field_value'] = self::get_field_value_from_payload( $payload, EngageBay_LearnDash_Preferences_Section::get_setting_value( 'quiz_completed_contact_field_value' ) );
				}

				break;
			case EngageBay_LearnDash_Preferences_Section::ESSAY_SUBMITTED:
				if ( in_array( 'update_deal_field', $actions ) ) {
					$additional_info['deal_field']       = EngageBay_LearnDash_Preferences_Section::get_setting_value( 'essay_submitted_deal_field' );
					$additional_info['deal_field_value'] = self::get_field_value_from_payload( $payload, EngageBay_LearnDash_Preferences_Section::get_setting_value( 'essay_submitted_deal_field_value' ) );
				}
				break;
		}

		return $additional_info;
	}

	public static function get_field_value_from_payload( array $payload, string $value ) {
		preg_match_all( '/\{(.*?)\}/', $value, $matches );
		foreach ( $matches[1] as $match ) {
			if ( str_contains( $match, '_' ) ) {
				[ $key, $property ] = explode( '_', $match );
				if ( array_key_exists( $key, $payload ) && is_array( $payload[ $key ] ) && array_key_exists( $property, $payload[ $key ] ) ) {
					$value = html_entity_decode( str_replace( '{' . $match . '}', $payload[ $key ][ $property ], $value ) );
				}
			}
		}

		return $value;
	}

	public static function get_fields( string $type = 'contact' | 'deal' ) {
		$url     = self::BASE_API_URL . '/learndash/field-info?type=' . $type;
		$headers = [
			'Authorization' => EngageBay_LearnDash_Settings_Section::rest_api_key(),
			'ebwhitelist'   => true,
			'Content-Type'  => 'application/json',
			'Accept'        => 'application/json'
		];

		$response = wp_remote_request( $url, [ 'headers' => $headers ] );

//		$response_code = wp_remote_retrieve_response_code( $response );
		$result        = wp_remote_retrieve_body( $response );
//		if ( $response_code !== 200 ) {
//			file_put_contents( "error_fields.json", $result, FILE_APPEND );
//		}

		return json_decode( $result, true );
	}
}
