<?php
/*
Plugin Name: Email Marketing Services - Integration for Jetpack CRM
Plugin URI: https://wordpress.org/plugins/ems-jetpack-crm/
Description: Import all of your contacts from popular newsletter services to Jetpack CRM.
Version: 1.0.0
Author: Jahidur Nadim
Author URI: https://github.com/nadim1992
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: ems-jetpack-crm
Domain Path: /languages
*/

// Do not call this file directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * EMS_Jetpack_CRM class
 *
 * @class EMS_Jetpack_CRM The class that holds the entire EMS_Jetpack_CRM plugin
 */
final class EMS_Jetpack_CRM {

    /**
     * Plugin version.
     *
     * @var string
     */
    public $version = '1.0.0';

    /**
     * Holds various class instances.
     *
     * @var array
     */
    private $container = [];

    /**
     * Constructor for the EMS_Jetpack_CRM class.
     *
     * Sets up all the appropriate hooks and actions
     * within our plugin.
     */
    public function __construct() {
        $this->define_constants();

        register_activation_hook( EMS_JETPACK_CRM_FILE, [ $this, 'activate' ] );
        register_deactivation_hook( EMS_JETPACK_CRM_FILE, [ $this, 'deactivate' ] );

        require_once EMS_JETPACK_CRM_PATH . '/vendor/woocommerce/action-scheduler/action-scheduler.php';

        add_action( 'plugins_loaded', [ $this, 'init_plugin' ] );
    }

    /**
     * Initializes the EMS_Jetpack_CRM() class.
     *
     * Checks for an existing EMS_Jetpack_CRM() instance
     * and if it doesn't find one, creates it.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new EMS_Jetpack_CRM();
        }

        return $instance;
    }

    /**
     * Magic getter to bypass referencing plugin.
     *
     * @param $prop
     *
     * @return mixed
     */
    public function __get( $prop ) {
        if ( array_key_exists( $prop, $this->container ) ) {
            return $this->container[ $prop ];
        }

        return $this->{$prop};
    }

    /**
     * Magic isset to bypass referencing plugin.
     *
     * @param $prop
     *
     * @return mixed
     */
    public function __isset( $prop ) {
        return isset( $this->{$prop} ) || isset( $this->container[ $prop ] );
    }

    /**
     * Define the constants.
     *
     * @return void
     */
    public function define_constants() {
        define( 'EMS_JETPACK_CRM_VERSION', $this->version );
        define( 'EMS_JETPACK_CRM_FILE', __FILE__ );
        define( 'EMS_JETPACK_CRM_PATH', dirname( EMS_JETPACK_CRM_FILE ) );
        define( 'EMS_JETPACK_CRM_INCLUDES', EMS_JETPACK_CRM_PATH . '/includes' );
        define( 'EMS_JETPACK_CRM_URL', plugins_url( '', EMS_JETPACK_CRM_FILE ) );
        define( 'EMS_JETPACK_CRM_ASSETS', EMS_JETPACK_CRM_URL . '/assets' );
    }

    /**
     * Load the plugin after all plugis are loaded.
     *
     * @return void
     */
    public function init_plugin() {
        $this->includes();
        $this->init_hooks();

        do_action( 'ems_jetpack_crm_loaded' );
    }

    /**
     * Used for activation function.
     *
     * Nothing being called here yet.
     */
    public function activate() {
        $installed = get_option( 'ems_jetpack_crm_installed' );

        if ( ! $installed ) {
            update_option( 'ems_jetpack_crm_installed', time() );
        }

        update_option( 'ems_jetpack_crm_version', EMS_JETPACK_CRM_VERSION );
    }

    /**
     * Used for deactivation function.
     *
     * Nothing being called here yet.
     */
    public function deactivate() {}

    /**
     * Include the required files.
     *
     * @return void
     */
    public function includes() {
        require_once EMS_JETPACK_CRM_PATH . '/vendor/autoload.php';

        do_action( 'ems_jetpack_crm_include_services' );

        require_once EMS_JETPACK_CRM_INCLUDES . '/services/sendinblue/Sendinblue.php';
        require_once EMS_JETPACK_CRM_INCLUDES . '/services/services-base.php';

        require_once EMS_JETPACK_CRM_INCLUDES . '/Assets.php';

        if ( $this->is_request( 'admin' ) ) {
            require_once EMS_JETPACK_CRM_INCLUDES . '/Admin.php';
        }
    }

    /**
     * Initialize the hooks.
     *
     * @return void
     */
    public function init_hooks() {
        add_action( 'init', [ $this, 'init_classes' ] );
        add_action( 'admin_init', [ $this, 'check_if_jetpack_crm_active' ] );

        // Localize our plugin
        add_action( 'init', [ $this, 'localization_setup' ] );
    }

    /**
     * Instantiate the required classes
     *
     * @return void
     */
    public function init_classes() {
        if ( $this->is_request( 'admin' ) ) {
            $this->container['admin'] = new EMS\Admin();
        }

        $this->container['assets'] = new EMS\Assets();
    }

    /**
    * Check Jetpack CRM active or not.
    *
    * @return void
    */
    public function check_if_jetpack_crm_active() {
        if ( ! isset( $_GET['page'] ) || 'ems-jetpack-crm' !== $_GET['page'] ) {
            return;
        }

        if ( is_plugin_inactive( 'zero-bs-crm/ZeroBSCRM.php' ) ) {
            add_action( 'admin_notices', function() { ?>
                <div class="notice notice-error">
                    <p>Please first activate Jetpack CRM plugin.</p>
                </div>
                <?php exit;
            } );
        }
    }

    /**
     * Initialize plugin for localization.
     *
     * @uses load_plugin_textdomain()
     */
    public function localization_setup() {
        load_plugin_textdomain( 'ems-jetpack-crm', false, dirname( plugin_basename( EMS_JETPACK_CRM_FILE ) ) . '/languages/' );
    }

    /**
     * What type of request is this?
     *
     * @param  string $type admin, ajax, cron or frontend.
     *
     * @return bool
     */
    private function is_request( $type ) {
        switch ( $type ) {
            case 'admin' :
                return is_admin();

            case 'cron' :
                return defined( 'DOING_CRON' );
        }
    }

}

EMS_Jetpack_CRM::init();

