<?php
namespace Emlm\Woocommerce;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class LicenseKeyInputSystem {

    /**
     * Constructor to initialize the license key input system
     */
    public function __construct() {
        // Add license key input field in the order edit page (for downloadable products)
        add_action( 'woocommerce_after_order_itemmeta', array( $this, 'add_license_key_input_field' ), 10, 3 );

        // Save license key only when the order is marked as completed
        add_action( 'woocommerce_order_status_completed', array( $this, 'save_license_key_on_completed' ), 10, 1 );

    }

    /**
     * Add the license key input field in the order edit page for downloadable products (admin)
     */
    public function add_license_key_input_field( $item_id, $item, $product ) {
        // Check if the item is a downloadable product
        if ( $product->is_downloadable() ) {
            // Get the existing license key (if any)
            $license_key = wc_get_order_item_meta( $item_id, 'License Key', true );
            ?>
            <p class="form-field license-key-field">
                <label for="emlm_license_key_<?php echo esc_attr( $item_id ); ?>"><?php echo esc_html__( 'License Key:', 'emlm-license-key-manager' ); ?></label>
                <input type="text" name="emlm_license_key[<?php echo esc_attr( $item_id ); ?>]" id="emlm_license_key_<?php echo esc_attr( $item_id ); ?>" value="<?php echo esc_attr( $license_key ); ?>" placeholder="<?php echo esc_attr__( 'Enter license key', 'emlm-license-key-manager' ); ?>" />
            </p>
            <?php
        }
    }

    /**
     * Save the license key when the order status is changed to completed (admin)
     */
    public function save_license_key_on_completed( $order_id ) {
        // Check if the nonce is valid
        if ( !isset( $_POST['woocommerce_meta_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['woocommerce_meta_nonce'] ) ), 'woocommerce_save_data' ) ) {
            return; // Nonce check failed, exit without saving
        }
        // Get the order object
        $order = wc_get_order( $order_id );

        // Loop through the items in the order
        foreach ( $order->get_items() as $item_id => $item ) {
            $product = $item->get_product();

            // Check if the product is downloadable and the license key is set
            if ( $product->is_downloadable() && isset( $_POST['emlm_license_key'][$item_id] ) ) {
                $license_key = sanitize_text_field( wp_unslash( $_POST['emlm_license_key'][$item_id] ) );
                wc_update_order_item_meta( $item_id, 'License Key', $license_key );
            }
        }
    }

}
