<?php
namespace Emlm\RestAPIs;
/**
 * Class License Verify API
 *
 * @package License Key Manager
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class LicenseVerify {
    public function __construct() {
        // Register REST API route
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }

    public function register_routes() {
        register_rest_route( 'emlm/v1', '/activate/', array(
            'methods'             => 'POST',
            'callback'            => array( $this, 'activate_license' ),
            'permission_callback' => array( $this, 'check_api_keys' ),
            'args'                => array(
                'emlm_license_key'   => array(
                    'required'          => true,
                    'validate_callback' => function ( $param ) {
                        return is_string( $param );
                    },
                ),
                'emlm_license_email' => array(
                    'required'          => true,
                    'validate_callback' => function ( $param ) {
                        return is_email( $param );
                    },
                ),
            ),
        ) );
    }

    public function check_api_keys( $request ) {
        $consumer_key = $request->get_header( 'Consumer-Key' );
        $consumer_secret = $request->get_header( 'Consumer-Secret' );

        $stored_key = \Emlm\Helpers\APIKeyGenerator::get_consumer_key();
        $stored_secret = \Emlm\Helpers\APIKeyGenerator::get_consumer_secret();

        if ( $consumer_key === $stored_key && $consumer_secret === $stored_secret ) {
            return true;
        }

        return new \WP_Error( 'invalid_api_key', 'Invalid API Key or Secret', array( 'status' => 401 ) );
    }

    public function activate_license( $request ) {
        $license_key = sanitize_text_field( $request->get_param( 'emlm_license_key' ) );
        $license_email = sanitize_email( $request->get_param( 'emlm_license_email' ) );
        $license_domain = $this->get_domain_name();

        $args = array(
            'post_type'      => 'emlm_listing',
            'meta_query'     => array(  /* phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query */
                array(
                    'key'     => '_emlm_generate_key',
                    'value'   => $license_key,
                    'compare' => '=',
                ),
            ),
            'posts_per_page' => 1,
        );

        $query = new \WP_Query( $args );

        if ( $query->have_posts() ) {
            $post_id = $query->posts[0]->ID;

            // Check if the license has expired
            $expired_response = $this->is_license_expired( $post_id );
            if ( $expired_response !== false ) {
                return $expired_response;
            }

            $activations = get_post_meta( $post_id, '_emlm_activations', true );
            $max_activations = get_post_meta( $post_id, '_emlm_maximum_activations', true );
            $license_type = get_post_meta( $post_id, '_emlm_license_type', true );
            if ( empty( $activations ) ) {
                $activations = array();
            }

            if ( in_array( $license_domain, $activations ) ) {
                return new \WP_REST_Response(
                    array( 'status' => 'error', 'message' => 'This domain is already activated for this license.' ),
                    400
                );
            }

            if ( count( $activations ) >= $max_activations && $license_type == 'limitation' ) {
                return new \WP_REST_Response(
                    array( 'status' => 'error', 'message' => 'Maximum number of activations reached.' ),
                    403
                );
            }

            $activations[] = $license_domain;
            update_post_meta( $post_id, '_emlm_activations', $activations );
            update_option( 'emlm_license_key', $license_key );
            update_option( 'emlm_license_email', $license_email );
            update_option( 'emlm_license_domain', $license_domain );

            // Retrieve additional product data
            $product_id = get_post_meta( $post_id, '_emlm_product_id', true );
            $owner_name = get_post_meta( $post_id, '_emlm_owener_name', true );
            $product_type = wp_get_post_terms( $post_id, 'emlm_product_types', array( 'fields' => 'names' ) );
            $license_type = get_post_meta( $post_id, '_emlm_license_type', true );
            $validation_days = get_post_meta( $post_id, '_emlm_validation_days', true );
            $status = get_post_meta( $post_id, '_emlm_license_status', true );

            // Prepare response data
            $data = array(
                "license_id"          => $post_id,
                "license_key"         => $license_key,
                "owner_name"          => $owner_name,
                "product_id"          => $product_id,
                "product_type"        => !empty( $product_type ) ? implode( ', ', $product_type ) : '',
                "license_type"        => $license_type,
                "validation_days"     => $validation_days,
                "maximum_activations" => $max_activations,
                "activations"         => count( $activations ),
                "status"              => $status,
            );

            return new \WP_REST_Response(
                array( 'status' => 'success', 'message' => 'License activated successfully.', 'data' => $data ),
                200
            );

        } else {
            update_option( 'emlm_license_key', '' );
            return new \WP_REST_Response(
                array( 'status' => 'error', 'message' => 'Invalid license key.' ),
                404
            );
        }
    }

    private function is_license_expired( $post_id ) {
        $license_status = get_post_meta( $post_id, '_emlm_license_status', true );

        if ( $license_status == 'expired' ) {
            return new \WP_REST_Response(
                array( 'status' => 'error', 'message' => 'The license key has expired.' ),
                403
            );
        } elseif ( $license_status == 'cancelled' ) {
            return new \WP_REST_Response(
                array( 'status' => 'error', 'message' => 'The license key has been cancelled.' ),
                403
            );
        } elseif ( $license_status == 'deactivated' ) {
            return new \WP_REST_Response(
                array( 'status' => 'error', 'message' => 'The license key has been deactivated.' ),
                403
            );
        } elseif ( $license_status == 'sold' ) {
            return new \WP_REST_Response(
                array( 'status' => 'sold', 'message' => 'The license key has been sold.' ),
                403
            );
        }

        return false;
    }

    private function get_domain_name() {
        $home_url = home_url();
        $parsed_url = wp_parse_url( $home_url );
        return $parsed_url['host'];
    }
}
