<?php
namespace Emlm\RestAPIs;
/**
 * Class License Stats API
 *
 * @package License Key Manager
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class LicenseStats {
    /**
     * Constructor - Initializes the API and registers routes.
     */
    public function __construct() {
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }

    /**
     * Register REST API route to fetch license stats.
     */
    public function register_routes() {
        register_rest_route( 'emlm/v1', '/license-stats/', array(
            'methods'             => 'GET',
            'callback'            => array( $this, 'get_license_stats' ),
            'permission_callback' => function() {
                    return current_user_can( 'manage_options' );
                },
        ) );
    }

    /**
     * Fetch monthly license stats dynamically based on post publish date.
     *
     * @return WP_REST_Response License stats.
     */
    public function get_license_stats() {
        global $wpdb;

        // Define a cache key
        $cache_key = 'emlm_license_stats';

        // Try to get cached data
        $data = wp_cache_get( $cache_key );

        // If no cached data, query the database
        if ( $data === false ) {
            // Use a prepared statement to safely query the database
            $query = $wpdb->prepare( "
                SELECT DATE_FORMAT(post_date, '%%Y-%%m') AS month, COUNT(*) as count
                FROM {$wpdb->posts}
                WHERE post_type = %s
                  AND post_status = %s
                GROUP BY month
                ORDER BY month ASC
            ", 'emlm_listing', 'publish' ); // Safe values passed to the query

            // Execute the query
            $results = $wpdb->get_results( $query ); /* phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared */

            // Prepare the data in the format expected by the frontend
            $data = array();
            foreach ( $results as $row ) {
                $data[] = array(
                    'month' => gmdate( 'F', strtotime( $row->month . '-01' ) ), // Convert YYYY-MM to full month name
                    'count' => intval( $row->count ),
                );
            }

            // Cache the data for 1 hour (3600 seconds)
            wp_cache_set( $cache_key, $data, '', 3600 );
        }

        return new \WP_REST_Response( $data, 200 );
    }

}
