<?php
namespace Emlm\RestAPIs;
/**
 * Class All License API
 *
 * @package License Key Manager
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AllLicense {
    /**
     * Constructor - Initializes the API and registers routes
     */
    public function __construct() {
        // Register REST API routes
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }

    /**
     * Register the REST API routes
     */
    public function register_routes() {
        // Endpoint to get all licenses
        register_rest_route( 'emlm/v1', '/alllicense/', array(
            'methods'             => 'GET',
            'callback'            => array( $this, 'get_all_licenses' ),
            'permission_callback' => array( $this, 'check_api_keys' ),
        ) );
    }

    /**
     * Check for valid API keys in headers
     *
     * @return bool|WP_Error
     */
    public function check_api_keys() {
        // Retrieve the consumer key and secret from the headers
        $consumer_key = isset( $_SERVER['HTTP_CONSUMER_KEY'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_CONSUMER_KEY'] ) ) : '';
        $consumer_secret = isset( $_SERVER['HTTP_CONSUMER_SECRET'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_CONSUMER_SECRET'] ) ) : '';

        // Retrieve stored keys (implement your own way of getting stored keys)
        $stored_key = \Emlm\Helpers\APIKeyGenerator::get_consumer_key();
        $stored_secret = \Emlm\Helpers\APIKeyGenerator::get_consumer_secret();

        // Check if provided keys match stored keys
        if ( $consumer_key === $stored_key && $consumer_secret === $stored_secret ) {
            return true; // Valid keys
        }

        return new \WP_Error( 'invalid_api_key', 'Invalid API Key or Secret', array( 'status' => 401 ) );
    }

    /**
     * Get all licenses
     *
     * @return WP_REST_Response
     */
    public function get_all_licenses() {
        // Query all license posts of type 'emlm_listing'
        $args = array(
            'post_type'      => 'emlm_listing',
            'posts_per_page' => -1, // Get all posts
        );

        $licenses_query = new \WP_Query( $args );
        $licenses = array();

        // Loop through the posts and gather the relevant data
        if ( $licenses_query->have_posts() ) {
            while ( $licenses_query->have_posts() ) {
                $licenses_query->the_post();
                $post_id = get_the_ID();

                // Gather relevant license data
                $license_data = array(
                    'license_id'          => $post_id,
                    'product_id'          => get_post_meta( $post_id, '_emlm_product_id', true ),
                    'license_key'         => get_post_meta( $post_id, '_emlm_generate_key', true ),
                    'owner_name'          => get_post_meta( $post_id, '_emlm_owener_name', true ),
                    'license_type'        => get_post_meta( $post_id, '_emlm_license_type', true ),
                    'validation_days'     => get_post_meta( $post_id, '_emlm_validation_days', true ),
                    'maximum_activations' => get_post_meta( $post_id, '_emlm_maximum_activations', true ),
                    'license_status'      => get_post_meta( $post_id, '_emlm_license_status', true ),
                );

                $licenses[] = $license_data; // Add license data to the array
            }
        }

        wp_reset_postdata(); // Reset post data after custom query

        return new \WP_REST_Response(
            array(
                'status' => 'success',
                'data'   => $licenses,
            ),
            200// HTTP status code
        );
    }
}
