<?php
namespace Emlm\Helpers;
/**
 * Class License Status Update
 *
 * @package License Key Manager
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class LicenseStatusUpdate {

    /**
     * Constructor - Hooks into the 'admin_init' to check and update license status.
     */
    public function __construct() {
        // Hook into 'admin_init' to update license status
        add_action( 'admin_init', array( $this, 'update_all_licenses_status' ) );
    }

    /**
     * Update all licenses status when admin_init is triggered
     */
    public function update_all_licenses_status() {
        // Query all license posts (assuming custom post type 'emlm_listing')
        $args = array(
            'post_type'      => 'emlm_listing',
            'posts_per_page' => -1, // Get all license posts
        );

        $query = new \WP_Query( $args );

        // Loop through each license post and update its status
        if ( $query->have_posts() ) {
            while ( $query->have_posts() ) {
                $query->the_post();
                $post_id = get_the_ID();

                // Update the status of each license
                $this->update_license_status( $post_id );
            }
        }

        // Reset post data
        wp_reset_postdata();
    }

    /**
     * Check if the license is expired and update its status.
     *
     * @param int $post_id License post ID
     */
    private function update_license_status( $post_id ) {
        $license_status = get_post_meta( $post_id, '_emlm_license_status', true );

        // Check if the license has expired
        $is_expired = $this->is_license_expired( $post_id );

        if ( $license_status == 'active' ) {
            // Determine new license status
            $new_status = $is_expired ? 'expired' : 'active';

            // Update the license status meta field
            update_post_meta( $post_id, '_emlm_license_status', $new_status );
        }

    }

    /**
     * Check if the license is expired based on issue date and validation days.
     *
     * @param int $post_id License post ID
     * @return bool True if expired, false otherwise
     */
    private function is_license_expired( $post_id ) {
        // Get the issue date and validation days from the post meta
        $license_type = get_post_meta( $post_id, '_emlm_license_type', true );
        $issue_date = get_post_meta( $post_id, '_emlm_issue_date', true );
        $validation_days = get_post_meta( $post_id, '_emlm_validation_days', true );

        if ( $license_type == 'unlimited' ) {
            // If there is no issue date or validation days, assume the license doesn't expire
            if ( empty( $issue_date ) || empty( $validation_days ) ) {
                return false;
            }
        }

        // Calculate the expiration date by adding validation days to the issue date
        $expiration_date = gmdate( 'Y-m-d', strtotime( "$issue_date + $validation_days days" ) );

        // Get the current date
        $current_date = current_time( 'Y-m-d' );

        // Return true if the current date is greater than the expiration date
        return $current_date > $expiration_date;
    }
}
