<?php
namespace Emlm\Helpers;

use Emlm\Emails\LicenseExpiredEmail;

/**
 * Class Helper
 *
 * @package License Key Manager
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Helper {
    /**
     * Constructor.
     */
    public function __construct() {
        // Constructor body
        add_action( 'admin_init', array( $this, 'emlm_check_license_expiration' ) );
    }

    /**
     * Get total number of licenses.
     *
     * @return int Total number of licenses.
     */
    public static function get_total_license() {
        $args = array(
            'post_type'      => 'emlm_listing',
            'post_status'    => 'publish',
            'posts_per_page' => -1, // Get all posts
            'fields' => 'ids', // We only need the IDs
        );

        $query = new \WP_Query( $args );
        return (int) $query->found_posts; // Return only the number (as an integer)
    }

    /**
     * Get number of active licenses.
     *
     * @return int Number of active licenses.
     */
    public static function get_active_license() {
        $args = array(
            'post_type'      => 'emlm_listing',
            'post_status'    => 'publish',
            'posts_per_page' => -1, // Get all posts
            'meta_query' => array(  /* phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query */
                array(
                    'key'   => '_emlm_license_status',
                    'value' => 'active', // We're checking for active licenses
                    'compare' => '=',
                ),
            ),
            'fields'         => 'ids',
        );

        $query = new \WP_Query( $args );
        return (int) $query->found_posts; // Return only the number (as an integer)
    }

    /**
     * Get number of expired licenses.
     *
     * @return int Number of expired licenses.
     */
    public static function get_expired_license() {
        $args = array(
            'post_type'      => 'emlm_listing',
            'post_status'    => 'publish',
            'posts_per_page' => -1, // Get all posts
            'meta_query' => array(  /* phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query */
                array(
                    'key'   => '_emlm_license_status',
                    'value' => 'expired', // We're checking for expired licenses
                    'compare' => '=',
                ),
            ),
            'fields'         => 'ids',
        );

        $query = new \WP_Query( $args );
        return (int) $query->found_posts; // Return only the number (as an integer)
    }

    public static function emlm_check_license_expiration() {
        // Query all licenses (assuming 'emlm_listing' is your custom post type)
        $args = array(
            'post_type'      => 'emlm_listing',
            'posts_per_page' => -1,
        );

        $licenses = new \WP_Query( $args );

        if ( $licenses->have_posts() ) {
            while ( $licenses->have_posts() ) {
                $licenses->the_post();
                $post_id = get_the_ID();

                // Get the license status and check if the email has already been sent
                $get_license_status = get_post_meta( $post_id, '_emlm_license_status', true );
                $email_sent = get_post_meta( $post_id, '_emlm_license_email_expired_sent', true );

                if ( $get_license_status == 'expired' && !$email_sent ) {
                    // If the license is expired and the email hasn't been sent yet
                    LicenseExpiredEmail::handle_email_template( $post_id );

                    // Mark the email as sent by updating the post meta
                    update_post_meta( $post_id, '_emlm_license_email_expired_sent', 1 );
                }
            }
        }
        wp_reset_postdata(); // Reset after custom query
    }

}
