<?php
namespace Emlm\Helpers;

/**
 * Class GenerateKey.
 *
 * @package License Key Manageer
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GenerateKey {
    private $emlm_options;
    private $character;
    private $character_type;
    private $license_length;
    private $prefix;
    private $suffix;
    private $delimiter;
    private $key_section;
    private $license_expiration;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->emlm_options = get_option( 'emlm_options' );
        $this->character = isset( $this->emlm_options['allow_characters'] ) ? $this->emlm_options['allow_characters'] : 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $this->character_type = isset( $this->emlm_options['character_type'] ) ? $this->emlm_options['character_type'] : 'alphanumeric';
        $this->license_length = isset( $this->emlm_options['license_length'] ) ? (int) $this->emlm_options['license_length'] : 16;
        $this->prefix = isset( $this->emlm_options['license_prefix'] ) ? $this->emlm_options['license_prefix'] : '';
        $this->suffix = isset( $this->emlm_options['license_suffix'] ) ? $this->emlm_options['license_suffix'] : '';
        $this->delimiter = isset( $this->emlm_options['license_delimiters'] ) ? $this->emlm_options['license_delimiters'] : '-';
        $this->key_section = isset( $this->emlm_options['license_key_section'] ) ? (int) $this->emlm_options['license_key_section'] : 4;
        $this->license_expiration = isset( $this->emlm_options['license_expiration'] ) ? $this->emlm_options['license_expiration'] : null;
    }

    /**
     * Generates a license key based on the dynamic options.
     *
     * @return string The generated license key.
     */
    public function generate_license_key() {
        // Calculate length of each section
        $section_length = $this->license_length / $this->key_section;

        $characters = $this->get_allowed_characters();
        $key = '';

        // Generate each section of the license key
        for ( $i = 0; $i < $this->key_section; $i++ ) {
            $section = '';
            for ( $j = 0; $j < $section_length; $j++ ) {
                $section .= $characters[random_int( 0, strlen( $characters ) - 1 )];
            }
            $key .= $section . $this->delimiter;
        }

        // Trim the last delimiter
        $license_key = rtrim( $key, $this->delimiter );

        // Add prefix and suffix if they are set
        if ( !empty( $this->prefix ) ) {
            $license_key = $this->prefix . $this->delimiter . $license_key;
        }
        if ( !empty( $this->suffix ) ) {
            $license_key .= $this->delimiter . $this->suffix;
        }

        return $license_key;
    }

    /**
     * Get allowed characters based on the character type setting.
     *
     * @return string A string of allowed characters for the license key.
     */
    private function get_allowed_characters() {
        switch ( $this->character_type ) {
        case 'numberic':
            return '0123456789';
        case 'alphabetic':
            return 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        case 'alphanumeric':
        default:
            return 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        }
    }
}
