<?php
namespace Emlm\ExportImport;
/**
 * Export and Import
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class ExportImport {

    /**
     * Constructor - Hooks the export and import actions
     */
    public function __construct() {
        add_action( 'admin_post_emlm_export_licenses', array( $this, 'emlm_export_license_data' ) );
        add_action( 'admin_post_emlm_import_licenses', array( $this, 'emlm_import_license_data' ) );
    }

    /**
     * Export license data to CSV
     */
    public function emlm_export_license_data() {
        // Verify nonce
        if ( !isset( $_GET['emlm_export_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['emlm_export_nonce'] ) ), 'emlm_export_nonce_action' ) ) {
            wp_die( 'Nonce verification failed' );
        }

        // Initialize WP_Filesystem
        global $wp_filesystem;
        if ( false === ( $creds = request_filesystem_credentials( '', '', false, false, null ) ) ) {
            return; // If credentials couldn't be retrieved, return
        }
        if ( !WP_Filesystem( $creds ) ) {
            return; // If WP_Filesystem initialization failed, return
        }

        // Create temporary file for CSV export
        $csv_file = 'php://output'; // We're using 'php://output' for direct download

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=emlm_license_data_export.csv' );

        $output = $wp_filesystem->get_contents( $csv_file );

        // Write headers to CSV
        $output .= implode( ',', array( 'Post ID', 'License Key', 'Product ID', 'Owner Name', 'Product Type', 'License Type', 'Validation Days', 'Status', 'Max Activation' ) ) . "\n";

        // Get license data
        $args = array(
            'post_type'      => 'emlm_listing',
            'posts_per_page' => -1,
        );
        $licenses = new \WP_Query( $args );

        if ( $licenses->have_posts() ) {
            while ( $licenses->have_posts() ) {
                $licenses->the_post();
                $post_id = get_the_ID();

                // Fetch post meta data
                $emlm_license_key       = get_post_meta( $post_id, '_emlm_generate_key', true );
                $emlm_product_id        = get_post_meta( $post_id, '_emlm_product_id', true );
                $emlm_owner_name        = get_post_meta( $post_id, '_emlm_owener_name', true );
                $emlm_product_type      = wp_get_post_terms( $post_id, 'emlm_product_types', array( 'fields' => 'names' ) );
                $emlm_license_type      = get_post_meta( $post_id, '_emlm_license_type', true );
                $emlm_validation_days   = get_post_meta( $post_id, '_emlm_validation_days', true );
                $emlm_status            = get_post_meta( $post_id, '_emlm_license_status', true );
                $emlm_max_activation    = get_post_meta( $post_id, '_emlm_maximum_activations', true );

                // Convert product type to string
                $emlm_product_type_str = !empty( $emlm_product_type ) ? implode( ', ', $emlm_product_type ) : '';

                // Add row data to output
                $output .= implode( ',', array(
                    $post_id,
                    $emlm_license_key,
                    $emlm_product_id,
                    $emlm_owner_name,
                    $emlm_product_type_str,
                    $emlm_license_type,
                    $emlm_validation_days,
                    $emlm_status,
                    $emlm_max_activation,
                ) ) . "\n";
            }
        }

        echo esc_html( $output );
        exit();
    }

    /**
     * Import license data from CSV
     */
    public function emlm_import_license_data() {
        // Verify nonce
        if ( !isset( $_POST['emlm_import_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['emlm_import_nonce'] ) ), 'emlm_import_nonce_action' ) ) {
            wp_die( 'Nonce verification failed' );
        }

        // Validate and sanitize file input
        if ( isset( $_FILES['emlm_license_csv'] ) && !empty( $_FILES['emlm_license_csv']['tmp_name'] ) ) {
            $file = sanitize_file_name( $_FILES['emlm_license_csv']['tmp_name'] );

            // Initialize WP_Filesystem
            global $wp_filesystem;
            if ( false === ( $creds = request_filesystem_credentials( '', '', false, false, null ) ) ) {
                return; // If credentials couldn't be retrieved, return
            }
            if ( !WP_Filesystem( $creds ) ) {
                return; // If WP_Filesystem initialization failed, return
            }

            // Open file for reading
            if ( ( $handle = $wp_filesystem->get_contents_array( $file ) ) !== FALSE ) {
                // Skip the first row (headers)
                $rows = array_slice( $handle, 1 );

                // Process each row
                foreach ( $rows as $row ) {
                    $data = str_getcsv( $row );

                    // Assuming the CSV has data in the same order as we exported
                    list( $post_id, $emlm_license_key, $emlm_product_id, $emlm_owner_name, $emlm_product_type_str, $emlm_license_type, $emlm_validation_days, $emlm_status, $emlm_max_activation ) = $data;

                    // Insert or update post
                    $post_args = array(
                        'post_type'   => 'emlm_listing',
                        'post_status' => 'publish',
                        'post_title'  => 'License id #' . $post_id,
                    );
                    $post_id = wp_insert_post( $post_args );

                    // Update meta
                    update_post_meta( $post_id, '_emlm_generate_key', sanitize_text_field( $emlm_license_key ) );
                    update_post_meta( $post_id, '_emlm_product_id', sanitize_text_field( $emlm_product_id ) );
                    update_post_meta( $post_id, '_emlm_owener_name', sanitize_text_field( $emlm_owner_name ) );
                    update_post_meta( $post_id, '_emlm_license_type', sanitize_text_field( $emlm_license_type ) );
                    update_post_meta( $post_id, '_emlm_validation_days', sanitize_text_field( $emlm_validation_days ) );
                    update_post_meta( $post_id, '_emlm_license_status', sanitize_text_field( $emlm_status ) );
                    update_post_meta( $post_id, '_emlm_maximum_activations', sanitize_text_field( $emlm_max_activation ) );

                    // Set product types
                    if ( !empty( $emlm_product_type_str ) ) {
                        wp_set_post_terms( $post_id, array_map( 'sanitize_text_field', explode( ', ', $emlm_product_type_str ) ), 'emlm_product_types' );
                    }
                }
            }
        }
    }
}
