<?php

namespace EmailCraft\App\Repositories;

defined( "ABSPATH" ) || exit;

use EmailCraft\App\DTO\TemplateTypeDTO;
use EmailCraft\App\Models\TemplateType;
use EmailCraft\App\DTO\TemplateDTO;
use EmailCraft\App\DTO\TemplateReadDTO;
use EmailCraft\App\Models\Post;
use EmailCraft\App\Repositories\TemplateTypeRepository;
use EmailCraft\WpMVC\Database\Query\Builder;

class TemplateRepository {
    public TemplateTypeRepository $template_type_repository;

    public function __construct( TemplateTypeRepository $template_type_repository ) {
        $this->template_type_repository = $template_type_repository;
    }

    public function get_query_builder():Builder {
        return Post::query( 'post' )->where( 'post_type', 'emailcraft' );
    }

    public function get( TemplateReadDTO $dto ) {
        $query = $this->get_query_builder()->select( 'ID', 'post_title as title', 'post_content as content', 'post_status as status', 'post_modified as updated_at' )
                ->where_in( 'post.post_status', ['publish', 'draft'] );

        if ( $dto->get_search() ) {
            $query->where( 'post_title', 'like', '%' . $dto->get_search() . '%' );
        }

        $total_query = clone $query;

        $query->with( 'template_type' );

        if ( $dto->get_order_by() ) {
            $query->order_by( $dto->get_order_by(), $dto->get_order_direction() );
        } else {
            $query->order_by_desc( "ID" );
        }

        $templates = array_map(
            function( $template )  {
                $template->status = 'publish' === $template->status;
                return $template;
            }, $query->pagination( $dto->get_per_page(), $dto->get_page() ),
        );

        return [
            'total' => $total_query->count( 'post.ID' ),
            'items' => $templates
        ];
    }

    public function create( TemplateDTO $dto ) {
        $post_id = wp_insert_post(
            [
                'post_title'   => $dto->get_title(),
                'post_content' => $dto->get_content(),
                // 'post_status' => $dto->get_status(),
                'post_type'    => 'emailcraft',
            ]
        );

        $template_type_dto = ( new TemplateTypeDTO )->set_post_id( $post_id )
            ->set_integration( $dto->get_integration() )
            ->set_template_type( $dto->get_template_type() );

        $this->template_type_repository->create( $template_type_dto );

        return $post_id;
    }

    public function get_template( string $integration, string $template_type ) {
        $cache_key = 'emailcraft_template_' . $integration . '_' . $template_type;
        $template  = wp_cache_get( $cache_key, 'emailcraft' );

        if ( $template === false ) {
            $template = TemplateType::query( 'template' )->select( 'template.post_id, post_content' )->where( 'template.integration', $integration )->where( 'template.template_type', $template_type )
            ->join(
                Post::get_table_name() . ' as post',
                function ( $join ) {
                    $join->on_column( 'post.ID', '=', 'template.post_id' )
                     ->on( 'post.post_status', 'publish' );
                }
            )->first();

            wp_cache_set( $cache_key, $template, 'emailcraft', 3600 );
        }

        return $template;
    }
 
    public function delete_by_id( int $id ) {
        return wp_delete_post( $id );
    }
}