<?php

namespace EmailCraft\App\Providers;

defined( "ABSPATH" ) || exit;

use EmailCraft\WpMVC\Contracts\Provider;

class PostTypeServiceProvider implements Provider {
    public function boot() {
        add_action( 'init', [ $this, 'register_post_type' ] );
        add_filter( 'allowed_block_types_all', [ $this, 'filter_allowed_blocks' ], 10, 2 );
        add_action( 'admin_init', [$this, 'redirect_to_template_page'] );
        add_filter( 'block_categories_all', [$this, 'filter_block_categories_all'] );
    }

    /**
     * Filters the default array of categories for block types.
     *
     * @param array[] $block_categories Array of categories for block types.
     * @return array[] Array of categories for block types.
     */
    public function filter_block_categories_all( array $block_categories ) : array {
        $block_categories[] = [
            'slug'  => 'emailcraft-general',
            'title' => esc_html__( 'EmailCraft General', 'emailcraft' ),
            'icon'  => ''
        ];

        return $block_categories;
    }

    public function redirect_to_template_page() {
        global $pagenow;

        //phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( $pagenow !== 'edit.php' || empty( $_GET['post_type'] ) || 'emailcraft' !== $_GET['post_type'] ) {
            return;
        }

        wp_safe_redirect( admin_url( 'admin.php?page=emailcraft' ) );
        exit();
    }

    public function register_post_type() {
        $labels = [
            'name'               => __( 'Emails', 'emailcraft' ),
            'singular_name'      => __( 'Email', 'emailcraft' ),
            'menu_name'          => __( 'All Emails', 'emailcraft' ),
            'name_admin_bar'     => __( 'Email', 'emailcraft' ),
            'add_new'            => __( 'Add New', 'emailcraft' ),
            'add_new_item'       => __( 'Add New Email', 'emailcraft' ),
            'new_item'           => __( 'New Email', 'emailcraft' ),
            'edit_item'          => __( 'Edit Email', 'emailcraft' ),
            'view_item'          => __( 'View Email', 'emailcraft' ),
            'all_items'          => __( 'All Emails', 'emailcraft' ),
            'search_items'       => __( 'Search Emails', 'emailcraft' ),
            'not_found'          => __( 'No emails found.', 'emailcraft' ),
            'not_found_in_trash' => __( 'No emails found in Trash.', 'emailcraft' )
        ];

        $args = [
            'labels'          => $labels,
            'public'          => false,
            'show_ui'         => true,
            'show_in_menu'    => false,
            'menu_position'   => null,
            'menu_icon'       => 'dashicons-email-alt2',
            'capability_type' => 'post',
            'supports'        => [ 'title', 'editor', 'revisions', 'custom-fields' ],
            'has_archive'     => false,
            'show_in_rest'    => true,
        ];

        register_post_type( emailcraft_app_config( 'post_type' ), $args );

        register_post_meta(
            emailcraft_app_config( 'post_type' ), '_emailcraft_settings', [
                'type'          => 'object',
                'single'        => true,
                'show_in_rest'  => [
                    'schema' => [
                        'type'                 => 'object',
                        'additionalProperties' => true, // this allows dynamic/unstructured data
                    ],
                ],
                'auth_callback' => function() {
                    return current_user_can( 'edit_posts' );
                },
            ]
        );
    }

    /**
     * Restrict allowed blocks for the emailcraft post type.
     */
    public function filter_allowed_blocks( $allowed_blocks, $editor_context ) {
        if ( empty( $editor_context->post->post_type ) || emailcraft_app_config( 'post_type' ) !== $editor_context->post->post_type ) {
            return $allowed_blocks;
        }

        $blocks = emailcraft_config( 'blocks' );
        $blocks = array_keys( $blocks );

        return apply_filters( 'emailcraft_allowed_blocks',  $blocks );
    }
}