<?php

namespace EmailCraft\App\Helpers;

defined( 'ABSPATH' ) || exit;

use WP_User;
use EmailCraft\App\DTO\ContentDTO;

class SmartTag {
    public static function get_categories( ContentDTO $dto ): array {
        $categories = [
            'current_template' => __( 'Current Template', 'emailcraft' ),
            'general'          => __( 'General', 'emailcraft' ),
            'user'             => __( 'User', 'emailcraft' ),
            'date_and_time'    => __( 'Date and Time', 'emailcraft' ),
        ];
        return apply_filters( 'emailcraft_smarttag_categories', $categories, $dto );
    }

    public function process( ContentDTO $dto ) {
        $args          = apply_filters( 'emailcraft_smarttag_args', $dto->get_args() );
        $is_test_email = apply_filters( 'emailcraft_is_test_email', false );

        $tags    = static::get_tags( $dto );
        $content = $dto->get_content();

        if ( $is_test_email ) {
            foreach ( $tags as $tag ) {
                $content = str_replace( $tag['value'], $tag['demo_value'], $content );
            }
        } else {
            foreach ( $tags as $tag ) {
                if ( isset( $tag['data_callback'] ) && is_callable( $tag['data_callback'] ) ) {
                    $content = str_replace( $tag['value'], call_user_func( $tag['data_callback'], $args ), $content );
                }
            }
        }
        return $content;
    }

    public static function get_tags( ContentDTO $dto ): array {
        $tags = [
            [
                'label'         => __( 'Site Name', 'emailcraft' ),
                'value'         => '{site_name}',
                'demo_value'    => 'EmailCraft',
                'data_callback' => function() {
                    return get_bloginfo( 'name' );
                },
                'category'      => 'general',
            ],
            [
                'label'         => __( 'Site URL', 'emailcraft' ),
                'value'         => '{site_url}',
                'demo_value'    => get_bloginfo( 'url' ),
                'data_callback' => function() {
                    return get_bloginfo( 'url' );
                },
                'category'      => 'general',
            ],
            [
                'label'         => __( 'Site Logo', 'emailcraft' ),
                'value'         => '{site_logo}',
                'demo_value'    => static::get_wp_site_logo_url(),
                'data_callback' => function() {
                    return static::get_wp_site_logo_url();
                },
                'category'      => 'general',
            ],
            [
                'label'         => __( 'Site Description', 'emailcraft' ),
                'value'         => '{site_description}',
                'demo_value'    => get_bloginfo( 'description' ),
                'data_callback' => function() {
                    return get_bloginfo( 'description' );
                },
                'category'      => 'general',
            ],
            [
                'label'         => __( 'Site Admin Email', 'emailcraft' ),
                'value'         => '{site_admin_email}',
                'demo_value'    => get_bloginfo( 'admin_email' ),
                'data_callback' => function() {
                    return get_bloginfo( 'admin_email' );
                },
                'category'      => 'general',
            ],
            [
                'label'         => __( 'User ID', 'emailcraft' ),
                'value'         => '{user_id}',
                'demo_value'    => 1,
                'data_callback' => function( $args ) {
                    $user = static::get_user( $args );
                    return $user->ID;
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'User Name', 'emailcraft' ),
                'value'         => '{user_name}',
                'demo_value'    => 'John Doe',
                'data_callback' => function( $args ) {
                    $user = static::get_user( $args );
                    return $user->user_login;
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'User Display Name', 'emailcraft' ),
                'value'         => '{user_display_name}',
                'demo_value'    => 'John Doe',
                'data_callback' => function( $args ) {
                    $user = static::get_user( $args );
                    return $user->display_name ?: $user->user_login;
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'User Email', 'emailcraft' ),
                'value'         => '{user_email}',
                'demo_value'    => 'john.doe@example.com',
                'data_callback' => function( $args ) {
                    $user = static::get_user( $args );
                    return $user->user_email;
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'User Role', 'emailcraft' ),
                'value'         => '{user_role}',
                'demo_value'    => 'Subscriber',
                'data_callback' => function( $args ) {
                    $user  = static::get_user( $args );
                    $roles = $user->roles;
                    return ! empty( $roles ) ? implode( ', ', $roles ) : __( 'No Role', 'emailcraft' );
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'User Password Reset URL', 'emailcraft' ),
                'value'         => '{user_password_reset_url}',
                'demo_value'    => get_bloginfo( 'url' ),
                'data_callback' => function( $args ) {
                    $user = static::get_user( $args );
                    $key  = get_password_reset_key( $user );
                    return add_query_arg(
                        [
                            'action' => 'rp',
                            'key'    => $key,
                            'login'  => rawurlencode( $user->user_login ),
                        ], wp_login_url()
                    );
                },
                'category'      => 'user',
            ],
            [
                'label'         => __( 'Current Date', 'emailcraft' ),
                'value'         => '{current_date}',
                'demo_value'    => date_i18n( get_option( 'date_format' ) ),
                'data_callback' => function() {
                    return date_i18n( get_option( 'date_format' ) );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Time', 'emailcraft' ),
                'value'         => '{current_time}',
                'demo_value'    => date_i18n( get_option( 'time_format' ) ),
                'data_callback' => function() {
                    return date_i18n( get_option( 'time_format' ) );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Date and Time', 'emailcraft' ),
                'value'         => '{current_date_time}',
                'demo_value'    => date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ),
                'data_callback' => function() {
                    return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Year', 'emailcraft' ),
                'value'         => '{current_year}',
                'demo_value'    => date_i18n( 'Y' ),
                'data_callback' => function() {
                    return date_i18n( 'Y' );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Month', 'emailcraft' ),
                'value'         => '{current_month}',
                'demo_value'    => date_i18n( 'F' ),
                'data_callback' => function() {
                    return date_i18n( 'F' );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Day', 'emailcraft' ),
                'value'         => '{current_day}',
                'demo_value'    => date_i18n( 'd' ),
                'data_callback' => function() {
                    return date_i18n( 'd' );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Hour', 'emailcraft' ),
                'value'         => '{current_hour}',
                'demo_value'    => date_i18n( 'H' ),
                'data_callback' => function() {
                    return date_i18n( 'H' );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Minute', 'emailcraft' ),
                'value'         => '{current_minute}',
                'demo_value'    => date_i18n( 'i' ),
                'data_callback' => function() {
                    return date_i18n( 'i' );
                },
                'category'      => 'date_and_time',
            ],
            [
                'label'         => __( 'Current Second', 'emailcraft' ),
                'value'         => '{current_second}',
                'demo_value'    => date_i18n( 's' ),
                'data_callback' => function() {
                    return date_i18n( 's' );
                },
                'category'      => 'date_and_time',
            ]
        ];

        return apply_filters( 'emailcraft_smarttag_tags', $tags, $dto );
    }

    private static function get_user( array $args ) : WP_User {
        if ( isset( $args['user'] ) && $args['user'] instanceof WP_User ) {
            return $args['user'];
        }
        return wp_get_current_user();
    }

    private static function get_wp_site_logo_url() {
        $logo_id = get_theme_mod( 'custom_logo' );
        
        if ( $logo_id ) {
            $logo = wp_get_attachment_image_src( $logo_id, 'full' );
            if ( ! empty( $logo[0] ) ) {
                return $logo[0];
            }
        }

        return get_site_icon_url();
    }
}