<?php
/**
 * Plugin Name: Email Preview for WooCommerce
 * Plugin URI: https://kamalyon.com/email-preview-for-woocommerce/
 * Description: Añade un botón de vista previa a todos los emails de WooCommerce (incluyendo soporte para WooCommerce Subscriptions).
 * Version: 1.0.5
 * Author: Kamalyon
 * Author URI: https://kamalyon.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: email-preview-for-woocommerce
 * Requires Plugins: woocommerce
 * WC requires at least: 3.0.0
 * WC tested up to: 10.3.5
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Evitar acceso directo
}

/**
 * Si deseas modo debug, defínelo a true y asegúrate de que WP_DEBUG también esté activo.
 */
if ( ! defined( 'EPFWC_DEBUG' ) ) {
    define( 'EPFWC_DEBUG', false );
}

/**
 * Al activar el plugin, registra el endpoint y hace flush_rewrite_rules() solo una vez.
 */
register_activation_hook( __FILE__, 'epfwc_activate_plugin' );
function emaiprfo_activate_plugin() {
    // Activamos el flag para que se actualicen las reglas de reescritura.
    update_option( 'epfwc_flush_rules', 'yes' );
}

/**
 * Al desactivar, puedes (opcionalmente) volver a flush_rewrite_rules()
 */
register_deactivation_hook( __FILE__, 'epfwc_deactivate_plugin' );
function emaiprfo_deactivate_plugin() {
    flush_rewrite_rules();
}

/**
 * Clase principal del plugin
 */
class EPFWC_WC_Email_Preview {

    /**
     * Constructor
     */
    public function __construct() {
        // Hook para procesar las reglas de reescritura solo si está pendiente
        add_action( 'init', array( $this, 'maybe_flush_rewrite_rules' ), 20 );

        // Añadir el endpoint donde capturamos las vistas previas
        add_action( 'init', array( $this, 'add_preview_endpoint' ) );

        // Añadir botón de preview en la edición individual de cada email
        add_action( 'woocommerce_email_settings_after', array( $this, 'add_preview_button' ), 10, 1 );

        // Manejar la solicitud de vista previa
        add_action( 'template_redirect', array( $this, 'handle_preview_request' ) );

        // (Opcional) script de debug en consola
        add_action( 'admin_footer', array( $this, 'add_debug_script' ) );
    }

    /**
     * Si el flag emaiprfo_flush_rules está en "yes", hace flush_rewrite_rules() y lo pone en "no".
     */
    public function maybe_flush_rewrite_rules() {
        if ( 'yes' === get_option( 'epfwc_flush_rules' ) ) {
            flush_rewrite_rules();
            update_option( 'epfwc_flush_rules', 'no' );
        }
    }

    /**
     * Registrar el endpoint para email-preview-for-woocommerce
     */
    public function add_preview_endpoint() {
        add_rewrite_endpoint( 'email-preview-for-woocommerce', EP_ALL );
    }

    /**
     * Agregar el botón de vista previa en la página de edición de cada email
     */
    public function add_preview_button( $email ) {
        // Verificar nonce para seguridad
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'woocommerce-settings' ) ) {
            return;
        }

        // Obtener la sección actual (ej: 'wc_email_new_order')
        $section = isset( $_GET['section'] ) ? sanitize_title( wp_unslash( $_GET['section'] ) ) : '';

        if ( ! empty( $section ) ) {
            // Quitar prefijos
            $email_id = str_replace( array( 'wc_email_', 'wcs_email_' ), '', $section );

            // Construir la URL para la vista previa
            $preview_url = add_query_arg(
                array(
                    'email-preview-for-woocommerce' => '1',
                    'email_id'                      => $email_id,
                    'nonce'                         => wp_create_nonce( 'email-preview-for-woocommerce' ),
                ),
                home_url()
            );

            echo '<div class="button-padding">';
            // Botón de vista previa
            echo '<a href="' . esc_url( $preview_url ) . '" class="button button-secondary" target="_blank">';
            echo esc_html__( 'Preview Email', 'email-preview-for-woocommerce' );
            echo '</a>';
            echo '</div>';
        }
    }

    /**
     * Manejar la solicitud de vista previa
     */
    public function handle_preview_request() {
        // Comprobar si viene el parámetro email-preview-for-woocommerce=1
        if ( ! isset( $_GET['email-preview-for-woocommerce'] ) || '1' !== $_GET['email-preview-for-woocommerce'] ) {
            return;
        }

        // Solo forzamos el debug si WP_DEBUG y EPFWC_DEBUG están activos
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && EPFWC_DEBUG ) {
            if ( function_exists( 'wp_debug_mode' ) ) {
                wp_debug_mode();
            }
        }

        // Verificar nonce
        $nonce = isset( $_GET['nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['nonce'] ) ) : '';
        if ( ! $nonce || ! wp_verify_nonce( $nonce, 'email-preview-for-woocommerce' ) ) {
            wp_die(
                esc_html__( 'Security check failed. Please try again.', 'email-preview-for-woocommerce' ),
                esc_html__( 'Error', 'email-preview-for-woocommerce' ),
                array( 'response' => 403 )
            );
        }

        // Verificar permisos
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die(
                esc_html__( 'You do not have permission to access this page.', 'email-preview-for-woocommerce' ),
                esc_html__( 'Error', 'email-preview-for-woocommerce' ),
                array( 'response' => 403 )
            );
        }

        // Obtener ID del email
        $email_id = isset( $_GET['email_id'] ) ? sanitize_title( wp_unslash( $_GET['email_id'] ) ) : '';
        if ( empty( $email_id ) ) {
            wp_die(
                esc_html__( 'Email ID is required.', 'email-preview-for-woocommerce' ),
                esc_html__( 'Error', 'email-preview-for-woocommerce' )
            );
        }

        // ¿Modo seguro?
        $safe_mode = ( isset( $_GET['safe_mode'] ) && '1' === $_GET['safe_mode'] );

        try {
            // Obtener instancias de emails de WooCommerce
            $mailer = WC()->mailer();
            $emails = $mailer->get_emails();

            $available_emails = array_map(
                static function( $e ) {
                    return $e->id;
                },
                $emails
            );

            // Localizar la instancia del email
            $found_email = null;
            foreach ( $emails as $email_obj ) {
                if ( $email_id === $email_obj->id ) {
                    $found_email = $email_obj;
                    break;
                }
            }

            if ( ! $found_email ) {
                echo '<h1>' . esc_html__( 'Email not found', 'email-preview-for-woocommerce' ) . '</h1>';
                /* Traducible con placeholders */
                printf(
                    '<p>%s</p>',
                    sprintf(
                        // translators: %s is the ID of the email template
                        esc_html__( "The email with ID '%s' was not found.", 'email-preview-for-woocommerce' ),
                        esc_html( $email_id )
                    )
                );
                printf(
                    '<p>%s %s</p>',
                    esc_html__( 'Available emails:', 'email-preview-for-woocommerce' ),
                    esc_html( implode( ', ', $available_emails ) )
                );
                printf(
                    '<p>%s</p>',
                    sprintf(
                        // translators: %s is the ID of the email template
                        esc_html__( 'Current URL: %s', 'email-preview-for-woocommerce' ),
                        esc_url( isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '' )
                    )
                );
                exit;
            }

            // Generar la vista previa
            $this->generate_email_preview( $found_email, $safe_mode );

        } catch ( Exception $e ) {
            echo '<h1>' . esc_html__( 'Error generating preview', 'email-preview-for-woocommerce' ) . '</h1>';
            echo '<p>' . esc_html__( 'An error occurred while trying to generate the email preview.', 'email-preview-for-woocommerce' ) . '</p>';
            printf(
                '<p><strong>%s</strong> %s</p>',
                esc_html__( 'Error message:', 'email-preview-for-woocommerce' ),
                esc_html( $e->getMessage() )
            );
            printf(
                '<p><strong>%s</strong> %s (line %d)</p>',
                esc_html__( 'File:', 'email-preview-for-woocommerce' ),
                esc_html( $e->getFile() ),
                (int) $e->getLine()
            );
            echo '<p><strong>' . esc_html__( 'Trace:', 'email-preview-for-woocommerce' ) . '</strong></p>';
            echo '<pre>' . esc_html( $e->getTraceAsString() ) . '</pre>';
            exit;
        }
    }

    /**
     * Generar la vista previa del email
     */
    private function generate_email_preview( $email, $safe_mode = false ) {
        $sample_data = $this->get_sample_data_for_email( $email );
        $content     = $this->get_email_content( $email, $sample_data, $safe_mode );

        echo wp_kses_post( $content );
        exit;
    }

    /**
     * Definir datos de ejemplo según el ID del email
     */
    private function get_sample_data_for_email( $email ) {
        $sample_order = $this->get_sample_order();

        switch ( $email->id ) {
            case 'customer_new_account':
                return array(
                    'user_login' => 'customer@example.com',
                    'user_pass'  => 'password123',
                    'user_id'    => 1,
                );
            case 'customer_reset_password':
                return array(
                    'user_login' => 'customer@example.com',
                    'reset_key'  => 'sample-reset-key',
                );
            case 'customer_note':
                return array(
                    'order'         => $sample_order,
                    'customer_note' => esc_html__( 'This is a sample note for the customer.', 'email-preview-for-woocommerce' ),
                );

            // Ejemplo de correos de suscripciones
            case 'new_renewal_order':
            case 'new_switch_order':
            case 'customer_processing_renewal_order':
            case 'customer_completed_renewal_order':
            case 'customer_on_hold_renewal_order':
            case 'customer_completed_switch_order':
            case 'customer_renewal_invoice':
            case 'cancelled_subscription':
            case 'expired_subscription':
            case 'suspended_subscription':
                return array(
                    'order'        => $sample_order,
                    'subscription' => $this->get_sample_subscription(),
                );

            default:
                // Emails de pedido normales
                return array( 'order' => $sample_order );
        }
    }

    /**
     * Obtener un pedido de muestra (real o simulado)
     */
    private function get_sample_order() {
        $orders = wc_get_orders(
            array(
                'limit'   => 1,
                'orderby' => 'date',
                'order'   => 'DESC',
                'type'    => 'shop_order',
            )
        );

        if ( ! empty( $orders ) ) {
            return $orders[0];
        }

        // Crear pedido simulado en memoria
        $mock_order = new WC_Order();
        $mock_order->set_id( 0 );
        $mock_order->set_billing_first_name( 'Cliente' );
        $mock_order->set_billing_last_name( 'Ejemplo' );
        $mock_order->set_billing_email( 'cliente@ejemplo.com' );
        $mock_order->set_billing_phone( '123456789' );
        $mock_order->set_billing_address_1( 'Calle Ejemplo 123' );
        $mock_order->set_billing_city( 'Ciudad Ejemplo' );
        $mock_order->set_billing_postcode( '12345' );
        $mock_order->set_billing_country( 'ES' );
        $mock_order->set_total( 99.99 );
        $mock_order->set_status( 'processing' );

        $item = new WC_Order_Item_Product();
        $item->set_name( 'Producto de Ejemplo' );
        $item->set_quantity( 1 );
        $item->set_total( 99.99 );
        $mock_order->add_item( $item );

        return $mock_order;
    }

    /**
     * Obtener una suscripción de muestra (WooCommerce Subscriptions)
     */
    private function get_sample_subscription() {
        if ( class_exists( 'WC_Subscription' ) && function_exists( 'wcs_get_subscriptions' ) ) {
            $subscriptions = wcs_get_subscriptions(
                array(
                    'limit'   => 1,
                    'orderby' => 'date',
                    'order'   => 'DESC',
                )
            );
            if ( ! empty( $subscriptions ) ) {
                return reset( $subscriptions );
            }
        }
        return null;
    }

    /**
     * Generar el contenido HTML del email
     */
    private function get_email_content( $email, $sample_data, $safe_mode = false ) {
        // Guardar formato original
        $original_format = '';
        if ( method_exists( $email, 'get_email_type' ) ) {
            $original_format = $email->get_email_type();
        } elseif ( property_exists( $email, 'email_type' ) ) {
            $original_format = $email->email_type;
        }

        // Forzar a HTML
        if ( method_exists( $email, 'set_email_type' ) ) {
            $email->set_email_type( 'html' );
        } elseif ( property_exists( $email, 'email_type' ) ) {
            $email->email_type = 'html';
        }

        ob_start();

        // Si no es modo seguro, llamamos a trigger() con los parámetros adecuados
        if ( ! $safe_mode && method_exists( $email, 'trigger' ) ) {
            $order_id = ( isset( $sample_data['order'] ) ) ? $sample_data['order']->get_id() : 0;

            switch ( $email->id ) {
                // Suscripciones que usan trigger($order_id, $subscription)
                case 'new_renewal_order':
                case 'new_switch_order':
                case 'customer_processing_renewal_order':
                case 'customer_completed_renewal_order':
                case 'customer_on_hold_renewal_order':
                case 'customer_completed_switch_order':
                case 'customer_renewal_invoice':
                    if ( isset( $sample_data['subscription'] ) && $sample_data['subscription'] ) {
                        $email->trigger( $order_id, $sample_data['subscription'] );
                    } else {
                        $email->trigger( $order_id, $sample_data['order'] );
                    }
                    break;

                // Suscripciones que usan trigger($subscription)
                case 'cancelled_subscription':
                case 'expired_subscription':
                case 'suspended_subscription':
                    if ( isset( $sample_data['subscription'] ) && $sample_data['subscription'] ) {
                        $email->trigger( $sample_data['subscription'] );
                    } else {
                        echo '<p>' . esc_html__( 'No sample subscription found for preview.', 'email-preview-for-woocommerce' ) . '</p>';
                        ob_end_clean();
                        return '';
                    }
                    break;

                // Emails de pedido
                default:
                    if ( isset( $sample_data['order'] ) ) {
                        $email->trigger( $order_id, $sample_data['order'] );
                    } elseif ( 'customer_new_account' === $email->id ) {
                        $email->trigger(
                            ( isset( $sample_data['user_id'] ) ? $sample_data['user_id'] : 1 ),
                            ( isset( $sample_data['user_login'] ) ? $sample_data['user_login'] : 'customer@example.com' ),
                            ( isset( $sample_data['user_pass'] ) ? $sample_data['user_pass'] : 'password123' )
                        );
                    } elseif ( 'customer_reset_password' === $email->id ) {
                        $email->trigger(
                            ( isset( $sample_data['user_login'] ) ? $sample_data['user_login'] : 'customer@example.com' ),
                            ( isset( $sample_data['reset_key'] ) ? $sample_data['reset_key'] : 'sample-reset-key' )
                        );
                    }
                    break;
            }
        } elseif ( $safe_mode ) {
            // Modo seguro: no llamamos trigger(), solo configuramos propiedades
            $this->setup_email_properties( $email, $sample_data );
        }

        // Obtener el HTML
        $email_content = '';
        if ( method_exists( $email, 'get_content_html' ) ) {
            $email_content = $email->get_content_html();
        } else {
            $email_content = '<p>' . esc_html__( 'This email type does not support HTML preview.', 'email-preview-for-woocommerce' ) . '</p>';
        }

        ob_end_clean();

        // Normalmente get_content_html() ya incluye header y footer
        $final_content = $email_content;

        // Aplicar estilos inline
        if ( method_exists( $email, 'style_inline' ) ) {
            $final_content = $email->style_inline( $final_content );
        }

        // Restaurar formato original
        if ( ! empty( $original_format ) ) {
            if ( method_exists( $email, 'set_email_type' ) ) {
                $email->set_email_type( $original_format );
            } elseif ( property_exists( $email, 'email_type' ) ) {
                $email->email_type = $original_format;
            }
        }

        return $final_content;
    }

    /**
     * Configurar propiedades mínimas en modo seguro
     */
    private function setup_email_properties( $email, $sample_data ) {
        // Emails de pedido
        if ( isset( $sample_data['order'] ) ) {
            if ( method_exists( $email, 'set_object' ) ) {
                $email->set_object( $sample_data['order'] );
            } elseif ( property_exists( $email, 'object' ) ) {
                $email->object = $sample_data['order'];
            }
            if ( property_exists( $email, 'order_id' ) ) {
                $email->order_id = $sample_data['order']->get_id();
            }
            if ( property_exists( $email, 'placeholders' ) && is_array( $email->placeholders ) ) {
                if ( isset( $email->placeholders['{order_number}'] ) ) {
                    $email->placeholders['{order_number}'] = $sample_data['order']->get_order_number();
                }
                if ( isset( $email->placeholders['{order_date}'] ) ) {
                    $email->placeholders['{order_date}'] = wc_format_datetime( $sample_data['order']->get_date_created() );
                }
            }
        }

        // Emails de cuenta
        if ( 'customer_new_account' === $email->id && isset( $sample_data['user_login'] ) ) {
            if ( property_exists( $email, 'user_login' ) ) {
                $email->user_login = $sample_data['user_login'];
            }
            if ( property_exists( $email, 'user_pass' ) ) {
                $email->user_pass = isset( $sample_data['user_pass'] ) ? $sample_data['user_pass'] : 'password123';
            }
        }
    }

}

// Inicializar el plugin solo si WooCommerce está activo
add_action( 'plugins_loaded', function() {
    if ( class_exists( 'WooCommerce' ) ) {
        new EPFWC_WC_Email_Preview();
    }
} );

add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'custom_order_tables',
            __FILE__,
            true
        );
    }
} );

// Enganchar la función al hook 'admin_enqueue_scripts' para cargar los recursos en el admin
add_action( 'admin_enqueue_scripts', 'epfwc_enqueue_admin_assets' );

function emaiprfo_enqueue_admin_assets( $hook ) {
    // Limitar solo a las páginas relevantes de WooCommerce (ajustar según sea necesario)
    if ( strpos( $hook, 'woocommerce' ) === false && !in_array($hook, ['woocommerce_page_wc-settings', 'toplevel_page_email-preview-for-woocommerce'])) {
        return;
    }

    // Ruta base del plugin
    $plugin_url  = plugin_dir_url(__FILE__);
    $plugin_path = plugin_dir_path(__FILE__);

    // Registrar y encolar el script de administración
    wp_register_script(
        'epfwc-admin-script',
        $plugin_url . 'assets/js/admin.js',
        ['jquery'],
        filemtime($plugin_path . 'assets/js/admin.js'),  // Usar la fecha de modificación para versión
        true  // Cargar al final (footer)
    );
    wp_enqueue_script('epfwc-admin-script');

    // Añadir el inline script (para funcionalidad JS personalizada)
    wp_add_inline_script( 'epfwc-admin-script', "
        (function($) {
            $(document).ready(function() {
                $('.epfw-email-preview-button').on('click', function(e) {
                    e.preventDefault();
                    const email = $(this).data('email');
                    const subscription = $(this).data('subscription') || '';
                    const url = new URL( window.location.href );
                    url.searchParams.set('epfw_preview_email', email);
                    if (subscription) {
                        url.searchParams.set('subscription_id', subscription);
                    }
                    window.open(url.toString(), '_blank');
                });
            });
        })(jQuery);
    " );

    // Registrar y encolar los estilos de administración
    wp_register_style(
        'epfwc-admin-style',
        $plugin_url . 'assets/css/admin.css',
        [],  // Dependencias vacías
        filemtime($plugin_path . 'assets/css/admin.css')  // Usar la fecha de modificación para versión
    );
    wp_enqueue_style('epfwc-admin-style');
}
