<?php namespace EmailLog\Core\UI\Page;

use EmailLog\Core\DB\TableManager;
use EmailLog\Core\UI\ListTable\LogListTable;

/**
 * Log List Page.
 *
 * @since 2.0
 */
class LogListPage extends BasePage {
	/**
	 * @var LogListTable
	 */
	protected $log_list_table;

	/**
	 * Page slug.
	 */
	const PAGE_SLUG = 'email-log';

	/**
	 * Nonce Field.
	 */
	const LOG_LIST_ACTION_NONCE_FIELD = 'el-log-list-nonce-field';

	/**
	 * Nonce name.
	 */
	const LOG_LIST_ACTION_NONCE = 'el-log-list-nonce';

	/**
	 * Capability to manage email logs.
	 *
	 * @since 2.1.0
	 */
	const CAPABILITY = 'manage_email_logs';

	/**
	 * Setup hooks.
	 */
	public function load() {
		parent::load();

		add_filter( 'set-screen-option', array( $this, 'save_screen_options' ), 10, 3 );

		add_action( 'admin_enqueue_scripts', array( $this, 'load_view_logs_assets' ), 1 );
	}

	/**
	 * Register page.
	 *
	 * @inheritdoc
	 */
	public function register_page() {
		add_menu_page(
			__( 'Email Log', 'email-log' ),
			__( 'Email Log', 'email-log' ),
			self::CAPABILITY,
			self::PAGE_SLUG,
			array( $this, 'render_page' ),
			'dashicons-email-alt',
			26
		);

		$this->page = add_submenu_page(
			self::PAGE_SLUG,
			__( 'View Logs', 'email-log'),
			__( 'View Logs', 'email-log'),
			self::CAPABILITY,
			self::PAGE_SLUG,
			array( $this, 'render_page' )
		);

		add_action( "load-{$this->page}", array( $this, 'load_page' ) );

		/**
		 * Fires before loading log list page.
		 *
		 * @since 2.0
		 *
		 * @param string $page Page slug.
		 */
		do_action( 'el_load_log_list_page', $this->page );
	}

	/**
	 * Render page.
	 */
	public function render_page() {
		if ( ! current_user_can( self::CAPABILITY ) ) {
			return;
		}

		add_thickbox();

		$this->log_list_table->prepare_items();
		?>
		<div class="wrap">
			<h2><img class="el-logo" src="<?php echo esc_url(EMAIL_LOG_URL . 'assets/img/logo-64x64.png'); ?>" /><?php esc_html_e( 'Email Log', 'email-log' ); ?></h2>
			<?php settings_errors(); ?>
            <div class="email-log-body-wrapper">
                <form id="email-logs-list" method="get">
                    <input type="hidden" name="page" value="<?php echo esc_attr( self::PAGE_SLUG ); ?>">
                    <?php $this->log_list_table->search_box( __( 'Search Logs', 'email-log' ), 'search_id' ); ?>

                    <?php
                    // Disable the output of referrer hidden field, since it will be generated by the log list table.
                    wp_nonce_field( self::LOG_LIST_ACTION_NONCE, self::LOG_LIST_ACTION_NONCE_FIELD, false );
                    $this->log_list_table->display();
                    ?>
                </form>
            </div>
            <div class="email-log-sidebar-wrapper">
                <?php \EmailLog\Core\EmailLog::wp_kses_wf($this->sidebar()); ?>
            </div>
		</div>
		<?php
		$this->render_page_footer();
	}

	/**
	 * Load page.
	 */
	public function load_page() {
		// Add screen options
		$this->get_screen()->add_option(
			'per_page',
			array(
				'label'   => __( 'Entries per page', 'email-log' ),
				'default' => 20,
				'option'  => 'per_page',
			)
		);

		$this->log_list_table = new LogListTable( $this );
	}

	/**
	 * Gets the per page option.
	 *
	 * @return int Number of logs a user wanted to be displayed in a page.
	 */
	public function get_per_page() {
		$screen = get_current_screen();
		$option = $screen->get_option( 'per_page', 'option' );

		$per_page = get_user_meta( get_current_user_id(), $option, true );

		if ( empty( $per_page ) || $per_page < 1 ) {
			$per_page = $screen->get_option( 'per_page', 'default' );
		}

		return $per_page;
	}

	/**
	 * Get nonce args.
	 *
	 * @return array Nonce args.
	 */
	public function get_nonce_args() {
		return array(
			self::LOG_LIST_ACTION_NONCE_FIELD => wp_create_nonce( self::LOG_LIST_ACTION_NONCE ),
		);
	}

	/**
	 * Get TableManager instance.
	 *
	 * @return TableManager TableManager instance.
	 */
	public function get_table_manager() {
		$email_log = email_log();

		return $email_log->table_manager;
	}

	/**
	 * Saves Screen options.
	 *
	 * @since Genesis
	 *
	 * @param bool|int $status Screen option value. Default false to skip.
	 * @param string   $option The option name.
	 * @param int      $value  The number of rows to use.
	 *
	 * @return bool|int
	 */
	public function save_screen_options( $status, $option, $value ) {
		if ( 'per_page' == $option ) {
			return $value;
		} else {
			return $status;
		}
	}

	/**
	 * Loads assets on the Log List page.
	 *
	 * @since 2.0.0
	 *
	 * @param string $hook The current admin page.
	 */
	public function load_view_logs_assets( $hook ) {
		// Don't load assets if not View Logs page.
		if ( 'toplevel_page_email-log' !== $hook && 'email-log_page_email-log-settings' !== $hook) {
			return;
		}

		$email_log      = email_log();

		wp_register_script('insertionQ', EMAIL_LOG_URL . 'assets/js/insQ.min.js', array( 'jquery' ), '1.0.4', true );
		wp_enqueue_script('jquery-ui-position');
    wp_enqueue_script('jquery-effects-core');
    wp_enqueue_script("jquery-effects-blind");
    wp_enqueue_script('jquery-ui-accordion');
    wp_enqueue_script('jquery-ui-dialog');
    wp_enqueue_script( 'el-view-logs', EMAIL_LOG_URL . 'assets/js/view-logs.js', array( 'insertionQ', 'jquery-ui-core', 'jquery-ui-datepicker', 'jquery-ui-tooltip', 'jquery-ui-tabs' ), $email_log->get_version(), true );
    wp_enqueue_style('el-jquery-ui-css', EMAIL_LOG_URL . 'assets/css/email-log-jquery-ui.min.css', array());
    wp_enqueue_style('wp-jquery-ui-dialog');

	}
}
