<?php

// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnnecessaryPrepare
// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder

if ( ! defined( 'ABSPATH' ) ) exit;

class Model_EINVAPPMY
{
    private $config;
    private $wpdb;

    /**
     * WooCommerce High Performance Order Storage
     * - if true then can retrieve order from wc_orders table
     * - if false then can retrieve order from wp_posts table
     * 
     * @var bool
     */
    private $use_hpos = true;

    public function __construct($config) {
        global $wpdb;
        $this->config = $config;
        $this->wpdb = $wpdb;

        // Ensure session is available for reading
        if (!session_id() && session_status() === PHP_SESSION_NONE) {
            session_start();
        }

        $session_use_hpos = sanitize_text_field($_SESSION['use_hpos'] ?? false);

        // get the flag so that can know retrieve order from wc_orders table or wp_posts table
        if (!empty($session_use_hpos)) {
            $this->use_hpos = $session_use_hpos;
        } else {
            // if dint store in session, then check and store in session.
            $check_order_table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->prefix . 'wc_orders'));
            $this->use_hpos = !empty($check_order_table_exists);

            // Ensure session is open for writing
            if (!session_id() || session_status() === PHP_SESSION_NONE) {
                session_start();
            }
            $_SESSION['use_hpos'] = $this->use_hpos;
            // Close session after writing to avoid interfering with REST API
            session_write_close();
        }
    }

    public function getOrdersData()
    {
        global $wpdb;

        $result = [];
        $current_month = [];
        $current_month['orders'] = [];

        if ($this->use_hpos) {
            $table_name = $wpdb->prefix . 'wc_orders';
            $sql = "SELECT DATE(date_created_gmt) AS date, SUM(total_amount) AS total, COUNT(id) AS count 
                    FROM %i 
                    WHERE MONTH(date_created_gmt) = MONTH(CURDATE()) 
                    AND YEAR(date_created_gmt) = YEAR(CURDATE())
                    AND status IS NOT NULL
                    GROUP BY DATE(date_created_gmt)
                    ORDER BY date ASC
                ";
            $sql = $wpdb->prepare($sql, $table_name);
        } else {
           $sql = "SELECT DATE(p.post_date_gmt) AS date,
                    SUM(m.meta_value) AS total,
                    COUNT(p.ID) AS count
                    FROM {$wpdb->posts} p
                    INNER JOIN {$wpdb->postmeta} m ON p.ID = m.post_id
                    WHERE p.post_type = 'shop_order'
                    AND m.meta_key = '_order_total'
                    AND MONTH(p.post_date_gmt) = MONTH(CURDATE())
                    AND YEAR(p.post_date_gmt) = YEAR(CURDATE())
                    AND p.post_status NOT IN ('auto-draft', 'trash')
                    GROUP BY DATE(p.post_date_gmt)
                    ORDER BY date ASC
                ";
        }

        $current_month['orders'] = $wpdb->get_results($sql, ARRAY_A);
        $result['current_month'] = $current_month;

        $day = wp_date('d');

        $result['past_month'] = [];
        $result['past_month']['orders'] = [];

        if ($day < 7) {
            $past_month = array();
            $past_month['orders'] = array();

            $last_month = wp_date('Y-m-d', strtotime('first day of previous month'));
            $last_month_year = wp_date('Y', strtotime($last_month));
            $last_month_month = wp_date('m', strtotime($last_month));

            if ($this->use_hpos) {
                $table_name = $wpdb->prefix . 'wc_orders';
                $sql = $wpdb->prepare(
                    "SELECT DATE(date_created_gmt) AS date, SUM(total_amount) AS total, COUNT(id) AS count 
                    FROM %i 
                    WHERE MONTH(date_created_gmt) = %d 
                    AND YEAR(date_created_gmt) = %d
                    AND status IS NOT NULL
                    GROUP BY DATE(date_created_gmt)
                    ORDER BY date ASC
                    ",
                    $table_name,
                    (int) $last_month_month,
                    (int) $last_month_year
                );
            } else {
                $sql = $wpdb->prepare(
                    "SELECT DATE(p.post_date_gmt) AS date,
                            SUM(m.meta_value) AS total,
                            COUNT(p.ID) AS count
                     FROM {$wpdb->posts} p
                     INNER JOIN {$wpdb->postmeta} m ON p.ID = m.post_id
                     WHERE p.post_type = 'shop_order'
                       AND m.meta_key = '_order_total'
                       AND MONTH(p.post_date_gmt) = %d
                       AND YEAR(p.post_date_gmt) = %d
                       AND p.post_status NOT IN ('auto-draft', 'trash')
                     GROUP BY DATE(p.post_date_gmt)
                     ORDER BY date ASC",
                    (int) $last_month_month,
                    (int) $last_month_year
                );
            }
            
            $past_month['orders'] = $wpdb->get_results($sql, ARRAY_A);
            $result['past_month'] = $past_month;
        }

        return $result;
    }

    public function addConsolidatedInvoice($data) {
        global $wpdb;
    
        $invoice = $data['invoiceDetails'];
        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';
    
        // Prepare data with proper sanitization
        $insert_data = [
            'invoice_id'    => sanitize_text_field($invoice['store_order_id']),
            'invoice_code'  => sanitize_text_field($invoice['invoice_code']),
            'myinvois_uuid' => sanitize_text_field($invoice['uuid']),
            'description'   => sanitize_text_field($invoice['items'][0]['name']),
            'currency_code' => sanitize_text_field($invoice['currency_code']),
            'total'         => (float) $invoice['total'],
            'tax_total'     => (float) $invoice['tax_total'],
            'status'        => (int) $invoice['status'],
            'date_added'    => current_time('mysql', true) // GMT time
        ];
    
        // Use wpdb->insert() for automatic escaping
        $wpdb->insert(
            $table_name,
            $insert_data,
            [
                '%s', // invoice_id
                '%s', // invoice_code
                '%s', // myinvois_uuid
                '%s', // description
                '%s', // currency_code
                '%f', // total
                '%f', // tax_total
                '%d', // status
                '%s'  // date_added
            ]
        );
    
        return $wpdb->insert_id;
    }

    public function editConsolidatedInvoice($data = [], $where = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';

        // Prepare SET clause with placeholders
        $set_clauses = [];
        $set_values = [];

        if (isset($data['description'])) {
            $set_clauses[] = 'description = %s';
            $set_values[] = sanitize_text_field($data['description']);
        }

        if (isset($data['currency_code'])) {
            $set_clauses[] = 'currency_code = %s';
            $set_values[] = sanitize_text_field($data['currency_code']);
        }

        if (isset($data['total'])) {
            $set_clauses[] = 'total = %f';
            $set_values[] = (float) $data['total'];
        }

        if (isset($data['tax_total'])) {
            $set_clauses[] = 'tax_total = %f';
            $set_values[] = (float) $data['tax_total'];
        }

        if (isset($data['status'])) {
            $set_clauses[] = 'status = %d';
            $set_values[] = (int) $data['status'];
        }

        // Prepare WHERE clause
        $where_clauses = ['1=1']; // Default true condition
        $where_values = [];

        if (!empty($where['invoice_code'])) {
            $where_clauses[] = 'invoice_code = %s';
            $where_values[] = sanitize_text_field($where['invoice_code']);
        }

        if (!empty($where['myinvois_uuid'])) {
            $where_clauses[] = 'myinvois_uuid = %s';
            $where_values[] = sanitize_text_field($where['myinvois_uuid']);
        }


        // Build the query
        $sql = $wpdb->prepare(
            "UPDATE `" . esc_sql($table_name) . "` 
            SET " . implode(', ', $set_clauses) . "
            WHERE " . implode(' AND ', $where_clauses),
            array_merge($set_values, $where_values)
        );

        // Execute the query
        $result = $wpdb->query($sql);

        return $result !== false;
    }

    public function deleteConsolidatedInvoice($invoice_code)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';

        $sql = "DELETE FROM " . esc_sql($table_name) . " WHERE 1 AND invoice_code = %s";
        $sql = $wpdb->prepare($sql, sanitize_text_field($invoice_code));

        $result = $wpdb->query($sql);

        return $result !== false;
    }

    public function getConsolidatedEinvoiceSubmissions($filter = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';
        $values = [];

        $sql = "SELECT * FROM " . esc_sql($table_name) . " WHERE 1";

        if (!empty($filter['invoice_code'])) {
            $sql .= " AND invoice_code = %s";
            $values[] = sanitize_text_field($filter['invoice_code']);
        }

        if (!empty($values)) {
            $sql = $wpdb->prepare($sql, $values);
        }

        $query = $wpdb->get_results($sql, ARRAY_A);

        return $query;
    }

    public function getConsolidatedEinvoiceSubmission($filter = []) {
        $results = $this->getConsolidatedEinvoiceSubmissions($filter);

        if (empty($results)) {
            return [];
        }

        return $results[0];
    }

    public function getCurrentMonthConsolidated() {
        global $wpdb;

        $startDate = wp_date('Y-m-01 00:00:00');
        $endDate = wp_date('Y-m-t 23:59:59');
        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';

        $sql = "SELECT * 
                FROM " . esc_sql($table_name) . " 
                WHERE date_added >= %s 
                AND date_added <= %s 
                ORDER BY date_added DESC 
                LIMIT 1";

        $sql = $wpdb->prepare($sql, $startDate, $endDate);

        $query = $wpdb->get_row($sql, ARRAY_A);

        return $query;
    }
    public function getConsolidatedTotalSales($month, $year, $data = array()) {
        $tmptotal = 0;
        $tmptax = 0;

        if ($this->use_hpos) {
            $orders = $this->getConsoliodatedTotalSales_hpos($month, $year, $data);
        } else {
            $orders = $this->getConsoliodatedTotalSales_wp_posts($month, $year, $data);
        }

        if (empty($orders)) {
            foreach ($orders as $order) {
                $order_info = wc_get_order($order->order_id);
                // wordpress or woocommerce dint have currency exchange rate
                // so we default set to 1
                $currency_value = 1;

                // Adjust total
                $order_total = $order->total / $currency_value;
                $tmptotal += $order_total;

                $total_tax = $order_info->get_total_tax();
                $tmptax += $total_tax / $currency_value;
            }

        }

        return [
            'total' => $tmptotal,
            'tax_total' => $tmptax,
        ];
    }

    private function getConsoliodatedTotalSales_hpos($month, $year, $data = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'wc_orders';
        $sql = $wpdb->prepare(
            "SELECT id AS order_id, total_amount AS total, currency AS currency_code 
                FROM %i 
                WHERE YEAR(date_created_gmt) = %d 
                AND MONTH(date_created_gmt) = %d",
            $table_name,
            (int) $year,
            (int) $month
        );

        if (!empty($data) && !empty($data['statuses'])) {
            $placeholders = implode(',', array_fill(0, count($data['statuses']), '%s'));

            $sql .= $wpdb->prepare(
                " AND `status` IN (" . $placeholders . ")",
                $data['statuses']
            );
        }

        $orders = $wpdb->get_results($sql, ARRAY_A);

        return $orders;
    }

    private function getConsoliodatedTotalSales_wp_posts($month, $year, $data = []) {
        global $wpdb;

        $sql = "SELECT p.ID AS order_id, 
                MAX(CASE WHEN pm1.meta_key = '_order_total' THEN pm1.meta_value END) AS total,
                MAX(CASE WHEN pm2.meta_key = '_order_currency' THEN pm2.meta_value END) AS currency_code
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_order_total'
                LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_order_currency'
                WHERE 1
                AND p.post_type = 'shop_order'
                AND YEAR(p.post_date_gmt) = %d
                AND MONTH(p.post_date_gmt) = %d
                ";
        $prepared_data = [
            (int) $year,
            (int) $month
        ];

        if (!empty($data) && !empty($data['statuses'])) {
            $placeholders = implode(',', array_fill(0, count($data['statuses']), '%s'));

            $sql .= " AND p.post_status IN (" . $placeholders . ")";
            $prepared_data = array_merge($prepared_data, $data['statuses']);
        }

        $sql = $wpdb->prepare($sql, $prepared_data);

        $orders = $wpdb->get_results($sql, ARRAY_A);

        return $orders;
    }

    // TODO: this function seem not used, might need to remove it after confirmed
    public function getOrdersForConsolidation($month, $year, $validatedOrderIds = array())
    {
        global $wpdb;

        $statusIds = EINVAPPMY_Helper::get_option('consolidated_order_statuses');
        $placeholders = implode(',', array_fill(0, count($statusIds), '%s'));

        $table_name = $wpdb->prefix . 'wc_orders';
        $sql = $wpdb->prepare(
            "SELECT id AS order_id, 
            date_created_gmt AS date_added, 
            total_amount AS total,
            status as order_status
                FROM " . esc_sql($table_name) . "
                WHERE status IN (" . $placeholders . ")
                AND MONTH(date_created_gmt) = %d
                AND YEAR(date_created_gmt) = %d",
            $table_name,
            $statusIds,
            (int) $month,
            (int) $year
        );

        if (!empty($validatedOrderIds)) {
            $placeholders = implode(',', array_fill(0, count($validatedOrderIds), '%d'));
            $sql .= $wpdb->prepare(
                " AND id NOT IN (" . $placeholders . ")",
                $validatedOrderIds
            );
        }

        $sql .= $wpdb->prepare("GROUP BY id ORDER BY date_created_gmt ASC");

        $query = $wpdb->get_results($sql, ARRAY_A);

        return $query;
    }

    public function getOrders($filter = [])
    {
        global $wpdb;

        if ($this->use_hpos) {
            return $this->getOrders_hpos($filter);
        } else {
            return $this->getOrders_wp_posts($filter);
        }
    }

    private function getOrders_hpos($filter = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'wc_orders';
        $sql = $wpdb->prepare("SELECT * FROM %i WHERE 1", $table_name);

        if (!empty($filter['order_statuses'])) {
            $placeholders = implode(',', array_fill(0, count($filter['order_statuses']), '%s'));
           
            $sql .= $wpdb->prepare(
                " AND `status` IN (" . $placeholders . ")",
                $filter['order_statuses']
            );
        }

        if (!empty($filter['month'])) {
            $sql .= $wpdb->prepare(
                " AND MONTH(date_created_gmt) = %d",
                (int) $filter['month']
            );
        }

        if (!empty($filter['year'])) {
            $sql .= $wpdb->prepare(
                " AND YEAR(date_created_gmt) = %d",
                (int) $filter['year']
            );
        }

        if (!empty($filter['order_ids'])) {
            $placeholders = implode(',', array_fill(0, count($filter['order_ids']), '%d'));
            $sql .= $wpdb->prepare(
                " AND id IN (" . $placeholders . ")",
                $filter['order_ids']
            );
        }

        if (!empty($filter['exclude_order_ids'])) {
            $placeholders = implode(',', array_fill(0, count($filter['exclude_order_ids']), '%d'));
            $sql .= $wpdb->prepare(
                " AND id NOT IN (" . $placeholders . ")",
                $filter['exclude_order_ids']
            );
        }

        if (!empty($filter['group_by'])) {
            $sql .= $wpdb->prepare(" GROUP BY %i", $filter['group_by']);
        }

        if (!empty($filter['sort_by']) && !empty($filter['sort_order'])) {
            $sql .= $wpdb->prepare(" ORDER BY %i %5s", $filter['sort_by'], $filter['sort_order']);
        }

        $query = $wpdb->get_results($sql, ARRAY_A);

        return $query;
    }

    private function getOrders_wp_posts($filter = []) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT p.ID AS id,
                p.post_date_gmt AS date_created_gmt,
                MAX(CASE WHEN pm_total.meta_key = '_order_total' THEN pm_total.meta_value END) AS total_amount,
                MAX(CASE WHEN pm_currency.meta_key = '_order_currency' THEN pm_currency.meta_value END) AS currency,
                p.post_status AS status
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm_total ON p.ID = pm_total.post_id AND pm_total.meta_key = '_order_total'
                LEFT JOIN {$wpdb->postmeta} pm_currency ON p.ID = pm_currency.post_id AND pm_currency.meta_key = '_order_currency'
                WHERE p.post_type = 'shop_order'
            ");
        
         // Prepare values for WHERE clauses
        $values = [];

        if (!empty($filter['order_statuses'])) {
            $placeholders = implode(',', array_fill(0, count($filter['order_statuses']), '%s'));

            $sql .= $wpdb->prepare(" AND p.post_status IN ($placeholders)", $filter['order_statuses']);
        }

        // Date filtering
        if (!empty($filter['month'])) {
            $sql .= $wpdb->prepare(" AND MONTH(p.post_date_gmt) = %d", (int)$filter['month']);
        }

        if (!empty($filter['year'])) {
            $sql .= $wpdb->prepare(" AND YEAR(p.post_date_gmt) = %d", (int)$filter['year']);
        }

        // Order IDs filtering
        if (!empty($filter['order_ids'])) {
            $placeholders = implode(',', array_fill(0, count($filter['order_ids']), '%d'));

            $sql .= $wpdb->prepare(" AND p.ID IN ($placeholders)", $filter['order_ids']);
        }

        // Excluded Order IDs
        if (!empty($filter['exclude_order_ids'])) {
            $placeholders = implode(',', array_fill(0, count($filter['exclude_order_ids']), '%d'));

            $sql .= $wpdb->prepare(" AND p.ID NOT IN ($placeholders)", $filter['exclude_order_ids']);
        }

        // Group By
        if (!empty($filter['group_by'])) {
            $sql .= $wpdb->prepare(" GROUP BY p.%1s", $filter['group_by']);
        }

        // Sorting
        if (!empty($filter['sort_by']) && !empty($filter['sort_order'])) {
            if ($filter['sort_by'] == 'date_created_gmt') {
                $filter['sort_by'] = 'post_date_gmt';
            }

            $sql .= $wpdb->prepare(" ORDER BY p.%1s %5s", $filter['sort_by'], $filter['sort_order']);
        }

        $orders = $wpdb->get_results($sql, ARRAY_A);

        return $orders;
    }

    public function formatOrders($orders = [])
    {
        if (empty($orders)) {
            return [];
        }

        $customer_ids = array_filter(array_unique(array_column($orders, 'customer_id')));
        $customers = $this->getCustomers(['customer_ids' => $customer_ids]);

        if (!empty($customers)) {
            $customers = array_combine(
                array_column($customers, 'customer_id'),
                array_values($customers)
            );
        }

        $order_statuses = wc_get_order_statuses();
        $formatted_orders = [];

        foreach ($orders as $order) {
            $customer_info = $customers[$order['customer_id']] ?? [];

            if (!empty($customer_info)) {
                $customer_name = $customer_info['first_name'] . ' ' . $customer_info['last_name'];
            } else {
                $customer_name = "Guest";
            }

            $formatted_orders[$order['id']] = array_merge($order, [
                'date_added' => wp_date('F d, Y', strtotime($order['date_created_gmt'])),
                'customer_name' => $customer_name,
                'status_text' => $order_statuses[$order['status']] ?? '',
                'formatted_total_amount' => number_format($order['total_amount'], 2, '.', ''),
                'formatted_tax_amount' => number_format($order['tax_amount'], 2, '.', ''),
            ]);
        }

        return $formatted_orders;
    }

    public function getCustomers($filter = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'wc_customer_lookup';

        $sql = $wpdb->prepare("SELECT * FROM %i WHERE 1", $table_name);

        if (!empty($filter['customer_ids'])) {
            $placeholders = implode(',', array_fill(0, count($filter['customer_ids']), '%d'));

            $sql .= $wpdb->prepare(" AND customer_id IN ($placeholders)", $filter['customer_ids']);
        }

        $query = $wpdb->get_results($sql, ARRAY_A);

        return $query;
    }

    public function addConsolidatedEinvoiceOrder($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_orders';
        $insert_data = [
            'order_id' => (int) $data['order_id'],
            'consolidate_einvocie_submission_id' => (int) $data['consolidate_einvocie_submission_id']
        ];

        $wpdb->insert($table_name, $insert_data, ['%d', '%d']);

        return $wpdb->insert_id;
    }

    public function deleteConsolidatedEinvoiceOrder($filter = []) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_orders';
        $values = [];

        $sql = $wpdb->prepare("DELETE FROM %i WHERE 1", $table_name);

        if (!empty($filter['consolidate_einvocie_submission_id'])) {
            $sql .= $wpdb->prepare(" AND consolidate_einvocie_submission_id = %d", (int) $filter['consolidate_einvocie_submission_id']);
        }

        if (!empty($filter['order_id'])) {
            $sql .= $wpdb->prepare(" AND order_id = %d", (int) $filter['order_id']);
        }

        $wpdb->query($sql);
    }

    public function getConsolidatedEinvoiceOrders($filter = [])
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'consolidated_einvoice_orders';
        $values = [];

        $sql = $wpdb->prepare("SELECT * FROM %i WHERE 1", $table_name);

        if (!empty($filter['consolidate_einvocie_submission_id'])) {
            $sql .= $wpdb->prepare(" AND consolidate_einvocie_submission_id = %d", (int) $filter['consolidate_einvocie_submission_id']);
        }

        $query = $wpdb->get_results($sql, ARRAY_A);

        return $query;
    }

    public function setupDBTables()
    {
        global $wpdb;
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        $table_name = $wpdb->prefix . 'consolidated_einvoice_submission';

        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) != $table_name) {
            $charset_collate = $wpdb->get_charset_collate();

            $sql = "CREATE TABLE $table_name (
                id int(11) NOT NULL AUTO_INCREMENT,
                invoice_id varchar(100) NOT NULL,
                invoice_code varchar(100) NOT NULL, 
                myinvois_uuid varchar(250) NOT NULL,
                description varchar(200) NOT NULL,
                currency_code varchar(3) NOT NULL,
                total decimal(15,4) NOT NULL DEFAULT 0.0000,
                tax_total decimal(15,4) NOT NULL DEFAULT 0.0000,
                status tinyint(1) NOT NULL DEFAULT 0,
                date_added datetime NOT NULL,
                PRIMARY KEY (id)
            ) $charset_collate;";

            dbDelta($sql);
        }

        $table_name = $wpdb->prefix . 'consolidated_einvoice_orders';

        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) != $table_name) {
            $charset_collate = $wpdb->get_charset_collate();

            $sql = "CREATE TABLE $table_name (
                id int(11) NOT NULL AUTO_INCREMENT,
                order_id BIGINT(20) NOT NULL,
                consolidate_einvocie_submission_id INT(11) NOT NULL,
                PRIMARY KEY (id)
            ) $charset_collate;";

            dbDelta($sql);
        }
    }
}