<?php

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

class EINVAPPMY_Library
{
    public $admin_code;

    private $staging_url;
    private $live_url;
    private $channel;
    private $url = "";
    private $config;
    private $auth_token;

    public $url_dev_endpoint = 'https://dev-api.einvoiceapp.my';
    public $url_live_endpoint = 'https://api.einvoiceapp.my/';
    public $url_endpoint;

    private $origin_domain;
    private $last_error_array = null; // Store last error array for controller access
    private $api_endpoints = [
        'get_invoices' => 'get_invoices',
        'get_credit_notes' => 'get_credit_notes',
        'get_debit_notes' => 'get_debit_notes',
        'get_refund_notes' => 'get_refund_notes',
        'create_credit_note' => 'create_credit_note',
        'create_debit_note' => 'create_debit_note',
        'create_refund_note' => 'create_refund_note',
        'create_invoice' => 'create_invoice',
        'client_get_auth_token' => 'get_authToken',
        'get_client_details' => 'get_client_details',
        'create_clients' => 'create_clients',
        'update_client' => 'update_client',
        'monthly_summary' => 'monthly_summary',
        'validated_totals' => 'validated_totals',
        'export_invoices' => 'export_invoices',
        'create_consolidatedinvoice' => 'create_consolidatedinvoice',
        'cancel_invoice' => 'cancel_invoice',
        'get_subscription_info' => 'get_subscription_info',
        'get_subscription_status' => 'get_subscription_status',
    ];
    public function __construct($config) {
        $this->config = $config;

        $api_config = $config['api'];
        $this->staging_url = $api_config['staging']['url'];
        $this->live_url = $api_config['live']['url'];
        $this->admin_code = $config['admin_code'];
        $this->origin_domain = sanitize_text_field($_SERVER['HTTP_HOST']);
        $this->channel = $config['channel'];

        $live_status = EINVAPPMY_Helper::get_option('status');
        $auth_token = EINVAPPMY_Helper::get_option('myinvois_token');

        if ($live_status == 2) {
            $this->url = $this->live_url;
            $this->auth_token = 'Basic ' . $auth_token;
        } elseif ($live_status == 1) {
            $this->url = $this->staging_url;
            $this->auth_token = 'Basic ' . (!empty($auth_token) ? $auth_token : "");
        }
    }

    public function getOriginDomain() {
        return $this->origin_domain;
    }

    /**
     * Get the URL for invoice PDF trails
     * 
     * Constructs a URL to access the PDF trails for a given invoice UUID.
     * Optionally adds a query parameter to trigger PDF download.
     *
     * @param array $query_params {
     *     Query parameters for the URL.
     *     
     *     @type string $invoice_uuid Required. The UUID of the invoice.
     * }
     * @param bool  $download_pdf  Optional. Whether to add PDF download parameter. Default false.
     * 
     * @return string The constructed PDF trails URL, or empty string if invoice_uuid is not provided.
     */
    public function getInvoiceTrailsPDFUrl($query_params = [], $download_pdf = false) {
        if (empty($query_params) || empty($query_params['invoice_uuid'])) {
            return '';
        }

        $invoice_uuid = sanitize_text_field($query_params['invoice_uuid'] ?? '');
        
        $url = trailingslashit($this->url) . 'pdf-trails/' . $invoice_uuid;

        if ($download_pdf) {
            $url = add_query_arg('pdf', 1, $url);
        }

        return $url;
    }

    public function getInvoiceUrl($invoice_code = '') {
        if (empty($invoice_code)) {
            return '';
        }

        $invoice_code = sanitize_text_field($invoice_code);

        return trailingslashit($this->url) . 'invoice/' . $invoice_code;
    }

    public function getInvoicePDFUrl($uuid = "") {
        if (empty($uuid)) {
            return '';
        }

        $uuid = sanitize_text_field($uuid);

        $base_url = trailingslashit($this->url) . 'pdf/' . $uuid;
        $url = add_query_arg('pdf', 1, $base_url);

        return $url;
    }

    public function getPdfUrl($unique_string = '') {
        if (empty($unique_string)) {
            return '';
        }

        $unique_string = sanitize_text_field($unique_string);

        $base_url = trailingslashit($this->url) . 'pdf/' . $unique_string;
        $url = add_query_arg('pdf', 1, $base_url);

        return $url;
    }

    public function getCreditNoteUrl($credit_note_code = "") {
        if (empty($credit_note_code)) {
            return '';
        }

        $credit_note_code = sanitize_text_field($credit_note_code);

        return trailingslashit($this->url) . 'credit-note/' . $credit_note_code;
    }

    public function getCreditNotePdfUrl($unique_string = "") {
        if (empty($unique_string)) {
            return '';
        }

        $unique_string = sanitize_text_field($unique_string);
        $base_url = trailingslashit($this->url) . 'pdf-credit-note/' . $unique_string;
        $url = add_query_arg('pdf', 1, $base_url);

        return $url;
    }

    public function getDebitNoteUrl($debit_note_code = "") {
        if (empty($debit_note_code)) {
            return '';
        }

        $debit_note_code = sanitize_text_field($debit_note_code);

        return trailingslashit($this->url) . 'debit-note/' . $debit_note_code;
    }

    public function getDebitNotePdfUrl($unique_string = "") {
        if (empty($unique_string)) {
            return '';
        }

        $unique_string = sanitize_text_field($unique_string);
        $base_url = trailingslashit($this->url) . 'pdf-debit-note/' . $unique_string;
        $url = add_query_arg('pdf', 1, $base_url);

        return $url;
    }

    public function getRefundNoteUrl($refund_note_code = "") {
        if (empty($refund_note_code)) {
            return '';
        }

        $refund_note_code = sanitize_text_field($refund_note_code);

        return trailingslashit($this->url) . 'refund-note/' . $refund_note_code;
    }

    public function getRefundNotePdfUrl($unique_string = "") {
        if (empty($unique_string)) {
            return '';
        }

        $unique_string = sanitize_text_field($unique_string);
        $base_url = trailingslashit($this->url) . 'pdf-refund-note/' . $unique_string;
        $url = add_query_arg('pdf', 1, $base_url);

        return $url;
    }

    /**
     * Make API request with error handling and logging
     * 
     * @param string       $endpoint_key          API endpoint key from $this->api_endpoints (or full URL)
     * @param string       $method                HTTP method ('POST', 'GET', or 'PUT')
     * @param array|string|null $body            Request body data (will be JSON encoded if array)
     * @param array        $headers               Additional headers (will be merged with defaults)
     * @param string       $function_name         Function name for logging context
     * @param bool|string  $error_check           How to check for errors: true=check 'success' field, 'handleResponseDataError'=use handleResponseDataError method, false=no check
     * @param int          $timeout               Request timeout in seconds
     * @param array        $query_params          Query parameters for GET requests (will be appended to URL)
     * @param array        $suppress_log_messages List of error messages that should NOT be logged
     * @return array Response data
     * @throws Exception On API error
     */
    private function makeApiRequest($endpoint_key, $method = 'POST', $body = null, $headers = [], $function_name = '', $error_check = true, $timeout = 30, $query_params = [], $suppress_log_messages = []) {
        $api_endpoint = '';
        $response_body = '';
        $response_data = null;
        
        try {
            // Check if endpoint_key is a full URL or needs to be resolved
            if (filter_var($endpoint_key, FILTER_VALIDATE_URL)) {
                $api_endpoint = $endpoint_key;
            } else {
                $api_endpoint = $this->getApiEndpoints($endpoint_key);
                if (!$api_endpoint) {
                    throw new Exception("Invalid API endpoint key: {$endpoint_key}");
                }
            }

            // Add query parameters for GET requests
            if (!empty($query_params) && strtoupper($method) === 'GET') {
                $url_parts = parse_url($api_endpoint);
                $existing_params = [];
                if (isset($url_parts['query'])) {
                    parse_str($url_parts['query'], $existing_params);
                }
                $all_params = array_merge($existing_params, $query_params);
                $api_endpoint = $url_parts['scheme'] . '://' . $url_parts['host'] . ($url_parts['path'] ?? '');
                if (!empty($all_params)) {
                    $api_endpoint .= '?' . http_build_query($all_params);
                }
            }

            // Prepare default headers
            $default_headers = [
                'Content-Type' => 'application/json'
            ];
            
            // Merge with provided headers
            $request_headers = array_merge($default_headers, $headers);

            // Prepare request arguments
            $request_args = [
                'method' => strtoupper($method),
                'headers' => $request_headers,
                'timeout' => $timeout,
            ];

            // Add body for POST/PUT requests
            if (in_array(strtoupper($method), ['POST', 'PUT']) && $body !== null) {
                $request_args['body'] = is_string($body) ? $body : wp_json_encode($body);
            }

            // Make the request
            if (strtoupper($method) === 'GET') {
                $response = wp_remote_get($api_endpoint, $request_args);
            } elseif (strtoupper($method) === 'PUT') {
                $response = wp_remote_request($api_endpoint, $request_args);
            } else {
                $response = wp_remote_post($api_endpoint, $request_args);
            }

            if (is_wp_error($response)) {
                throw new Exception("HTTP Request Failed: " . $response->get_error_message());
            }

            $response_body = wp_remote_retrieve_body($response);
            $response_data = json_decode($response_body, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Invalid JSON response from API");
            }

            if (empty($response_data)) {
                throw new Exception("Invalid response from API");
            }

            // Check for errors based on error_check parameter
            if ($error_check === true) {
                // Check for 'success' field
                if (empty($response_data['success'])) {
                    throw new Exception("API request failed");
                }
            } elseif ($error_check === 'handleResponseDataError') {
                // Use handleResponseDataError method
                $error_result = $this->handleResponseDataError($response_data);
                
                // Only throw exception if there are errors
                if ($error_result !== true) {
                    // Store error array for controller access
                    if (is_array($error_result) && isset($error_result['errors'])) {
                        $this->last_error_array = $error_result['errors'];

                        if (!empty($error_result['errors'])) {
                            // Collect all error messages from nested arrays
                            $error_messages = [];
                            foreach ($error_result['errors'] as $field => $messages) {
                                if (is_array($messages)) {
                                    // If messages is an array, add each message
                                    foreach ($messages as $message) {
                                        $error_messages[] = $message;
                                    }
                                } else {
                                    // If it's a single message string
                                    $error_messages[] = $messages;
                                }
                            }
                            
                            // Implode all messages and throw as exception
                            $error_message = implode(' ', $error_messages);
                            throw new Exception($error_message);
                        } else {
                            // If errors array exists but is empty, use fallback message
                            throw new Exception($error_result['message'] ?? 'Validation errors occurred');
                        }
                    } else {
                        // Fallback for string messages
                        throw new Exception($error_result);
                    }
                }
            }
            // If error_check is false, skip error checking

            return $response_data;
        } catch (Exception $e) {
            // If this error message is in the suppression list, do not log it
            if (!empty($suppress_log_messages) && in_array($e->getMessage(), $suppress_log_messages, true)) {
                throw $e;
            }

            // Log error with context
            $log_message = sprintf(
                '%s() - %s | Endpoint: %s | Origin Domain: %s',
                $function_name ?: 'makeApiRequest',
                $e->getMessage(),
                $api_endpoint,
                $this->origin_domain
            );
            
            // Add response body details if available
            if (!empty($response_body)) {
                $log_message .= sprintf(' | Response Body: %s', substr($response_body, 0, 500));
                
                // Add JSON error if JSON parsing failed
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $log_message .= sprintf(' | JSON Error: %s', json_last_error_msg());
                }
            }
            
            // Add API error details if response data is available
            if (!empty($response_data) && isset($response_data['error'])) {
                $api_error = $response_data['error'];
                $log_message .= sprintf(
                    ' | API Error: %s | Full Response: %s',
                    is_array($api_error) ? wp_json_encode($api_error) : $api_error,
                    wp_json_encode($response_data)
                );
            }
            
            EINVAPPMY_Helper::write_custom_log('einvoiceapp', $log_message, 'error');
            
            throw $e;
        }
    }

    public function getClientAuthToken() {
        try {
            $response_data = $this->makeApiRequest(
                'client_get_auth_token',
                'POST',
                [
                    'origin_domain' => $this->origin_domain,
                    'admin_code' => $this->admin_code
                ],
                [],
                'getClientAuthToken',
                true
            );

            return $this->returnResponse(true, ['auth_token' => $response_data['auth_token']]);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function getClientData() {
        try {
            $auth_token_response = $this->getClientAuthToken();

            if (
                empty($auth_token_response)
                || empty($auth_token_response['success'])
                || empty($auth_token_response['data'])
                || empty($auth_token_response['data']['auth_token'])
            ) {
                throw new Exception("Failed to get client auth token");
            }

            $auth_token = $auth_token_response['data']['auth_token'];
            $response_data = $this->makeApiRequest(
                'get_client_details',
                'GET',
                null,
                [
                    'Authorization' => 'Basic ' . $auth_token,
                ],
                'getClientData',
                'handleResponseDataError'
            );

            return $this->returnResponse(true, $response_data['details']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function createClientData($payload = []) {
        try {
            if (empty($payload)) {
                throw new Exception('Payload is required');
            }

            $response_data = $this->makeApiRequest(
                'create_clients',
                'POST',
                $payload,
                [],
                'createClientData',
                'handleResponseDataError'
            );

            $data = [
                'auth_token' => $response_data['auth_token'],
                'details' => $response_data['details']
            ];

            return $this->returnResponse(true, $data, null, $response_data['message']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function saveClientData($payload) {
        try {
            $token = $this->getClientAuthToken();
            $token_data = $token['data'] ?? [];

            if (empty($token_data['auth_token'])) {
                throw new Exception('Failed to get client auth token');
            }

            $auth_token = $token_data['auth_token'];
            $response_data = $this->makeApiRequest(
                'update_client',
                'PUT',
                $payload,
                [
                    'Authorization' => 'Basic ' . $auth_token,
                ],
                'saveClientData',
                'handleResponseDataError'
            );

            return $this->returnResponse(true, $response_data['client'], null, $response_data['message']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function createInvoice($payload) {
        try {
            if (empty($payload)) {
                throw new Exception('Payload is required');
            }

            $payload = array_merge($payload, ['channel' => $this->channel]);

            $response_data = $this->makeApiRequest(
                'create_invoice',
                'POST',
                $payload,
                [
                    'Authorization' => $this->auth_token,
                ],
                'createInvoice',
                'handleResponseDataError'
            );

            $data = [
                'eInvoiceUrl' => $response_data['eInvoiceUrl'],
                'eInvoiceCode' => $response_data['eInvoiceCode'],
            ];

            return $this->returnResponse(true, $data, null, $response_data['message']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            $error_array = $this->getLastErrorArray();
            $error_message = $e->getMessage();
            
            // Include error array in response if available
            $response = $this->returnResponse(false, null, $error_message);
            if ($error_array !== null) {
                $response['error_array'] = $error_array;
                // Reset error array after use to avoid stale data
                $this->last_error_array = null;
            }
            
            return $response;
        }
    }

    public function getInvoices($order_id = null, $month_added = 0, $year_added = 0, $date_start = null, $date_end = null, $invoice_type = null, $invoice_code = null) {
        try {
            // Build query parameters
            $query_params = [];
            if (!empty($order_id)) {
                $query_params['order_id'] = $order_id;
            }
            if (!empty($month_added)) {
                $query_params['month_added'] = $month_added;
            }
            if (!empty($year_added)) {
                $query_params['year_added'] = $year_added;
            }
            if (!empty($date_start)) {
                $query_params['datestart'] = $date_start;
            }
            if (!empty($date_end)) {
                $query_params['dateend'] = $date_end;
            }
            // 0 = regular invoice, 1 = consolidated invoice
            if (isset($invoice_type) && in_array($invoice_type, [0, 1])) {
                $query_params['invoice_type'] = (int)$invoice_type;
            }

            if (!empty($invoice_code)) {
                $query_params['invoice_code'] = $invoice_code;
            }

            $response_data = $this->makeApiRequest(
                'get_invoices',
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token
                ],
                'getInvoices',
                'handleResponseDataError',
                30,
                $query_params
            );

            return $this->returnResponse(true, $response_data['data']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function getNonInvoiceDocument($invoice_id = null, $document_type = null, $filter = []) {
        try {
            // Build query parameters
            $query_params = [];
            if (!empty($invoice_id)) {
                $query_params['invoice_id'] = $invoice_id;
            }
            if (!empty($filter['month_added'])) {
                $query_params['month_added'] = $filter['month_added'];
            }
            if (!empty($filter['year_added'])) {
                $query_params['year_added'] = $filter['year_added'];
            }
            if (!empty($filter['filter_date_start'])) {
                $query_params['datestart'] = $filter['filter_date_start'];
            }
            if (!empty($filter['filter_date_end'])) {
                $query_params['dateend'] = $filter['filter_date_end'];
            }

            $response_data = $this->makeApiRequest(
                'get_' . $document_type,
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token
                ],
                'getNonInvoiceDocument',
                'handleResponseDataError',
                30,
                $query_params
            );

            return $this->returnResponse(true, $response_data['data']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function createNonInvoiceDocument($document_type, $invoice_code, $total = 0, $note_desc = '') {
        try {
            if (empty($invoice_code)) {
                throw new Exception('Invoice code is required');
            }
            if (empty($document_type)) {
                throw new Exception('Document type is required');
            }

            $payload = ['invoice_code' => $invoice_code];

            if ($total > 0) {
                $payload['total'] = $total;
            }
            if (!empty($note_desc)) {
                $payload['note_desc'] = $note_desc;
            }

            $response_data = $this->makeApiRequest(
                'create_' . $document_type,
                'POST',
                $payload,
                [
                    'Authorization' => $this->auth_token,
                ],
                'createNonInvoiceDocument',
                'handleResponseDataError'
            );

            switch ($document_type) {
                case 'credit_note':
                    $data = [
                        'credit_note_code' => $response_data['credit_note_code'],
                        'credit_note_url' => $response_data['credit_note_url'],
                        'original_invoice_uuid' => $response_data['original_invoice_uuid'],
                    ];
                    break;
                case 'debit_note':
                    $data = [
                        'debit_note_code' => $response_data['debit_note_code'],
                        'debit_note_url' => $response_data['debit_note_url'],
                        'original_invoice_uuid' => $response_data['original_invoice_uuid'],
                    ];
                    break;
                case 'refund_note':
                    $data = [
                        'refund_note_code' => $response_data['refund_note_code'],
                        'refund_note_url' => $response_data['refund_note_url'],
                        'original_invoice_uuid' => $response_data['original_invoice_uuid'],
                    ];
                    break;
            }

            return $this->returnResponse(true, $data, null, $response_data['message']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function getSummary() {
        try {
            $response_data = $this->makeApiRequest(
                'monthly_summary',
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token,
                ],
                'getSummary',
                false // Custom error handling below
            );

            // Custom error handling: check for empty data
            if (empty($response_data['data'])) {
                $message = isset($response_data['error']) ? $response_data['error'] : "No Summary found";
                throw new Exception($message);
            }

            return $this->returnResponse(true, $response_data['data']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function getTotalConsolidatedInvoices($month = 0, $year = 0) {
        try {
            if ($month <= 0 || $year <= 0) {
                throw new Exception('Month and year are required');
            }

            $response_data = $this->makeApiRequest(
                'validated_totals',
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token,
                ],
                'getTotalConsolidatedInvoices',
                false, // Custom error handling below
                30,
                [
                    'month' => $month,
                    'year' => $year
                ]
            );

            // Custom error handling: check for error field
            if (isset($response_data['error'])) {
                throw new Exception($response_data['error']);
            }

            return $this->returnResponse(true, $response_data);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    // this function currently is not used
    public function exportInvoices($start_date, $end_date, $document_type) {
        try {
            // Build query parameters
            $query_params = [];
            if (!empty($start_date)) {
                $query_params['datestart'] = $start_date;
            }
            if (!empty($end_date)) {
                $query_params['dateend'] = $end_date;
            }
            if (!empty($document_type)) {
                $query_params['document_type'] = $document_type;
            }

            $response_data = $this->makeApiRequest(
                'export_invoices',
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token,
                ],
                'exportInvoices',
                false, // Custom error handling below
                30,
                $query_params
            );

            // Custom error handling: check for error field
            if (isset($response_data['error'])) {
                throw new Exception($response_data['error']);
            }

            // Clean null bytes and return raw result (not wrapped in returnResponse)
            $result = preg_replace('/\x00/', '', $response_data);

            return $result;
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function createConsolidatedInvoice($payload = []) {
        try {
            if (empty($payload)) {
                throw new Exception('Payload is required');
            }

            $payload = array_merge($payload, ['channel' => $this->channel]);

            $response_data = $this->makeApiRequest(
                'create_consolidatedinvoice',
                'POST',
                $payload,
                [
                    'Authorization' => $this->auth_token,
                ],
                'createConsolidatedInvoice',
                'handleResponseDataError'
            );

            $data = $response_data;
            unset($data['success']);

            return $this->returnResponse(true, $data);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function cancelInvoice($invoice_code = '') {
        try {
            if (empty($invoice_code)) {
                throw new Exception('Invoice code is required');
            }

            $response_data = $this->makeApiRequest(
                'cancel_invoice',
                'POST',
                ['invoice_code' => $invoice_code],
                [
                    'Authorization' => $this->auth_token,
                ],
                'cancelInvoice',
                'handleResponseDataError'
            );

            return $this->returnResponse(true, $response_data, null, $response_data['message'] ?? '');
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    public function cancelConsolidated($invoice_code) {
        return $this->cancelInvoice($invoice_code);
    }

    public function getSubscriptionInfo() {
        try {
            $response_data = $this->makeApiRequest(
                'get_subscription_info',
                'GET',
                null,
                [
                    'Authorization' => $this->auth_token,
                ],
                'getSubscriptionInfo',
                'handleResponseDataError',
                30,
                [],
                [
                    'Permission denied. You account didnt require our subscription service.',
                ]
            );

            $data = [
                'subscription_payment_url' => $response_data['subscription_payment_url'] ?? '',
            ];

            return $this->returnResponse(true, $data);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    private function getAuthToken() {
        $auth_token = EINVAPPMY_Helper::get_option('myinvois_token');

        if (!empty($auth_token)) {
            return 'Basic ' . $auth_token;
        }

        $auth_token = $this->getClientAuthToken();
        $auth_token_data = $auth_token['data'] ?? [];

        if (isset($auth_token_data['auth_token'])) {
            return 'Basic ' . $auth_token_data['auth_token'];
        }

        return '';
    }

    public function getSubscriptionStatus() {
        try {
            $auth_token = $this->getAuthToken();

            $response_data = $this->makeApiRequest(
                'get_subscription_status',
                'GET',
                null,
                [
                    'Authorization' => $auth_token,
                ],
                'getSubscriptionStatus',
                'handleResponseDataError'
            );

            return $this->returnResponse(true, $response_data['data']);
        } catch (Exception $e) {
            // Error already logged in makeApiRequest(), just return error response
            return $this->returnResponse(false, null, $e->getMessage());
        }
    }

    private function getApiEndpoints($endpoint) {
        $api_url = $this->url . 'api/';

        if (!empty($endpoint)) {
            return isset($this->api_endpoints[$endpoint]) ? $api_url . $this->api_endpoints[$endpoint] : false;
        }

        $api_endpoints = [];

        foreach ($this->api_endpoints as $key => $value) {
            $api_endpoints[$key] = $this->url . $value;
        }

        return $api_endpoints;
    }

    public function getConsolidateStatuses() {
        return $this->config['consolidated_status_text'];
    }

    public function getConsolidatedStatusText($status) {
        return $this->config['consolidated_status_text'][$status] ?? '';
    }

    public function getStatusLabelBackgroundColor($status) {
        return $this->config['status_label_background_color'][$status] ?? $this->config['default_status_label_background_color'];
    }

    public function getInvoiceTypes($invoice_type = null) {
        if ($invoice_type === null || !is_numeric($invoice_type)) {
            return $this->config['invoice_types'];
        }

        return $this->config['invoice_types'][$invoice_type];
    }

    public function getNotesStatusText($status) {
        return $this->config['notes_status_text'][$status] ?? '';
    }

    public function isSubscriptionActive() {
        $saved_subscription_status = get_option('einvappmy_user_subscription_status');

        // default set need to check subscription status from api
        $is_required_get_status_from_api = true;
        // a flag allow to refresh the subscription status from api
        $refresh_subsription_status = sanitize_text_field($_GET['refresh_subsription_status'] ?? 0);

        if (!empty($saved_subscription_status) && $refresh_subsription_status <= 0) {
            $expire_timestamp = strtotime($saved_subscription_status['expired_at']);

            // still havent expired, subscription is active
            if ($expire_timestamp > time()) {
                // since still active, then no need to get the status from api
                $is_required_get_status_from_api = false;

                return true;
            }
        }

        if ($refresh_subsription_status > 0) {
            $is_required_get_status_from_api = true;
        }

        // first time check subscription OR subscription expired
        if ($is_required_get_status_from_api) {
            $subscription_status_response = $this->getSubscriptionStatus();

            if (isset($subscription_status_response['success']) && $subscription_status_response['success']) {
                $subsription_status = $subscription_status_response['data'];

                // api result shown that subscription is not active
                if (!$subsription_status['subscription_active']) {
                    // TODO: thing a way to reduce the number of api call
                    $client_details = $this->getClientData();

                    // check if the account is excluded from subscription service
                    if (!empty($client_details['data']) && $client_details['data']['subscription_exempt']) {
                        return true;
                    }

                    // if the subscription is in trial, then return true
                    if ($subsription_status['is_in_trial']) {
                        return true;
                    }

                    return false;
                } else {
                    // subscription still active, then update the local data
                    update_option('einvappmy_user_subscription_status', $subsription_status);

                    return true;
                }
            } else {
                return false;
            }
        }

        return false;
    }

    private function returnResponse($success = true, $data = null, $error = null, $message = null, $others = []) {
        $result = [
            'success' => $success,
            'data' => $data,
            'error' => $error,
            'message' => $message,
        ];

        return array_merge($result, $others);
    }

    private function handleResponseDataError($response_data = null) {
        if (empty($response_data)) {
            return 'Invalid response from API';
        }

        $error_array = [];

        if (isset($response_data['error'])) {
            if (is_array($response_data['error'])) {
                $error_array = $response_data['error'];
            } else {
                return $response_data['error'];
            }
        }

        if (isset($response_data['errors'])) {
            $error_array = $response_data['errors'];
        }

        if (!empty($error_array)) {
            // Return error array for controller to process
            return [
                'errors' => $error_array,
                'message' => 'Validation errors occurred'
            ];
        }

        if (empty($response_data['success']) || $response_data['success'] !== true) {
            if (isset($response_data['message'])) {
                return $response_data['message'];
            }

            return 'API operation failed without error details';
        }

        return true;
    }

    /**
     * Get the last error array from API response
     * 
     * @return array|null Error array or null if no errors
     */
    public function getLastErrorArray() {
        return $this->last_error_array;
    }
}