<?php

// phpcs:disable audit.php.lang.security.file.inclusion-arg

/**
 * O2OE Invoice Helper Class
 * 
 * Contains utility functions and template loader for the plugin
 */

if (!defined('ABSPATH')) {
    exit('Access denied.');
}

class EINVAPPMY_Helper
{

    /**
     * Plugin configuration
     * @var array
     */
    private static $config = null;

    public static function get_order_info_page_url($id = 0) {
        if (version_compare(WC()->version, '7.1.0', '>=')) {
            $url = admin_url('admin.php?page=wc-orders&action=edit&id=' . $id);
        } else {
            $url = admin_url('post.php?post=' . $id . '&action=edit');
        }

        return $url;
    }

    /**
     * Load and get plugin configuration
     * 
     * @return array Plugin configuration
     */
    public static function get_config() {
        if (self::$config === null) {
            self::load_config();
        }

        return self::$config;
    }

    /**
     * Load configuration from config file
     */
    private static function load_config() {
        $config_file = EINVAPPMY_PLUGIN_PATH . 'includes/config.php';

        if (file_exists($config_file)) {
            self::$config = include $config_file;
        } else {
            // Fallback default config
            self::$config = array(
                'setting_name' => 'einvappmy_settings',
                'option_name'  => 'einvappmy_options',
            );

            self::log('Config file not found, using defaults', 'warning');
        }

        // Ensure config is an array
        if (!is_array(self::$config)) {
            self::$config = array();
            self::log('Invalid config file format', 'error');
        }
    }

    /**
     * Get specific config value
     * 
     * @param string $key Config key
     * @param mixed $default Default value if key not found
     * @return mixed Config value or default
     */
    public static function get_config_value($key, $default = null) {
        $config = self::get_config();
        return isset($config[$key]) ? $config[$key] : $default;
    }

    public static function load_template($template_name, $data = array(), $return_html = false) {
        $allowed_template_names = [
            'admin/consolidated-order-list.php',
            'admin/consolidated-page.php',
            'admin/consolidated-history.php',
            'admin/einvoice-request-page.php',
            'admin/notes-page.php',
            'admin/listing-page.php',
            'admin/settings-page.php',
            'admin/wc-order-einvoice.php',
            'admin/log-listing.php',
            'storefront/wc-order-einvoice.php',
            'storefront/wc-checkout-success-einvoice.php',
            'components/subscription.php',
        ];

        $template_name = sanitize_text_field($template_name);

        if (!in_array($template_name, $allowed_template_names, true)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log("EInvoiceApp Malaysia: Template not allowed - {$template_name}");
            }
            return $return_html ? '' : null;
        }
        
        $template_path = self::locate_template($template_name);

          // Verify the template path is inside your plugin's template directory
        $base_dir = realpath(EINVAPPMY_PLUGIN_PATH . 'templates/');
        $real_template_path = realpath($template_path);

        if (!$template_path || strpos($real_template_path, $base_dir) !== 0) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log("EInvoiceApp Malaysia: Template path invalid or outside allowed directory - {$template_name}");
            }
            return $return_html ? '' : null;
        }

        try {
            if ($return_html) {
                $ob_level = ob_get_level();
                ob_start();
            }
            if (!empty($data) && is_array($data)) {
                extract($data, EXTR_SKIP);
            }
            include $real_template_path;
            if ($return_html) {
                if (ob_get_level() > $ob_level) {
                    $content = ob_get_contents();
                    ob_end_clean();
                    return $content;
                } else {
                    return '';
                }
            }
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log("O2OE Invoice: Template error - " . $e->getMessage());
            }
            if ($return_html && ob_get_level() > 0) {
                ob_end_clean();
            }
            return $return_html ? '' : null;
        }
        return null;
    }

    /**
     * Get template HTML (returns instead of echoing)
     * 
     * @param string $template_name Template file name
     * @param array $data Data to pass to template
     * @return string
     */
    public static function get_template_html($template_name, $data = array()) {
        return self::load_template($template_name, $data, true);
    }

    /**
     * Locate template file
     * 
     * Checks in this order:
     * 1. Theme override: /wp-content/themes/your-theme/o2oeinvoice/template-name.php
     * 2. Plugin template: /wp-content/plugins/o2oeinvoice/templates/template-name.php
     * 
     * @param string $template_name Template file name
     * @return string|false Full path to template or false if not found
     */
    private static function locate_template($template_name) {
        // Remove any leading slashes
        $template_name = ltrim($template_name, '/');

        // Check plugin templates directory
        $plugin_template = EINVAPPMY_PLUGIN_PATH . 'templates/' . $template_name;

        if (file_exists($plugin_template)) {
            return $plugin_template;
        }

        return false;
    }

    /**
     * Render admin notice template
     * 
     * @param string $message Notice message
     * @param string $type Notice type (success, error, warning, info)
     * @param bool $dismissible Whether notice is dismissible
     */
    public static function admin_notice($message, $type = 'info', $dismissible = true) {
        $data = array(
            'message'     => $message,
            'type'        => $type,
            'dismissible' => $dismissible
        );

        self::load_template('admin/notice.php', $data);
    }

    /**
     * Get plugin option with default fallback (UPDATED to use config)
     * 
     * @param string $option_name Option name
     * @param mixed $default Default value
     * @return mixed Option value or default
     */
    public static function get_option($option_name, $default = null) {
        $option_key = self::get_config_value('option_name', 'einvappmy_options');
        $options = get_option($option_key, array());
        return isset($options[$option_name]) ? $options[$option_name] : $default;
    }

    /**
     * Update plugin option (UPDATED to use config)
     * 
     * @param string $option_name Option name
     * @param mixed $value Option value
     * @return bool Success
     */
    public static function update_option($option_name, $value) {
        $option_key = self::get_config_value('option_name', 'einvappmy_options');
        $options = get_option($option_key, array());
        $options[$option_name] = $value;
        return update_option($option_key, $options);
    }

    /**
     * Get all plugin options (NEW method using config)
     * 
     * @return array All plugin options
     */
    public static function get_all_options() {
        $option_key = self::get_config_value('option_name', 'einvappmy_options');
        return get_option($option_key, array());
    }

    /**
     * Update all plugin options (NEW method using config)
     * 
     * @param array $options Options array
     * @return bool Success
     */
    public static function update_all_options($options) {
        $option_key = self::get_config_value('option_name', 'einvappmy_options');
        return update_option($option_key, $options);
    }

    /**
     * Log debug message (only if WP_DEBUG is enabled)
     * 
     * @param string $message Log message
     * @param string $level Log level (info, warning, error)
     */
    public static function log($message, $level = 'info') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            if (is_array($message) || is_object($message)) {
                $message = json_encode($message);
            }
            
            $formatted_message = sprintf(
                '[O2OE Invoice][%s] %s',
                strtoupper($level),
                $message
            );
            error_log($formatted_message);
        }
    }

    /**
     * Write custom log to file in plugin directory
     * 
     * @param string $filename Log filename (without extension, .log will be added)
     * @param string $message Log message
     * @param string $level Log level (info, warning, error)
     * @return bool Success status
     */
    public static function write_custom_log($filename, $message, $level = 'info') {
        // Sanitize filename and add .log extension if not present
        $filename = sanitize_file_name($filename);
        if (!str_ends_with($filename, '.log')) {
            $filename .= '.log';
        }
        
        // Create logs directory in plugin folder
        $log_dir = EINVAPPMY_PLUGIN_PATH . 'logs/';
        
        // Create logs directory if it doesn't exist
        if (!file_exists($log_dir)) {
            if (!wp_mkdir_p($log_dir)) {
                return false;
            }
            
            // Create .htaccess file to protect log directory from web access
            $htaccess_file = $log_dir . '.htaccess';
            $htaccess_content = "Order Deny,Allow\nDeny from all\n";
            file_put_contents($htaccess_file, $htaccess_content);
        }
        
        // Prepare log file path
        $log_file = $log_dir . $filename;
        
        // Format message with timestamp and level
        if (is_array($message) || is_object($message)) {
            $message = json_encode($message, JSON_PRETTY_PRINT);
        }
        
        $timestamp = wp_date('Y-m-d H:i:s');
        $formatted_message = sprintf(
            "[%s][%s] %s\n",
            $timestamp,
            strtoupper($level),
            $message
        );
        
        // Write to log file (create if not exists, append if exists)
        $result = file_put_contents($log_file, $formatted_message, FILE_APPEND | LOCK_EX);
        
        return $result !== false;
    }

    /**
     * Read custom log file content
     * 
     * @param string $filename Log filename
     * @param int $lines Number of lines to read from end (0 = all lines)
     * @return array|false Array of log entries or false on failure
     */
    public static function read_custom_log($filename, $lines = 0) {
        // Sanitize filename and add .log extension if not present
        $filename = sanitize_file_name($filename);
        if (!str_ends_with($filename, '.log')) {
            $filename .= '.log';
        }
        
        $log_dir = EINVAPPMY_PLUGIN_PATH . 'logs/';
        $log_file = $log_dir . $filename;
        
        // Check if file exists
        if (!file_exists($log_file)) {
            return false;
        }
        
        // Read file content
        $content = file_get_contents($log_file);
        if ($content === false) {
            return false;
        }
        
        // Split into lines and remove empty lines
        $log_lines = array_filter(explode("\n", trim($content)));
        
        // Get specified number of lines from the end
        if ($lines > 0 && count($log_lines) > $lines) {
            $log_lines = array_slice($log_lines, -$lines);
        }
        
        // Parse each line into structured data
        $log_entries = [];
        $current_entry = null;
        
        foreach ($log_lines as $line) {
            // Parse format: [timestamp][level] message
            if (preg_match('/^\[([^\]]+)\]\[([^\]]+)\] (.+)$/', $line, $matches)) {
                // Save previous entry if exists
                if ($current_entry !== null) {
                    $log_entries[] = $current_entry;
                }
                
                // Start new entry
                $current_entry = [
                    'timestamp' => $matches[1],
                    'level' => $matches[2],
                    'message' => $matches[3],
                    'raw' => $line
                ];
            } else {
                // This is a continuation line (part of previous entry's message)
                if ($current_entry !== null) {
                    // Append to current entry's message
                    $current_entry['message'] .= "\n" . $line;
                    $current_entry['raw'] .= "\n" . $line;
                } else {
                    // No previous entry, treat as standalone line
                    $log_entries[] = [
                        'timestamp' => '',
                        'level' => 'INFO',
                        'message' => $line,
                        'raw' => $line
                    ];
                }
            }
        }
        
        // Don't forget to add the last entry
        if ($current_entry !== null) {
            $log_entries[] = $current_entry;
        }
        
        return array_reverse($log_entries); // Show newest entries first
    }

    /**
     * Get list of available custom log files
     * 
     * @return array Array of log files with metadata
     */
    public static function get_custom_log_files() {
        $log_dir = EINVAPPMY_PLUGIN_PATH . 'logs/';
        
        if (!file_exists($log_dir)) {
            return [];
        }
        
        $files = glob($log_dir . '*.log');
        $log_files = [];
        
        foreach ($files as $file) {
            $filename = basename($file);
            $log_files[] = [
                'filename' => $filename,
                'filepath' => $file,
                'size' => filesize($file),
                'size_formatted' => size_format(filesize($file)),
                'modified' => filemtime($file),
                'modified_formatted' => wp_date('Y-m-d H:i:s', filemtime($file))
            ];
        }
        
        // Sort by modified time (newest first)
        usort($log_files, function($a, $b) {
            return $b['modified'] - $a['modified'];
        });
        
        return $log_files;
    }

    /**
     * Clear custom log file content
     * 
     * @param string $filename Log filename
     * @return bool Success status
     */
    public static function clear_custom_log($filename) {
        // Sanitize filename and add .log extension if not present
        $filename = sanitize_file_name($filename);
        if (!str_ends_with($filename, '.log')) {
            $filename .= '.log';
        }
        
        $log_dir = EINVAPPMY_PLUGIN_PATH . 'logs/';
        $log_file = $log_dir . $filename;
        
        if (!file_exists($log_file)) {
            return false;
        }
        
        return file_put_contents($log_file, '') !== false;
    }

    /**
     * Delete custom log file
     * 
     * @param string $filename Log filename
     * @return bool Success status
     */
    public static function delete_custom_log($filename) {
        // Sanitize filename and add .log extension if not present
        $filename = sanitize_file_name($filename);
        if (!str_ends_with($filename, '.log')) {
            $filename .= '.log';
        }
        
        $log_dir = EINVAPPMY_PLUGIN_PATH . 'logs/';
        $log_file = $log_dir . $filename;
        
        if (!file_exists($log_file)) {
            return false;
        }
        
        return unlink($log_file);
    }

    /**
     * Check if WooCommerce is active and loaded
     * 
     * @return bool
     */
    public static function is_woocommerce_active() {
        return class_exists('WooCommerce');
    }

    /**
     * Format currency amount
     * 
     * @param float $amount Amount to format
     * @return string Formatted amount
     */
    public static function format_currency($amount) {
        if (self::is_woocommerce_active()) {
            return wc_price($amount);
        }

        return '$' . number_format($amount, 2);
    }

    /**
     * Get current user capabilities for plugin
     * 
     * @return array User capabilities
     */
    public static function get_user_capabilities() {
        return array(
            'can_manage_settings' => current_user_can('manage_options'),
            'can_view_orders'     => current_user_can('edit_shop_orders'),
            'can_export_invoices' => current_user_can('manage_woocommerce')
        );
    }

    public static function get_default_country_state() {
        $default_country = get_option('woocommerce_default_country');

        if (empty($default_country)) {
            return [];
        }

        $country_state = explode(':', $default_country);
        $countries = WC()->countries->get_countries();
        $states = WC()->countries->get_states($country_state[0]);

        $result = [
            'country'      => $country_state[0],
            'country_name' => $countries[$country_state[0]] ?? '',
            'state'        => $country_state[1],
            'state_name'   => $states[$country_state[1]] ?? ''
        ];

        return $result;
    }

    public static function get_country_by_code($code) {
        $countries = WC()->countries->get_countries();
        return $countries[$code] ?? '';
    }

    public static function get_state_by_code($country_code, $state_code) {
        $states = WC()->countries->get_states($country_code);
        return $states[$state_code] ?? '';
    }

    public static function get_consolidated_status_text($status) {
        $config = self::get_config();
        
        return $config['consolidated_status_text'][$status] ?? '';
    } 
}