<?php

if ( ! defined( 'ABSPATH' ) ) exit;

class EINVAPPMY_Order_Handler
{
    private $config;
    private $model_einvappmy;
    private $einvappmy_library;

    public function __construct($config) {
        $this->config = $config;
        $this->model_einvappmy = new Model_EINVAPPMY($this->config);
        $this->einvappmy_library = new EINVAPPMY_Library($this->config);

        // add_action('woocommerce_order_details_after_order_table', array($this, 'render_einvoice_section'));
        add_action('woocommerce_view_order', array($this, 'render_einvoice_section'), 25);
        add_action('woocommerce_thankyou', array($this, 'render_einvoice_checkout_success_section'), 25);
        
        add_action('wp_ajax_einvappmy_myeinvoice_check_status', array($this, 'check_einvoice_status'));
        add_action('wp_ajax_nopriv_einvappmy_myeinvoice_check_status', array($this, 'check_einvoice_status'));

        add_action('wp_ajax_einvappmy_storefront_myeinvoice_request', array($this, 'myeinvoice_request'));
        add_action('wp_ajax_nopriv_einvappmy_storefront_myeinvoice_request', array($this, 'myeinvoice_request'));
    }

    /**
     * Get WooCommerce order object with backward compatibility
     * 
     * Supports both old (WooCommerce 2.x) and new (WooCommerce 3.0+) methods
     *
     * @param int $order_id The order ID
     * @return WC_Order|false Order object or false on failure
     */
    private function get_order($order_id) {
        // WooCommerce 3.0+: wc_get_order()
        // WooCommerce 2.x: new WC_Order()
        if (function_exists('wc_get_order')) {
            return wc_get_order($order_id);
        } else {
            return new WC_Order($order_id);
        }
    }

    public function enqueue_einvoice_scripts($data = []): void {
        wp_enqueue_style('einvappmy-storefront-style', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/storefront.css', [], '1.0.1');
        wp_enqueue_script('jquery'); // Ensure jQuery is loaded
        wp_enqueue_script('einvappmy-qrcode', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/qrcode.min.js', array('jquery'), null, true);
        wp_enqueue_script('einvappmy-script', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/o2oei-script.js?v=' . time(), array('jquery'), null, true);
        
        // Localize script to pass data
        wp_localize_script('einvappmy-script', 'einvappmyData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'order_id' => $data['order_id'], // Make sure $order_id is available here
            'date_added' => $data['date_added'],
            'nonce' => wp_create_nonce('einvappmy_nonce')
        ));
    }

    public function render_einvoice_checkout_success_section($order_id) {
        wp_enqueue_style('einvappmy-storefront-style', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/storefront.css', [], '1.0.1');
        $output_data = [];
        $einapp_option = get_option($this->config['option_name'], []);

        $is_myinvoice_section_to_show = true;
        $output_data['myinvois_token'] = sanitize_text_field($einapp_option['myinvois_token'] ?? '');
        $output_data['order_id'] = absint($order_id);

        $order_info = $this->get_order($order_id);

        if (
            empty($output_data['myinvois_token']) 
            || $einapp_option['status'] == 0 
            || ($einapp_option['status'] == 1 && !current_user_can( 'manage_options' ))
            || !$this->einvappmy_library->isSubscriptionActive()
            || empty($order_info)
        ) {
            $is_myinvoice_section_to_show = false;
        } else {
            $is_myinvoice_section_to_show = true;
        }

        $output_data['link_order_info_einvoice_section'] = $order_info->get_view_order_url() . '#einvappmy-myinvoice-section';
        $output_data['is_myinvoice_section_to_show'] = $is_myinvoice_section_to_show;

        EINVAPPMY_Helper::load_template('storefront/wc-checkout-success-einvoice.php', $output_data);
    }

    public function render_einvoice_section($order_id) {
        $output_data = [];
        $output_data['myinvois_token'] = EINVAPPMY_Helper::get_option('myinvois_token');
        $output_data['order_id'] = $order_id;

        $order = $this->get_order($order_id);
        $output_data['date_added'] = $order->get_date_created()->date('Y-m-d H:i:s');

        $this->enqueue_einvoice_scripts($output_data);

        $myinvois_options = get_option($this->config['option_name'], []);
        $status = $myinvois_options['status'] ?? 0;
        $is_myinvoice_section_to_show = false;

        // no token, disable or (staging but is not admin login)
        if (empty($output_data['myinvois_token']) || $status == 0 || ($status == 1 && !current_user_can( 'manage_options' ))) {
            $is_myinvoice_section_to_show = false;
        } else {
            $is_myinvoice_section_to_show = true;
        }

        // if the website owner still havent subscribe to the service, then disable the myinvoice section
        if (!$this->einvappmy_library->isSubscriptionActive()) {
            $is_myinvoice_section_to_show = false;
        }

        $output_data['is_myinvoice_section_to_show'] = $is_myinvoice_section_to_show;

        EINVAPPMY_Helper::load_template('storefront/wc-order-einvoice.php', $output_data);
    }

    public function check_einvoice_status() {
        $date_added = sanitize_text_field($_GET['date_added']);
        $date_added_monthyear = DateTime::createFromFormat('Y-m-d H:i:s', $date_added);
        $current_date = new DateTime();

        $allow_create_invoice_statuses = $this->config['allow_create_invoice_statuses'];

        if ($date_added_monthyear->format('Y-m') == $current_date->format('Y-m')) {
            $date_added_same_month = true;
        } else {
            $date_added_same_month = false;
        }

        if ($date_added_same_month) {
            $order_id = absint($_GET['order_id']);
            $order_info = $this->get_order($order_id);

            $allow_create_invoice_statuses = $this->config['allow_create_invoice_statuses'];
            $cancel_or_refund_order_statuses = $this->config['cancel_or_refund_order_statuses'];
            $pending_order_statuses = $this->config['pending_order_statuses'];
            $fallback_message_order_statuses = $this->config['fallback_message_order_statuses'];

            // if order status is refund or cancelled, then show message invoice creation is not available for canceled or refunded order
            if (in_array($order_info->get_status(), $cancel_or_refund_order_statuses)) {
                wp_send_json([
                    'success' => false,
                    'error' => esc_html__('Invoice creation is not available for canceled or refunded order', 'einvoiceapp-malaysia')
                ]);
            }

            // if order status is pending, then show message invoice creation is not available for pending order
            if (in_array($order_info->get_status(), $pending_order_statuses)) {
                wp_send_json([
                    'success' => false,
                    'error' => esc_html__('Your invoice will be available to create once your payment is made. Please complete the payment first.', 'einvoiceapp-malaysia')
                ]);
            }

            // fallback message invoice generation is not available for this order at the moment
            if (in_array($order_info->get_status(), $fallback_message_order_statuses) || !in_array($order_info->get_status(), $allow_create_invoice_statuses)) {
                wp_send_json([
                    'success' => false,
                    'error' => esc_html__('Invoice creation is not available for this order at the moment. Please try again later.', 'einvoiceapp-malaysia')
                ]);
            }

            $invoice = $this->einvappmy_library->getInvoices($order_id);
            $success = false;
            
            if (!empty($invoice['data']) && count($invoice['data']) == 1) {
                $success = true;
                $invoice_code = $invoice['data'][0]['invoice_code'];
                $invoice['invoice_url'] = $this->einvappmy_library->getInvoiceUrl($invoice_code);
                $invoice['status'] = $this->einvappmy_library->getConsolidatedStatusText($invoice['data'][0]['status']);
                $invoice['status_color'] = $this->einvappmy_library->getStatusLabelBackgroundColor($invoice['data'][0]['status']);
                
                // Add PDF URL if status is 1 (Validated)
                if ($invoice['data'][0]['status'] == 1) {
                    $invoice['pdf_url'] = $this->einvappmy_library->getPdfUrl($invoice['data'][0]['uuid']);
                }
            }

            $output = $invoice;

            if (!$success) {
                $output['error'] = $invoice['error'];
            }
            
            wp_send_json($output);
        } else {
            wp_send_json([
                'success' => false,
                'error' => esc_html__('Expired', 'einvoiceapp-malaysia')
            ]);
        }
    }

    public function myeinvoice_request() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_nonce')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $order_id = absint($_POST['order_id']);

        if (empty($order_id)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Order ID is required', 'einvoiceapp-malaysia')
            ]);
        }

        $order_info = wc_get_order($order_id);

        if (empty($order_info)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Order not found', 'einvoiceapp-malaysia')
            ]);
        }

        $order_products = $order_info->get_items();
        $order_totals = $order_info->get_total();

        $payload_order_products = [];

        foreach ($order_products as $order_product) {
            $price = $order_product->get_total() / $order_product->get_quantity();

            $payload_order_products[] = [
                'name' => $order_product['name'],
                'type' => 'sale',
                'price' => number_format($price, 2, '.', ''),
                'quantity' => $order_product->get_quantity(),
                'item_tax_total' => number_format($order_product->get_total_tax(), 2, '.', ''),
                'item_price_total' => number_format($order_product->get_total(), 2, '.', '')
            ];
        }

        if ($order_info->get_shipping_total() > 0) {
            $payload_order_products[] = [
               'name' => 'Delivery Fee',
               'type' => 'sale',
               'price' => number_format(abs($order_info->get_shipping_total()), 2, '.', ''),
               'quantity' => 1,
               'item_tax_total' => 0,
               'item_price_total' => number_format(abs($order_info->get_shipping_total()), 2, '.', '')
            ];
        }

        $coupon_total_discount_amount = 0;
        $used_coupons = $order_info->get_coupon_codes();
        if (!empty($used_coupons)) {
            foreach ($used_coupons as $used_coupon) {
                $coupon_info = new WC_Coupon($used_coupon);
                $coupon_total_discount_amount += $coupon_info->get_amount();
            }
        }

        if ($coupon_total_discount_amount > 0) {
            $payload_order_products[] = array(
                'name' => 'Coupon Discount',
                'type' => 'discount',
                'price' => number_format(abs($coupon_total_discount_amount), 2, '.', ''),
                'quantity' => 1,
                'item_tax_total' => 0,
                'item_price_total' => number_format(abs($coupon_total_discount_amount), 2, '.', ''),
            );
        }

        $customer_name = $order_info->get_billing_first_name() . ' ' . $order_info->get_billing_last_name();
        $states = $this->config['states'];

        $address1 = $order_info->get_shipping_address_1();
        $address2 = $order_info->get_shipping_address_2();
        $city = $order_info->get_shipping_city();
        $postcode = $order_info->get_shipping_postcode();
        $state = $order_info->get_shipping_state();
        $country = $order_info->get_shipping_country();
        $country_code = $order_info->get_shipping_country();

        $address1 = empty($address1) ? $order_info->get_billing_address_1() : $address1;
        $address2 = empty($address2) ? $order_info->get_billing_address_2() : $address2;
        $city = empty($city) ? $order_info->get_billing_city() : $city;
        $postcode = empty($postcode) ? $order_info->get_billing_postcode() : $postcode;
        $state = empty($state) ? $order_info->get_billing_state() : $state;
        $country = empty($country) ? $order_info->get_billing_country() : $country;
        $country_code = empty($country_code) ? $order_info->get_billing_country() : $country_code;

        $country = EINVAPPMY_Helper::get_country_by_code($country);
        $state = $states[$state]['value'];

        $payload = [
            'order_id' => $order_id,
            'order_date' => $order_info->get_date_created()->date('Y-m-d H:i:s'),
            'customer_name' => $customer_name,
            'telephone' => $order_info->get_billing_phone(),
            'email' => $order_info->get_billing_email(),
            'address1' => $address1, // Shipping address line 1
            'address2' => $address2, // Shipping address line 2
            'city' => $city, // Shipping city
            'postcode' => $postcode, // Shipping postcode
            'state' => $state, // Shipping state
            'country' => $country, // Shipping country
            'country_code' => $country_code, // You may need to map this to ISO code if needed
            'currency_code' => get_woocommerce_currency(), // Get the currency code for the order
            'items' => $payload_order_products,
            'order_tax_total' => number_format($order_info->get_total_tax(), 2, '.', ''),
            'order_total' => number_format($order_totals, 2, '.', ''),
        ];

        $result = $this->einvappmy_library->createInvoice($payload);

        wp_send_json([
            'success' => isset($result['success']),
            'message' => $result['message'] ?? "",
            'data' => $result['data'] ?? []
        ]);
    }
}