<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class EINVAPPMY_WC_Order
{
    private $config;
    private $einvappmy_library;

    public function __construct($config) {
        $this->config = $config;
        $this->einvappmy_library = new EINVAPPMY_Library($this->config);

        $this->init();
    }

    public function init() {
        // hook into woocommerce order page and add the einvoice section
        add_action('add_meta_boxes', array($this, 'add_order_meta_boxes'));
        add_action('wp_ajax_einvappmy_myeinvoice_request', array($this, 'request_myeinvoice'));
        add_action('wp_ajax_einvappmy_myeinvoice_create_document', array($this, 'request_myeinvoice_create_document'));
    }

    public function request_myeinvoice(): void {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_myeinvoice_request')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }


        $post_data_order_id = sanitize_text_field(wp_unslash($_POST['order_id']));
        $post_data_create_date = sanitize_text_field(wp_unslash($_POST['create_date']));

        if (empty($post_data_order_id)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Order ID is required', 'einvoiceapp-malaysia')
            ]);
        }

        $order_info = wc_get_order($post_data_order_id);

        if (empty($order_info)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Order not found', 'einvoiceapp-malaysia')
            ]);
        }

        $order_products = $order_info->get_items();
        $order_totals = $order_info->get_total();

        if (empty($order_products)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('No products found in the order', 'einvoiceapp-malaysia')
            ]);
        }

        $payload_order_products = [];

        foreach ($order_products as $order_product) {
            $price = $order_product->get_total() / $order_product->get_quantity();

            $payload_order_products[] = [
                'name'             => $order_product['name'],
                'type'             => 'sale',
                'price'            => number_format($price, 2, '.', ''),
                'quantity'         => $order_product->get_quantity(),
                'item_tax_total'   => number_format($order_product->get_total_tax(), 2, '.', ''),
                'item_price_total' => number_format($order_product->get_total(), 2, '.', '')
            ];
        }

        if ($order_info->get_shipping_total() > 0) {
            $payload_order_products[] = [
                'name'             => 'Delivery Fee',
                'type'             => 'sale',
                'price'            => number_format(abs($order_info->get_shipping_total()), 2, '.', ''),
                'quantity'         => 1,
                'item_tax_total'   => 0,
                'item_price_total' => number_format(abs($order_info->get_shipping_total()), 2, '.', '')
            ];
        }

        $coupon_total_discount_amount = 0;
        $used_coupons = $order_info->get_coupon_codes();
        if (!empty($used_coupons)) {
            foreach ($used_coupons as $used_coupon) {
                $coupon_info = new WC_Coupon($used_coupon);
                $coupon_total_discount_amount += $coupon_info->get_amount();
            }
        }

        if ($coupon_total_discount_amount > 0) {
            $payload_order_products[] = array(
                'name'             => 'Coupon Discount',
                'type'             => 'discount',
                'price'            => number_format(abs($coupon_total_discount_amount), 2, '.', ''),
                'quantity'         => 1,
                'item_tax_total'   => 0,
                'item_price_total' => number_format(abs($coupon_total_discount_amount), 2, '.', ''),
            );
        }

        $customer_name = $order_info->get_billing_first_name() . ' ' . $order_info->get_billing_last_name();
        $states = $this->config['states'];

        $address1 = $order_info->get_shipping_address_1();
        $address2 = $order_info->get_shipping_address_2();
        $city = $order_info->get_shipping_city();
        $postcode = $order_info->get_shipping_postcode();
        $state = $order_info->get_shipping_state();
        $country = $order_info->get_shipping_country();
        $country_code = $order_info->get_shipping_country();

        $address1 = empty($address1) ? $order_info->get_billing_address_1() : $address1;
        $address2 = empty($address2) ? $order_info->get_billing_address_2() : $address2;
        $city = empty($city) ? $order_info->get_billing_city() : $city;
        $postcode = empty($postcode) ? $order_info->get_billing_postcode() : $postcode;
        $state = empty($state) ? $order_info->get_billing_state() : $state;
        $country = empty($country) ? $order_info->get_billing_country() : $country;
        $country_code = empty($country_code) ? $order_info->get_billing_country() : $country_code;

        $country = EINVAPPMY_Helper::get_country_by_code($country);
        $state = $states[$state]['value'];

        if (empty($country)) {
            $country = $this->config['default_country'];
        }

        // Validate payload fields before sending to middleware
        $validation_result = $this->validate_order_for_myeinvoice(
            $post_data_order_id, 
            $order_info, 
            $order_totals, 
            $payload_order_products,
            [
                'create_date' => $post_data_create_date,
            ]
        );

        if (!$validation_result['valid']) {
            wp_send_json([
                'success' => false,
                'message' => implode(' ', $validation_result['errors']),
            ]);
        }

        $payload = [
            'order_id'        => $validation_result['order_id'],
            'order_date'      => $validation_result['order_date'],
            'customer_name'   => $customer_name,
            'telephone'       => $order_info->get_billing_phone(),
            'email'           => $order_info->get_billing_email(),
            'address1'        => $address1, // Shipping address line 1
            'address2'        => $address2, // Shipping address line 2
            'city'            => $city, // Shipping city
            'postcode'        => $postcode, // Shipping postcode
            'state'           => $state, // Shipping state
            'country'         => $country, // Shipping country
            'country_code'    => $country_code, // You may need to map this to ISO code if needed
            'currency_code'   => $validation_result['currency_code'], // Get the currency code for the order
            'items'           => $payload_order_products,
            'order_tax_total' => number_format($validation_result['order_tax_total_raw'], 2, '.', ''),
            'order_total'     => number_format($validation_result['order_total_raw'], 2, '.', ''),
        ];

        $result = $this->einvappmy_library->createInvoice($payload);

        // Process error array if present
        $error_html = '';
        if (!empty($result['error_array']) && is_array($result['error_array'])) {
            $error_html = $this->process_api_errors($result['error_array']);
        }

        wp_send_json([
            // Ensure this is a real boolean, not just "field exists"
            'success' => !empty($result['success']) && $result['success'] === true,
            'message' => $result['message'] ?? '',
            'error'   => $error_html ?: ($result['error'] ?? ''),
            'error_array' => $result['error_array'] ?? null,
            'data'    => $result['data'] ?? []
        ]);
    }

    /**
     * Process API error array and create user-friendly messages
     * 
     * @param array $error_array Error array from API response
     * @return string HTML formatted error message
     */
    private function process_api_errors($error_array) {
        if (empty($error_array) || !is_array($error_array)) {
            return '';
        }

        // Field name mapping for user-friendly display
        $field_labels = [
            'customer_name' => esc_html__('Customer Name', 'einvoiceapp-malaysia'),
            'address1' => esc_html__('Address', 'einvoiceapp-malaysia'),
            'city' => esc_html__('City', 'einvoiceapp-malaysia'),
            'postcode' => esc_html__('Postcode', 'einvoiceapp-malaysia'),
            'state' => esc_html__('State', 'einvoiceapp-malaysia'),
            'country' => esc_html__('Country', 'einvoiceapp-malaysia'),
            'country_code' => esc_html__('Country Code', 'einvoiceapp-malaysia'),
            'telephone' => esc_html__('Phone Number', 'einvoiceapp-malaysia'),
            'email' => esc_html__('Email Address', 'einvoiceapp-malaysia'),
            'items' => esc_html__('Order Items', 'einvoiceapp-malaysia'),
            'order_total' => esc_html__('Order Total', 'einvoiceapp-malaysia'),
            'order_tax_total' => esc_html__('Tax Total', 'einvoiceapp-malaysia'),
            'currency_code' => esc_html__('Currency', 'einvoiceapp-malaysia'),
        ];

        $errors_html = '<strong>' . esc_html__('We found some issues with the information provided:', 'einvoiceapp-malaysia') . '</strong><br>';
        $errors_html .= '<ul>';

        foreach ($error_array as $field => $messages) {
            $field_label = isset($field_labels[$field]) ? $field_labels[$field] : ucfirst(str_replace('_', ' ', $field));
            $message_array = (array) $messages;

            foreach ($message_array as $message) {
                // Convert technical API messages to user-friendly ones
                $friendly_message = $this->convert_error_message_to_friendly($message);

                $errors_html .= sprintf(
                    '<li><strong>%s:</strong> %s</li>',
                    esc_html($field_label),
                    esc_html($friendly_message)
                );
            }
        }

        $errors_html .= '</ul>';

        return $errors_html;
    }

    /**
     * Convert technical API error messages to user-friendly ones
     * 
     * @param string $message Original error message from API
     * @return string User-friendly error message
     */
    private function convert_error_message_to_friendly($message) {
        $message_lower = strtolower($message);

        // Handle "must have a value" messages
        if (stripos($message, 'must have a value') !== false || stripos($message, 'required') !== false) {
            return esc_html__('This field is required. Please provide this information.', 'einvoiceapp-malaysia');
        }

        // Handle "must be a valid" messages
        if (stripos($message, 'must be a valid') !== false) {
            return esc_html__('The format is invalid. Please check and correct it.', 'einvoiceapp-malaysia');
        }

        // Handle "must be greater than" messages
        if (stripos($message, 'must be greater than') !== false) {
            return esc_html__('The value is too small. Please check the amount.', 'einvoiceapp-malaysia');
        }

        // Handle "must be less than" messages
        if (stripos($message, 'must be less than') !== false) {
            return esc_html__('The value is too large. Please check the amount.', 'einvoiceapp-malaysia');
        }

        // Remove technical terms like "The [field] field" and "must"
        $friendly = $message;
        
        // Remove "The [field] field" prefix
        $friendly = preg_replace('/^The\s+\w+\s+field\s+/i', '', $friendly);
        
        // Replace "must" with more friendly language
        $friendly = preg_replace('/\bmust\b/i', 'needs to', $friendly);
        
        // Capitalize first letter
        $friendly = ucfirst(trim($friendly));

        return esc_html($friendly);
    }

    public function request_myeinvoice_create_document() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_myeinvoice_create_document')) {
            wp_send_json([
                'success' => false,
                'message' => 'Invalid nonce'
            ]);
        }

        $post_data_document_type = sanitize_text_field(wp_unslash($_POST['document_type']));
        $post_data_invoice_code = sanitize_text_field(wp_unslash($_POST['invoice_code']));

        if (empty($post_data_document_type)) {
            wp_send_json([
                'success' => false,
                'message' => 'Document type is required'
            ]);
        }

        if (empty($post_data_invoice_code)) {
            wp_send_json([
                'success' => false,
                'message' => 'Invoice code is required'
            ]);
        }

        $result = $this->einvappmy_library->createNonInvoiceDocument($post_data_document_type, $post_data_invoice_code);

        wp_send_json([
            'success' => isset($result['success']),
            'message' => $result['message'] ?? "",
            'data'    => $result
        ]);
    }

    /**
     * Validate order fields before sending e-invoice request to middleware.
     *
     * @param int      $order_id
     * @param WC_Order $order_info
     * @param float    $order_totals
     * @param array    $items
     * @return array{
     *   valid: bool,
     *   errors: string[],
     *   order_id?: int,
     *   order_date?: string,
     *   currency_code?: string,
     *   order_tax_total_raw?: float,
     *   order_total_raw?: float
     * }
     */
    private function validate_order_for_myeinvoice($order_id, $order_info, $order_totals, $items, $others = []) {
        $errors = [];

        // order_id > 0
        $validated_order_id = absint($order_id);
        if ($validated_order_id <= 0) {
            $errors[] = esc_html__('Invalid order ID.', 'einvoiceapp-malaysia');
        }

        if (!empty($others['create_date'])) {
            if ($others['create_date'] == 'order_creation_date') {
                $order_date_obj = $order_info->get_date_created();
            } elseif ($others['create_date'] == 'order_paid_date') {
                $order_date_obj = $order_info->get_date_paid();
            } elseif ($others['create_date'] == 'order_completed_date') {
                $order_date_obj = $order_info->get_date_completed();
            }
        } else {
            // fallback will use order creation date
            $order_date_obj = $order_info->get_date_created();
        }

        $order_date_str = $order_date_obj ? $order_date_obj->date('Y-m-d H:i:s') : '';
        if (empty($order_date_str) || !DateTime::createFromFormat('Y-m-d H:i:s', $order_date_str)) {
            $errors[] = esc_html__('Invalid order date.', 'einvoiceapp-malaysia');
        }

        // currency_code string (non-empty)
        $currency_code = get_woocommerce_currency();
        if (empty($currency_code) || !is_string($currency_code)) {
            $errors[] = esc_html__('Invalid currency code.', 'einvoiceapp-malaysia');
        }

        // order_tax_total >= 0
        $order_tax_total_raw = (float) $order_info->get_total_tax();
        if ($order_tax_total_raw < 0) {
            $errors[] = esc_html__('Order tax total must be greater than or equal to 0.', 'einvoiceapp-malaysia');
        }

        // order_total >= 0
        $order_total_raw = (float) $order_totals;
        if ($order_total_raw < 0) {
            $errors[] = esc_html__('Order total must be greater than or equal to 0.', 'einvoiceapp-malaysia');
        }

        // items array (non-empty)
        if (empty($items) || !is_array($items)) {
            $errors[] = esc_html__('Order items data is invalid.', 'einvoiceapp-malaysia');
        }

        return [
            'valid'             => empty($errors),
            'errors'            => $errors,
            'order_id'          => $validated_order_id,
            'order_date'        => $order_date_str,
            'currency_code'     => $currency_code,
            'order_tax_total_raw' => $order_tax_total_raw,
            'order_total_raw'   => $order_total_raw,
        ];
    }

    public function add_order_meta_boxes() {
        $screen = function_exists('wc_get_page_screen_id') ? wc_get_page_screen_id('shop-order') : 'shop_order';
        $myinvois_token = EINVAPPMY_Helper::get_option('myinvois_token');
        $myinvois_status = EINVAPPMY_Helper::get_option('status');

        if (!empty($myinvois_token) || $myinvois_status != 0) {
            // Add for traditional post-based orders
            add_meta_box(
                'einvappmy-einvoice',                    // Meta box ID
                'E-Invoice',                      // Title
                array($this, 'render_einvoice_section_html'), // Callback
                $screen,                             // Screen
                'advanced',                              // Context (normal, side, advanced)
                'high'                               // Priority
            );
        }
    }

    public function render_einvoice_section_html($order) {
        wp_enqueue_script('qrcode-js', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/qrcode.min.js', array('jquery'), null, true);
        wp_enqueue_script('wc-order-einvoice', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/wc-order-einvoice.js', array('jquery'), null, true);

        if (isset($order->ID)) {
            $order = wc_get_order($order->ID);
        }

        if ($order->get_status() == 'auto-draft') {
            echo '<p>This order is still in draft status. Please save the order first before generating e-invoice.</p>';
            return;
        }

        $output = [];
        $einvoice_status = $this->checkEInvoiceStatus($order);
        $output = array_merge($output, $einvoice_status);
        $output['order_id'] = $order->get_id();
        $output['myinvois_token'] = EINVAPPMY_Helper::get_option('myinvois_token');
        $output['config'] = $this->config;

        $status_text = $output['invoice_data']['status_text'] ?? '';

        if (empty($status_text)) {
            $output['status_text'] = $this->einvappmy_library->getConsolidatedStatusText(0);
        }

        $status_color_hex = "";
        switch ($status_text) {
            case 'Validated':
                $status_color_hex = sanitize_hex_color('#40c90a');
                break;
            case "Invalid":
                $status_color_hex = sanitize_hex_color('#ff0000');
                break;
            case "Submitted":
                $status_color_hex = sanitize_hex_color('#fcb322');
                break;
            default:
                $status_color_hex = sanitize_hex_color('#999');
                break;
        }

        $output['status_color_hex'] = $status_color_hex;

        $output['is_subscription_active'] = true;
        $output['is_subscription_active'] = $einvoice_status['is_subscription_required'] ? false : true;
        
        if (!$this->einvappmy_library->isSubscriptionActive()) {
            $output['is_subscription_active'] = false;
        }

        $subscription_info_response = $this->einvappmy_library->getSubscriptionInfo();
        $output['subscription_url'] = $subscription_info_response['data']['subscription_payment_url'] ?? '';

        // Localize script to pass data
        wp_localize_script('wc-order-einvoice', 'einvappmyWCOrderData', array(
            'ajax_url' => esc_url(admin_url('admin-ajax.php')),
            'nonce' => esc_js(wp_create_nonce('einvappmy_myeinvoice_request')),
            'order_id' => absint($order->get_id()),
            'nonce_create_document' => esc_js(wp_create_nonce('einvappmy_myeinvoice_create_document'))
        ));

        EINVAPPMY_Helper::load_template('admin/wc-order-einvoice.php', $output);
    }

    private function checkEInvoiceStatus($order_info) {
        if (method_exists($order_info, 'get_id')) {
            $order_id = $order_info->get_id();
        } else {
            $order_id = $order_info->ID;
        }

        $output = [];

        $order_user_id = $order_info->get_user_id();
        $order_user_data = get_userdata($order_user_id);

        $date_created = $order_info->get_date_created();
        $current_date = new DateTime('now', new DateTimeZone('Asia/Kuala_Lumpur'));
        $date_added_same_month = $date_created->format('Y-m') === $current_date->format('Y-m');

        // if the order is not added in the same month, return expired
        if (!$date_added_same_month) {
            $output['error'] = 'Expired';

            return $output;
        }

        // validate for name and address, there are not mandatory but need to display warning message if empty
        // to let user know that they will need to provided when go to middleware form
        $customer_name = $order_info->get_billing_first_name() . ' ' . $order_info->get_billing_last_name();
        $email = $order_info->get_billing_email();
        $telephone = $order_info->get_billing_phone();

        $address1 = $order_info->get_shipping_address_1();
        $address2 = $order_info->get_shipping_address_2();
        $city = $order_info->get_shipping_city();
        $postcode = $order_info->get_shipping_postcode();
        $state = $order_info->get_shipping_state();

        // fallback to billing address if shipping address is empty
        $address1 = empty($address1) ? $order_info->get_billing_address_1() : $address1;
        $address2 = empty($address2) ? $order_info->get_billing_address_2() : $address2;
        $city = empty($city) ? $order_info->get_billing_city() : $city;
        $postcode = empty($postcode) ? $order_info->get_billing_postcode() : $postcode;
        $state = empty($state) ? $order_info->get_billing_state() : $state;

        // fallback to user data if customer name is empty
        $customer_name = empty($customer_name) ? $order_user_data->first_name . ' ' . $order_user_data->last_name : $customer_name;

        $validation_warning_fields = [];
        if (empty($customer_name)) {
            $validation_warning_fields[] = 'Customer Name';
        }
        if (empty($email)) {
            $validation_warning_fields[] = 'Email Address';
        }
        if (empty($telephone)) {
            $validation_warning_fields[] = 'Phone Number';
        }
        if (empty($address1)) {
            $validation_warning_fields[] = 'Address';
        }
        if (empty($city)) {
            $validation_warning_fields[] = 'City';
        }
        if (empty($postcode)) {
            $validation_warning_fields[] = 'Postcode';
        }
        if (empty($state)) {
            $validation_warning_fields[] = 'State';
        }

        $is_allow_create_invoice = true;
        $allow_create_invoice_statuses = $this->config['allow_create_invoice_statuses'];
        $cancel_or_refund_order_statuses = $this->config['cancel_or_refund_order_statuses'];
        $pending_order_statuses = $this->config['pending_order_statuses'];
        $fallback_message_order_statuses = $this->config['fallback_message_order_statuses'];
        $error_message = '';

        if (in_array($order_info->get_status(), $cancel_or_refund_order_statuses)) {
            $error_message = esc_html__('Invoice creation is not available for canceled or refunded order', 'einvoiceapp-malaysia');
            $is_allow_create_invoice = false;
        }
        if (in_array($order_info->get_status(), $pending_order_statuses)) {
            $error_message = esc_html__('Invoice creation will be available once the payment is made', 'einvoiceapp-malaysia');
            $is_allow_create_invoice = false;
        }
        if (in_array($order_info->get_status(), $fallback_message_order_statuses) || !in_array($order_info->get_status(), $allow_create_invoice_statuses)) {
            $error_message = sprintf(
                /* translators: %1$s: current order status */
                esc_html__('Invoice cannot be created. Current order status is "%1$s". Only paid orders (Processing or Completed) can have invoices.', 'einvoiceapp-malaysia'),
                $order_info->get_status()
            );
            $is_allow_create_invoice = false;
        }

        $order_paid_date = $order_info->get_date_paid();
        $order_completed_date = $order_info->get_date_completed();

        $is_order_paid = $order_paid_date ? true : false;
        $is_order_completed = $order_completed_date ? true : false;

        $is_ui_display_not_requested = false;
        $is_ui_display_checking = false;
        $is_ui_display_expired = false;
        $is_ui_display_success = false;
        $is_ui_display_error = false;
        $is_subscription_required = false;
        $invoices = $this->einvappmy_library->getInvoices($order_id);
        $invoice_data = [];

        if (!$is_allow_create_invoice) { 
            $is_ui_display_error = true;
            $is_ui_display_checking = false;
            $is_ui_display_not_requested = false;
            $is_ui_display_expired = false;
        } else {
            if (!empty($invoices['error'])) {
                if ($invoices['error'] == 'No invoices found') {
                    $is_ui_display_not_requested = true;
                    $is_ui_display_checking = false;
                    $is_ui_display_expired = false;
                } elseif ($invoices['error'] == 'Expired') {
                    $is_ui_display_expired = true;
                    $is_ui_display_not_requested = false;
                    $is_ui_display_checking = false;
                } elseif ($invoices['error'] == 'Subscription required') {
                    $is_subscription_required = true;
                    $is_ui_display_checking = false;
                    $is_ui_display_not_requested = false;
                    $is_ui_display_expired = false;
                }
            }  elseif (!empty($invoices['data'])) {
                $is_ui_display_checking = false;
                $is_ui_display_not_requested = false;
                $is_ui_display_expired = false;
                $is_ui_display_success = true;
    
                $invoice_data = $invoices['data'][0] ?? [];
                $invoice_code = $invoice_data['invoice_code'] ?? '';
                $invoice_data['invoice_url'] = $this->einvappmy_library->getInvoiceUrl($invoice_code);
                $invoice_data['status_text'] = $this->einvappmy_library->getConsolidatedStatusText($invoice_data['status']);
    
                if ($invoice_data['status'] == 1) {
                    $invoice_data['pdf_url'] = $this->einvappmy_library->getPdfUrl($invoice_data['uuid']);
    
                    $invoice_id = $invoice_data['id'];
    
                    $credit_notes = $this->einvappmy_library->getNonInvoiceDocument($invoice_id, 'credit_notes');
                    $invoice_data['credit_note_status'] = isset($credit_notes['success']);
    
                    if (!empty($credit_notes['success'])) {
                        $credit_note_data = $credit_notes['data'][0] ?? [];
    
                        $credit_note_data['status_text'] = $this->einvappmy_library->getConsolidatedStatusText($credit_note_data['status']);
                        $credit_note_data['status_color_hex'] = $this->einvappmy_library->getStatusLabelBackgroundColor($credit_note_data['status']);
    
                        $invoice_data['credit_note_url'] = $this->einvappmy_library->getCreditNoteUrl($credit_note_data['credit_note_code']);
                        $invoice_data['credit_note_pdf_url'] = $this->einvappmy_library->getCreditNotePdfUrl($credit_note_data['uuid']);
                        $invoice_data['credit_note'] = $credit_note_data;
                    }
    
                    $debit_notes = $this->einvappmy_library->getNonInvoiceDocument($invoice_id, 'debit_notes');
                    $invoice_data['debit_note_status'] = isset($debit_notes['success']);
    
                    if (!empty($debit_notes['success'])) {
                        $debit_note_data = $debit_notes['data'][0] ?? [];
    
                        $debit_note_data['status_text'] = $this->einvappmy_library->getConsolidatedStatusText($debit_note_data['status']);
                        $debit_note_data['status_color_hex'] = $this->einvappmy_library->getStatusLabelBackgroundColor($debit_note_data['status']);
    
                        $invoice_data['debit_note_url'] = $this->einvappmy_library->getDebitNoteUrl($debit_note_data['debit_note_code']);
                        $invoice_data['debit_note_pdf_url'] = $this->einvappmy_library->getDebitNotePdfUrl($debit_note_data['uuid']);
                        $invoice_data['debit_note'] = $debit_note_data;
                    }
    
                    $refund_notes = $this->einvappmy_library->getNonInvoiceDocument($invoice_id, 'refund_notes');
                    $invoice_data['refund_note_status'] = isset($refund_notes['success']);
    
                    if (!empty($refund_notes['success'])) {
                        $refund_note_data = $refund_notes['data'][0] ?? [];
    
                        $refund_note_data['status_text'] = $this->einvappmy_library->getConsolidatedStatusText($refund_note_data['status']);
                        $refund_note_data['status_color_hex'] = $this->einvappmy_library->getStatusLabelBackgroundColor($refund_note_data['status']);
    
                        $invoice_data['refund_note_url'] = $this->einvappmy_library->getRefundNoteUrl($refund_note_data['refund_note_code']);
                        $invoice_data['refund_note_pdf_url'] = $this->einvappmy_library->getRefundNotePdfUrl($refund_note_data['uuid']);
                        $invoice_data['refund_note'] = $refund_note_data;
                    }
                }
            }
        }

        if (empty($invoices['data'])) {
            $output['error'] = 'No invoice found';
        }

        return [
            'is_date_added_same_month'    => $date_added_same_month,
            'is_ui_display_not_requested' => $is_ui_display_not_requested,
            'is_ui_display_checking'      => $is_ui_display_checking,
            'is_ui_display_expired'       => $is_ui_display_expired,
            'is_ui_display_success'       => $is_ui_display_success,
            'is_ui_display_error'        => $is_ui_display_error,
            'error_message'               => $error_message,
            'is_subscription_required'    => $is_subscription_required,
            'invoice_data'                => $invoice_data,
            'validation_warning_fields'   => $validation_warning_fields,
            'is_order_paid'               => $is_order_paid,
            'is_order_completed'          => $is_order_completed,
        ];
    }
}