<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


// Admin Setting page

class EINVAPPMY_Admin_Settings
{
    private $config;
    private $einvappmy_library;
    private $model_einvappmy;

    public function __construct($config) {
        $this->config = $config;
        $this->model_einvappmy = new Model_EINVAPPMY($this->config);
        $this->einvappmy_library = new EINVAPPMY_Library($this->config);

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_post_einvappmy_save_settings', array($this, 'save_form_settings'));
        add_action('admin_post_save_einvappmy_settings', array($this, 'save_form_settings'));

        add_action('wp_ajax_einvappmy_export_invoices', array($this, 'export_invoices'));
        add_action('wp_ajax_einvappmy_download_consolidated_excel', array($this, 'download_consolidated_excel'));
        add_action('wp_ajax_einvappmy_submit_consolidated', array($this, 'submit_consolidated'));
        add_action('wp_ajax_einvappmy_cancel_consolidated', array($this, 'cancelConsolidatedOrder'));
        add_action('wp_ajax_einvappmy_consolidated_fetch_orders', array($this, 'consolidated_fetch_orders'));
        add_action('wp_ajax_einvappmy_get_consolidated_invoice', array($this, 'get_consolidated_invoice'));
        add_action('wp_ajax_clear_einvoice_logs', array($this, 'clear_einvoice_logs'));
    }

    public function add_admin_menu() {
        add_menu_page(
            'EInvoice App',
            'EInvoice App',
            'manage_options',
            'einvappmy-main',
            array($this, 'render_main_page'),
            'dashicons-admin-generic',
        );

        // Add the main submenu item (rename the first item to "Listing")
        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Dashboard', // Page title
            'Dashboard', // Menu title
            'manage_options', // Capability
            'einvappmy-main', // Menu slug (same as parent to replace the default item)
            array($this, 'render_main_page') // Callback function
        );

        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Invoices', // Page title
            'Invoices', // Menu title
            'manage_options', // Capability
            'einvappmy-einvoice-request', // Menu slug (same as parent to replace the default item)
            array($this, 'render_einvoice_request_page') // Callback function
        );

        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Notes', // Page title
            'Notes', // Menu title
            'manage_options', // Capability
            'einvappmy-einvoice-notes', // Menu slug (same as parent to replace the default item)
            array($this, 'render_notes_page') // Callback function
        );

        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Consolidated', // Page title
            'Consolidated', // Menu title
            'manage_options', // Capability
            'einvappmy-consolidated', // Menu slug (same as parent to replace the default item)
            array($this, 'render_consolidated_page') // Callback function
        );

        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Consolidated History', // Page title
            'Consolidated History', // Menu title
            'manage_options', // Capability
            'einvappmy-consolidated-history', // Menu slug (same as parent to replace the default item)
            array($this, 'render_consolidated_history_page') // Callback function
        );

        // Add the sub-menu item
        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Settings', // Page title
            'Settings', // Menu title
            'manage_options', // Capability
            'einvappmy-settings', // Menu slug
            array($this, 'render_settings_page') // Callback function
        );

        // add log page
        add_submenu_page(
            'einvappmy-main', // Parent slug
            'Logs', // Page title
            'Logs', // Menu title
            'manage_options', // Capability
            'einvappmy-logs', // Menu slug
            array($this, 'render_logs_page') // Callback function
        );
    }

    public function render_logs_page() {
        wp_enqueue_style('einvappmy-style', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/style.css');
        wp_enqueue_script('wc-admin-log-listing', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-log-listing.js', array('jquery'), null, true);

        $output_data = [];

        $log_filename = 'einvoiceapp.log';
        $log_histories = EINVAPPMY_Helper::read_custom_log($log_filename, 100);

        // Localize script to pass data
        wp_localize_script('wc-admin-log-listing', 'einvappmyLogData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('clear_logs_nonce'),
        ));

        EINVAPPMY_Helper::load_template('admin/log-listing.php', array_merge($output_data, [
            'page_title' => 'Logs Listing',
            'log_filename' => $log_filename,
            'log_histories' => $log_histories,
        ]));
    }

    /**
     * AJAX handler to clear log files
     */
    public function clear_einvoice_logs() {
        // Verify nonce
        check_ajax_referer('clear_logs_nonce', 'nonce');

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }

        // Clear the log file
        $log_filename = 'einvoiceapp.log';
        $result = EINVAPPMY_Helper::clear_custom_log($log_filename);

        if ($result) {
            wp_send_json_success(['message' => 'Logs cleared successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to clear logs']);
        }
    }

    public function render_main_page() {
        wp_enqueue_script('jquery-ui-datepicker');
        wp_enqueue_style('jquery-ui-css', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/jqueryui/ui-lightness/jquery-ui.css');
        wp_enqueue_style('einvappmy-style', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/style.css');
        wp_enqueue_script('bootstrap-js', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/bootstrap/boostrap.bundle.min.js', array('jquery'), null, true);
        wp_enqueue_script('wc-admin-listing-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-listing-page.js', array('jquery'), null, true);

        $output_data = [];
        $myinvois_token = EINVAPPMY_Helper::get_option('myinvois_token');
        $status = EINVAPPMY_Helper::get_option('status');

        $this->page_permission_check();

        $summary = $this->einvappmy_library->getSummary();
        $summary_data = !empty($summary['data']) ? $summary['data'] : [];

        $storeOrdersData = $this->model_einvappmy->getOrdersData();
        if (isset($storeOrdersData)) {
            $output_data['store_orders_data'] = $storeOrdersData;
        }

        $consolidatedDate = wp_date("F Y", strtotime("-1 month"));
        $consolidatedSubmission = $this->model_einvappmy->getCurrentMonthConsolidated();
        $consolidatedStatus = 0;
        $consolidated_invoice_code = '';
        $consolidated_uuid = '';
        $consolidated_pdf_url = '';

        // for staging only
        // $consolidatedSubmission['invoice_id'] = "o2oeC-525";

        if (!empty($consolidatedSubmission)) {
            $middlewareInvoice = $this->einvappmy_library->getInvoices($consolidatedSubmission['invoice_id']);
            $isSuccess = ($middlewareInvoice['success'] ?? 0) == 1;

            $invoice_code = "";
            $myinvois_uuid = "";
            $description = "";
            $currency_code = "";
            $total = "";
            $tax_total = "";
            $status = "";

            if ($isSuccess && !empty($middlewareInvoice['data']) && !empty($middlewareInvoice['data'][0])) {
                $invoice_code = $middlewareInvoice['data'][0]['invoice_code'];
                $myinvois_uuid = $middlewareInvoice['data'][0]['uuid'];
                $description = $middlewareInvoice['data'][0]['items'][0]['name'];
                $currency_code = $middlewareInvoice['data'][0]['currency_code'];
                $total = $middlewareInvoice['data'][0]['total'];
                $tax_total = $middlewareInvoice['data'][0]['tax_total'];
                $status = $middlewareInvoice['data'][0]['status'];
            } elseif (!empty($consolidatedSubmission)) {
                $invoice_code = $consolidatedSubmission['invoice_code'];
                $myinvois_uuid = $consolidatedSubmission['myinvois_uuid'];
                $description = $consolidatedSubmission['description'];
                $currency_code = $consolidatedSubmission['currency_code'];
                $total = $consolidatedSubmission['total'];
                $tax_total = $consolidatedSubmission['tax_total'];
                $status = $consolidatedSubmission['status'];
            }

            $data = array(
                "invoice_code"  => $invoice_code,
                "myinvois_uuid" => $myinvois_uuid,
                "description"   => $description,
                "currency_code" => $currency_code,
                "total"         => $total,
                "tax_total"     => $tax_total,
                "status"        => $status
            );

            if ($isSuccess) {
                $this->model_einvappmy->editConsolidatedInvoice($data);
            }

            $consolidated_invoice_code = $data['invoice_code'];
            $consolidated_uuid = $data['myinvois_uuid'];
            $consolidated_pdf_url = $this->einvappmy_library->getPdfUrl($data['myinvois_uuid']);
            $consolidatedStatus = $data['status'];
            $consolidatedTotal = $data['total'];
            $consolidatedTaxTotal = $data['tax_total'];
            $consolidatedCurrencyCode = $data['currency_code'];
        } else {
            $consolidatedStatuses = implode(',', EINVAPPMY_Helper::get_option('consolidated_order_statuses') ?? []);
            $requiredMonth = wp_date("m", strtotime("-1 month"));
            $requiredYear = wp_date("Y", strtotime("-1 month"));

            $filter_data = [
                'statuses' => EINVAPPMY_Helper::get_option('consolidated_order_statuses'),
            ];

            $totalConsolidatedSales = $this->model_einvappmy->getConsolidatedTotalSales($requiredMonth, $requiredYear, $filter_data);
            $totalValidatedSales = $this->einvappmy_library->getTotalConsolidatedInvoices($requiredMonth, $requiredYear);
            $totalValidatedSales = $totalValidatedSales['data'] ?? [];

            $total_validated_amount = $totalValidatedSales['total_validated_amount'] ?? 0;
            $total_validated_tax = $totalValidatedSales['total_validated_tax'] ?? 0;

            $consolidatedTotal = $totalConsolidatedSales['total'] - $total_validated_amount;
            $consolidatedTaxTotal = $totalConsolidatedSales['tax_total'] - $total_validated_tax;
            $consolidatedCurrencyCode = get_woocommerce_currency();
        }

        $currency_symbol = get_option('woocommerce_currency');
        $formatted_total = $currency_symbol . number_format($consolidatedTotal, 2, '.', '');
        $formatted_tax = $currency_symbol . number_format($consolidatedTaxTotal, 2, '.', '');

        $output_data = array_merge($output_data, [
            'summary_data'              => $summary_data,
            'consolidatedDate'          => $consolidatedDate,
            'consolidatedCurrencyCode'  => $consolidatedCurrencyCode,
            'consolidated_status_code'  => $consolidatedStatus,
            'consolidated_invoice_code' => $consolidated_invoice_code,
            'consolidated_uuid'         => $consolidated_uuid,
            'consolidated_pdf_url'      => $consolidated_pdf_url,
            'consolidated_total_raw'    => $consolidatedTotal,
            'consolidated_tax_raw'      => $consolidatedTaxTotal,
            'consolidated_status'       => EINVAPPMY_Helper::get_consolidated_status_text($consolidatedStatus),
            'consolidated_total'        => $formatted_total,
            'consolidated_tax'          => $formatted_tax,
            'myinvois_token'            => $myinvois_token,
            'page_title'                => 'EInvoice App Dashboard',
            'store_orders_data'         => $storeOrdersData,
        ]);
        EINVAPPMY_Helper::load_template('admin/listing-page.php', $output_data);
    }

    public function export_invoices() {
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'einvappmy_export_invoices')) {
            wp_die(
                esc_html__('Security check failed. Please try again.', 'einvoiceapp-malaysia'),
                esc_html__('Security Error', 'einvoiceapp-malaysia'),
                ['response' => 403]
            );
        }

        $start_date = sanitize_text_field(wp_unslash($_GET['start_date'] ?? ''));
        $end_date = sanitize_text_field(wp_unslash($_GET['end_date'] ?? ''));
        $document_type = sanitize_text_field(wp_unslash($_GET['document_type'] ?? ''));

        if (!empty($start_date) && !DateTime::createFromFormat('Y-m-d', $start_date)) {
            wp_die(esc_html__('Invalid start date format.', 'einvoiceapp-malaysia'));
        }
        if (!empty($end_date) && !DateTime::createFromFormat('Y-m-d', $end_date)) {
            wp_die(esc_html__('Invalid end date format.', 'einvoiceapp-malaysia'));
        }

        $result = $this->einvappmy_library->exportInvoices($start_date, $end_date, $document_type);

        // Clean any existing output buffers to prevent HTML contamination
        while (ob_get_level()) {
            ob_end_clean();
        }

        // Set appropriate headers for CSV export
        if (!headers_sent()) {
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="export_' . $document_type . '_' . wp_date('Y-m-d') . '.csv"');
            header('Content-Length: ' . strlen($result));
            header('Pragma: no-cache');
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: 0');
        }

        echo wp_kses($result, []);

        // Ensure we exit cleanly
        wp_die();
    }

    public function submit_consolidated() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_consolidated_submit')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $descriptionDate = sanitize_text_field(wp_unslash($_POST['description_date']) ?? '');
        $currencyCode = sanitize_text_field(wp_unslash($_POST['currency_code']) ?? 'MYR');
        $totalAmt = floatval(wp_unslash($_POST['total']) ?? 0);
        $totalTaxAmt = floatval(wp_unslash($_POST['tax_total']) ?? 0);
        $order_ids = sanitize_text_field(wp_unslash($_POST['selected_order_ids']) ?? []);

        if (empty($descriptionDate)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Description date is required', 'einvoiceapp-malaysia')
            ]);
        }
        if ($totalAmt < 0) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Total amount is required', 'einvoiceapp-malaysia')
            ]);
        }
        
        if ($totalTaxAmt < 0) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Total tax amount is required', 'einvoiceapp-malaysia')
            ]);
        }
        
        if (!DateTime::createFromFormat('F Y', $descriptionDate)) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid description date format', 'einvoiceapp-malaysia')
            ]);
        }
        
        $timestamp = strtotime($descriptionDate);
        $requiredYear = wp_date("y", $timestamp);
        $requiredMonth = wp_date("n", $timestamp);
        $order_id = 'WC-' . $requiredMonth . '-' . $requiredYear;

        $description = $descriptionDate . ' Consolidated Invoices';

        $payload = array(
            'order_id'        => $order_id,
            'order_date'      => wp_date("Y-m-d H:i:s"),
            'customer_name'   => 'Consolidated Buyers',
            'telephone'       => 'NA',
            'email'           => 'NA',
            'address1'        => 'NA',
            'address2'        => 'NA',
            'city'            => 'NA',
            'postcode'        => 'NA',
            'state'           => 'NA',
            'country'         => 'MALAYSIA',
            'country_code'    => 'MY',
            'currency_code'   => $currencyCode,
            'order_tax_total' => $totalTaxAmt,
            'order_total'     => $totalAmt,
            'items'           => array(
                array(
                    'name'             => $description,
                    'price'            => $totalAmt,
                    'quantity'         => 1,
                    'item_tax_total'   => $totalTaxAmt,
                    'item_price_total' => $totalAmt
                )
            )
        );

        $submissionResponse = $this->einvappmy_library->createConsolidatedInvoice($payload);
        $response = array();

        if ($submissionResponse && isset($submissionResponse['success']) && $submissionResponse['success'] == 1) {
            $currentMonthSubmission = $this->model_einvappmy->getCurrentMonthConsolidated();
            $consolidated_submission_id = 0;
            $submission_response_data = $submissionResponse['data'] ?? [];

            if (
                !empty($currentMonthSubmission)
                && ($currentMonthSubmission['invoice_code'] == $submission_response_data['eInvoiceCode'])
                && ($currentMonthSubmission['myinvois_uuid'] == $submission_response_data['invoiceDetails']['uuid'])
            ) {
                $consolidated_submission_id = $currentMonthSubmission['id'];

                $data = array(
                    "invoice_code"  => $submission_response_data['eInvoiceCode'],
                    "myinvois_uuid" => $submission_response_data['invoiceDetails']['uuid'],
                    "description"   => $submission_response_data['invoiceDetails']['items'][0]['name'],
                    "currency_code" => $submission_response_data['invoiceDetails']['currency_code'],
                    "total"         => $submission_response_data['invoiceDetails']['total'],
                    "tax_total"     => $submission_response_data['invoiceDetails']['tax_total'],
                    "status"        => $submission_response_data['invoiceDetails']['status']
                );

                $where = [
                    'invoice_code'  => $currentMonthSubmission['invoice_code'],
                    'myinvois_uuid' => $currentMonthSubmission['myinvois_uuid']
                ];

                $this->model_einvappmy->editConsolidatedInvoice($data, $where);

            } else {
                $consolidated_submission_id = $this->model_einvappmy->addConsolidatedInvoice($submission_response_data);
            }

            // NOTE: no need to save db because when fetch order will check with api to exclude valid orders
            // if (!empty($order_ids) && $consolidated_submission_id > 0) {
            //     foreach ($order_ids as $order_id) {
            //         $this->model_einvappmy->addConsolidatedEinvoiceOrder([
            //             'consolidate_einvocie_submission_id' => $consolidated_submission_id,
            //             'order_id'                           => $order_id
            //         ]);
            //     }
            // }

            $response = [
                'success' => true,
                'message' => esc_html__('The consolidated e-Invoice has been submitted successfully.', 'einvoiceapp-malaysia'),
                'uuid'    => $submission_response_data['invoiceDetails']['uuid']
            ];
        } else {
            // Default fallback error
            $errorMessage = esc_html__('Error: No response received. Please check with the web development team for assistance.', 'einvoiceapp-malaysia');

            // Laravel validator error
            if (isset($submissionResponse['error']) && is_array($submissionResponse['error'])) {
                $validationErrors = [];
                foreach ($submissionResponse['error'] as $field => $messages) {
                    foreach ((array) $messages as $msg) {
                        $field = str_replace('_', ' ', $field);

                        $validationErrors[] = strtoupper($field) . ": " . $msg;
                        // $validationErrors[] = $msg;
                    }
                }
                $errorMessage = implode("; ", $validationErrors);

                // Duplicate or other error message
            } elseif (isset($submissionResponse['error']) && is_string($submissionResponse['error'])) {
                $errorMessage = $submissionResponse['error'];
            }

            $response = [
                'success'    => false,
                'message'    => $errorMessage
            ];
        }

        wp_send_json($response);
    }

    public function cancelConsolidatedOrder() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_consolidated_cancel')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $invoice_code = sanitize_text_field(wp_unslash($_POST['invoice_code']));

        $response = $this->einvappmy_library->cancelConsolidated($invoice_code);

        if ($response && isset($response['success']) && $response['success'] == true) {
            $consolidated_einvoice_submission = $this->model_einvappmy->getConsolidatedEinvoiceSubmission(['invoice_code' => $invoice_code]);

            $this->model_einvappmy->deleteConsolidatedInvoice($invoice_code);
            $this->model_einvappmy->deleteConsolidatedEinvoiceOrder([
                'consolidate_einvocie_submission_id' => $consolidated_einvoice_submission['id']
            ]);

            $response = array(
                'success' => true,
                'msg'     => $response['message']
            );
        } else {
            $response = array(
                'success'    => false,
                'error_desc' => isset($response['error']) ? $response['error'] : esc_html__('Failed to cancel consolidated invoice', 'einvoiceapp-malaysia')
            );
        }

        wp_send_json($response);
    }

    public function download_consolidated_excel() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_consolidated_download_excel')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $post_data = [];
        $post_data['order_ids'] = json_decode(sanitize_text_field(wp_unslash($_POST['order_ids'])), true);

        $consolidated_date = sanitize_text_field($_POST['consolidated_date']);
        // Convert the date string to a timestamp
        $timestamp = strtotime($consolidated_date);

        // Get the required month and year
        $requiredMonth = wp_date("n", $timestamp); // n = month without leading zeros
        $requiredYear = wp_date("Y", $timestamp);  // Y = full year

        // Get validated orders to exclude
        $totalValidatedSales = $this->einvappmy_library->getTotalConsolidatedInvoices($requiredMonth, $requiredYear);
        $totalValidatedSales = $totalValidatedSales['data'] ?? [];
        // get the order ids that already validated
        $validatedOrderIds = isset($totalValidatedSales['order_ids']) ? $totalValidatedSales['order_ids'] : array();

        // order to retrieve from database, the those already validated will be excluded.
        $order_ids = array_diff($post_data['order_ids'], $validatedOrderIds ?? []);

        // Get orders for consolidation
        $orders = $this->model_einvappmy->getOrders([
            'month'      => $requiredMonth,
            'year'       => $requiredYear,
            'order_ids'  => $order_ids,
            'sort_by'    => 'date_created_gmt',
            'sort_order' => 'ASC',
            'group_by'   => 'id'
        ]);

        $orders = $this->model_einvappmy->formatOrders($orders);
        $get_invoices_response = $this->einvappmy_library->getInvoices(null, $requiredMonth, $requiredYear);
        $einvoice_list = [];

        if (isset($get_invoices_response['success']) && !empty($get_invoices_response['data'])) {
            $store_order_ids = array_column($get_invoices_response['data'], 'store_order_id');
            $einvoice_list = array_combine($store_order_ids, $get_invoices_response['data']);
        }

        // Format orders for Excel
        $formattedOrders = array();
        $currency_symbol = get_option('woocommerce_currency');

        if (!empty($orders)) {
            foreach ($orders as $order) {
                $order_info = wc_get_order($order['id']);
                $einvoice_info = $einvoice_list[$order['id']] ?? [];

                $einvoice_status_text = !empty($einvoice_info['status']) ? $this->einvappmy_library->getConsolidatedStatusText($einvoice_info['status']) : '';
                $uuid = $einvoice_info['myinvois_uuid'] ?? '';
                $validation_url = $einvoice_info['validation_url'] ?? '';

                $formattedOrders[] = array(
                    'Order ID'         => $order['id'],
                    'Date'             => wp_date('d/m/Y', strtotime($order['date_created_gmt'])),
                    'Customer'         => $order_info->get_formatted_billing_full_name(),
                    'Payment Status'   => wc_get_order_status_name($order['status']),
                    'Currency'         => $order_info->get_currency(),
                    'Total'            => number_format($order_info->get_total(), 2, '.', ''),
                    'Tax'              => number_format($order_info->get_total_tax(), 2, '.', ''),
                    'E-Invoice Status' => $einvoice_status_text,
                    'UUID'             => $uuid,
                    'Validation URL'   => $validation_url
                );
            }
        }

        // Create CSV content
        $csv_content = '';

        // Add headers
        if (!empty($formattedOrders)) {
            $headers = array_keys($formattedOrders[0]);
            $csv_content .= implode(',', $headers) . "\n";

            // Add data rows
            foreach ($formattedOrders as $order) {
                $row = [];
                foreach ($order as $value) {
                    // Escape quotes and wrap in quotes if contains comma, quote or newline
                    if (is_string($value) && (strpos($value, ',') !== false || strpos($value, '"') !== false || strpos($value, "\n") !== false)) {
                        $value = '"' . str_replace('"', '""', $value) . '"';
                    }
                    $row[] = $value;
                }
                $csv_content .= implode(',', $row) . "\n";
            }
        }

        // Clean any existing output buffers to prevent HTML contamination
        while (ob_get_level()) {
            ob_end_clean();
        }

        // Prevent any caching and set headers for CSV download
        if (!headers_sent()) {
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="consolidated_orders.csv"');
            header('Content-Length: ' . strlen($csv_content));
            header('Pragma: no-cache');
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: 0');
        }

        // Output CSV content
        echo wp_kses($csv_content, []);
        
        // Ensure we exit cleanly
        wp_die();
    }

    /**
     * Validate settings form submission (currently helpline fields).
     *
     * @param array|null $post_data
     * @return array {
     *   @type array $messages List of validation error messages (escaped strings)
     *   @type array $fields   List of field keys that have errors
     * }
     */
    private function validate_form_submission($post_data = null) {
        if (!is_array($post_data)) {
            return [
                'messages' => [],
                'fields'   => [],
            ];
        }

        // save the error message to display at the top of the form
        $errors = [];
        // save the field that has error, use to show the field with error
        $error_fields = [];

        // Validate helpline email
        $helpline_email = isset($post_data['helpline_email']) ? trim($post_data['helpline_email']) : '';
        if (!empty($helpline_email)) {
            // Basic email format check
            if (!is_email($helpline_email)) {
                $errors[] = esc_html__('Helpline email format is invalid.', 'einvoiceapp-malaysia');
                $error_fields[] = 'helpline_email';
            }

            // Disallow @o2o.my domain
            $at_position = strrpos($helpline_email, '@');
            if ($at_position !== false) {
                $domain = strtolower(substr($helpline_email, $at_position + 1));
                if ($domain === 'o2o.my') {
                    $errors[] = esc_html__('Helpline email cannot use the @o2o.my domain.', 'einvoiceapp-malaysia');
                    $error_fields[] = 'helpline_email';
                }
            }
        }

        // Validate helpline phone
        $helpline_phone = isset($post_data['helpline_phone']) ? trim($post_data['helpline_phone']) : '';
        if (!empty($helpline_phone)) {
            // Normalise phone number to digits only for comparison
            $normalised_phone = preg_replace('/\D+/', '', $helpline_phone);
            $blocked_phone = '60376207020'; // 603 7620 7020

            if ($normalised_phone === $blocked_phone) {
                $errors[] = esc_html__('This helpline phone number is not allowed. Please provide your own contact number.', 'einvoiceapp-malaysia');
                $error_fields[] = 'helpline_phone';
            }
        }

        return [
            'messages' => $errors,
            'fields'   => array_values(array_unique($error_fields)),
        ];
    }

    /**
     * Submit or Save button in settings page is clicked will trigger this function
     * 
     * @param mixed $options
     * @return void
     */
    public function save_form_settings($options) {
        // Verify nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_save_settings')) {
            wp_die(esc_html__('Security check failed', 'einvoiceapp-malaysia'));
        }

        $save_action = isset($_POST['save_action']) ? sanitize_text_field(wp_unslash($_POST['save_action'])) : '';
        $post_data = isset($_POST['einvappmy_options']) ? array_map('sanitize_text_field', $_POST['einvappmy_options']) : [];
        $save_data = [];

        if ($save_action == 'environment') {
            $save_data['status'] = isset($post_data['status']) ? absint($post_data['status']) : 0;

            update_option($this->config['option_name'], $save_data);

            // when environment change then delete save status
            // so that when page is refresh the subscription status will be get from api based on the new environment
            delete_option('einvappmy_user_subscription_status');

            wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&settings-updated=true'));
            exit();
        } elseif ($save_action == 'company_information') {
            $options = get_option($this->config['option_name'], []);
            $status = isset($options['status']) ? $options['status'] : 0;

            $myinvois_token = isset($post_data['myinvois_token']) ? sanitize_text_field(wp_unslash($post_data['myinvois_token'])) : '';

            $save_data['status'] = $status;
            $save_data['myinvois_token'] = $myinvois_token;

            update_option($this->config['option_name'], $save_data);

            foreach ($post_data as $key => $value) {
                $post_data[$key] = sanitize_text_field($value);
            }

            $option_country = isset($post_data['country']) ? sanitize_text_field($post_data['country']) : '';
            $country = EINVAPPMY_Helper::get_country_by_code($option_country);

            // $submit_method = $this->config['submit_methods'][$post_data['submit_method']]['value'] ?? 'auto';

            // Validate helpline fields (and any future validation) before sending to API
            $validation_results = $this->validate_form_submission($post_data);
            $validation_errors  = $validation_results['messages'] ?? [];
            $error_fields       = $validation_results['fields'] ?? [];

            // if got error
            if (!empty($validation_errors)) {
                $allowed_html = [
                    'strong' => [],
                    'br'     => [],
                    'ul'     => [],
                    'li'     => [],
                ];

                $errors = '<strong>' . esc_html__('Form validation errors:', 'einvoiceapp-malaysia') . '</strong><br>';
                $errors .= '<ul>';
                foreach ($validation_errors as $message) {
                    $errors .= '<li>' . esc_html($message) . '</li>';
                }
                $errors .= '</ul>';

                // Ensure session is open for writing
                if (!session_id() || session_status() === PHP_SESSION_NONE) {
                    session_start();
                }
                $_SESSION['einvappmy_error'] = wp_kses($errors, $allowed_html);
                $_SESSION['einvappmy_error_fields'] = $error_fields;
                $_SESSION['einvappmy_form_post_data'] = $post_data;
                // Close session after writing to avoid interfering with REST API
                session_write_close();

                wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&error=api-failed'));
                exit();
            }

            try {
                $payload = array(
                    'company_name'                 => sanitize_text_field($post_data['company_name']),
                    'email'                        => sanitize_email($post_data['email']),
                    'telephone'                    => sanitize_text_field($post_data['telephone']),
                    'address1'                     => sanitize_text_field($post_data['address_1']),
                    'address2'                     => sanitize_text_field($post_data['address_2']),
                    'city'                         => sanitize_text_field($post_data['city']),
                    'postcode'                     => sanitize_text_field($post_data['postcode']),
                    'country'                      => $country,
                    'state'                        => sanitize_text_field($post_data['state']),
                    'company_tin'                  => sanitize_text_field($post_data['company_tin']),
                    'company_identification_type'  => sanitize_text_field($post_data['company_identification_type']),
                    'company_identification_value' => sanitize_text_field($post_data['company_identification_value']),
                    'company_sst'                  => sanitize_text_field($post_data['company_sst']),
                    'company_ttx'                  => sanitize_text_field($post_data['company_ttx']),
                    'classification_code'          => sanitize_text_field($post_data['classification_code']),
                    'classification_name'          => sanitize_text_field($post_data['classification_name']),
                    'myinvois_id'                  => sanitize_text_field($post_data['myinvois_id']),
                    'myinvois_secret'              => sanitize_text_field($post_data['myinvois_secret']),
                    'submit_method'                => sanitize_text_field($post_data['submit_method']),
                    'name'                         => sanitize_text_field($post_data['name']),
                    'admin_code'                   => sanitize_text_field($post_data['admin_code']),
                    'site_domain'                  => sanitize_text_field($post_data['site_domain']),
                    'helpline_email'               => sanitize_text_field($post_data['helpline_email']),
                    'helpline_phone'               => sanitize_text_field($post_data['helpline_phone'])
                );

                if (!empty($myinvois_token)) {
                    $action = 'update';
                    $response = $this->einvappmy_library->saveClientData($payload);
                } else {
                    $action = 'create';
                    $response = $this->einvappmy_library->createClientData($payload);
                }

                // Ensure session is open for unsetting
                if (!session_id() || session_status() === PHP_SESSION_NONE) {
                    session_start();
                }
                unset($_SESSION['einvappmy_error'], $_SESSION['einvappmy_form_post_data']);
                // Close session after writing to avoid interfering with REST API
                session_write_close();

                if (empty($response)) {
                    throw new Exception('No response from API server');
                }

                if (isset($response['error'])) {
                    throw new Exception($response['error']);
                }
    
                if (isset($response['errors'])) {
                    $allowed_html = [
                        'strong' => [],
                        'br' => [],
                        'ul' => [],
                        'li' => [],
                        'em' => [],
                        'span' => ['class' => []]
                    ];

                    if (is_array($response['errors'])) {
                        $errors = '<strong>' . esc_html__('Please correct the following issues:', 'einvoiceapp-malaysia') . '</strong><br>';
                        $errors .= '<ul>';
        
                        foreach ($response['errors'] as $field => $message) {
                            $errors .= sprintf(
                                '<li><strong>%s:</strong> %s</li>',
                                esc_html(ucfirst($field)),
                                esc_html(implode(', ', (array)$message))
                            );
                        }
    
                        $errors .= '</ul>';

                        throw new Exception(wp_kses($errors, $allowed_html));
                    } else {
                        throw new Exception(wp_kses($response['errors'], $allowed_html));
                    }
                }

                if (empty($response['success']) || $response['success'] !== true) {
                    if (isset($response['message'])) {
                        throw new Exception(esc_html($response['message']));
                    }

                    throw new Exception(esc_html__('API operation failed without error details', 'einvoiceapp-malaysia'));
                }
    
                wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&settings-updated=true'));
                exit();
            } catch (Exception $e) {
                // Ensure session is open for writing
                if (!session_id() || session_status() === PHP_SESSION_NONE) {
                    session_start();
                }
               
                $_SESSION['einvappmy_error'] = $e->getMessage();
                $_SESSION['einvappmy_form_post_data'] = $post_data;
                // Close session after writing to avoid interfering with REST API
                session_write_close();

                wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&error=api-failed'));
                exit();
            }
        }
    }

    /**
     * Render the settings page
     * When user access to the setting page, this function responsible to load the setting and the page.
     * 
     * @return void
     */
    public function render_settings_page() {
        // Enqueue Select2 CSS
        wp_enqueue_style('select2-css', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/select2/select2.min.css');
        wp_enqueue_style('select2-bootstrap-css', EINVAPPMY_PLUGIN_ASSETS_URL . 'css/select2/select2-bootstrap.min.css');
        // Enqueue Select2 JS
        wp_enqueue_script('select2-js', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/select2/select2.min.js', array('jquery'), null, true);
        wp_enqueue_script('wc-admin-settings-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-settings-page.js', array('jquery'), null, true);

        // in setting page double check if the table is created
        $model_einvappmy = new Model_EINVAPPMY($this->config);
        $model_einvappmy->setupDBTables();

        $error_message = '';
        $form_post_data = [];
        
        // Ensure session is available for reading/writing
        if (!session_id() || session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        if (isset($_SESSION['einvappmy_error'])) {
            $error_message = wp_kses($_SESSION['einvappmy_error'], ['br' => [], 'strong' => [], 'ul' => [], 'li' => []]);
            unset($_SESSION['einvappmy_error']);
        }

        if (isset($_SESSION['einvappmy_form_post_data'])) {
            if (is_array($_SESSION['einvappmy_form_post_data'])) {
                $form_post_data = array_map('sanitize_text_field', $_SESSION['einvappmy_form_post_data']);
            } else {
                $form_post_data = sanitize_text_field($_SESSION['einvappmy_form_post_data']);
            }

            unset($_SESSION['einvappmy_form_post_data']);
        }

        // Handle error fields for field highlighting
        $error_fields = [];
        if (isset($_SESSION['einvappmy_error_fields'])) {
            if (is_array($_SESSION['einvappmy_error_fields'])) {
                $error_fields = array_map('sanitize_text_field', $_SESSION['einvappmy_error_fields']);
            } else {
                $error_fields = [sanitize_text_field($_SESSION['einvappmy_error_fields'])];
            }
            unset($_SESSION['einvappmy_error_fields']);
        }
        
        // Close session after reading/writing to avoid interfering with REST API
        session_write_close();

        // get the saved settings
        $options = get_option($this->config['option_name'], []);
        $myinvois_token = $options['myinvois_token'] ?? '';
        $module_status = $options['status'] ?? 0;

        // try to get the client details from middleware
        $client_details = $this->einvappmy_library->getClientData();
        $client_details = $client_details['data'] ?? [];

        // if got client details, get the auth token
        if (!empty($client_details)) {
            $myinvois_token = $this->einvappmy_library->getClientAuthToken();
            $myinvois_token = $myinvois_token['data']['auth_token'] ?? '';

            $options['myinvois_token'] = $myinvois_token;
            update_option($this->config['option_name'], $options);
        }

        $statuses = $this->config['environments'];
        $submit_methods = $this->config['submit_methods'];
        $identification_types = $this->config['identification_types'];

        $order_statuses = wc_get_order_statuses();

        $country_list = WC()->countries->get_countries();
        $countries = array_filter($country_list, function ($country, $country_code) {
            return stripos($country, 'Malaysia') !== false;
        }, ARRAY_FILTER_USE_BOTH);

        $default_country = EINVAPPMY_Helper::get_default_country_state();
        $selected_country = $options['country'] ?? '';

        if (empty($selected_country)) {
            $selected_country = $default_country['country'];
        }

        $states = $this->config['states'];
        $selected_state = $options['state'] ?? '';

        if (empty($selected_state)) {
            $selected_state = $default_country['state'];
        }

        $site_domain = $client_details['site_domain'] ?? $this->einvappmy_library->getOriginDomain();
        $site_name = get_bloginfo('name');

        // get the payment url to make subscription
        $subscription_info_response = $this->einvappmy_library->getSubscriptionInfo();
        $subscription_url = $subscription_info_response['data']['subscription_payment_url'] ?? '';

        // check user subscription status
        $is_subscription_active = $this->einvappmy_library->isSubscriptionActive();

        // if the account is excluded from subscription service, set the subscription status to active
        if (!empty($client_details) && $client_details['subscription_exempt']) {
            $is_subscription_active = true;
        }

        $subscription_details = [];
        $user_subscription_status = get_option('einvappmy_user_subscription_status');

        // TODO: now take live data, need to figure a better way for handle this
        $user_subscription_status_response = $this->einvappmy_library->getSubscriptionStatus();
        
        if (!empty($user_subscription_status_response) && isset($user_subscription_status_response['data'])) {
            $user_subscription_status = $user_subscription_status_response['data'];
        }

        if (!empty($user_subscription_status)) {
            $subscription_details['is_subscription_active'] = $user_subscription_status['subscription_active'];
            $subscription_details['package_plan'] = !empty($user_subscription_status['subscription_details']['package_plan']) ? $user_subscription_status['subscription_details']['package_plan'] : 
            '';

            $expire_date = $user_subscription_status['expired_at'] ? wp_date('F d, Y', strtotime($user_subscription_status['expired_at'])) : '';
            $subscription_details['expire_date'] = $expire_date;

            $subscription_details['is_in_trial'] = $user_subscription_status['is_in_trial'] ?? false;
            $subscription_details['trail_remaining_days'] = $user_subscription_status['trial_days_remaining'] ?? 0;

            if (!empty($user_subscription_status['trial_end_date'])) {
                $expire_date = wp_date('F d, Y', strtotime($user_subscription_status['trial_end_date']));
                $subscription_details['expire_date'] = $expire_date;
            }
        }

        // Localize script to pass data
        wp_localize_script('wc-admin-settings-page', 'einvappmySettingsData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'get_msi_sub_categories_codes_url' => esc_url(EINVAPPMY_PLUGIN_ASSETS_URL . 'json/MSICSubCategoryCodes.json'),
            'client_details' => $client_details,
            'config' => $this->config,
        ));

        EINVAPPMY_Helper::load_template('admin/settings-page.php', [
            'options'                => $options,
            'page_title'             => 'EInvoice App Settings',
            'classification_code'    => $client_details['classification_code'] ?? '',
            'myinvois_token'         => $myinvois_token,
            'config'                 => $this->config,
            'statuses'               => $statuses,
            'client_details'         => $client_details,
            'submit_methods'         => $submit_methods,
            'order_statuses'         => $order_statuses,
            'identification_types'   => $identification_types,
            'countries'              => $countries,
            'selected_country'       => $selected_country,
            'selected_state'         => $selected_state,
            'states'                 => $states,
            'site_domain'            => $site_domain,
            'site_name'              => $site_name,
            'subscription_url'       => $subscription_url,
            'is_subscription_active' => $is_subscription_active,
            'module_status'          => $module_status,
            'user_subscription_status' => $user_subscription_status,
            'error_message'          => $error_message,
            'subscription_details'   => $subscription_details,
            'form_post_data'         => $form_post_data,
            'error_fields'           => $error_fields,
        ]);
    }

    public function render_consolidated_page() {
        $this->page_permission_check();

        wp_enqueue_script('wc-admin-consolidated-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-consolidated-page.js', array('jquery'), null, true);

        $output_data = [];
        $output_data['page_title'] = 'Consolidated E-invoices';
        $output_data['order_statuses'] = wc_get_order_statuses();
        $output_data['default_consolidated_order_statuses'] = $this->config['default_consolidated_order_statuses'];
        $output_data['consolidated_currency_code'] = get_option('woocommerce_currency');

        $output_data['consolidated_month'] = wp_date('Y F', strtotime('-1 month'));
        $output_data['consolidated_date'] = wp_date("F Y", strtotime("-1 month"));

        $consolidated_date_info = $this->get_consolidated_date_info();
        $output_data['consolidated_month'] = $consolidated_date_info['formatted_month_YF'];
        $output_data['consolidated_date'] = $consolidated_date_info['formatted_month_FY'];

         // Localize script to pass data
        wp_localize_script('wc-admin-consolidated-page', 'einvappmyConsolidatedData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce_consolidated_get' => wp_create_nonce('einvappmy_consolidated_get'),
            'nonce_consolidated_submit' => wp_create_nonce('einvappmy_consolidated_submit'),
            'nonce_consolidated_cancel' => wp_create_nonce('einvappmy_consolidated_cancel'),
            'nonce_consolidated_download_excel' => wp_create_nonce('einvappmy_consolidated_download_excel'),
            'nonce_consolidated_get_orders' => wp_create_nonce('einvappmy_consolidated_get_orders'),
            'loading_image_url' => get_admin_url() . 'images/loading.gif',
            'consolidated_date' => $output_data['consolidated_date'],
            'consolidated_currency_code' => $output_data['consolidated_currency_code'],
            'cancel_button_text' => esc_html__('Cancel Consolidated E-invoice', 'einvoiceapp-malaysia')
        ));

        EINVAPPMY_Helper::load_template('admin/consolidated-page.php', $output_data);
    }

    private function page_permission_check() {
        $myinvois_token = EINVAPPMY_Helper::get_option('myinvois_token');

        // did not have token mean the account still havent created
        if (empty($myinvois_token)) {
            wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&has_token=false'));
            exit();
        }

        $is_subscription_active = $this->einvappmy_library->isSubscriptionActive();

        if (!$is_subscription_active) {
            wp_safe_redirect(admin_url('admin.php?page=einvappmy-settings&subscription_active=false'));
            exit();
        }
    }

    private function get_consolidated_date_info() {
        $year = wp_date('Y');
        $month = wp_date('m', strtotime('-1 month'));

        // if today date is larger than 7th of the month, then the month is current month
        if (wp_date('d') > 7) {
            $month = wp_date('m');
        }

        // Create a DateTime object for the first day of the target month
        $date = DateTime::createFromFormat('Y-m', "$year-$month");

        return [
            'year'               => $year,
            'month'              => $month,
            'formatted_month_FY' => $date->format('F Y'),  // July 2023
            'formatted_month_YF' => $date->format('Y F')   // 2023 July
        ];
    }

    public function consolidated_fetch_orders() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'einvappmy_consolidated_get_orders')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $consolidate_order_statuses = isset($_GET['order_statuses']) ? array_map('sanitize_text_field', $_GET['order_statuses']) : [];
        $view = sanitize_text_field($_GET['view'] ?? '');

        $consolidated_date_info = $this->get_consolidated_date_info();
        $month = $consolidated_date_info['month'];
        $year = $consolidated_date_info['year'];

        // get the order that has been validated
        $validated_details = $this->einvappmy_library->getTotalConsolidatedInvoices($month, $year);
        $validated_details = $validated_details['data'] ?? [];
        $validated_order_ids = $validated_details['order_ids'] ?? [];

        // NOTE: directly exclude those orders that already been validated in api enough
        // get submitted consolidated
        // $current_month_consolidated = $this->model_einvappmy->getCurrentMonthConsolidated();
        // $current_month_consolidated_id = $current_month_consolidated['id'] ?? 0;

        // if ($current_month_consolidated_id > 0) {
        //     // get all the order ids that has been submitted
        //     $order_ids = $this->model_einvappmy->getConsolidatedEinvoiceOrders([
        //         'consolidate_einvocie_submission_id' => $current_month_consolidated_id
        //     ]);

        //     $order_ids = array_column($order_ids, 'order_id');
        //     // merge the order ids that has been validated and the order ids that has been submitted
        //     $validated_order_ids = array_merge($validated_order_ids, $order_ids);
        // }

        $orders = $this->model_einvappmy->getOrders([
            'order_statuses'    => $consolidate_order_statuses,
            'month'             => $month,
            'year'              => $year,
            'sort_by'           => 'date_created_gmt',
            'sort_order'        => 'ASC',
            'group_by'          => 'id',
            'exclude_order_ids' => $validated_order_ids // for orders that has been validated, exclude them from the list
        ]);
        $orders = $this->model_einvappmy->formatOrders($orders);

        $total_order_amount = array_sum(array_column($orders, 'total_amount'));
        $formatted_total_order_amount = number_format($total_order_amount, 2, '.', '');

        $total_order_tax_amount = array_sum(array_column($orders, 'tax_amount'));
        $formatted_total_order_tax_amount = number_format($total_order_tax_amount, 2, '.', '');

        $result_data = null;

        if ($view == 'html') {
            $result_data = EINVAPPMY_Helper::load_template('admin/consolidated-order-list.php', [
                'orders'                       => $orders,
                'total_orders'                 => count($orders),
                'formatted_total_order_amount' => $formatted_total_order_amount,
            ], true);
        }

        wp_send_json([
            'success' => true,
            'data'    => [
                'html'                   => $result_data,
                'total_order_amount'     => $formatted_total_order_amount,
                'total_orders'           => count($orders),
                'total_order_tax_amount' => $formatted_total_order_tax_amount
            ]
        ]);
    }

    public function get_consolidated_invoice() {
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'einvappmy_consolidated_get')) {
            wp_send_json([
                'success' => false,
                'message' => esc_html__('Invalid nonce', 'einvoiceapp-malaysia')
            ]);
        }

        $submitted_consolidated = $this->model_einvappmy->getCurrentMonthConsolidated();

        if (empty($submitted_consolidated)) {
            wp_send_json([
                'success'    => false,
                'error_desc' => esc_html__('No consolidated invoice found', 'einvoiceapp-malaysia')
            ]);
            exit();
        }

        $einvoice_info_response = $this->einvappmy_library->getInvoices($submitted_consolidated['invoice_id']);
        if (isset($einvoice_info_response['success']) && !empty($einvoice_info_response['data'])) {
            $einvoice_info = $einvoice_info_response['data'][0];
        }

        if (!empty($einvoice_info)) {
            $validation_url = $einvoice_info['validation_url'];
            $uuid = $einvoice_info['uuid'];
            $currency_code = $einvoice_info['currency_code'];
            $total_amount = $einvoice_info['total'];
            $tax_amount = $einvoice_info['tax_total'];
            $status = $einvoice_info['status'];
            $date_added = $einvoice_info['created_at'];
            $invoice_code = $einvoice_info['invoice_code'];

            if ($einvoice_info['status'] != $submitted_consolidated['status']) {
                // if the status in local db is different from the status in api, then update the status in local db
                $this->model_einvappmy->editConsolidatedInvoice([
                    'status' => $einvoice_info['status']
                ], [
                    'invoice_code' => $submitted_consolidated['invoice_code']
                ]);
            }
        } else {
            $validation_url = '';
            $uuid = $submitted_consolidated['myinvois_uuid'];
            $currency_code = $submitted_consolidated['currency_code'];
            $total_amount = $submitted_consolidated['total'];
            $tax_amount = $submitted_consolidated['tax_total'];
            $status = $submitted_consolidated['status'];
            $date_added = $submitted_consolidated['date_added'];
            $invoice_code = $submitted_consolidated['invoice_code'];
        }

        $currency_symbol = get_woocommerce_currency_symbol($currency_code);
        $formatted_total_amount = $currency_symbol . ' ' . number_format($total_amount, 2, '.', '');

        $status_text = $this->einvappmy_library->getConsolidatedStatusText($status);
        $status_color = $this->einvappmy_library->getStatusLabelBackgroundColor($status);
        $date_time = wp_date('d/m/Y (H:ia)', strtotime($date_added));

        $results = [
            'amount'         => $formatted_total_amount,
            'status_text'    => $status_text,
            'status_color'   => $status_color,
            'validation_url' => $validation_url,
            'date_time'      => $date_time,
            'invoice_code'   => $invoice_code
        ];

        wp_send_json([
            'success' => true,
            'data'    => $results
        ]);
    }

    public function render_einvoice_request_page() {
        $this->page_permission_check();

        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';

        // If an action is requested...
        if (!empty($action)) {
            // ...then a nonce MUST be present and valid.
            if (!isset($_GET['nonce'])) {
                wp_die(esc_html__('Security nonce is missing.', 'einvoiceapp-malaysia'));
            }

            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'einvappmy_einvoice_request_page')) {
                wp_die(esc_html__('Security check failed. Please try again.', 'einvoiceapp-malaysia'));
            }
        }
        
        // Enqueue jQuery UI Dialog and styles
        wp_enqueue_script('jquery-ui-dialog');
        wp_enqueue_style('wp-jquery-ui-dialog');

        wp_enqueue_script('wc-admin-einvoices-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-invoices-page.js', array('jquery', 'jquery-ui-dialog'), time(), true);

        $output_data = [];
        $output_data['page_title'] = esc_html__('Invoices', 'einvoiceapp-malaysia');
        $consolidate_statuses = $this->einvappmy_library->getConsolidateStatuses();
        $new_consolidate_statuses = [];

        foreach ($consolidate_statuses as $status_value => $status_text) {
            $new_consolidate_statuses[$status_value] = [
                'value' => $status_value,
                'text'  => $status_text,
                'style' => $this->einvappmy_library->getStatusLabelBackgroundColor($status_value),
                'url'   => add_query_arg('tab', $status_value, admin_url('admin.php?page=einvappmy-einvoice-request'))
            ];
        }

        $output_data['success_message'] = sanitize_text_field($_GET['success_message'] ?? '');
        $output_data['error_message'] = sanitize_text_field($_GET['error_message'] ?? '');
        $output_data['document_url'] = esc_url($_GET['document_url'] ?? '');

        $active_tab = absint($_GET['tab'] ?? 0);
        $filter_date_start = sanitize_text_field($_GET['filter_date_start'] ?? '');
        $filter_date_end = sanitize_text_field($_GET['filter_date_end'] ?? '');
       
        $invoice_code = sanitize_text_field($_GET['invoice_code'] ?? '');
        $amount = isset($_GET['amount']) ? floatval($_GET['amount']) : 0;
        $description = isset($_GET['description']) ? sanitize_textarea_field(wp_unslash($_GET['description'])) : '';

        $url_params = [];
        $url_params['tab'] = $active_tab;

        if (!empty($filter_date_start)) {
            $url_params['filter_date_start'] = $filter_date_start;
        }

        if (!empty($filter_date_end)) {
            $url_params['filter_date_end'] = $filter_date_end;
        }

        if (!empty($action) && !empty($invoice_code)) {
            switch ($action) {
                case 'cancel':
                    $response = $this->einvappmy_library->cancelInvoice($invoice_code);
                    break;
                case 'create-credit-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'credit_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
                case 'create-debit-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'debit_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
                case 'create-refund-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'refund_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
            }

            if (!empty($response['success'])) {
                $url_params['success_message'] = $response['message'];
                
                // Check if there's a document URL in the response to add to the success message
                if (!empty($response['data'])) {
                    $document_url = '';
                    
                    if (!empty($response['data']['debit_note_url'])) {
                        $document_url = $response['data']['debit_note_url'];
                    } elseif (!empty($response['data']['credit_note_url'])) {
                        $document_url = $response['data']['credit_note_url'];
                    } elseif (!empty($response['data']['refund_note_url'])) {
                        $document_url = $response['data']['refund_note_url'];
                    }
                    
                    if (!empty($document_url)) {
                        $url_params['document_url'] = $document_url;
                    }
                }

                wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
            } else {
                $url_params['error_message'] = $response['error'];
                wp_safe_redirect(admin_url('admin.php?page=einvappmy-einvoice-request&' . http_build_query($url_params)));
            }

            exit();
        }

        $consolidated_date_info = $this->get_consolidated_date_info();

        if (empty($filter_date_start) && empty($filter_date_end)) {
            $filter_date_start = wp_date('Y-m-01');
            $filter_date_end = wp_date('Y-m-t');
            $year_added = wp_date('Y');
            $month_added = wp_date('m');

            // $invoices = $this->einvappmy_library->getInvoices(null, $consolidated_date_info['month'], $consolidated_date_info['year']);
            $invoices = $this->einvappmy_library->getInvoices(null, $month_added, $year_added);
        } else {
            // Validate filter start date
            if (!DateTime::createFromFormat('Y-m-d', $filter_date_start)) {
                $filter_date_start = wp_date('Y-m-01');
            }

            // Validate filter end date  
            if (!DateTime::createFromFormat('Y-m-d', $filter_date_end)) {
                $filter_date_end = wp_date('Y-m-t');
            }

            $filter_date_start = wp_date('Y-m-d', strtotime($filter_date_start));
            $filter_date_end = wp_date('Y-m-d', strtotime($filter_date_end));

            $invoices = $this->einvappmy_library->getInvoices(null, 0, 0, $filter_date_start, $filter_date_end, 0);
        }

        $valid_status = 1;
        $cancel_status = 4;
        $invoice_list = [];
        $export_list = [];

        if (!empty($invoices['success']) && !empty($invoices['data'])) {
            $invoices = $invoices['data'];

            foreach ($invoices as $invoice) {
                // for export csv, will include all invoices
                if ($invoice['status'] != $active_tab && $action != 'export') {
                    continue;
                }

                $store_order_info_url = esc_url(admin_url('post.php?post=' . $invoice['store_order_id'] . '&action=edit'));
                $date_created = wp_date('d/m/Y (H:ia)', strtotime($invoice['created_at']));
                $currency_symbol = get_woocommerce_currency_symbol($invoice['currency_code']);
                $formatted_total_amount = number_format($invoice['total'], 2, '.', '');

                // Generate nonce for einvoice request page actions
                $nonce = wp_create_nonce('einvappmy_einvoice_request_page');

                $pdf_url = "";
                $cancel_invoice_url = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-einvoice-request&action=cancel&invoice_code=' . $invoice['invoice_code'])));
                $create_credit_note_url = "";
                $create_debit_note_url = "";
                $create_refund_note_url = "";

                if ($invoice['status'] == $cancel_status) {
                    $cancel_invoice_url = '';
                }   

                if (!empty($invoice['uuid']) && $invoice['status'] == $valid_status) {
                    $pdf_url = esc_url($this->einvappmy_library->getInvoicePDFUrl($invoice['uuid']));
                    $create_credit_note_url = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-einvoice-request&action=create-credit-note&invoice_code=' . $invoice['invoice_code'])));
                    $create_debit_note_url = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-einvoice-request&action=create-debit-note&invoice_code=' . $invoice['invoice_code'])));
                    $create_refund_note_url = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-einvoice-request&action=create-refund-note&invoice_code=' . $invoice['invoice_code'])));
                }

                $status_text = $this->einvappmy_library->getConsolidatedStatusText($invoice['status']);

                $invoice_list[] = [
                    'order_id'               => $invoice['store_order_id'],
                    'customer_name'          => $invoice['customer_name'],
                    'order_info_url'         => $store_order_info_url,
                    'date_created'           => $date_created,
                    'total_amount'           => $currency_symbol . ' ' . $formatted_total_amount,
                    'invoice_type'           => $invoice['invoice_type'],
                    'invoice_type_text'      => $this->einvappmy_library->getInvoiceTypes($invoice['invoice_type'])['text'],
                    'uuid'                   => $invoice['uuid'] ?? '',
                    'cancel_invoice_url'     => $cancel_invoice_url,
                    'validation_url'         => $invoice['validation_url'],
                    'pdf_url'                => $pdf_url,
                    'create_credit_note_url' => $create_credit_note_url,
                    'create_debit_note_url'  => $create_debit_note_url,
                    'create_refund_note_url' => $create_refund_note_url,
                    'date_timestamp'       => strtotime($invoice['created_at'])
                ];

                if ($action == 'export') {
                    $export_list[] = [
                        'Date'            => wp_date('Y-m-d', strtotime($invoice['created_at'])),
                        'Order ID'        => $invoice['store_order_id'],
                        'Customer Name'   => $invoice['customer_name'],
                        'Document Status' => $status_text,
                        'Total Amount'    => $formatted_total_amount,
                        'PDF URL'         => $pdf_url,
                    ];
                }
            }
        }

        if (!empty($invoice_list)) {
            $sort_field = array_column($invoice_list, 'date_timestamp');
            array_multisort($sort_field, SORT_DESC, $invoice_list);
        }

        if (!empty($export_list) && $action == 'export') {
            $sort_field = array_column($export_list, 'Date');
            array_multisort($sort_field, SORT_DESC, $export_list);
            
            $this->export_csv($export_list, 'Invoices.csv');
            exit();
        }

        $link_export = esc_url(add_query_arg($url_params, admin_url('admin.php?page=einvappmy-einvoice-request&action=export')));

        $output_data['active_tab'] = $active_tab;
        $output_data['filter_date_start'] = $filter_date_start;
        $output_data['filter_date_end'] = $filter_date_end;
        $output_data['invoice_list'] = $invoice_list;
        $output_data['consolidate_statuses'] = $new_consolidate_statuses;
        $output_data['link_export'] = $link_export;

         // Localize script to pass data
        wp_localize_script('wc-admin-einvoices-page', 'einvappmyEinvoicesData', array(
            'einvoices_page_url' => admin_url('admin.php?page=einvappmy-einvoice-request'),
        ));

        EINVAPPMY_Helper::load_template('admin/einvoice-request-page.php', $output_data);
    }

    public function render_notes_page() {
        $this->page_permission_check();

        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';

        // If an action is requested...
        if (!empty($action)) {
            // ...then a nonce MUST be present and valid.
            if (!isset($_GET['nonce'])) {
                wp_die(esc_html__('Security nonce is missing.', 'einvoiceapp-malaysia'));
            }

            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'einvappmy_notes_page')) {
                wp_die(esc_html__('Security check failed. Please try again.', 'einvoiceapp-malaysia'));
            }
        }

        wp_enqueue_script('wc-admin-notes-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-notes-page.js', array('jquery'), null, true);

        $output_data = [];
        $output_data['page_title'] = 'Notes';

        $output_data['success_message'] = sanitize_text_field($_GET['success_message'] ?? '');
        $output_data['error_message'] = sanitize_text_field($_GET['error_message'] ?? '');

        $active_tab = sanitize_text_field($_GET['tab'] ?? 'credit_notes');
        $status_active_tab = sanitize_text_field($_GET['status_tab'] ?? 0);
        $filter_date_start = sanitize_text_field($_GET['filter_date_start'] ?? '');
        $filter_date_end = sanitize_text_field($_GET['filter_date_end'] ?? '');


        if (empty($filter_date_start) && empty($filter_date_end)) {
            $filter_date_start = wp_date('Y-m-01');
            $filter_date_end = wp_date('Y-m-t');
            $year_added = wp_date('Y');
            $month_added = wp_date('m');

            $notes = $this->einvappmy_library->getNonInvoiceDocument(null, $active_tab, [
                'month_added' => $month_added,
                'year_added'  => $year_added
            ]);
        } else {
            // Validate filter start date
            if (!DateTime::createFromFormat('Y-m-d', $filter_date_start)) {
                $filter_date_start = wp_date('Y-m-01');
            }

            // Validate filter end date  
            if (!DateTime::createFromFormat('Y-m-d', $filter_date_end)) {
                $filter_date_end = wp_date('Y-m-t');
            }

            $filter_date_start = wp_date('Y-m-d', strtotime($filter_date_start));
            $filter_date_end = wp_date('Y-m-d', strtotime($filter_date_end));

            $notes = $this->einvappmy_library->getNonInvoiceDocument(null, $active_tab, [
                'filter_date_start' => $filter_date_start,
                'filter_date_end'   => $filter_date_end
            ]);
        }

        $valid_status = 1;
        $note_list = [];
        $export_list = [];

        $url_params = [];
        $url_params['tab'] = $active_tab;
        $url_params['status_tab'] = $status_active_tab;

        if (!empty($filter_date_start)) {
            $url_params['filter_date_start'] = $filter_date_start;
        }

        if (!empty($filter_date_end)) {
            $url_params['filter_date_end'] = $filter_date_end;
        }

        if (!empty($notes['success']) && !empty($notes['data'])) {
            $notes = $notes['data'];

            foreach ($notes as $note) {
                // for export csv, will include all invoices
                if ($note['status'] != $status_active_tab && $action != 'export') {
                    continue;
                }

                $pdf_url = "";

                if (!empty($note['uuid']) && $note['status'] == $valid_status) {
                    $pdf_url = $this->einvappmy_library->getInvoicePDFUrl($note['uuid']);
                }

                $link_order_info = admin_url('post.php?post=' . $note['original_invoice_store_order_id'] . '&action=edit');
                $date_created = wp_date('d/m/Y (H:ia)', strtotime($note['created_at']));
                $currency_symbol = get_woocommerce_currency_symbol($note['currency_code']);
                $formatted_total_amount = number_format($note['total'], 2, '.', '');
                $total_amount_text = $currency_symbol . ' ' . $formatted_total_amount;

                $note_list[] = [
                    'order_id' => $note['original_invoice_store_order_id'],
                    'order_info_url' => $link_order_info,
                    'original_invoice_uuid' => $note['original_invoice_uuid'],
                    'original_invoice_validation_url' => $note['original_invoice_validation_url'],
                    'customer_name' => $note['customer_name'],
                    'date_created' => $date_created,
                    'date_timestamp' => strtotime($note['created_at']),
                    'total_amount_text' => $total_amount_text,
                    'uuid' => $note['uuid'] ?? '--',
                    'validation_url' => $note['validation_url'],
                ];

                $status_text = $this->einvappmy_library->getNotesStatusText($note['status']);
                
                if ($action == 'export') {
                    $export_list[] = [
                        'Date'            => wp_date('Y-m-d', strtotime($note['created_at'])),
                        'Order ID'        => $note['original_invoice_store_order_id'],
                        'Customer Name'   => $note['customer_name'],
                        'Document Status' => $status_text,
                        'Total Amount'    => $formatted_total_amount,
                        'PDF URL'         => $pdf_url,
                    ];
                }
            }
        }

        if (!empty($note_list)) {
            $sort_field = array_column($note_list, 'date_timestamp');
            array_multisort($sort_field, SORT_DESC, $note_list);
        }

        if (!empty($export_list) && $action == 'export') {
            $sort_field = array_column($export_list, 'Date');
            array_multisort($sort_field, SORT_DESC, $export_list);

            $filename = ucwords($active_tab) . '.csv';
            $this->export_csv($export_list, $filename);
            exit();
        }

        $note_statuses = $this->config['notes_status_text'];
        $new_note_statuses = [];

        foreach ($note_statuses as $status_value => $status_text) {
            $status_url_params = array_merge($url_params, ['status_tab' => $status_value, 'tab' => $active_tab]);

            $new_note_statuses[$status_value] = [
                'value' => $status_value,
                'text'  => $status_text,
                'style' => $this->einvappmy_library->getStatusLabelBackgroundColor($status_value),
                'url'   => add_query_arg($status_url_params, admin_url('admin.php?page=einvappmy-einvoice-notes'))
            ];
        }

        $notes_types = $this->config['notes'];

        foreach ($notes_types as $note_type_value => $notes_type) {
            $note_type_url_params = array_merge($url_params, ['tab' => $notes_type['value2']]);

            $output_data['notes_types'][$notes_type['value2']] = [
                'text' => $notes_type['text'],
                'value' => $notes_type['value2'],
                'url' => add_query_arg($note_type_url_params, admin_url('admin.php?page=einvappmy-einvoice-notes'))
            ];
        }

        $link_export = add_query_arg($url_params, admin_url('admin.php?page=einvappmy-einvoice-notes&action=export'));


        $output_data['active_tab'] = $active_tab;
        $output_data['status_active_tab'] = $status_active_tab;
        $output_data['filter_date_start'] = $filter_date_start;
        $output_data['filter_date_end'] = $filter_date_end;
        $output_data['note_list'] = $note_list;
        $output_data['note_statuses'] = $new_note_statuses;
        $output_data['link_export'] = $link_export;

        // Localize script to pass data
        wp_localize_script('wc-admin-notes-page', 'einvappmyNotesData', array(
            'note_page_url' => admin_url('admin.php?page=einvappmy-einvoice-notes'),
        ));

        EINVAPPMY_Helper::load_template('admin/notes-page.php', $output_data);
    }

    private function export_csv($export_data = null, $filename = 'consolidated_orders.csv') {
        // Clean any existing output buffers to prevent HTML contamination
        while (ob_get_level()) {
            ob_end_clean();
        }

        // Create CSV content
        $csv_content = '';

        // Add headers
        if (!empty($export_data)) {
            $headers = array_keys($export_data[0]);
            $csv_content .= implode(',', $headers) . "\n";

            // Add data rows
            foreach ($export_data as $data) {
                $row = [];
                foreach ($data as $value) {
                    // Escape quotes and wrap in quotes if contains comma, quote or newline
                    if (is_string($value) && (strpos($value, ',') !== false || strpos($value, '"') !== false || strpos($value, "\n") !== false)) {
                        $value = '"' . str_replace('"', '""', $value) . '"';
                    }
                    $row[] = $value;
                }
                $csv_content .= implode(',', $row) . "\n";
            }
        }

        // Prevent any caching
        if (!headers_sent()) {
            // Set headers for CSV download
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . strlen($csv_content));
            header('Pragma: no-cache');
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: 0');
        }

        // Output CSV content
        echo wp_kses($csv_content, []);
        
        // Exit cleanly without any WordPress footer
        exit();
    }

    public function render_consolidated_history_page() {
        $this->page_permission_check();

        // Enqueue jQuery UI Dialog and styles
        wp_enqueue_script('jquery-ui-dialog');
        wp_enqueue_style('wp-jquery-ui-dialog');
        
        wp_enqueue_script('wc-admin-consolidated-history-page', EINVAPPMY_PLUGIN_ASSETS_URL . 'js/admin-consolidated-history-page.js', array('jquery', 'jquery-ui-dialog'), time(), true);

        $output_data = [];
        $output_data['page_title'] = 'Consolidated History';
        $output_data['consolidated_listing'] = [];
        $output_data['success_message'] = sanitize_text_field($_GET['success_message'] ?? '');
        $output_data['error_message'] = sanitize_text_field($_GET['error_message'] ?? '');
        $output_data['document_url'] = esc_url($_GET['document_url'] ?? '');

        $active_tab = absint($_GET['tab'] ?? 2);
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        $invoice_code = isset($_GET['invoice_code']) ? sanitize_text_field(wp_unslash($_GET['invoice_code'])) : '';
        $amount = isset($_GET['amount']) ? floatval($_GET['amount']) : 0;
        $description = isset($_GET['description']) ? sanitize_textarea_field(wp_unslash($_GET['description'])) : '';

        // If an action is requested...
        if (!empty($action)) {
            // ...then a nonce MUST be present and valid.
            if (!isset($_GET['nonce'])) {
                wp_die(esc_html__('Security nonce is missing.', 'einvoiceapp-malaysia'));
            }

            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'einvappmy_consolidated_history_action')) {
                wp_die(esc_html__('Security check failed. Please try again.', 'einvoiceapp-malaysia'));
            }
        }

        $url_params = [];
        $url_params['tab'] = $active_tab;
        $url_params['invoice_code'] = $invoice_code;

        if (!empty($action) && !empty($invoice_code)) {
            switch ($action) {
                case 'cancel':
                    $response = $this->einvappmy_library->cancelInvoice($invoice_code);
                    
                    // If cancel is successful, delete the local database records
                    if (!empty($response['success'])) {
                        // Get the consolidated invoice record
                        $consolidated_einvoice_submission = $this->model_einvappmy->getConsolidatedEinvoiceSubmission(['invoice_code' => $invoice_code]);
                        
                        // Delete the consolidated invoice record
                        $this->model_einvappmy->deleteConsolidatedInvoice($invoice_code);
                        
                        // Delete the related consolidated orders
                        if (!empty($consolidated_einvoice_submission['id'])) {
                            $this->model_einvappmy->deleteConsolidatedEinvoiceOrder([
                                'consolidate_einvocie_submission_id' => $consolidated_einvoice_submission['id']
                            ]);
                        }
                    }
                    break;
                case 'create-credit-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'credit_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
                case 'create-debit-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'debit_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
                case 'create-refund-note':
                    // Validate amount and description
                    if (empty($amount) || $amount <= 0) {
                        $url_params['error_message'] = 'Invalid amount. Please enter a valid amount greater than 0.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    if (empty($description)) {
                        $url_params['error_message'] = 'Description is required.';
                        wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
                        exit();
                    }
                    $document_type = 'refund_note';
                    $response = $this->einvappmy_library->createNonInvoiceDocument($document_type, $invoice_code, $amount, $description);
                    break;
            }

            if (!empty($response['success'])) {
                $url_params['success_message'] = $response['message'];
                
                // Check if there's a document URL in the response to add to the success message
                if (!empty($response['data'])) {
                    $document_url = '';
                    
                    if (!empty($response['data']['debit_note_url'])) {
                        $document_url = $response['data']['debit_note_url'];
                    } elseif (!empty($response['data']['credit_note_url'])) {
                        $document_url = $response['data']['credit_note_url'];
                    } elseif (!empty($response['data']['refund_note_url'])) {
                        $document_url = $response['data']['refund_note_url'];
                    }
                    
                    if (!empty($document_url)) {
                        $url_params['document_url'] = $document_url;
                    }
                }

                wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
            } else {
                $url_params['error_message'] = $response['error'];
                wp_safe_redirect(admin_url('admin.php?page=einvappmy-consolidated-history&' . http_build_query($url_params)));
            }

            exit();
        }

        $consolidated_listing = $this->model_einvappmy->getConsolidatedEinvoiceSubmissions();

        $date_start = null;
        $date_end = null;
        $api_invoice_listing = [];

        $consolidated_valid_statuses = $this->config['consolidated_valid_statuses'];

        if (!empty($consolidated_listing)) {
            // Extract all date_added values
            $dates = array_column($consolidated_listing, 'date_added');
            
            // Filter out empty dates
            $dates = array_filter($dates, function($date) {
                return !empty($date);
            });
            
            if (!empty($dates)) {
                // Find earliest (minimum) and latest (maximum) dates
                // Dates are in format: 2025-10-01 10:10:10
                $date_start_raw = min($dates);
                $date_end_raw = max($dates);
                
                // Extract date portion only (Y-m-d) for API requests
                $date_start = wp_date('Y-m-d', strtotime($date_start_raw));
                $date_end = wp_date('Y-m-d', strtotime($date_end_raw));
            }

            $api_invoices = $this->einvappmy_library->getInvoices(null, 0, 0, $date_start, $date_end, 1);
            if (!empty($api_invoices['data']) && $api_invoices['success']) {
                $invoice_codes = array_column($api_invoices['data'], 'invoice_code');
                $api_invoice_listing = array_combine($invoice_codes, $api_invoices['data']);
            }

            foreach ($consolidated_listing as $consolidated_item) {
                $api_invoice_data = $api_invoice_listing[$consolidated_item['invoice_code']] ?? [];

                // if the api invoice data is not empty, and the status in local db is different from the status in api, then update the status in local db
                if (!empty($api_invoice_data) && $consolidated_item['status'] != $api_invoice_data['status']) {
                    $this->model_einvappmy->editConsolidatedInvoice([
                        'status' => $api_invoice_data['status']
                    ], [
                        'invoice_code' => $consolidated_item['invoice_code']
                    ]);
                }

                // Skip if API invoice data is empty
                if (empty($api_invoice_data)) {
                    continue;
                }

                // if the invoice status is not the same as the active tab, then skip
                if ($api_invoice_data['status'] != $active_tab) {
                    continue;
                }

                $formatted_date = wp_date('d/m/Y (H:ia)', strtotime($consolidated_item['date_added']));
                $currency_symbol = get_woocommerce_currency_symbol($api_invoice_data['currency_code']);
                $formatted_total = number_format($consolidated_item['total'], 2, '.', '');

                $trails_pdf_url = $this->einvappmy_library->getInvoiceTrailsPDFUrl(['invoice_uuid' => $consolidated_item['myinvois_uuid']]);

                $url_params = [];
                $url_params['tab'] = $active_tab;

                $link_cancel_invoice = '';
                $link_create_debit_note = '';
                $link_create_credit_note = '';
                $link_create_refund_note = '';

                if (!empty($api_invoice_data['uuid']) && in_array($api_invoice_data['status'], $consolidated_valid_statuses)) {
                    // Generate nonce for consolidated history actions
                    $nonce = wp_create_nonce('einvappmy_consolidated_history_action');
                    
                    $link_cancel_invoice = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-consolidated-history&action=cancel&invoice_code=' . $consolidated_item['invoice_code'])));
                    $link_create_debit_note = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-consolidated-history&action=create-debit-note&invoice_code=' . $consolidated_item['invoice_code'])));
                    $link_create_credit_note = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-consolidated-history&action=create-credit-note&invoice_code=' . $consolidated_item['invoice_code'])));
                    $link_create_refund_note = esc_url(add_query_arg(array_merge($url_params, ['nonce' => $nonce]), admin_url('admin.php?page=einvappmy-consolidated-history&action=create-refund-note&invoice_code=' . $consolidated_item['invoice_code'])));
                }

                $output_data['consolidated_listing'][] = [
                    'invoice_code' => $consolidated_item['invoice_code'],
                    'date_added' => $consolidated_item['date_added'],
                    'formatted_date' => $formatted_date,
                    'formatted_total' => $currency_symbol . ' ' . $formatted_total,
                    'uuid' => $consolidated_item['myinvois_uuid'],
                    'validation_url' => $api_invoice_data['validation_url'] ?? '',
                    'trails_pdf_url' => $trails_pdf_url,
                    'link_cancel_invoice' => $link_cancel_invoice,
                    'link_create_debit_note' => $link_create_debit_note,
                    'link_create_credit_note' => $link_create_credit_note,
                    'link_create_refund_note' => $link_create_refund_note,
                ];

            }
        }

        $consolidated_statuses = $this->config['consolidated_statuses'];
        // make it descending order, so that the tab can show submitted at the top
        $consolidated_statuses = array_reverse($consolidated_statuses);


        $output_data['consolidated_statuses'] = $consolidated_statuses;
        $output_data['active_tab'] = $active_tab;
        $output_data['date_start'] = $date_start;
        $output_data['date_end'] = $date_end;
        
        EINVAPPMY_Helper::load_template('admin/consolidated-history.php', $output_data);
    }
}