<?php
/**
 * Setting layout view.
 *
 * @package Edidev\AiAssistantForPerfection42\Views
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Views;

use Edidev\AiAssistantForPerfection42\Includes\Setting\SettingProvider;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class SettingLayout
 */
class SettingLayout {


	/**
	 * Twig template engine instance.
	 *
	 * @var mixed
	 */
	private $twig_template_engine;

	/**
	 * Arguments.
	 *
	 * @var array
	 */
	private array $args;

	/**
	 * Page slug name.
	 */
	const PAGE_NAME = 'edidev-ai-assistant-for-perfection42';

	/**
	 * Setting provider instance.
	 *
	 * @var SettingProvider|null
	 */
	protected ?SettingProvider $setting_provider = null;

	/**
	 * Constructor.
	 *
	 * @param array $args Arguments.
	 */
	public function __construct( $args = array() ) {
		global $edidev_twig_template_engine;

		$this->twig_template_engine = $edidev_twig_template_engine;
		$this->setting_provider     = new SettingProvider();
		$this->args                 = $args;

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ), 19 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ), 20 );
		// Stop printing WP Admin Notices on this page; messages are rendered inline in our template.
	}

	/**
	 * Check if view is loadable.
	 *
	 * @return bool
	 */
	protected function is_loadable(): bool {
		$result = ! empty( $_GET['page'] ) && self::PAGE_NAME === $_GET['page']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return $result;
	}

	/**
	 * Enqueue styles.
	 *
	 * @return void
	 */
	public function enqueue_styles(): void {
		if ( ! $this->is_loadable() ) {
			return;
		}
	}

	/**
	 * Enqueue scripts.
	 *
	 * @return void
	 */
	public function enqueue_scripts(): void {
		if ( ! $this->is_loadable() ) {
			return;
		}
	}

	/**
	 * Render the template.
	 *
	 * @return void
	 */
	public function render(): void {
		// Handle POST for settings save.
		if ( ! empty( $_POST['submitted-action'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			// Capability gate.
			if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown -- 'manage_woocommerce' is provided by WooCommerce.
				wp_die( esc_html__( 'You do not have permission.', 'edidev-ai-assistant-for-perfection42' ), 403 );
			}

			// Nonce verification.
			check_admin_referer( 'edidev_ai_assistant_settings_save', 'edidev_ai_assistant_nonce' );

			// Action sanitization.
			$action = isset( $_POST['submitted-action'] ) ? sanitize_key( wp_unslash( $_POST['submitted-action'] ) ) : '';

			switch ( $action ) {
				case 'save-general-tab':
					$posted = array(
						'username' => isset( $_POST['username'] ) ? sanitize_text_field( wp_unslash( $_POST['username'] ) ) : '',
						'password' => isset( $_POST['password'] ) ? sanitize_text_field( wp_unslash( $_POST['password'] ) ) : '',
					);
					$this->setting_provider->save_generate_tab( $posted );
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'sign-in':
					$posted = array(
						'username' => isset( $_POST['username'] ) ? sanitize_text_field( wp_unslash( $_POST['username'] ) ) : '',
						'password' => isset( $_POST['password'] ) ? sanitize_text_field( wp_unslash( $_POST['password'] ) ) : '',
					);
					$this->setting_provider->save_generate_tab( $posted );
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'sign-out':
					$this->setting_provider->sign_out();
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'start-signup-verification':
					$posted = array(
						'name'     => isset( $_POST['signup_name'] ) ? sanitize_text_field( wp_unslash( $_POST['signup_name'] ) ) : '',
						'email'    => isset( $_POST['signup_email'] ) ? sanitize_email( wp_unslash( $_POST['signup_email'] ) ) : '',
						'password' => isset( $_POST['signup_password'] ) ? sanitize_text_field( wp_unslash( $_POST['signup_password'] ) ) : '',
					);
					$this->setting_provider->start_signup_verification( $posted );
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'complete-signup':
					$posted = array(
						'otp_code'   => isset( $_POST['otp_code'] ) ? sanitize_text_field( wp_unslash( $_POST['otp_code'] ) ) : '',
						'session_id' => isset( $_POST['signup_session_id'] ) ? sanitize_text_field( wp_unslash( $_POST['signup_session_id'] ) ) : '',
					);
					$this->setting_provider->complete_signup_verification( $posted );
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'resend-signup-otp':
					$this->setting_provider->resend_signup_otp();
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				case 'cancel-signup-verification':
					$this->setting_provider->cancel_signup_verification();
					wp_safe_redirect( SettingProvider::GENERAL_TAB_URL );
					exit;
				default:
					break;
			}
		}

		$this->view();
	}

	/**
	 * Render view.
	 *
	 * @return void
	 */
	private function view(): void {
		$template = $this->twig_template_engine->load( 'SettingLayout.twig' );

		$data = $this->setting_provider->get_generate_tab();

		// Prepare nonce field HTML for the settings form.
		$nonce_field = wp_nonce_field( 'edidev_ai_assistant_settings_save', 'edidev_ai_assistant_nonce', true, false );

		$labels = array(
			'page_title_label'            => esc_html__( 'EdiDev AI Assistant for Perfection42', 'edidev-ai-assistant-for-perfection42' ),
			'save_label'                  => esc_html__( 'Save', 'edidev-ai-assistant-for-perfection42' ),
			'signup_label'                => esc_html__( 'Sign Up', 'edidev-ai-assistant-for-perfection42' ),
			'panel_general_label'         => esc_html__( 'General', 'edidev-ai-assistant-for-perfection42' ),
			'username_label'              => esc_html__( 'Email', 'edidev-ai-assistant-for-perfection42' ),
			'password_label'              => esc_html__( 'Password', 'edidev-ai-assistant-for-perfection42' ),
			'name_label'                  => esc_html__( 'Name', 'edidev-ai-assistant-for-perfection42' ),
			'email_label'                 => esc_html__( 'Email', 'edidev-ai-assistant-for-perfection42' ),
			'title_login_panel'           => esc_html__( 'Connect your Perfection42 account', 'edidev-ai-assistant-for-perfection42' ),
			'title_signup_panel'          => esc_html__( 'Create a new Perfection42 account', 'edidev-ai-assistant-for-perfection42' ),
			'dont_have_account_yet_label' => esc_html__( "Don't have an account yet?", 'edidev-ai-assistant-for-perfection42' ),
			'sign_up_button_label'        => __( 'Sign up on Perfection42 (opens in new tab)', 'edidev-ai-assistant-for-perfection42' ),
			'sign_up_helper_text'         => __( 'After creating your account, return here to log in and connect.', 'edidev-ai-assistant-for-perfection42' ),
			'forgot_password_label'       => esc_html__( 'Forgot password?', 'edidev-ai-assistant-for-perfection42' ),
			'sign_in_label'               => esc_html__( 'Sign In', 'edidev-ai-assistant-for-perfection42' ),
			'sign_out_label'              => esc_html__( 'Sign Out', 'edidev-ai-assistant-for-perfection42' ),
			'connected_as_label'          => esc_html__( 'Connected as', 'edidev-ai-assistant-for-perfection42' ),
			'show_password_label'         => esc_html__( 'Show', 'edidev-ai-assistant-for-perfection42' ),
			'hide_password_label'         => esc_html__( 'Hide', 'edidev-ai-assistant-for-perfection42' ),
			// Labels for Account panel (to avoid Twig translation filters).
			'account_title_label'         => esc_html__( 'Perfection42 Account', 'edidev-ai-assistant-for-perfection42' ),
			'loading_account_label'       => esc_html__( 'Loading account…', 'edidev-ai-assistant-for-perfection42' ),
			'check_status_label'          => esc_html__( 'Check Status', 'edidev-ai-assistant-for-perfection42' ),
			// New CTA labels.
			'my_projects_label'           => esc_html__( 'My Projects', 'edidev-ai-assistant-for-perfection42' ),
			'my_projects_aria_label'      => __( 'Open My Projects on Perfection42 Platform (opens in a new tab)', 'edidev-ai-assistant-for-perfection42' ),
			'my_profile_label'            => esc_html__( 'My Profile', 'edidev-ai-assistant-for-perfection42' ),
			'my_profile_aria_label'       => __( 'Open My Profile on Perfection42 Platform (opens in a new tab)', 'edidev-ai-assistant-for-perfection42' ),
			// Inline validation messages (client-side UX).
			'msg_required_name'           => esc_html__( 'Please enter your name.', 'edidev-ai-assistant-for-perfection42' ),
			'msg_required_email'          => esc_html__( 'Please enter your email address.', 'edidev-ai-assistant-for-perfection42' ),
			'msg_invalid_email'           => esc_html__( 'Please enter a valid email address.', 'edidev-ai-assistant-for-perfection42' ),
			'msg_required_password'       => esc_html__( 'Please enter your password.', 'edidev-ai-assistant-for-perfection42' ),
			'msg_min_password'            => esc_html__( 'Please enter a password of at least 8 characters.', 'edidev-ai-assistant-for-perfection42' ),
		);

		// Settings page disclosure paragraph (privacy/terms + non‑affiliation).
		$p42_privacy_url = 'https://REPLACE_WITH_PERFECTION42_PRIVACY_URL';
		$p42_terms_url   = 'https://REPLACE_WITH_PERFECTION42_TERMS_URL';
		$disclosure_html = sprintf(
		/* translators: 1: <a> to Privacy Policy, 2: </a>, 3: <a> to Terms of Service, 4: </a> */
			__( 'This plugin connects to Perfection42 for AI‑assisted product content. Data is sent only when you manually trigger an AI action. Not affiliated with WooCommerce/Automattic; Perfection42 is a third‑party service. See %1$sPrivacy Policy%2$s and %3$sTerms of Service%4$s.', 'edidev-ai-assistant-for-perfection42' ),
			'<a href="' . esc_url( $p42_privacy_url ) . '" target="_blank" rel="noopener noreferrer">',
			'</a>',
			'<a href="' . esc_url( $p42_terms_url ) . '" target="_blank" rel="noopener noreferrer">',
			'</a>'
		);

		$routes = array(
			'tab_general' => SettingProvider::GENERAL_TAB_URL,
		);

		$new_args = array_merge(
			$this->args,
			$data,
			$labels,
			$routes,
			array(
				'nonce_field'          => $nonce_field,
				'disclosure_paragraph' => wp_kses_post( $disclosure_html ),
				// Collect any settings-page specific admin notices so we can render them.
				// Inline in the Twig template just under the page title (outside forms).
				// Do NOT persist notices in WordPress options. Expect them to be
				// provided at render-time via $this->args (request-scoped only).
				'error_notices'        => ( isset( $this->args['error_notices'] )
					? array_map( 'sanitize_text_field', (array) $this->args['error_notices'] )
					: array() ),
				'success_notices'      => ( isset( $this->args['success_notices'] )
					? array_map( 'sanitize_text_field', (array) $this->args['success_notices'] )
					: array() ),
				// Signal to the Twig view which auth form should be visible initially after reload.
				// Priority: explicit query arg (?p42_auth_tab=signup) then transient flag set after failed signup.
				'show_signup_form'     => ( function () {
					$from_query = isset( $_GET['p42_auth_tab'] ) && 'signup' === sanitize_key( wp_unslash( $_GET['p42_auth_tab'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$from_transient = (bool) get_transient( 'edidev_p42_show_signup_form' );
					// Clear the transient after reading to avoid sticking state.
					if ( $from_transient ) {
						delete_transient( 'edidev_p42_show_signup_form' );
					}
					return ( $from_query || $from_transient );
				} )(),
			)
		);

		echo $template->render( $new_args ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Get instance.
	 *
	 * @return SettingLayout
	 */
	public static function instance(): SettingLayout {
		return new self();
	}
}
