<?php
/**
 * React components package.
 *
 * @package Edidev\AiAssistantForPerfection42\Packages
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Packages;

use Edidev\AiAssistantForPerfection42\Main;
use Edidev\AiAssistantForPerfection42\Includes\ServiceProvider\ServiceProvider;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class ReactComponents
 *
 * @package Edidev\AiAssistantForPerfection42\Packages
 */
class ReactComponents extends ServiceProvider {

	/**
	 * Asset manifest file path.
	 *
	 * @var string
	 */
	private string $asset_manifest_file_path = '';

	const SCRIPT_HANDLE_NAME_PREFIX = 'edidev-react-coms';
	const PAGE_NAME                 = 'edidev-ai-assistant-for-perfection42';

	/**
	 * Constructor.
	 *
	 * @param array $args Optional args.
	 */
	public function __construct( $args = array() ) {
		parent::__construct( $args );
		$this->asset_manifest_file_path = rtrim( Main::$plugin_dir, '/' ) . '/src/Packages/ReactComponents/asset-manifest.json';
		add_action( 'admin_init', array( $this, 'wordpress_admin_initialized' ) );
	}

	/**
	 * Check if loadable.
	 *
	 * @return bool
	 */
	protected function is_loadable(): bool {
		$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return ! empty( $page ) && self::PAGE_NAME === $page;
	}

	/**
	 * WordPress admin initialized.
	 *
	 * @return void
	 */
	public function wordpress_admin_initialized() {
		// add_filter( 'script_loader_tag', [ $this, 'wordpress_script_loader_tag' ], 10, 2 ). This hook modifies script tags.
		// add_action( 'admin_enqueue_scripts', [ $this, 'wordpress_enqueue_scripts' ] ). Enqueues React assets.
	}

	/**
	 * Modify script loader tag.
	 *
	 * @param string $tag    The <script> tag.
	 * @param string $handle The script handle.
	 * @return string
	 */
	public function wordpress_script_loader_tag( $tag, $handle ) {
		if ( ! preg_match( '/^' . self::SCRIPT_HANDLE_NAME_PREFIX . '-/', $handle ) ) {
			return $tag;
		}
		return str_replace( ' src', ' async defer src', $tag );
	}

	/**
	 * Parse asset manifest into array.
	 *
	 * @return array
	 */
	private function parse_asset_manifest_into_array(): array {
		if ( file_exists( $this->asset_manifest_file_path ) ) {
			$read_data     = file_get_contents( $this->asset_manifest_file_path ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
			$data_in_array = json_decode( $read_data, true );
			if ( ! empty( $data_in_array ) ) {
				return $data_in_array;
			}
		}
		return array();
	}

	/**
	 * Enqueue scripts.
	 *
	 * @return void
	 */
	public function wordpress_enqueue_scripts(): void {
		$asset_manifest = $this->parse_asset_manifest_into_array();

		if ( empty( $asset_manifest ) ) {
			return;
		}

		$site_url = rtrim( get_site_url(), '/' ) . Main::$plugin_dir_from_root . '/src/Packages/ReactComponents/';
		$site_url = str_replace( '\\', '/', $site_url );

		foreach ( $asset_manifest as $key => $value ) {
			if ( str_ends_with( $key, '.css' ) ) {
				$handle = self::SCRIPT_HANDLE_NAME_PREFIX . '-' . preg_replace( '/[^A-Za-z0-9_]/', '-', basename( $key, '.css' ) );
				wp_enqueue_style( $handle, $site_url . $value, array(), Main::PLUGIN_VERSION );
			}
		}

		foreach ( $asset_manifest as $key => $value ) {
			if ( str_ends_with( $key, '.js' ) ) {
				$handle = self::SCRIPT_HANDLE_NAME_PREFIX . '-' . preg_replace( '/[^A-Za-z0-9_]/', '-', basename( $key, '.js' ) );
				wp_enqueue_script( $handle, $site_url . $value, array(), Main::PLUGIN_VERSION, true );

				add_filter(
					'script_loader_tag',
					function ( $tag, $handle_name ) use ( $handle ) {
						if ( $handle_name === $handle ) {
							return str_replace( ' src', ' async defer src', $tag );
						}
						return $tag;
					},
					10,
					2
				);
			}
		}
	}

	/**
	 * Get instance.
	 *
	 * @param array $args Optional args.
	 * @return ReactComponents
	 */
	public static function instance( $args = array() ): ReactComponents {
		return new self( $args );
	}
}
