<?php
/**
 * Main plugin class file.
 *
 * @package Edidev\AiAssistantForPerfection42
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42;

use Edidev\AiAssistantForPerfection42\Includes\WooCommerce\WooCommerceProvider;
use Edidev\AiAssistantForPerfection42\Includes\RestApi\RestAPIProvider;
use Edidev\AiAssistantForPerfection42\Views\SettingLayout;
use Edidev\AiAssistantForPerfection42\Views\TwigTemplateEngine;

defined( 'ABSPATH' ) || exit;

/**
 * Main plugin class.
 */
class Main {

	const PLUGIN_NAME = 'EdiDev AI Assistant for Perfection42';

	const PLUGIN_VERSION                    = '1.0.12';
	const PLUGIN_MENU_URL                   = 'admin.php/?page=edidev-ai-assistant-for-perfection42';
	const COMPATIBILITY_WP_VERSION          = '6.0';
	const COMPATIBILITY_WOOCOMMERCE_VERSION = '6.0.0';

	/**
	 * Plugin directory path.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_dir = '';
	/**
	 * WordPress root directory path.
	 *
	 * @var string|null
	 */
	public static ?string $wordpress_root_dir = '';
	/**
	 * Plugin URL.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_url = '';
	/**
	 * Plugin basename.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_basename = '';
	/**
	 * Upload base directory.
	 *
	 * @var string|null
	 */
	public static ?string $upload_base_dir = '';
	/**
	 * Upload base URL.
	 *
	 * @var string|null
	 */
	public static ?string $upload_base_url = '';
	/**
	 * Plugin asset directory path.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_asset_dir = '';
	/**
	 * Plugin asset directory URL (with trailing slash).
	 *
	 * @var string|null
	 */
	private static ?string $plugin_asset_url = '';
	/**
	 * Plugin i18n directory path.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_i18n_dir = '';
	/**
	 * Plugin directory path from the WordPress root directory.
	 *
	 * @var string|null
	 */
	public static ?string $plugin_dir_from_root = '';

	const INSTALLED_PLUGIN   = 1;
	const UNINSTALLED_PLUGIN = 0;

	const ADMIN_NOTICE_ERROR_MESSAGE_KEY   = 'edidev_admin_notice_general_error_message';
	const ADMIN_NOTICE_SUCCESS_MESSAGE_KEY = 'edidev_admin_notice_general_success_message';

	/**
	 * Register privacy policy content for the plugin.
	 *
	 * This method adds privacy policy content that explains what data the plugin collects
	 * and how it interacts with external services.
	 *
	 * @link https://developer.wordpress.org/plugins/privacy/
	 * @return void
	 */
	public function register_privacy_policy() {
		if ( ! function_exists( 'wp_add_privacy_policy_content' ) ) {
			return;
		}

		$content = wp_kses_post(
			__(
				'This plugin connects to Perfection42 API to generate content or images.
				It may send product text, image URLs, or prompt text to perfection42.com.
				For details, visit: https://perfection42.com/privacy-policy',
				'edidev-ai-assistant-for-perfection42'
			)
		);

		wp_add_privacy_policy_content(
			'Perfection42 AI Product Assistant for WooCommerce',
			$content
		);
	}

	/**
	 * Constructor to initialize the plugin with necessary configurations.
	 *
	 * @param array $args Configuration arguments for the plugin.
	 */
	public function __construct( $args = array() ) {
		$GLOBALS['edidev_twig_template_engine'] = TwigTemplateEngine::INSTANCE();
		self::$plugin_dir                       = $args['plugin-dir'] ?? '';
		self::$wordpress_root_dir               = $args['wordpress-root-dir'] ?? '';
		self::$plugin_url                       = $args['plugin-url'] ?? '';
		self::$plugin_basename                  = $args['plugin-basename'] ?? '';
		$wp_upload__dir                         = wp_upload_dir();
		self::$upload_base_dir                  = $wp_upload__dir['basedir'];
		self::$upload_base_url                  = $wp_upload__dir['baseurl'];
		self::$plugin_dir_from_root             = str_replace( self::$wordpress_root_dir, '', self::$plugin_dir );
		self::$plugin_asset_dir                 = self::$plugin_dir . '/assets';
		self::$plugin_asset_url                 = trailingslashit( untrailingslashit( wp_normalize_path( self::$plugin_url . '/assets' ) ) );
		self::$plugin_i18n_dir                  = self::$plugin_dir . '/src/Packages';

		// Hook privacy policy registration.
		add_action( 'admin_init', array( $this, 'register_privacy_policy' ) );
	}

	/**
	 * Run the plugin by checking installation, compatibility, and setting up hooks.
	 *
	 * @return void
	 */
	public function run(): void {
		$is_plugin_installed = intval( get_option( 'perfection-ai-product-assistant', self::UNINSTALLED_PLUGIN ) );
		if ( self::INSTALLED_PLUGIN !== $is_plugin_installed ) {
			self::notice_error_message( self::PLUGIN_NAME . esc_html__( ' has NOT been installed correctly. Please deactivate and activate the plugin to resolve this problem.', 'edidev-ai-assistant-for-perfection42' ) );
			return;
		}

		if ( empty( self::$plugin_dir )
			|| empty( self::$wordpress_root_dir )
			|| empty( self::$plugin_url )
			|| empty( self::$plugin_basename )
		) {
			self::notice_error_message( self::PLUGIN_NAME . esc_html__( ' has NOT been installed correctly. Please deactivate and activate the plugin to resolve this problem.', 'edidev-ai-assistant-for-perfection42' ) );
			return;
		}

		if ( function_exists( 'is_multisite' ) && is_multisite() ) {
			self::self_deactivate_plugin();
			$_GET['deactivate'] = 1;
			unset( $_GET['activate'] );
			self::notice_error_message( self::PLUGIN_NAME . esc_html__( ' does NOT support multisite purposes at this moment but it would be coming soon.', 'edidev-ai-assistant-for-perfection42' ) );
			return;
		} elseif ( ! self::compatible_version_checker() ) {
			self::self_deactivate_plugin();
			$_GET['deactivate'] = 1;
			unset( $_GET['activate'] );
			self::notice_error_message( self::PLUGIN_NAME . esc_html__( ' could not be activated or deactivated automatically. The WordPress and WooCommerce versions are not compatible with this plugin. Our plugin supports WordPress versions from 4.7 and WooCommerce version from 3.6.0', 'edidev-ai-assistant-for-perfection42' ) );
			return;
		}

		add_filter( 'plugin_action_links_' . self::$plugin_basename, array( $this, 'admin_plugin_settings_link' ) );

		add_action( 'admin_notices', array( $this, 'admin_notices' ) );

		add_action( 'admin_menu', array( $this, 'register_sub_menu_item' ) );

		WooCommerceProvider::instance();

		RestAPIProvider::instance();
	}

	/**
	 * Display admin notices for errors and success messages.
	 *
	 * @return void
	 */
	public function admin_notices(): void {
		// Avoid printing core admin notices on our settings page — that page renders
		// its own inline messages within the custom layout.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only check
		$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
		if ( \Edidev\AiAssistantForPerfection42\Views\SettingLayout::PAGE_NAME === $page ) {
			return;
		}

		$error_notices = get_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY, array() );
		if ( ! empty( $error_notices ) ) {
			foreach ( $error_notices as $success_notice ) {
				echo "<div class='notice notice-error notice-box'>" . esc_html( $success_notice ) . '</div>';
			}
			delete_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY );
		}

		$success_notices = get_option( self::ADMIN_NOTICE_SUCCESS_MESSAGE_KEY, array() );
		if ( ! empty( $success_notices ) ) {
			foreach ( $success_notices as $success_notice ) {
				echo "<div class='notice notice-success notice-box'>" . esc_html( $success_notice ) . '</div>';
			}
			delete_option( self::ADMIN_NOTICE_SUCCESS_MESSAGE_KEY );
		}
	}


	/**
	 * Register the plugin submenu item in the WordPress admin menu.
	 *
	 * @return void
	 */
	public function register_sub_menu_item(): void {
		add_menu_page(
			esc_html__( 'Perfection42', 'edidev-ai-assistant-for-perfection42' ),
			esc_html__( 'Perfection42', 'edidev-ai-assistant-for-perfection42' ),
			'manage_options',
			SettingLayout::PAGE_NAME,
			array( SettingLayout::instance(), 'render' ),
			'dashicons-admin-tools',
			30
		);
	}

	/**
	 * Add a settings link to the plugin action links.
	 *
	 * @param array $links The existing action links.
	 * @return array The modified action links with the settings link added.
	 */
	public function admin_plugin_settings_link( $links ) {
		$setting_link = '<a href="' . esc_url( 'admin.php?page=edidev-ai-assistant-for-perfection42' ) . '">' . esc_html__( 'Settings', 'edidev-ai-assistant-for-perfection42' ) . '</a>';
		array_unshift( $links, $setting_link );

		return $links;
	}

	/**
	 * Install the plugin and set up necessary options.
	 *
	 * @return void
	 */
	public static function install(): void {
		add_option( 'perfection-ai-product-assistant', self::INSTALLED_PLUGIN );
	}

	/**
	 * Uninstall the plugin and clean up the database.
	 *
	 * @return void
	 */
	public static function uninstall(): void {
		delete_option( 'perfection-ai-product-assistant' );
	}

	/**
	 * Deactivate the plugin programmatically.
	 *
	 * @return void
	 */
	public static function self_deactivate_plugin(): void {
		deactivate_plugins( self::$plugin_basename );
	}

	/**
	 * Get current WooCommerce version.
	 *
	 * @return string
	 */
	public static function get_woocommerce_version(): string {
		// Prefer the WC_VERSION constant if WooCommerce is loaded.
		if ( defined( 'WC_VERSION' ) ) {
			$wc_version = constant( 'WC_VERSION' );
			if ( is_string( $wc_version ) ) {
				return $wc_version;
			}
		}
		// Fallback to the stored option set by WooCommerce.
		$version = get_option( 'woocommerce_version' );
		return is_string( $version ) ? $version : 'NULL';
	}

	/**
	 * Add an error message to be displayed in the admin area.
	 *
	 * @param string $message The error message to be added.
	 */
	public static function notice_error_message( $message ): void {
		$current_messages = get_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY, array() );
		if ( ! in_array( $message, $current_messages, true ) ) {
			$current_messages[] = $message;
			update_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY, $current_messages );
		}
	}

	/**
	 * Check if the current WordPress and WooCommerce versions are compatible with the plugin.
	 *
	 * @return bool
	 */
	public static function compatible_version_checker(): bool {
		return (
			// Using core hook name 'active_plugins' is intentional here to read active plugins list.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true )
			&& version_compare( get_bloginfo( 'version' ), self::COMPATIBILITY_WP_VERSION, '>=' )
			&& version_compare( self::get_woocommerce_version(), self::COMPATIBILITY_WOOCOMMERCE_VERSION, '>=' )
		);
	}

	/**
	 * Get instance of the class.
	 *
	 * @param array $args // Future use.
	 * @return Main
	 */
	public static function instance( $args = array() ): Main {
		return ( new self( $args ) );
	}

	/**
	 * Add privacy policy content related to the plugin.
	 *
	 * @return void
	 */
	public static function perfection42_privacy_policy_content() {
		$p42_privacy_url      = 'https://www.perfection42.com/privacy';
		$p42_api_overview_url = 'https://doc.clickup.com/9004041125/p/h/8caxgx5-49515/1e2e360b7d414ba';

		$html  = '<p><strong>Perfection42 AI Product Assistant for WooCommerce</strong> ';
		$html .= 'sends selected WooCommerce product fields (title, short/long description, categories, tags, and any images you provide) ';
		$html .= 'to the Perfection42 API solely to generate new product content (text, images, video). ';
		$html .= 'No customer or personally identifiable information is transmitted.</p>';

		$html .= '<p>Privacy policy: <a href="' . esc_url( $p42_privacy_url ) . '" target="_blank" rel="noopener noreferrer">'
			. esc_html( $p42_privacy_url ) . '</a><br/>';
		$html .= 'API overview: <a href="' . esc_url( $p42_api_overview_url ) . '" target="_blank" rel="noopener noreferrer">'
			. esc_html( $p42_api_overview_url ) . '</a></p>';

		echo wp_kses_post( $html );
	}

	/**
	 * Get the plugin's asset URL (with trailing slash)
	 *
	 * @return string
	 */
	public static function plugin_asset_url(): string {
		return ( '' !== self::$plugin_asset_url ) ? self::$plugin_asset_url : '';
	}

	/**
	 * Get the full URL for a relative asset path
	 *
	 * @param string $relative Relative path within assets directory.
	 * @return string
	 */
	public static function asset_url( string $relative = '' ): string {
		$base     = self::plugin_asset_url();
		$relative = ltrim( $relative, '/\\' );
		return trailingslashit( $base ) . $relative;
	}

	/**
	 * Get the absolute filesystem path to the assets directory
	 *
	 * @return string
	 */
	public static function asset_base_dir(): string {
		return ( '' !== self::$plugin_asset_dir ) ? self::$plugin_asset_dir : '';
	}
}
