<?php
/**
 * WooCommerce Provider Class
 *
 * @package Edidev\AiAssistantForPerfection42\Includes\WooCommerce
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Includes\WooCommerce;

use Edidev\AiAssistantForPerfection42\Main;
use Edidev\AiAssistantForPerfection42\Includes\ServiceProvider\ServiceProvider;
use Edidev\AiAssistantForPerfection42\Packages\ReactComponents;
use Edidev\AiAssistantForPerfection42\Includes\Setting\SettingProvider;
use Edidev\AiAssistantForPerfection42\Views\SettingLayout;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * WooCommerce Provider.
 */
class WooCommerceProvider extends ServiceProvider {

	/**
	 * WooCommerce Repository instance.
	 *
	 * @var WooCommerceRepository|null
	 */
	protected ?WooCommerceRepository $woo_commerce_repository = null;

	/**
	 * Twig template engine.
	 *
	 * @var object
	 */
	private $twig_template_engine;

	/**
	 * Setting provider instance.
	 *
	 * @var SettingProvider
	 */
	private $setting_provider;

	/**
	 * Constructor.
	 *
	 * @param array $args Constructor arguments.
	 */
	public function __construct( $args = array() ) {
		parent::__construct( $args );

		global $edidev_twig_template_engine;

		$this->twig_template_engine    = $edidev_twig_template_engine;
		$this->woo_commerce_repository = new WooCommerceRepository();
		$this->setting_provider        = new SettingProvider();

		add_action( 'manage_posts_extra_tablenav', array( $this, 'add_product_actions' ), 10, 1 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_perfection' ) );
		add_action( 'admin_footer', array( $this, 'render_perfection_ai_modal' ) );
		add_action( 'before_woocommerce_init', array( $this, 'declare_extension_compatibility_hpos' ) );
		add_action( 'edit_form_after_title', array( $this, 'add_action_to_product_edit' ), 10, 1 );
	}

	/**
	 * Declare WooCommerce HPOS compatibility.
	 *
	 * @return void
	 */
	public function declare_extension_compatibility_hpos(): void {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			$plugin_file = Main::$plugin_basename;
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', $plugin_file, true );
		}
	}

	/**
	 * Add product actions in product list table.
	 *
	 * @param string $which Top/bottom table nav.
	 * @return void
	 */
	public function add_product_actions( $which ) {
		if ( 'top' === $which && $this->is_loadable() ) {
			$this->woo_commerce_repository->add_product_actions_list();
		}
	}

	/**
	 * Enqueue admin scripts/styles for Perfection42.
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_admin_perfection( $hook ) {
		if ( ! $this->is_loadable() ) {
			return;
		}

		$react_component = new ReactComponents();
		$react_component->wordpress_enqueue_scripts();

		$this->load_edidev_wp();
		$this->load_edidev_custom();
	}

	/**
	 * Check if current screen is product list/edit.
	 *
	 * @return bool
	 */
	public function is_loadable(): bool {
		$screen = get_current_screen();
		return ! empty( $screen ) && ( 'edit-product' === $screen->id || 'product' === $screen->id );
	}

	/**
	 * Load WP-related assets.
	 *
	 * @return void
	 */
	public function load_edidev_wp() {
		$assets_url   = Main::plugin_asset_url();
		$is_litespeed = defined( 'LSCACHE_ADV_CACHE' ) || class_exists( 'LiteSpeed\\Core' ) || isset( $_SERVER['X-LSCACHE'] );

		$edidev_php_env_variable = array(
			'pathname'             => Main::PLUGIN_MENU_URL,
			'siteUrl'              => esc_url( get_site_url() ),
			'pluginRootUrl'        => esc_url( Main::$plugin_url ),
			'auth_perfection'      => wp_json_encode( $this->setting_provider->get_perfection_credentials() ),
			'setting_url'          => esc_url( admin_url( 'admin.php?page=' . SettingLayout::PAGE_NAME ) ),
			'api_endpoint'         => esc_url( get_site_url() . '/wp-json/edidev-ai-assistant-for-perfection42/v1' ),
			'wp_rest_nonce'        => wp_create_nonce( 'wp_rest' ),
			'litespeed_aggressive' => $is_litespeed,
		);

		$edidev_react_app_i18n = require Main::$plugin_i18n_dir . '/react-translations.php';

		wp_enqueue_script( 'edidev-wp-env-js', $assets_url . '/js/edidev-wp-env.js', array( 'jquery' ), Main::PLUGIN_VERSION, true );
		wp_add_inline_script( 'edidev-wp-env-js', 'window.EDIDEV_AI_ASSISTANT_FOR_PERFECTION42_WP_CONST = ' . wp_json_encode( $edidev_php_env_variable ), 'before' );
		wp_add_inline_script( 'edidev-wp-env-js', 'window.EDIDEV_AI_ASSISTANT_FOR_PERFECTION42_LOCALE_CONST = ' . wp_json_encode( $edidev_react_app_i18n ), 'before' );

		wp_enqueue_script(
			'edidev-wp-media-component-js',
			$assets_url . '/js/edidev-wp-media-component.js',
			array( 'jquery' ),
			Main::PLUGIN_VERSION,
			true
		);
	}

	/**
	 * Load custom plugin assets.
	 *
	 * @return void
	 */
	public function load_edidev_custom() {
		// New canonical way.
		$assets_url = Main::plugin_asset_url();

		wp_enqueue_style(
			'edidev-woo-products-css',
			$assets_url . '/css/modal-perfection.css',
			array(),
			Main::PLUGIN_VERSION
		);

		wp_enqueue_script(
			'edidev-woo-products-js',
			$assets_url . '/js/modal-perfection.js',
			array( 'jquery' ),
			Main::PLUGIN_VERSION,
			true
		);

		wp_enqueue_script(
			'perfection42-init',
			$assets_url . '/js/perfection42-init.js',
			array( 'jquery', 'wp-i18n' ),
			Main::PLUGIN_VERSION,
			true
		);

		wp_set_script_translations(
			'perfection42-init',
			'edidev-ai-assistant-for-perfection42',
			plugin_dir_path( dirname( __DIR__ ) ) . 'i18n/languages'
		);
	}

	/**
	 * Render Perfection AI modal.
	 *
	 * @return void
	 */
	public function render_perfection_ai_modal() {
		if ( ! $this->is_loadable() ) {
			return;
		}

		$template = $this->twig_template_engine->load( 'PerfectionAILayout.twig' );

		// Twig template already handles escaping.
		echo $template->render(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Add button to product edit page.
	 *
	 * @param WP_Post $post The current post object.
	 * @return void
	 */
	public function add_action_to_product_edit( $post ) {
		$this->woo_commerce_repository->add_product_actions( $post );
	}

	/**
	 * Singleton instance.
	 *
	 * @param array $args Arguments.
	 * @return WooCommerceProvider
	 */
	public static function instance( $args = array() ): WooCommerceProvider {
		return new self( $args );
	}
}
