<?php
/**
 * Setting repository class.
 *
 * @package Edidev\AiAssistantForPerfection42\Includes\Setting
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Includes\Setting;

/**
 * Repository for plugin settings.
 */
class SettingRepository {

	/**
	 * Prefix for OTP transients.
	 */
	const OTP_TRANSIENT_PREFIX = 'edidev_p42_signup_otp_';

	/**
	 * Get current admin user id (0 if none).
	 *
	 * @return int
	 */
	private function get_current_admin_id() {
		return function_exists( 'get_current_user_id' ) ? (int) get_current_user_id() : 0;
	}

	/**
	 * Normalize email (trim + lowercase).
	 *
	 * @param string $email Email address to normalize.
	 * @return string
	 */
	private function normalize_email( $email ) {
		return strtolower( trim( (string) $email ) );
	}

	/**
	 * Build transient key for OTP storage for current admin and email.
	 *
	 * @param string $email Email address to build the key for.
	 * @return string
	 */
	private function build_otp_key( $email ) {
		$admin_id = $this->get_current_admin_id();
		$email_n  = $this->normalize_email( $email );
		return self::OTP_TRANSIENT_PREFIX . $admin_id . '_' . md5( $email_n );
	}

	/**
	 * Filters: TTL, attempts, resend throttles.
	 */
	private function get_otp_config() {
		$ttl            = (int) apply_filters( 'edidev_p42_otp_ttl', 5 * MINUTE_IN_SECONDS );
		$max_attempts   = (int) apply_filters( 'edidev_p42_otp_max_attempts', 5 );
		$min_resend     = (int) apply_filters( 'edidev_p42_otp_min_resend_seconds', 60 );
		$max_sends_hour = (int) apply_filters( 'edidev_p42_otp_max_sends_per_hour', 5 );
		return array(
			'ttl'            => $ttl,
			'max_attempts'   => $max_attempts,
			'min_resend'     => $min_resend,
			'max_sends_hour' => $max_sends_hour,
		);
	}

	/**
	 * Send OTP to email: generate, store hash, and email the code.
	 *
	 * @param string $email Email address to send the OTP to.
	 * @return array|\WP_Error
	 */
	public function send_signup_otp( $email ) {
		$email = sanitize_email( $email );
		if ( false === filter_var( $email, FILTER_VALIDATE_EMAIL ) ) {
			return new \WP_Error( 'p42_email_invalid_format', __( 'Please enter a valid email address.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$config = $this->get_email_validation_config();
		$domain = $this->domain_from_email( $email );
		if ( empty( $domain ) || ! $this->can_receive_mail( $domain, ! empty( $config['require_mx'] ), ! empty( $config['allow_a_record_fallback'] ) ) ) {
			return new \WP_Error( 'p42_email_domain_unreachable', __( 'We could not verify that this email domain can receive messages. Please use a different email.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		if ( ! empty( $config['block_disposable'] ) && $this->is_disposable_domain( $domain ) ) {
			return new \WP_Error( 'p42_email_disposable_blocked', __( 'Temporary or disposable email addresses are not allowed. Please use a permanent email.', 'edidev-ai-assistant-for-perfection42' ) );
		}

		$key  = $this->build_otp_key( $email );
		$cfg  = $this->get_otp_config();
		$now  = time();
		$data = get_transient( $key );
		if ( ! is_array( $data ) ) {
			$data = array(); }
		$next_allowed = isset( $data['next_resend_at'] ) ? (int) $data['next_resend_at'] : 0;
		if ( $next_allowed > $now ) {
			$wait = $next_allowed - $now;
			/* translators: %d: number of seconds */
			return new \WP_Error( 'p42_otp_throttled', sprintf( __( 'Please wait %d seconds before requesting a new code.', 'edidev-ai-assistant-for-perfection42' ), $wait ) );
		}
		$sent_count   = isset( $data['sent_count'] ) ? (int) $data['sent_count'] : 0;
		$window_start = isset( $data['window_start'] ) ? (int) $data['window_start'] : $now;
		if ( $now - $window_start >= HOUR_IN_SECONDS ) {
			$sent_count   = 0;
			$window_start = $now;
		}
		if ( $sent_count >= $cfg['max_sends_hour'] ) {
			return new \WP_Error( 'p42_otp_rate_limited', __( 'Too many verification emails requested. Please try again later.', 'edidev-ai-assistant-for-perfection42' ) );
		}

		// Generate OTP and store hashed version only.
		$code    = (string) random_int( 100000, 999999 );
		$salt    = wp_generate_password( 20, true, true );
		$hash    = hash_hmac( 'sha256', $code, $salt );
		$expires = $now + (int) $cfg['ttl'];
		$new     = array(
			'hash'           => $hash,
			'salt'           => $salt,
			'email'          => $this->normalize_email( $email ),
			'created_at'     => $now,
			'expires_at'     => $expires,
			'attempts'       => 0,
			'sent_count'     => $sent_count + 1,
			'window_start'   => $window_start,
			'next_resend_at' => $now + (int) $cfg['min_resend'],
		);
		set_transient( $key, $new, $cfg['ttl'] );

		// Send email with wp_mail().
		$site = wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES );
		/* translators: %s: site name */
		$subject = sprintf( __( '[%s] Your verification code', 'edidev-ai-assistant-for-perfection42' ), $site );
		$minutes = max( 1, (int) floor( $cfg['ttl'] / 60 ) );
		/* translators: 1: verification code, 2: expiration time in minutes */
		$message = sprintf( __( 'Your Perfection42 verification code is %1$s. It expires in %2$d minutes. If you did not request this, you can ignore this email.', 'edidev-ai-assistant-for-perfection42' ), $code, $minutes );
		wp_mail( $email, $subject, $message );

		return array(
			'ttl'            => (int) $cfg['ttl'],
			'resend_in'      => (int) $cfg['min_resend'],
			'next_resend_at' => (int) $new['next_resend_at'],
		);
	}

	/**
	 * Verify OTP and return a signed email verification token on success.
	 *
	 * @param string $email Email address to verify.
	 * @param string $otp   One-time password code.
	 * @return array|\WP_Error
	 */
	public function verify_signup_otp( $email, $otp ) {
		$email_n = $this->normalize_email( sanitize_email( $email ) );
		$otp     = trim( (string) $otp );
		if ( '' === $email_n || '' === $otp ) {
			return new \WP_Error( 'p42_otp_invalid', __( 'Invalid verification request.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$key  = $this->build_otp_key( $email_n );
		$data = get_transient( $key );
		$cfg  = $this->get_otp_config();
		if ( ! is_array( $data ) ) {
			return new \WP_Error( 'p42_otp_missing', __( 'No verification code found. Please request a new code.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		if ( time() > (int) $data['expires_at'] ) {
			return new \WP_Error( 'p42_otp_expired', __( 'Your verification code has expired. Please request a new code.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$attempts = isset( $data['attempts'] ) ? (int) $data['attempts'] : 0;
		if ( $attempts >= (int) $cfg['max_attempts'] ) {
			return new \WP_Error( 'p42_otp_attempts_exceeded', __( 'Too many incorrect attempts. Please request a new code.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$calc = hash_hmac( 'sha256', $otp, (string) $data['salt'] );
		if ( ! hash_equals( (string) $data['hash'], $calc ) ) {
			$data['attempts'] = $attempts + 1;
			set_transient( $key, $data, max( 30, (int) ( $data['expires_at'] - time() ) ) );
			return new \WP_Error( 'p42_otp_incorrect', __( 'The verification code you entered is incorrect.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		// Success: issue signed token for final submit and delete transient.
		$token = $this->issue_email_verified_token( $email_n );
		delete_transient( $key );
		return array( 'token' => $token );
	}

	/**
	 * Issue an HMAC-signed token for verified email (short-lived).
	 *
	 * @param string $email_n Normalized email.
	 * @return string
	 */
	private function issue_email_verified_token( $email_n ) {
		$ttl     = (int) apply_filters( 'edidev_p42_otp_token_ttl', 2 * MINUTE_IN_SECONDS );
		$exp     = time() + $ttl;
		$sid     = wp_generate_password( 16, false, false );
		$payload = array(
			'email' => $email_n,
			'exp'   => $exp,
			'sid'   => $sid,
		);
		$json    = wp_json_encode( $payload );
		$secret  = defined( 'AUTH_KEY' ) ? AUTH_KEY : ( wp_salt( 'auth' ) );
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Used for encoding verification token payload.
		$body = rtrim( strtr( base64_encode( $json ), '+/', '-_' ), '=' );
		$sig  = hash_hmac( 'sha256', $body, $secret );
		return $body . '.' . $sig;
	}

	/**
	 * Verify signed token against email.
	 *
	 * @param string $token Signed email verification token.
	 * @param string $email Email address to verify against.
	 * @return bool|\WP_Error
	 */
	private function verify_email_verified_token( $token, $email ) {
		if ( empty( $token ) ) {
			return new \WP_Error( 'p42_email_otp_required', __( 'Please verify your email with the OTP code before creating an account.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$parts = explode( '.', (string) $token );
		if ( count( $parts ) !== 2 ) {
			return new \WP_Error( 'p42_email_otp_invalid', __( 'Invalid email verification token. Please verify again.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		list( $body, $sig ) = $parts;
		$secret             = defined( 'AUTH_KEY' ) ? AUTH_KEY : ( wp_salt( 'auth' ) );
		$calc               = hash_hmac( 'sha256', $body, $secret );
		if ( ! hash_equals( $sig, $calc ) ) {
			return new \WP_Error( 'p42_email_otp_invalid_sig', __( 'Invalid email verification token. Please verify again.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Used for decoding verification token payload.
		$json = base64_decode( strtr( $body, '-_', '+/' ) );
		$data = json_decode( (string) $json, true );
		if ( empty( $data ) || ! is_array( $data ) ) {
			return new \WP_Error( 'p42_email_otp_invalid_payload', __( 'Invalid email verification token. Please verify again.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		if ( time() > (int) ( $data['exp'] ?? 0 ) ) {
			return new \WP_Error( 'p42_email_otp_expired', __( 'Your email verification has expired. Please request a new code.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$email_n = $this->normalize_email( sanitize_email( $email ) );
		if ( (string) ( $data['email'] ?? '' ) !== $email_n ) {
			return new \WP_Error( 'p42_email_otp_email_mismatch', __( 'Verified email does not match. Please verify again.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		return true;
	}

	/**
	 * Option name for general settings.
	 */
	const GENERAL_SETTINGS_OPTION_NAME = 'wcedidevpa_general_settings';

	/**
	 * Option name for API settings.
	 */
	const API_SETTINGS_OPTION_NAME = 'wcedidevpa_api_settings';

	/**
	 * Constructor.
	 *
	 * @param array $args Optional args.
	 */
	public function __construct( $args = array() ) {
		// No init needed.
	}

	/**
	 * Validate login credentials by calling API.
	 *
	 * @param string $username Username.
	 * @param string $password Password.
	 * @return bool True if login successful, false otherwise.
	 */
	private function validate_login( $username, $password ): bool {
		if ( empty( $username ) || empty( $password ) ) {
			return false;
		}

		// Prepare request body.
		$request_body = array(
			'username' => $username,
			'password' => $password,
		);

		// Make API call to sign-in endpoint.
		$api_url  = 'https://api.perfection42.com/v1/sign-in';
		$response = wp_remote_post(
			$api_url,
			array(
				'method'      => 'POST',
				'timeout'     => 30,
				'headers'     => array(
					'Content-Type' => 'application/json',
				),
				'body'        => wp_json_encode( $request_body ),
				'data_format' => 'body',
			)
		);

		// Check for errors.
		if ( is_wp_error( $response ) ) {
			return false;
		}

		// Get response code.
		$response_code = wp_remote_retrieve_response_code( $response );

		// Consider success if status code is 200.
		return 200 === $response_code;
	}

	/**
	 * Save general tab data.
	 *
	 * @param array $posted_data Posted form data.
	 * @return bool
	 */
	public function save_general_tab( $posted_data = array() ): bool {
		$username = isset( $posted_data['username'] ) ? sanitize_text_field( $posted_data['username'] ) : '';
		$password = isset( $posted_data['password'] ) ? sanitize_text_field( $posted_data['password'] ) : '';

		// Require both username and password.
		if ( empty( $username ) || empty( $password ) ) {
			return false;
		}

		// Validate login credentials before saving.
		$is_valid = $this->validate_login( $username, $password );
		if ( ! $is_valid ) {
			return false;
		}

		$current_user   = wp_get_current_user();
		$data_to_update = array(
			'username'   => $username,
			'password'   => $password,
			'updated_by' => ! empty( $current_user )
				? sprintf( '%s|%s', $current_user->data->user_nicename, $current_user->data->user_email )
				: 'N/A',
			'updated_at' => gmdate( 'Y-m-d H:i:s' ),
		);

		return update_option( self::GENERAL_SETTINGS_OPTION_NAME, $data_to_update );
	}

	/**
	 * Clear stored credentials (used for sign-out).
	 *
	 * @return bool
	 */
	public function clear_credentials_and_workflows(): bool {
		$current_user   = wp_get_current_user();
		$data_to_update = array(
			'username'   => '',
			'password'   => '',
			'updated_by' => ! empty( $current_user )
				? sprintf( '%s|%s', $current_user->data->user_nicename, $current_user->data->user_email )
				: 'sign-out',
			'updated_at' => gmdate( 'Y-m-d H:i:s' ),
		);

		return update_option( self::GENERAL_SETTINGS_OPTION_NAME, $data_to_update );
	}

	/**
	 * Get general tab data.
	 *
	 * @return array
	 */
	public function get_general_tab(): array {
		return get_option( self::GENERAL_SETTINGS_OPTION_NAME, array() );
	}

	/**
	 * Get Perfection login credentials.
	 *
	 * @return array
	 */
	public function get_perfection_credentials(): array {
		$data = get_option( self::GENERAL_SETTINGS_OPTION_NAME, array() );

		return array(
			'username' => $data['username'] ?? '',
			'password' => $data['password'] ?? '',
		);
	}

	// Removed deprecated workflows storage and retrieval methods.

	/**
	 * Check debug mode.
	 *
	 * @return bool
	 */
	public function is_debug_mode(): bool {
		$data = get_option( self::GENERAL_SETTINGS_OPTION_NAME, array() );
		return ! empty( $data['is_debug_mode'] ) && (int) 1 === $data['is_debug_mode'];
	}

	/**
	 * Fetch Perfection42 account info using a valid ID token.
	 *
	 * This performs a server-side GET to Perfection42 "Get Account" endpoint
	 * with the provided Bearer token. On success, returns the decoded response
	 * array; on failure, returns a \WP_Error with one of the following codes:
	 *  - 'p42_account_fetch_failed' when the HTTP request fails entirely
	 *  - 'http_401' for unauthorized
	 *  - 'http_403' for forbidden
	 *  - 'http_error' for other HTTP errors (>= 400)
	 *
	 * @param string $id_token Bearer token from sign-in.
	 * @return array|\WP_Error
	 */
	public function fetch_account( string $id_token ) {
		$response = wp_remote_get(
			'https://api.perfection42.com/v1/api/account',
			array(
				'headers' => array( 'Authorization' => 'Bearer ' . $id_token ),
				'timeout' => 20,
			)
		);
		if ( is_wp_error( $response ) ) {
			return new \WP_Error( 'p42_account_fetch_failed', __( 'Unable to reach Perfection42 account service.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$code = wp_remote_retrieve_response_code( $response );
		$body = (string) wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );
		if ( $code >= 400 ) {
			$err_code = ( 401 === $code ) ? 'http_401' : ( ( 403 === $code ) ? 'http_403' : 'http_error' );
			$message  = $data['message'] ?? __( 'Failed to fetch account info from Perfection42.', 'edidev-ai-assistant-for-perfection42' );
			return new \WP_Error( $err_code, $message );
		}
		return is_array( $data ) ? $data : array();
	}

	/**
	 * Configuration for email validation; filterable for site-level overrides.
	 *
	 * @return array
	 */
	private function get_email_validation_config() {
		$defaults = array(
			'require_mx'              => true,
			'allow_a_record_fallback' => false,
			'block_disposable'        => true,
		);
		/**
		 * Filter: edidev_p42_email_validation_config
		 *
		 * @param array $defaults Default config values.
		 */
		return apply_filters( 'edidev_p42_email_validation_config', $defaults );
	}

	/**
	 * Extract domain part from an email address.
	 *
	 * @param string $email Email.
	 * @return string Domain or empty string.
	 */
	private function domain_from_email( $email ) {
		$at = strpos( $email, '@' );
		if ( false === $at ) {
			return '';
		}
		return strtolower( trim( substr( $email, $at + 1 ) ) );
	}

	/**
	 * Check if a domain can receive mail using DNS.
	 *
	 * @param string $domain Domain to check.
	 * @param bool   $require_mx Require MX records to pass.
	 * @param bool   $allow_a_fallback Allow A/AAAA fallback if no MX.
	 * @return bool
	 */
	private function can_receive_mail( $domain, $require_mx = true, $allow_a_fallback = false ) {
		if ( empty( $domain ) ) {
			return false;
		}
		// Suppress warnings if dns_get_record isn’t available or fails.
		$has_mx = false;
		if ( function_exists( 'dns_get_record' ) ) {
			$mx     = @dns_get_record( $domain, DNS_MX ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
			$has_mx = is_array( $mx ) && count( $mx ) > 0;
		}
		if ( $require_mx ) {
			return $has_mx;
		}
		if ( $has_mx ) {
			return true;
		}
		if ( ! $allow_a_fallback ) {
			return false;
		}
		// Fallback: A/AAAA records exist.
		if ( function_exists( 'dns_get_record' ) ) {
			$a    = @dns_get_record( $domain, DNS_A ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
			$aaaa = @dns_get_record( $domain, DNS_AAAA ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
			return ( is_array( $a ) && count( $a ) > 0 ) || ( is_array( $aaaa ) && count( $aaaa ) > 0 );
		}
		return false;
	}

	/**
	 * Check if a domain is considered disposable/temporary.
	 *
	 * @param string $domain Domain to check.
	 * @return bool
	 */
	private function is_disposable_domain( string $domain ): bool {
		$default_blocklist = array(
			'mailinator.com',
			'guerrillamail.com',
			'10minutemail.com',
			'tempmail.com',
			'trashmail.com',
			'dispostable.com',
			'getnada.com',
			'yopmail.com',
		);
		/**
		 * Filter: edidev_p42_disposable_domains
		 *
		 * @param array $default_blocklist Default list of disposable domains.
		 */
		$blocklist = apply_filters( 'edidev_p42_disposable_domains', $default_blocklist );
		$domain    = strtolower( $domain );
		foreach ( (array) $blocklist as $blocked ) {
			$blocked = strtolower( trim( (string) $blocked ) );
			if ( '' === $blocked ) {
				continue;
			}
			if ( $domain === $blocked || ( function_exists( 'str_ends_with' ) ? str_ends_with( $domain, '.' . $blocked ) : (bool) preg_match( '/\\.' . preg_quote( $blocked, '/' ) . '$/i', $domain ) ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Get minimum password length for signup (filterable).
	 *
	 * @return int
	 */
	private function get_password_min_length(): int {
		$min = 8;
		/**
		 * Filter: edidev_p42_password_min_length
		 *
		 * @param int $min Default minimum length (8).
		 */
		return (int) apply_filters( 'edidev_p42_password_min_length', $min );
	}

	/**
	 * Save sign up tab data.
	 *
	 * Calls the WordPress gateway endpoint to create a new Perfection42 account.
	 * Returns detailed error information when the remote API reports failure so
	 * callers can surface the exact message to the user.
	 *
	 * @param array $posted_data Posted form data.
	 * @return bool|array|\WP_Error True/array on success, WP_Error on failure.
	 */
	public function save_signup_tab( $posted_data = array() ) {
		// Validate required fields.
		$name     = $posted_data['name'] ?? '';
		$email    = $posted_data['email'] ?? '';
		$password = $posted_data['password'] ?? '';

		if ( empty( $name ) || empty( $email ) || empty( $password ) ) {
			return false;
		}

		// Server-side validation (authoritative) before hitting gateway.
		$name = trim( (string) $name );
		if ( strlen( $name ) < 2 ) {
			return new \WP_Error( 'p42_name_required', __( 'Please enter your name.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$email = sanitize_email( $email );
		if ( false === filter_var( $email, FILTER_VALIDATE_EMAIL ) ) {
			return new \WP_Error( 'p42_email_invalid_format', __( 'Please enter a valid email address.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		// Require a verified email token produced by OTP verification (AJAX flow).
		$verified_token = isset( $posted_data['email_verified_token'] ) ? (string) $posted_data['email_verified_token'] : '';

		$token_check = $this->verify_email_verified_token( $verified_token, $email );
		if ( is_wp_error( $token_check ) ) {
			return $token_check;
		}
		$min_len = method_exists( $this, 'get_password_min_length' ) ? $this->get_password_min_length() : 8;
		if ( strlen( (string) $password ) < (int) $min_len ) {
			return new \WP_Error( 'p42_password_too_short', __( 'Please enter a password of at least 8 characters.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$config = $this->get_email_validation_config();
		$domain = $this->domain_from_email( $email );
		if ( empty( $domain ) ) {
			return new \WP_Error( 'p42_email_missing_domain', __( 'Please enter a valid, deliverable email address.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		if ( ! $this->can_receive_mail( $domain, ! empty( $config['require_mx'] ), ! empty( $config['allow_a_record_fallback'] ) ) ) {
			return new \WP_Error( 'p42_email_domain_unreachable', __( 'We could not verify that this email domain can receive messages. Please use a different email.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		if ( ! empty( $config['block_disposable'] ) && $this->is_disposable_domain( $domain ) ) {
			return new \WP_Error( 'p42_email_disposable_blocked', __( 'Temporary or disposable email addresses are not allowed. Please use a permanent email.', 'edidev-ai-assistant-for-perfection42' ) );
		}

		// Prepare request body.
		$request_body = array(
			'name'     => $name,
			'email'    => $email,
			'password' => $password,
		);

		// Make API call to WordPress gateway sign-up endpoint.
		$api_url  = 'https://p42wp.com/wp-json/edidev-signup-gateway-perfection42/v1/signup';
		$response = wp_remote_post(
			$api_url,
			array(
				'method'      => 'POST',
				'timeout'     => 30,
				'headers'     => array(
					'Content-Type' => 'application/json',
				),
				'body'        => wp_json_encode( $request_body ),
				'data_format' => 'body',
			)
		);
		// Transport error.
		if ( is_wp_error( $response ) ) {
			return new \WP_Error( 'p42_signup_request_failed', __( 'Unable to reach Perfection42 sign-up service.', 'edidev-ai-assistant-for-perfection42' ) );
		}

		// Parse response.
		$response_code = wp_remote_retrieve_response_code( $response );
		$body_raw      = (string) wp_remote_retrieve_body( $response );
		$body_json     = json_decode( $body_raw, true );

		// Success cases (200/201). Return decoded body when available for further use.
		if ( in_array( $response_code, array( 200, 201 ), true ) ) {
			return is_array( $body_json ) ? $body_json : true;
		}

		// Failure: try to extract concrete error message from expected shape.
		$error_message = '';
		if ( is_array( $body_json ) ) {
			// According to provided shape: { success:false, code, message, error, data }.
			// Prefer detailed message if available at data.detail.
			$detail = $body_json['data']['detail'] ?? null;
			if ( ! empty( $detail ) ) {
				if ( is_string( $detail ) ) {
					$error_message = $detail;
				} elseif ( is_array( $detail ) ) {
					// Flatten array values to a readable string.
					$parts = array();
					foreach ( $detail as $k => $v ) {
						if ( is_string( $v ) ) {
							$parts[] = $v;
						} elseif ( is_array( $v ) ) {
							$parts[] = wp_json_encode( $v );
						} else {
							$parts[] = (string) $v;
						}
					}
					$error_message = trim( implode( ' ', array_filter( $parts ) ) );
				} else {
					// Scalar/object: cast to string/json for readability.
					$error_message = is_scalar( $detail ) ? (string) $detail : wp_json_encode( $detail );
				}
			} elseif ( ! empty( $body_json['error'] ) && is_string( $body_json['error'] ) ) {
				$error_message = $body_json['error'];
			} elseif ( ! empty( $body_json['message'] ) && is_string( $body_json['message'] ) ) {
				$error_message = $body_json['message'];
			}
		}
		if ( '' === $error_message ) {
			$error_message = __( 'New Perfection42 account has not been created. Something went wrong. Please try again!', 'edidev-ai-assistant-for-perfection42' );
		}

		$code = is_array( $body_json ) && ! empty( $body_json['code'] ) && is_string( $body_json['code'] )
			? $body_json['code']
			: 'p42_signup_failed';

		return new \WP_Error(
			$code,
			$error_message,
			array(
				'status'   => $response_code,
				'response' => $body_json,
			)
		);
	}
}
