<?php
/**
 * SettingProvider class file.
 *
 * @package Edidev\AiAssistantForPerfection42
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Includes\Setting;

use Edidev\AiAssistantForPerfection42\Includes\ServiceProvider\ServiceProvider;
use Edidev\AiAssistantForPerfection42\Main;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class SettingProvider
 *
 * Provides access to saving and retrieving plugin settings.
 */
class SettingProvider extends ServiceProvider {


	/**
	 * General tab URL.
	 */
	const GENERAL_TAB_URL = 'admin.php?page=edidev-ai-assistant-for-perfection42&tab=general';

	/**
	 * Sign up tab URL.
	 */
	const SIGNUP_TAB_URL = 'admin.php?page=edidev-ai-assistant-for-perfection42&tab=signup';

	/**
	 * Admin notice option key for error messages.
	 */
	const ADMIN_NOTICE_ERROR_MESSAGE_KEY = 'edidev-perfection_admin_notice_setting_page_error_message';

	/**
	 * Admin notice option key for success messages.
	 */
	const ADMIN_NOTICE_SUCCESS_MESSAGE_KEY = 'edidev-perfection_admin_notice_setting_page_success_message';

	/**
	 * Setting repository instance.
	 *
	 * @var SettingRepository|null
	 */
	protected ?SettingRepository $setting_repository = null;

	/**
	 * Constructor.
	 *
	 * @param array $args Optional arguments.
	 */
	public function __construct( $args = array() ) {
		parent::__construct( $args );
		$this->setting_repository = new SettingRepository();
		$this->hook_settings_assets();
		$this->hook_ajax();
	}

	/**
	 * Hook AJAX actions for async auth flows.
	 */
	public function hook_ajax(): void {
		add_action( 'wp_ajax_p42_sign_in', array( $this, 'ajax_sign_in' ) );
		add_action( 'wp_ajax_p42_sign_out', array( $this, 'ajax_sign_out' ) );
		add_action( 'wp_ajax_p42_sign_up', array( $this, 'ajax_sign_up' ) );
		add_action( 'wp_ajax_p42_get_account', array( $this, 'ajax_get_account' ) );
		// OTP (legacy AJAX endpoints kept for resend use).
		add_action( 'wp_ajax_p42_send_signup_otp', array( $this, 'ajax_send_signup_otp' ) );
		add_action( 'wp_ajax_p42_verify_signup_otp', array( $this, 'ajax_verify_signup_otp' ) );
		add_action( 'wp_ajax_p42_resend_signup_otp', array( $this, 'ajax_resend_signup_otp' ) );
	}

	/**
	 * Hook settings page assets
	 */
	public function hook_settings_assets(): void {
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_settings_assets' ) );
	}

	/**
	 * Enqueue assets only on the settings page.
	 *
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_settings_assets( $hook ): void {
		// Detect we are on this plugin's Settings page only.
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only usage in admin to determine if the settings page is being viewed.
		$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

		// Check if we're on the Perfection42 settings page.
		$is_settings = ( 'edidev-ai-assistant-for-perfection42' === $page )
			|| ( $screen && ( 'toplevel_page_edidev-ai-assistant-for-perfection42' === $screen->id || 'toplevel_page_edidev-ai-assistant-for-perfection42' === $screen->base ) );

		if ( ! $is_settings ) {
			return;
		}

		// Local Select2 assets.
		$css_select2 = Main::asset_url( 'vendor/select2/css/select2.min.css' );
		$js_select2  = Main::asset_url( 'vendor/select2/js/select2.full.min.js' );

		wp_register_style( 'edidev-select2', $css_select2, array(), '4.0.13' );
		wp_register_script( 'edidev-select2', $js_select2, array( 'jquery' ), '4.0.13', true );

		// Plugin admin CSS/JS for settings page.
		$css_admin = Main::asset_url( 'css/perfection42-admin.css' );
		$js_admin  = Main::asset_url( 'js/perfection42-init.js' );

		if ( file_exists( Main::asset_base_dir() . '/css/perfection42-admin.css' ) ) {
			wp_register_style(
				'edidev-perfection42-admin',
				$css_admin,
				array(),
				filemtime( Main::asset_base_dir() . '/css/perfection42-admin.css' )
			);
		}

		if ( file_exists( Main::asset_base_dir() . '/js/perfection42-init.js' ) ) {
			wp_register_script(
				'edidev-perfection42-init',
				$js_admin,
				array( 'jquery', 'edidev-select2', 'wp-i18n' ),
				filemtime( Main::asset_base_dir() . '/js/perfection42-init.js' ),
				true
			);
		}

		// Enqueue assets.
		wp_enqueue_style( 'edidev-select2' );
		wp_enqueue_script( 'edidev-select2' );

		if ( wp_style_is( 'edidev-perfection42-admin', 'registered' ) ) {
			wp_enqueue_style( 'edidev-perfection42-admin' );
		}
		if ( wp_script_is( 'edidev-perfection42-init', 'registered' ) ) {
			wp_enqueue_script( 'edidev-perfection42-init' );

			// Provide AJAX configuration and i18n to the settings script.
			wp_localize_script(
				'edidev-perfection42-init',
				'P42Settings',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( 'p42_settings_ajax' ),
					'i18n'    => array(
						'signingIn'         => __( 'Signing in…', 'edidev-ai-assistant-for-perfection42' ),
						'signingOut'        => __( 'Signing out…', 'edidev-ai-assistant-for-perfection42' ),
						'creatingAcc'       => __( 'Creating account…', 'edidev-ai-assistant-for-perfection42' ),
						'signIn'            => __( 'Sign In', 'edidev-ai-assistant-for-perfection42' ),
						'signOut'           => __( 'Sign Out', 'edidev-ai-assistant-for-perfection42' ),
						'signUp'            => __( 'Sign Up', 'edidev-ai-assistant-for-perfection42' ),
						'genericErr'        => __( 'Something went wrong. Please try again.', 'edidev-ai-assistant-for-perfection42' ),
						'account'           => __( 'Perfection42 Account', 'edidev-ai-assistant-for-perfection42' ),
						'checkStatus'       => __( 'Check Status', 'edidev-ai-assistant-for-perfection42' ),
						'refreshing'        => __( 'Refreshing…', 'edidev-ai-assistant-for-perfection42' ),
						'loadingAcct'       => __( 'Loading account…', 'edidev-ai-assistant-for-perfection42' ),
						'labelEmail'        => __( 'Email', 'edidev-ai-assistant-for-perfection42' ),
						'labelName'         => __( 'Name', 'edidev-ai-assistant-for-perfection42' ),
						'labelStatus'       => __( 'Status', 'edidev-ai-assistant-for-perfection42' ),
						'labelPlan'         => __( 'Plan', 'edidev-ai-assistant-for-perfection42' ),
						'labelCreated'      => __( 'Created', 'edidev-ai-assistant-for-perfection42' ),
						'labelUpdated'      => __( 'Updated', 'edidev-ai-assistant-for-perfection42' ),
						'labelId'           => __( 'ID', 'edidev-ai-assistant-for-perfection42' ),
						'labelPackageType'  => __( 'Package Type', 'edidev-ai-assistant-for-perfection42' ),
						'labelOrganization' => __( 'Organization', 'edidev-ai-assistant-for-perfection42' ),
						// New CTA labels for Account panel.
						'myProjects'        => __( 'My Projects', 'edidev-ai-assistant-for-perfection42' ),
						'myProfile'         => __( 'My Profile', 'edidev-ai-assistant-for-perfection42' ),
						'ariaOpenProjects'  => __( 'Open My Projects on Perfection42 Platform (opens in a new tab)', 'edidev-ai-assistant-for-perfection42' ),
						'ariaOpenProfile'   => __( 'Open My Profile on Perfection42 Platform (opens in a new tab)', 'edidev-ai-assistant-for-perfection42' ),
						// OTP related.
						'sendingCode'       => __( 'Sending…', 'edidev-ai-assistant-for-perfection42' ),
						'otpSent'           => __( 'We sent a 6-digit code to your email.', 'edidev-ai-assistant-for-perfection42' ),
						'verifying'         => __( 'Verifying…', 'edidev-ai-assistant-for-perfection42' ),
						'otpVerified'       => __( 'Email verified.', 'edidev-ai-assistant-for-perfection42' ),
					),
				)
			);
		}
	}

	/**
	 * AJAX: Sign in
	 */
	public function ajax_sign_in(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}

		$username = isset( $_POST['username'] ) ? sanitize_text_field( wp_unslash( $_POST['username'] ) ) : '';
		$password = isset( $_POST['password'] ) ? sanitize_text_field( wp_unslash( $_POST['password'] ) ) : '';

		if ( empty( $username ) || empty( $password ) ) {
			self::notice_error_message( esc_html__( 'Email and password are required. Please fill in both fields.', 'edidev-ai-assistant-for-perfection42' ) );
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Email and password are required. Please fill in both fields.', 'edidev-ai-assistant-for-perfection42' ) ) ), 200 );
		}

		// Try to save and validate via repository directly.
		$result = $this->setting_repository->save_general_tab(
			array(
				'username' => $username,
				'password' => $password,
			)
		);

		if ( $result ) {
			self::notice_success_message( esc_html__( 'Signed in successfully.', 'edidev-ai-assistant-for-perfection42' ) );
			$data = $this->setting_repository->get_general_tab();
			wp_send_json_success(
				array(
					'messages' => array( esc_html__( 'Signed in successfully.', 'edidev-ai-assistant-for-perfection42' ) ),
					'data'     => $data,
				)
			);
		} else {
			self::notice_error_message( esc_html__( 'Please check your email and password again.', 'edidev-ai-assistant-for-perfection42' ) );
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Please check your email and password again.', 'edidev-ai-assistant-for-perfection42' ) ) ) );
		}
	}

	/**
	 * AJAX: Sign out
	 */
	public function ajax_sign_out(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}

		$this->sign_out();
		wp_send_json_success( array( 'messages' => array( esc_html__( 'Signed out successfully.', 'edidev-ai-assistant-for-perfection42' ) ) ) );
	}

	/**
	 * AJAX: Sign up
	 */
	public function ajax_sign_up(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}

		$name                 = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$email                = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$password             = isset( $_POST['password'] ) ? sanitize_text_field( wp_unslash( $_POST['password'] ) ) : '';
		$email_verified_token = isset( $_POST['email_verified_token'] ) ? sanitize_text_field( wp_unslash( $_POST['email_verified_token'] ) ) : '';

		$result = $this->setting_repository->save_signup_tab(
			array(
				'name'                 => $name,
				'email'                => $email,
				'password'             => $password,
				'email_verified_token' => $email_verified_token,
			)
		);

		// If repository returned a WP_Error, surface the exact error message.
		if ( is_wp_error( $result ) ) {
			set_transient( 'edidev_p42_show_signup_form', 1, MINUTE_IN_SECONDS );
			$message = $result->get_error_message();
			self::notice_error_message( $message );
			wp_send_json_error( array( 'messages' => array( $message ) ) );
		}

		if ( $result ) {
			delete_transient( 'edidev_p42_show_signup_form' );
			self::notice_success_message( esc_html__( 'New Perfection42 account has been created!', 'edidev-ai-assistant-for-perfection42' ) );
			wp_send_json_success( array( 'messages' => array( esc_html__( 'New Perfection42 account has been created!', 'edidev-ai-assistant-for-perfection42' ) ) ) );
		}

		set_transient( 'edidev_p42_show_signup_form', 1, MINUTE_IN_SECONDS );
		// Fallback generic message if repository returned false (legacy behavior).
		$generic = esc_html__( 'New Perfection42 account has not been created. Something went wrong. Please try again!', 'edidev-ai-assistant-for-perfection42' );
		self::notice_error_message( $generic );
		wp_send_json_error( array( 'messages' => array( $generic ) ) );
	}

	/**
	 * AJAX: Send signup OTP
	 */
	public function ajax_send_signup_otp(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		if ( empty( $email ) ) {
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Please enter your email address.', 'edidev-ai-assistant-for-perfection42' ) ) ) );
		}
		$result = $this->setting_repository->send_signup_otp( $email );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'messages' => array( $result->get_error_message() ) ) );
		}
		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Verify signup OTP
	 */
	public function ajax_verify_signup_otp(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}
		$email  = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$otp    = isset( $_POST['otp'] ) ? sanitize_text_field( wp_unslash( $_POST['otp'] ) ) : '';
		$result = $this->setting_repository->verify_signup_otp( $email, $otp );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'messages' => array( $result->get_error_message() ) ) );
		}
		wp_send_json_success( $result );
	}

	/**
	 * Save general tab settings.
	 *
	 * @param array $posted_data Posted data.
	 */
	public function save_generate_tab( $posted_data ): void {
		$username = isset( $posted_data['username'] ) ? sanitize_text_field( $posted_data['username'] ) : '';
		$password = isset( $posted_data['password'] ) ? sanitize_text_field( $posted_data['password'] ) : '';

		// Require both username and password to be provided.
		if ( empty( $username ) || empty( $password ) ) {
			self::notice_error_message(
				esc_html__( 'Email and password are required. Please fill in both fields.', 'edidev-ai-assistant-for-perfection42' )
			);
			return;
		}

		// Validate login first - if validation fails, save_general_tab will return false.
		$result = $this->setting_repository->save_general_tab( $posted_data );

		if ( $result ) {
			self::notice_success_message(
				esc_html__( 'Signed in successfully.', 'edidev-ai-assistant-for-perfection42' )
			);
		} else {
			self::notice_error_message(
				esc_html__( 'Please check your email and password again.', 'edidev-ai-assistant-for-perfection42' )
			);
		}
	}

	/**
	 * Sign out: clear stored credentials and notify success.
	 *
	 * @return void
	 */
	public function sign_out(): void {
		// Clear cached token before clearing creds, if any.
		$creds = $this->setting_repository->get_perfection_credentials();
		if ( ! empty( $creds['username'] ) ) {
			$cache_key = $this->get_token_transient_key( $creds['username'] );
			delete_transient( $cache_key );
		}
		if ( method_exists( $this->setting_repository, 'clear_credentials_and_workflows' ) ) {
			$this->setting_repository->clear_credentials_and_workflows();
		} else {
			// Fallback: overwrite option directly if repository method is unavailable.
			update_option(
				\Edidev\AiAssistantForPerfection42\Includes\Setting\SettingRepository::GENERAL_SETTINGS_OPTION_NAME,
				array(
					'username'   => '',
					'password'   => '',
					'updated_by' => 'sign-out',
					'updated_at' => gmdate( 'Y-m-d H:i:s' ),
				)
			);
		}

		self::notice_success_message( esc_html__( 'Signed out successfully.', 'edidev-ai-assistant-for-perfection42' ) );
	}

	/**
	 * AJAX: Get Account info via Perfection42 API using stored credentials.
	 */
	public function ajax_get_account(): void {
		check_ajax_referer( 'p42_settings_ajax', 'nonce' );
		if ( ! ( current_user_can( 'manage_woocommerce' ) || current_user_can( 'manage_options' ) ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Permission denied.', 'edidev-ai-assistant-for-perfection42' ) ) ), 403 );
		}

		$creds    = $this->setting_repository->get_perfection_credentials();
		$username = $creds['username'] ?? '';
		$password = $creds['password'] ?? '';
		if ( empty( $username ) || empty( $password ) ) {
			wp_send_json_error( array( 'messages' => array( esc_html__( 'Not signed in. Please sign in first.', 'edidev-ai-assistant-for-perfection42' ) ) ), 200 );
		}

		$id_token = $this->get_cached_id_token( $username, $password );
		if ( is_wp_error( $id_token ) ) {
			wp_send_json_error( array( 'messages' => array( $id_token->get_error_message() ) ) );
		}

		$account = $this->setting_repository->fetch_account( $id_token );
		if ( is_wp_error( $account ) ) {
			// If unauthorized, clear cached token so next call refreshes.
			if ( in_array( $account->get_error_code(), array( 'p42_unauthorized', 'http_401', 'http_403' ), true ) ) {
				delete_transient( $this->get_token_transient_key( $username ) );
			}
			wp_send_json_error( array( 'messages' => array( $account->get_error_message() ) ) );
		}

		wp_send_json_success( array( 'account' => $account ) );
	}

	/**
	 * Get cached ID token or fetch a new one from Perfection42.
	 *
	 * @param string $username Username (email).
	 * @param string $password Password.
	 * @return string|\WP_Error
	 */
	private function get_cached_id_token( string $username, string $password ) {
		$cache_key = $this->get_token_transient_key( $username );
		$cached    = get_transient( $cache_key );
		if ( ! empty( $cached ) && is_string( $cached ) ) {
			return $cached;
		}
		$token = $this->fetch_id_token( $username, $password );
		if ( is_wp_error( $token ) ) {
			return $token;
		}
		// Cache for 20 minutes.
		set_transient( $cache_key, $token, 20 * MINUTE_IN_SECONDS );
		return $token;
	}

	/**
	 * Build transient key for token cache.
	 *
	 * @param string $username Username (email) used to generate a user-specific cache key.
	 * @return string Cache key for storing the ID token in a transient.
	 */
	private function get_token_transient_key( string $username ): string {
		return 'edidev_p42_id_token_' . md5( strtolower( trim( $username ) ) );
	}

	/**
	 * Fetch ID token from Perfection42 by signing in.
	 *
	 * @param string $username Username (email).
	 * @param string $password Password.
	 * @return string|\WP_Error
	 */
	private function fetch_id_token( string $username, string $password ) {
		$body     = wp_json_encode(
			array(
				'username' => $username,
				'password' => $password,
			)
		);
		$response = wp_remote_post(
			'https://api.perfection42.com/v1/sign-in',
			array(
				'headers' => array( 'Content-Type' => 'application/json' ),
				'body'    => $body,
				'timeout' => 20,
			)
		);
		if ( is_wp_error( $response ) ) {
			return new \WP_Error( 'p42_signin_failed', __( 'Unable to reach Perfection42 sign-in service.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		$code = wp_remote_retrieve_response_code( $response );
		$data = json_decode( (string) wp_remote_retrieve_body( $response ), true );
		if ( $code >= 400 ) {
			$message = $data['message'] ?? __( 'Sign-in failed. Please verify credentials.', 'edidev-ai-assistant-for-perfection42' );
			$err     = ( 401 === $code || 403 === $code ) ? 'p42_unauthorized' : 'p42_signin_failed_http';
			return new \WP_Error( $err, $message );
		}
		$token = $data['id_token'] ?? '';
		if ( empty( $token ) ) {
			return new \WP_Error( 'p42_missing_token', __( 'Sign-in succeeded but no token was returned.', 'edidev-ai-assistant-for-perfection42' ) );
		}
		return $token;
	}


	/**
	 * Get general tab settings.
	 *
	 * @return array
	 */
	public function get_generate_tab(): array {
		return $this->setting_repository->get_general_tab();
	}

	/**
	 * Add error message to admin notices.
	 *
	 * @param string $message Message to add.
	 */
	public static function notice_error_message( $message ): void {
		$current_messages = get_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY, array() );

		if ( ! in_array( $message, $current_messages, true ) ) {
			$current_messages[] = $message;
			update_option( self::ADMIN_NOTICE_ERROR_MESSAGE_KEY, $current_messages );
		}
	}

	/**
	 * Add success message to admin notices.
	 *
	 * @param string $message Message to add.
	 */
	public static function notice_success_message( $message ): void {
		$current_messages = get_option( self::ADMIN_NOTICE_SUCCESS_MESSAGE_KEY, array() );

		if ( ! in_array( $message, $current_messages, true ) ) {
			$current_messages[] = $message;
			update_option( self::ADMIN_NOTICE_SUCCESS_MESSAGE_KEY, $current_messages );
		}
	}

	/**
	 * Get Perfection credentials.
	 *
	 * @return array
	 */
	public function get_perfection_credentials(): array {
		return $this->setting_repository->get_perfection_credentials();
	}

	// Removed deprecated workflows accessor.

	/**
	 * Get instance of SettingProvider.
	 *
	 * @param array $args Optional args.
	 *
	 * @return SettingProvider
	 */
	public static function instance( $args ): SettingProvider {
		return new self( $args );
	}

	/**
	 * Save sign up tab settings.
	 *
	 * @param array $posted_data Posted data.
	 */
	public function save_signup_tab( $posted_data ): void {
		$result = $this->setting_repository->save_signup_tab( $posted_data );

		if ( is_wp_error( $result ) ) {
			// Keep form open and display the concrete error message from API.
			set_transient( 'edidev_p42_show_signup_form', 1, MINUTE_IN_SECONDS );
			self::notice_error_message( $result->get_error_message() );
			return;
		}

		if ( $result ) {
			// Ensure any previous flag to show the Sign-up form is cleared on success.
			delete_transient( 'edidev_p42_show_signup_form' );
			self::notice_success_message(
				esc_html__( 'New Perfection42 account has been created!', 'edidev-ai-assistant-for-perfection42' )
			);
			return;
		}

		// Fallback for legacy boolean false returns.
		set_transient( 'edidev_p42_show_signup_form', 1, MINUTE_IN_SECONDS );
		self::notice_error_message(
			esc_html__( 'New Perfection42 account has not been created. Something went wrong. Please try again!', 'edidev-ai-assistant-for-perfection42' )
		);
	}
}
