<?php
/**
 * REST API Repository Class
 *
 * @package Edidev\AiAssistantForPerfection42\Includes\RestApi
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Includes\RestApi;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class RestAPIRepository
 *
 * Handles REST API endpoints for image uploads and product management.
 */
class RestAPIRepository {
	/**
	 * Get translations for the React webapp.
	 *
	 * Route: GET /wp-json/edidev-ai-assistant-for-perfection42/v1/translations/{locale}
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error The REST response or error object.
	 */
	public function register_api_get_translations( WP_REST_Request $request ) {
		// Permissions: allow any logged-in user (or dev mode bypass), verify nonce if provided by client.
		if ( ! DevModeHelper::is_enabled() ) {
			if ( ! is_user_logged_in() && ! current_user_can( 'read' ) ) {
				return new WP_Error( 'rest_forbidden', __( 'You must be logged in to access translations.', 'edidev-ai-assistant-for-perfection42' ), array( 'status' => 403 ) );
			}
		}

		$locale = strtolower( (string) $request->get_param( 'locale' ) );
		// For initial phase, support only short code 'en'.
		if ( 'en' !== $locale ) {
			return new WP_Error(
				'unsupported_locale',
				__( 'Unsupported locale. Please use "en".', 'edidev-ai-assistant-for-perfection42' ),
				array(
					'status'    => 404,
					'supported' => array( 'en' ),
				)
			);
		}

		// Load translation map (flat key/value) from the existing package file.
		$translations_path = dirname( __DIR__, 2 ) . '/Packages/react-translations.php';
		$resources         = array();
		if ( file_exists( $translations_path ) ) {
			$loaded = include $translations_path; // returns an array of keys => translated strings via esc_html__().
			if ( is_array( $loaded ) ) {
				$resources = $loaded;
			}
		}

		// Build response payload.
		$payload  = array(
			'locale'    => 'en',
			'namespace' => 'translation',
			'resources' => $resources,
		);
		$response = new WP_REST_Response( $payload );

		// Dev mode: disable caching to ensure immediate reflection of edits.
		if ( DevModeHelper::is_enabled() ) {
			$response->set_headers(
				array(
					'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
					'Pragma'        => 'no-cache',
					'Expires'       => '0',
				)
			);
		} else {
			// Optionally allow short-lived caching in non-dev environments.
			$response->set_headers(
				array(
					'Cache-Control' => 'private, max-age=300',
				)
			);
		}

		return $response;
	}

	/**
	 * Validate request permissions.
	 *
	 * @param WP_REST_Request $unused_request Request object.
	 * @return bool|WP_Error True if has permission, WP_Error otherwise.
	 */
	public function validate_request_permission( $unused_request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		if ( DevModeHelper::is_enabled() ) {
			return true;
		}

		$nonce = isset( $_SERVER['HTTP_X_WP_NONCE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_WP_NONCE'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'edidev-ai-assistant-for-perfection42' ), array( 'status' => 403 ) );
		}

		if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'edit_products' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce defines this capability.
			return new WP_Error( 'rest_forbidden', __( 'You do not have sufficient permissions.', 'edidev-ai-assistant-for-perfection42' ), array( 'status' => 403 ) );
		}

		return true;
	}

	/**
	 * Handle image upload via REST API.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error The REST response or error object.
	 */
	public function register_api_upload_image( WP_REST_Request $request ) {
		$permission_check = $this->validate_request_permission( $request );
		if ( is_wp_error( $permission_check ) ) {
			return $permission_check;
		}

		// Accept file from either $request->get_file_params() or $_FILES and support common field names.
		$files = $request->get_file_params();

		// Fallback to global if needed (e.g., some servers bypass WP parsing).
		if ( empty( $files ) && ! empty( $_FILES ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- REST request nonce already validated in validate_request_permission()
			$files = $_FILES; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- REST request nonce already validated in validate_request_permission()
		}

		if ( empty( $files ) ) {
			return new WP_Error( 'no_file', 'No file uploaded', array( 'status' => 400 ) );
		}

		// Try common keys: 'file', 'image', or first file in the array.
		$upload = null;
		if ( isset( $files['file'] ) ) {
			$upload = $files['file'];
		} elseif ( isset( $files['image'] ) ) {
			$upload = $files['image'];
		} else {
			$first = reset( $files );
			if ( is_array( $first ) ) {
				$upload = $first;
			}
		}

		if ( empty( $upload ) || ! isset( $upload['name'], $upload['type'], $upload['tmp_name'], $upload['error'], $upload['size'] ) ) {
				return new WP_Error( 'invalid_file', 'Invalid upload payload', array( 'status' => 400 ) );
		}

		// Create a copy of the file array to avoid pass by reference issues.
		$type_value = isset( $upload['type'] ) ? ( function_exists( 'sanitize_mime_type' ) ? sanitize_mime_type( (string) $upload['type'] ) : sanitize_text_field( (string) $upload['type'] ) ) : '';
		$file_data  = array(
			'name'     => sanitize_file_name( (string) $upload['name'] ),
			'type'     => $type_value,
			'tmp_name' => (string) $upload['tmp_name'], // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Provided by PHP upload mechanism; validated by WP APIs.
			'error'    => (int) $upload['error'],
			'size'     => (int) $upload['size'],
		);
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		// Validate file size (10MB limit).
		$max_size = 10 * 1024 * 1024; // 10MB.
		if ( $file_data['size'] > $max_size ) {
			return new WP_Error( 'file_too_large', 'File size exceeds 10MB', array( 'status' => 400 ) );
		}

		// Validate MIME type.
		$allowed_types = array(
			'image/jpeg',
			'image/png',
			'image/gif',
			'image/webp',
			'image/svg+xml',
		);

		$filetype = wp_check_filetype_and_ext( $file_data['tmp_name'], $file_data['name'] );
		if ( empty( $filetype['type'] ) || ! in_array( $filetype['type'], $allowed_types, true ) ) {
			return new WP_Error( 'invalid_file_type', 'Only image files are allowed', array( 'status' => 400 ) );
		}

		// Handle file upload without manually requiring admin includes.
		$contents = file_get_contents( $file_data['tmp_name'] ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading a local uploaded file, not a remote URL.
		if ( false === $contents ) {
			return new WP_Error( 'upload_error', 'Unable to read uploaded file.', array( 'status' => 500 ) );
		}

		$bits = wp_upload_bits( $file_data['name'], null, $contents );
		if ( ! empty( $bits['error'] ) || empty( $bits['file'] ) ) {
			$return_message = ! empty( $bits['error'] ) ? $bits['error'] : 'Upload failed.';
			return new WP_Error( 'upload_error', $return_message, array( 'status' => 500 ) );
		}

		// Create attachment in media library.
		$attachment = array(
			'post_mime_type' => $filetype['type'],
			'post_title'     => pathinfo( $bits['file'], PATHINFO_FILENAME ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);

		$attachment_id = wp_insert_attachment( $attachment, $bits['file'] );
		if ( is_wp_error( $attachment_id ) ) {
			return new WP_Error( 'attachment_error', $attachment_id->get_error_message(), array( 'status' => 500 ) );
		}

		// Generate attachment metadata if available without forcing includes.
		if ( function_exists( 'wp_generate_attachment_metadata' ) && function_exists( 'wp_update_attachment_metadata' ) ) {
			$metadata = wp_generate_attachment_metadata( $attachment_id, $bits['file'] );
			wp_update_attachment_metadata( $attachment_id, $metadata );
		}

		return new WP_REST_Response(
			array(
				'id'       => $attachment_id,
				'url'      => wp_get_attachment_url( $attachment_id ),
				'filename' => $file_data['name'],
			),
			200
		);
	}

	/**
	 * Get product details by IDs.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The REST response or error object.
	 */
	public function register_api_get_product_by_ids( WP_REST_Request $request ) {
		$ids = $request->get_param( 'ids' );

		if ( empty( $ids ) || ! is_array( $ids ) ) {
			return new WP_Error( 'invalid_param', 'ids must be an array of product IDs', array( 'status' => 400 ) );
		}

		$products = array();
		foreach ( $ids as $id ) {
			$product = wc_get_product( $id );

			if ( ! $product ) {
				continue;
			}

			// Only expose public products to anonymous clients.
			$post_status = get_post_status( $id );
			if ( 'publish' !== $post_status || ( method_exists( $product, 'is_visible' ) && ! $product->is_visible() ) ) {
				continue;
			}

			// Categories.
			$categories_terms = wp_get_post_terms( $id, 'product_cat' );
			$categories       = array();
			foreach ( $categories_terms as $term ) {
				$categories[] = array(
					'id'   => $term->term_id,
					'name' => $term->name,
				);
			}

			// Tags.
			$tags_terms = wp_get_post_terms( $id, 'product_tag' );
			$tags       = array();
			foreach ( $tags_terms as $term ) {
				$tags[] = array(
					'id'   => $term->term_id,
					'name' => $term->name,
				);
			}

			// Featured image.
			$featured_image = get_the_post_thumbnail_url( $id, 'full' );

			// Gallery images.
			$gallery_ids    = $product->get_gallery_image_ids();
			$gallery_images = array();
			foreach ( $gallery_ids as $gid ) {
				$gallery_images[] = wp_get_attachment_url( $gid );
			}

			$callback = function ( $html ) {
				$html = preg_replace( '/width="\d+"/', 'width="100%"', $html );
				$html = preg_replace( '/height="\d+"/', 'height="100%"', $html );
				$html = str_replace( '<video', '<video style="width:100%;height:100%"', $html );
				return $html;
			};

			add_filter( 'wp_video_shortcode', $callback, 10, 1 );

			$description = $this->convert_playlist_to_video(
				// Intentionally using core hook 'the_content' to process WooCommerce content.
				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
				apply_filters( 'the_content', $product->get_description() )
			);

			$short_description = $this->convert_playlist_to_video(
				// Intentionally using core hook 'the_content' to process WooCommerce content.
				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
				apply_filters( 'the_content', $product->get_short_description() )
			);

			remove_filter( 'wp_video_shortcode', $callback, 10, 1 );

			$products[] = array(
				'id'                => $id,
				'name'              => $product->get_name(),
				'description'       => $description,
				'short_description' => $short_description,
				'categories'        => $categories,
				'tags'              => $tags,
				'featured_image'    => $featured_image,
				'gallery_images'    => $gallery_images,
				'permalink'         => get_permalink( $id ),
			);
		}

		return new WP_REST_Response( $products, 200 );
	}

	/**
	 * Convert playlist shortcodes to video shortcodes in the given content.
	 *
	 * @param WP_REST_Request $request The content to process.
	 * @return mixed The processed content with playlist shortcodes converted.
	 */
	public function register_api_update_product_info( WP_REST_Request $request ) {
		if ( ! DevModeHelper::is_enabled() ) {
			$nonce = isset( $_SERVER['HTTP_X_WP_NONCE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_WP_NONCE'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
				return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'edidev-ai-assistant-for-perfection42' ), array( 'status' => 403 ) );
			}
		}

		$params = $request->get_json_params();

		if ( empty( $params['id'] ) ) {
			return new WP_Error( 'missing_param', 'Product ID is required', array( 'status' => 400 ) );
		}

		$id      = intval( $params['id'] );
		$product = wc_get_product( $id );

		if ( ! $product ) {
			return new WP_Error( 'invalid_product', 'Invalid product ID', array( 'status' => 400 ) );
		}

		$handle = isset( $params['handle'] ) ? $params['handle'] : 'change';
		$type   = isset( $params['type'] ) ? $params['type'] : 'text';

		$process_content = function ( $content, $type, $product_id ) {
			$new_content = '';

			if ( 'text' === $type ) {
				$new_content = wp_kses_post( $content );
			} elseif ( 'images' === $type ) {
				$content       = esc_url_raw( $content );
				$attachment_id = $this->import_external_image( $content, $product_id );
				if ( $attachment_id ) {
					$image_url   = wp_get_attachment_url( $attachment_id );
					$image_data  = function_exists( 'wp_get_attachment_metadata' ) ? wp_get_attachment_metadata( $attachment_id ) : array();
					$width       = $image_data['width'] ?? 300;
					$height      = $image_data['height'] ?? 300;
					$new_content = sprintf(
						'<img class="alignnone size-medium wp-image-%1$d" src="%2$s" alt="" width="%3$d" height="%4$d" />',
						$attachment_id,
						esc_url( $image_url ),
						esc_attr( $width ),
						esc_attr( $height )
					);
				}
			} elseif ( 'videos' === $type ) {
				$content  = esc_url_raw( $content );
				$response = wp_remote_get( $content, array( 'timeout' => 30 ) );
				if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
					$body      = wp_remote_retrieve_body( $response );
					$file_name = basename( wp_parse_url( $content, PHP_URL_PATH ) );
					if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
						$query = wp_parse_url( $content, PHP_URL_QUERY );
						if ( $query ) {
							$query_params = array();
							parse_str( $query, $query_params );
							if ( ! empty( $query_params['response-content-disposition'] ) ) {
								if ( preg_match( '/filename="?(.*?)"?$/', $query_params['response-content-disposition'], $matches ) ) {
									$file_name = $matches[1];
								}
							}
						}
					}
					if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
						$file_name .= '.mp4';
					}
					$bits = wp_upload_bits( $file_name, null, $body );
					if ( empty( $bits['error'] ) && ! empty( $bits['file'] ) ) {
						$ft            = wp_check_filetype( $bits['file'] );
						$mime          = ! empty( $ft['type'] ) ? $ft['type'] : 'video/mp4';
						$attachment    = array(
							'post_mime_type' => $mime,
							'post_title'     => pathinfo( $bits['file'], PATHINFO_FILENAME ),
							'post_content'   => '',
							'post_status'    => 'inherit',
						);
						$attachment_id = wp_insert_attachment( $attachment, $bits['file'], $product_id );
						if ( ! is_wp_error( $attachment_id ) ) {
							$new_content = sprintf( '[video width="768" height="432" mp4="%s"][/video]', esc_url( wp_get_attachment_url( $attachment_id ) ) );
						}
					}
				}
			}

			return $new_content;
		};

		// Name.
		if ( ! empty( $params['product_title'] ) ) {
			if ( 'add' === $handle ) {
				$old = $product->get_name();
				$product->set_name( trim( $old . ' ' . $params['product_title'] ) );
			} else {
				$product->set_name( $params['product_title'] );
			}
		}

		// Description.
		if ( ! empty( $params['description'] ) ) {
			$new_content = $process_content( $params['description'], $type, $id );
			if ( ! empty( $new_content ) ) {
				if ( 'add' === $handle ) {
					$old = $product->get_description();
					$product->set_description( $old . "\r\n" . $new_content );
				} else {
					$product->set_description( $new_content );
				}
			}
		}

		// Short description.
		if ( ! empty( $params['short_description'] ) ) {
			$new_content = $process_content( $params['short_description'], $type, $id );
			if ( ! empty( $new_content ) ) {
				if ( 'add' === $handle ) {
					$old = $product->get_short_description();
					$product->set_short_description( $old . "\r\n" . $new_content );
				} else {
					$product->set_short_description( $new_content );
				}
			}
		}

		// Category (single).
		if ( ! empty( $params['category'] ) && is_string( $params['category'] ) ) {
			$cat_ids = array();

			$term = term_exists( $params['category'], 'product_cat' );
			if ( ! $term ) {
				$term = wp_insert_term( $params['category'], 'product_cat' );
			}

			if ( ! is_wp_error( $term ) && ! empty( $term['term_id'] ) ) {
				$cat_ids[] = intval( $term['term_id'] );
			}

			if ( ! empty( $cat_ids ) ) {
				if ( 'add' === $handle ) {
					$old     = wp_get_post_terms( $id, 'product_cat', array( 'fields' => 'ids' ) );
					$cat_ids = array_unique( array_merge( $old, $cat_ids ) );
				}
				wp_set_post_terms( $id, $cat_ids, 'product_cat' );
			}
		}

		// Tags.
		if ( ! empty( $params['tags'] ) && is_array( $params['tags'] ) ) {
			$tag_ids = array();
			foreach ( $params['tags'] as $tag ) {
				if ( is_array( $tag ) && ! empty( $tag['id'] ) ) {
					$tag_ids[] = intval( $tag['id'] ); // existing.
				} elseif ( is_string( $tag ) ) {
					$term = term_exists( $tag, 'product_tag' );
					if ( ! $term ) {
						$term = wp_insert_term( $tag, 'product_tag' );
					}
					if ( ! is_wp_error( $term ) && ! empty( $term['term_id'] ) ) {
						$tag_ids[] = intval( $term['term_id'] );
					}
				}
			}

			if ( 'add' === $handle ) {
				$old     = wp_get_post_terms( $id, 'product_tag', array( 'fields' => 'ids' ) );
				$tag_ids = array_unique( array_merge( $old, $tag_ids ) );
			}
			wp_set_post_terms( $id, $tag_ids, 'product_tag' );
		}

		// Featured image.
		if ( ! empty( $params['featured_image'] ) ) {
			$attachment_id = attachment_url_to_postid( $params['featured_image'] );
			if ( ! $attachment_id ) {
				$attachment_id = $this->import_external_image( $params['featured_image'], $id );
			}

			if ( $attachment_id ) {
				set_post_thumbnail( $id, $attachment_id );
				$product->set_image_id( $attachment_id );
			}
		}

		// Gallery images.
		if ( ! empty( $params['gallery'] ) && is_array( $params['gallery'] ) ) {
			$gallery_ids = array();
			foreach ( $params['gallery'] as $img ) {
				$attachment_id = attachment_url_to_postid( $img );
				if ( ! $attachment_id ) {
					$attachment_id = $this->import_external_image( $img, $id );
				}
				if ( $attachment_id ) {
					$gallery_ids[] = $attachment_id;
				}
			}

			if ( 'add' === $handle ) {
				$old         = $product->get_gallery_image_ids();
				$gallery_ids = array_unique( array_merge( $old, $gallery_ids ) );
			}
			$product->set_gallery_image_ids( $gallery_ids );
		}

		$product->save();

		wc_delete_product_transients( $id );
		$product = wc_get_product( $id );

		$categories_terms = wp_get_post_terms( $id, 'product_cat' );
		$categories       = array();
		foreach ( $categories_terms as $term ) {
			$categories[] = array(
				'id'   => $term->term_id,
				'name' => $term->name,
			);
		}

		$tags_terms = wp_get_post_terms( $id, 'product_tag' );
		$tags       = array();
		foreach ( $tags_terms as $term ) {
			$tags[] = array(
				'id'   => $term->term_id,
				'name' => $term->name,
			);
		}

		$featured_image = get_the_post_thumbnail_url( $id, 'full' );

		$gallery_ids    = $product->get_gallery_image_ids();
		$gallery_images = array();
		foreach ( $gallery_ids as $gid ) {
			$gallery_images[] = wp_get_attachment_url( $gid );
		}

		$callback = function ( $html ) {
			$html = preg_replace( '/width="\d+"/', 'width="100%"', $html );
			$html = preg_replace( '/height="\d+"/', 'height="100%"', $html );
			$html = str_replace( '<video', '<video style="width:100%;height:100%"', $html );
			return $html;
		};

		add_filter( 'wp_video_shortcode', $callback, 10, 1 );

		$description = $this->convert_playlist_to_video(
			// Intentionally using core hook 'the_content' to process WooCommerce content.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			apply_filters( 'the_content', $product->get_description() )
		);
		$short_description = $this->convert_playlist_to_video(
			// Intentionally using core hook 'the_content' to process WooCommerce content.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			apply_filters( 'the_content', $product->get_short_description() )
		);

		remove_filter( 'wp_video_shortcode', $callback, 10 );

		$response_data = array(
			'id'                => $id,
			'name'              => $product->get_name(),
			'description'       => $description,
			'short_description' => $short_description,
			'categories'        => $categories,
			'tags'              => $tags,
			'featured_image'    => $featured_image,
			'gallery_images'    => $gallery_images,
			'permalink'         => $product->get_permalink(),
			'message'           => 'Product updated successfully',
		);

		return new WP_REST_Response( $response_data, 200 );
	}

	/**
	 * Convert playlist shortcodes to video shortcodes in the given content.
	 *
	 * @param string $url The content to process.
	 * @param mixed  $product_id The product ID to associate the imported image with.
	 * @return string The processed content with playlist shortcodes converted.
	 */
	private function import_external_image( $url, $product_id ) {
		$url      = esc_url_raw( $url );
		$response = wp_remote_get( $url, array( 'timeout' => 30 ) );
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			return 0;
		}
		$body      = wp_remote_retrieve_body( $response );
		$file_name = basename( wp_parse_url( $url, PHP_URL_PATH ) );
		if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
			$query = wp_parse_url( $url, PHP_URL_QUERY );
			if ( $query ) {
				parse_str( $query, $params );
				if ( ! empty( $params['response-content-disposition'] ) ) {
					if ( preg_match( '/filename="?(.*?)"?$/', $params['response-content-disposition'], $matches ) ) {
						$file_name = $matches[1];
					}
				}
			}
		}
		if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
			$file_name .= '.jpg';
		}
		$bits = wp_upload_bits( $file_name, null, $body );
		if ( ! empty( $bits['error'] ) || empty( $bits['file'] ) ) {
			return 0;
		}
		$ft            = wp_check_filetype( $bits['file'] );
		$mime          = ! empty( $ft['type'] ) ? $ft['type'] : 'image/jpeg';
		$attachment    = array(
			'post_mime_type' => $mime,
			'post_title'     => pathinfo( $bits['file'], PATHINFO_FILENAME ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);
		$attachment_id = wp_insert_attachment( $attachment, $bits['file'], $product_id );
		if ( is_wp_error( $attachment_id ) ) {
			return 0;
		}
		if ( function_exists( 'wp_generate_attachment_metadata' ) && function_exists( 'wp_update_attachment_metadata' ) ) {
			$metadata = wp_generate_attachment_metadata( $attachment_id, $bits['file'] );
			wp_update_attachment_metadata( $attachment_id, $metadata );
		}
		return $attachment_id;
	}

	/**
	 * Convert playlist shortcodes to video shortcodes in the given content.
	 *
	 * @param WP_REST_Request $request The content to process.
	 * @return WP_REST_Response  The processed content with playlist shortcodes converted.
	 */
	public function register_api_update_multi_product_info( WP_REST_Request $request ): WP_REST_Response {
		if ( ! DevModeHelper::is_enabled() ) {
			$nonce = isset( $_SERVER['HTTP_X_WP_NONCE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_WP_NONCE'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
				return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'edidev-ai-assistant-for-perfection42' ), array( 'status' => 403 ) );
			}
		}
		$params = $request->get_json_params();

		if ( empty( $params ) || ! is_array( $params ) ) {
			return new WP_REST_Response(
				array(
					'error' => 'Products array is required',
					'code'  => 'missing_param',
				),
				400
			);
		}

		$results = array();

		foreach ( $params as $product_data ) {
			if ( empty( $product_data['id'] ) ) {
				$results[] = array(
					'id'      => null,
					'success' => false,
					'message' => 'Missing product ID',
				);
				continue;
			}

			$res = $this->update_single_product( $product_data );

			if ( is_wp_error( $res ) ) {
				$results[] = array(
					'id'      => $product_data['id'],
					'success' => false,
					'message' => $res->get_error_message(),
				);
			} else {
				$results[] = array_merge(
					$res->get_data(),
					array( 'success' => true )
				);
			}
		}

		return new WP_REST_Response( array( 'results' => $results ), 200 );
	}

	/**
	 * Update a single product based on provided parameters.
	 *
	 * @param array $params The parameters for updating the product.
	 * @return WP_Error|WP_REST_Response The response object or an error.
	 */
	private function update_single_product( array $params ) {
		$type = $params['type'] ?? 'text';

		if ( empty( $params['id'] ) ) {
			return new WP_Error( 'missing_param', 'Product ID is required', array( 'status' => 400 ) );
		}

		$id      = intval( $params['id'] );
		$product = wc_get_product( $id );

		if ( ! $product ) {
			return new WP_Error( 'invalid_product', 'Invalid product ID', array( 'status' => 400 ) );
		}

		$handle = isset( $params['handle'] ) ? $params['handle'] : 'change';

		$process_content = function ( $content, $type, $product_id ) {
			$new_content = '';

			if ( 'text' === $type ) {
				$new_content = $content;
			} elseif ( 'images' === $type ) {
				$attachment_id = $this->import_external_image( $content, $product_id );
				if ( $attachment_id ) {
					$image_url   = wp_get_attachment_url( $attachment_id );
					$image_data  = function_exists( 'wp_get_attachment_metadata' ) ? wp_get_attachment_metadata( $attachment_id ) : array();
					$width       = $image_data['width'] ?? 300;
					$height      = $image_data['height'] ?? 300;
					$new_content = sprintf(
						'<img class="alignnone size-medium wp-image-%1$d" src="%2$s" alt="" width="%3$d" height="%4$d" />',
						$attachment_id,
						esc_url( $image_url ),
						esc_attr( $width ),
						esc_attr( $height )
					);
				}
			} elseif ( 'videos' === $type ) {
				$content  = esc_url_raw( $content );
				$response = wp_remote_get( $content, array( 'timeout' => 30 ) );
				if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
					$body      = wp_remote_retrieve_body( $response );
					$file_name = basename( wp_parse_url( $content, PHP_URL_PATH ) );
					if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
						$query = wp_parse_url( $content, PHP_URL_QUERY );
						if ( $query ) {
							$query_params = array();
							parse_str( $query, $query_params );
							if ( ! empty( $query_params['response-content-disposition'] ) ) {
								if ( preg_match( '/filename="?(.*?)"?$/', $query_params['response-content-disposition'], $matches ) ) {
									$file_name = $matches[1];
								}
							}
						}
					}
					if ( ! pathinfo( $file_name, PATHINFO_EXTENSION ) ) {
						$file_name .= '.mp4';
					}
					$bits = wp_upload_bits( $file_name, null, $body );
					if ( empty( $bits['error'] ) && ! empty( $bits['file'] ) ) {
						$ft            = wp_check_filetype( $bits['file'] );
						$mime          = isset( $ft['type'] ) && $ft['type'] ? $ft['type'] : 'video/mp4';
						$attachment    = array(
							'post_mime_type' => $mime,
							'post_title'     => pathinfo( $bits['file'], PATHINFO_FILENAME ),
							'post_content'   => '',
							'post_status'    => 'inherit',
						);
						$attachment_id = wp_insert_attachment( $attachment, $bits['file'], $product_id );
						if ( ! is_wp_error( $attachment_id ) ) {
							$new_content = sprintf( '[video width="768" height="432" mp4="%s"][/video]', esc_url( wp_get_attachment_url( $attachment_id ) ) );
						}
					}
				}
			}

			return $new_content;
		};

		// Name.
		if ( ! empty( $params['product_title'] ) ) {
			if ( 'add' === $handle ) {
				$old = $product->get_name();
				$product->set_name( trim( $old . ' ' . $params['product_title'] ) );
			} else {
				$product->set_name( $params['product_title'] );
			}
		}

		// Description.
		if ( ! empty( $params['description'] ) ) {
			$new_content = $process_content( $params['description'], $type, $id );
			if ( ! empty( $new_content ) ) {
				if ( 'add' === $handle ) {
					$old = $product->get_description();
					$product->set_description( $old . "\r\n" . $new_content );
				} else {
					$product->set_description( $new_content );
				}
			}
		}

		// Short description.
		if ( ! empty( $params['short_description'] ) ) {
			$new_content = $process_content( $params['short_description'], $type, $id );
			if ( ! empty( $new_content ) ) {
				if ( 'add' === $handle ) {
					$old = $product->get_short_description();
					$product->set_short_description( $old . "\r\n" . $new_content );
				} else {
					$product->set_short_description( $new_content );
				}
			}
		}

		// Categories.
		if ( ! empty( $params['categories'] ) && is_array( $params['categories'] ) ) {
			$cat_ids = array();
			foreach ( $params['categories'] as $cat ) {
				if ( is_array( $cat ) && ! empty( $cat['id'] ) ) {
					$cat_ids[] = intval( $cat['id'] ); // existing.
				} elseif ( is_string( $cat ) ) {
					$term = term_exists( $cat, 'product_cat' );
					if ( ! $term ) {
						$term = wp_insert_term( $cat, 'product_cat' );
					}
					if ( ! is_wp_error( $term ) && ! empty( $term['term_id'] ) ) {
						$cat_ids[] = intval( $term['term_id'] );
					}
				}
			}

			if ( 'add' === $handle ) {
				$old     = wp_get_post_terms( $id, 'product_cat', array( 'fields' => 'ids' ) );
				$cat_ids = array_unique( array_merge( $old, $cat_ids ) );
			}
			wp_set_post_terms( $id, $cat_ids, 'product_cat' );
		}

		if ( ! empty( $params['tags'] ) && is_array( $params['tags'] ) ) {
			$tag_ids = array();
			foreach ( $params['tags'] as $tag ) {
				if ( is_array( $tag ) && ! empty( $tag['id'] ) ) {
					$tag_ids[] = intval( $tag['id'] ); // existing.
				} elseif ( is_string( $tag ) ) {
					$term = term_exists( $tag, 'product_tag' );
					if ( ! $term ) {
						$term = wp_insert_term( $tag, 'product_tag' );
					}
					if ( ! is_wp_error( $term ) && ! empty( $term['term_id'] ) ) {
						$tag_ids[] = intval( $term['term_id'] );
					}
				}
			}

			if ( 'add' === $handle ) {
				$old     = wp_get_post_terms( $id, 'product_tag', array( 'fields' => 'ids' ) );
				$tag_ids = array_unique( array_merge( $old, $tag_ids ) );
			}
			wp_set_post_terms( $id, $tag_ids, 'product_tag' );
		}

		// Featured image.
		if ( ! empty( $params['featured_image'] ) ) {
			$attachment_id = attachment_url_to_postid( $params['featured_image'] );
			if ( ! $attachment_id ) {
				$attachment_id = $this->import_external_image( $params['featured_image'], $id );
			}

			if ( $attachment_id ) {
				set_post_thumbnail( $id, $attachment_id );
				$product->set_image_id( $attachment_id );
			}
		}

		// Gallery images.
		if ( ! empty( $params['gallery'] ) && is_array( $params['gallery'] ) ) {
			$gallery_ids = array();
			foreach ( $params['gallery'] as $img ) {
				$attachment_id = attachment_url_to_postid( $img );
				if ( ! $attachment_id ) {
					$attachment_id = $this->import_external_image( $img, $id );
				}
				if ( $attachment_id ) {
					$gallery_ids[] = $attachment_id;
				}
			}

			if ( 'add' === $handle ) {
				$old         = $product->get_gallery_image_ids();
				$gallery_ids = array_unique( array_merge( $old, $gallery_ids ) );
			}
			$product->set_gallery_image_ids( $gallery_ids );
		}

		$product->save();

		wc_delete_product_transients( $id );
		$product = wc_get_product( $id );

		$categories_terms = wp_get_post_terms( $id, 'product_cat' );
		$categories       = array();
		foreach ( $categories_terms as $term ) {
			$categories[] = array(
				'id'   => $term->term_id,
				'name' => $term->name,
			);
		}

		// Tags.
		$tags_terms = wp_get_post_terms( $id, 'product_tag' );
		$tags       = array();
		foreach ( $tags_terms as $term ) {
			$tags[] = array(
				'id'   => $term->term_id,
				'name' => $term->name,
			);
		}

		// Featured image.
		$featured_image = get_the_post_thumbnail_url( $id, 'full' );

		// Gallery images.
		$gallery_ids    = $product->get_gallery_image_ids();
		$gallery_images = array();
		foreach ( $gallery_ids as $gid ) {
			$gallery_images[] = wp_get_attachment_url( $gid );
		}

		$callback = function ( $html ) {
			$html = preg_replace( '/width="\d+"/', 'width="100%"', $html );
			$html = preg_replace( '/height="\d+"/', 'height="100%"', $html );
			$html = str_replace( '<video', '<video style="width:100%;height:100%"', $html );
			return $html;
		};

		add_filter( 'wp_video_shortcode', $callback, 10, 1 );

		$description = $this->convert_playlist_to_video(
			// Intentionally using core hook 'the_content' to process WooCommerce content.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			apply_filters( 'the_content', $product->get_description() )
		);

		$short_description = $this->convert_playlist_to_video(
			// Intentionally using core hook 'the_content' to process WooCommerce content.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			apply_filters( 'the_content', $product->get_short_description() )
		);

		remove_filter( 'wp_video_shortcode', $callback, 10, 1 );

		$response_data = array(
			'id'                => $id,
			'name'              => $product->get_name(),
			'description'       => $description,
			'short_description' => $short_description,
			'categories'        => $categories,
			'tags'              => $tags,
			'featured_image'    => $featured_image,
			'gallery_images'    => $gallery_images,
			'permalink'         => $product->get_permalink(),
			'message'           => 'Product updated successfully',
		);

		return new WP_REST_Response( $response_data, 200 );
	}

	/**
	 * Convert WordPress playlist shortcode to video HTML
	 *
	 * @param mixed $content The content to process.
	 */
	public function convert_playlist_to_video( $content ) {
		if ( preg_match_all( '/<script type="application\/json" class="wp-playlist-script">(.*?)<\/script>/s', $content, $matches ) ) {
			foreach ( $matches[1] as $json ) {
				$playlist_data = json_decode( $json, true );

				$video_html = '';
				if ( ! empty( $playlist_data['tracks'] ) ) {
					foreach ( $playlist_data['tracks'] as $track ) {
						$src   = esc_url( $track['src'] );
						$type  = esc_attr( $track['type'] );
						$title = esc_html( $track['title'] );

						$video_html .= "<video controls preload='none'>";
						$video_html .= "<source src='{$src}' type='{$type}' />";
						$video_html .= 'Your browser does not support the video tag.';
						$video_html .= '</video>';
						if ( ! empty( $title ) ) {
							$video_html .= "<p>{$title}</p>";
						}
					}
				}

				$content = preg_replace( '/<div class="wp-playlist[\s\S]*?<script type="application\/json" class="wp-playlist-script">.*?<\/script>\s*<\/div>/s', $video_html, $content, 1 );
			}
		}

		return $content;
	}
}
