<?php
/**
 * REST API Provider Class
 *
 * @package Edidev\AiAssistantForPerfection42\Includes\RestApi
 */

// phpcs:disable WordPress.Files.FileName.NotHyphenatedLowercase, WordPress.Files.FileName.InvalidClassFileName

namespace Edidev\AiAssistantForPerfection42\Includes\RestApi;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class RestAPIProvider
 *
 * Handles the registration of REST API routes.
 */
class RestAPIProvider {
	/**
	 * REST API Repository instance.
	 *
	 * @var RestAPIRepository|null
	 */
	protected ?RestAPIRepository $rest_api_repository = null;

	/**
	 * Constructor.
	 *
	 * @param array $args // Future use.
	 */
	public function __construct( $args = array() ) {
		// Mark unused parameter as intentionally unused.
		unset( $args );

		$this->rest_api_repository = new RestAPIRepository();

		add_action( 'rest_api_init', array( $this, 'handle_register_rest_api' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @return void
	 */
	public function handle_register_rest_api(): void {
		register_rest_route(
			'edidev-ai-assistant-for-perfection42/v1',
			'/translations/(?P<locale>[A-Za-z\-]+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->rest_api_repository, 'register_api_get_translations' ),
				'permission_callback' => function () {
					if ( DevModeHelper::is_enabled() ) {
						return true;
					}
					return current_user_can( 'read' ) || is_user_logged_in();
				},
				'args'                => array(
					'locale' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		register_rest_route(
			'edidev-ai-assistant-for-perfection42/v1',
			'/upload-image',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->rest_api_repository, 'register_api_upload_image' ),
				'permission_callback' => function ( $unused_request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
					if ( DevModeHelper::is_enabled() ) {
						return true;
					}
					return current_user_can( 'manage_woocommerce' ) || current_user_can( 'edit_products' ); // phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce defines these capabilities.
				},
			)
		);

		register_rest_route(
			'edidev-ai-assistant-for-perfection42/v1',
			'/products-by-ids',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->rest_api_repository, 'register_api_get_product_by_ids' ),
				'permission_callback' => function () {
					if ( DevModeHelper::is_enabled() ) {
						return true;
					}
					return current_user_can( 'read' ) || is_user_logged_in();
				},
				'args'                => array(
					'ids' => array(
						'required'          => true,
						'type'              => 'array',
						'items'             => array(
							'type' => 'integer',
						),
						'sanitize_callback' => function ( $param ) {
							return array_map( 'absint', (array) $param );
						},
					),
				),
			)
		);

		register_rest_route(
			'edidev-ai-assistant-for-perfection42/v1',
			'/update-product-simple',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this->rest_api_repository, 'register_api_update_product_info' ),
				'permission_callback' => function ( $unused_request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
					if ( DevModeHelper::is_enabled() ) {
						return true;
					}
					return current_user_can( 'manage_woocommerce' ) || current_user_can( 'edit_products' ); // phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce defines these capabilities.
				},
				'args'                => array(
					'id'                => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
					'product_title'     => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'description'       => array(
						'type'              => 'string',
						'sanitize_callback' => function ( $param, $request, $key ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
							// Mark intentionally unused parameter to satisfy code sniffer.
							unset( $key );
							// If the payload type is 'images' or 'videos', bypass wp_kses_post sanitization and return raw string.
							// Otherwise, sanitize HTML content using wp_kses_post as usual.
							$type = ( is_object( $request ) && method_exists( $request, 'get_param' ) ) ? $request->get_param( 'type' ) : null;
							if ( in_array( $type, array( 'images', 'videos' ), true ) ) {
								return is_string( $param ) ? $param : '';
							}

							return wp_kses_post( (string) $param );
						},
					),
					'short_description' => array(
						'type'              => 'string',
						'sanitize_callback' => 'wp_kses_post',
					),
					'handle'            => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'enum'              => array( 'add', 'change' ),
						'default'           => 'change',
					),
					'type'              => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'enum'              => array( 'text', 'images', 'videos' ),
						'default'           => 'text',
					),
				),
			)
		);

		register_rest_route(
			'edidev-ai-assistant-for-perfection42/v1',
			'/update-multi-product-simple',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this->rest_api_repository, 'register_api_update_multi_product_info' ),
				'permission_callback' => function ( $unused_request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
					if ( DevModeHelper::is_enabled() ) {
						return true;
					}
					return current_user_can( 'manage_woocommerce' ) || current_user_can( 'edit_products' ); // phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce defines these capabilities.
				},
				'args'                => array(
					'*' => array(
						'type'       => 'object',
						'properties' => array(
							'id'                => array(
								'required'          => true,
								'type'              => 'integer',
								'sanitize_callback' => 'absint',
							),
							'product_title'     => array(
								'type'              => 'string',
								'sanitize_callback' => 'sanitize_text_field',
							),
							'description'       => array(
								'type'              => 'string',
								'sanitize_callback' => 'wp_kses_post',
							),
							'short_description' => array(
								'type'              => 'string',
								'sanitize_callback' => 'wp_kses_post',
							),
							'handle'            => array(
								'type'              => 'string',
								'sanitize_callback' => 'sanitize_text_field',
								'enum'              => array( 'add', 'change' ),
								'default'           => 'change',
							),
							'type'              => array(
								'type'              => 'string',
								'sanitize_callback' => 'sanitize_text_field',
								'enum'              => array( 'text', 'images', 'videos' ),
								'default'           => 'text',
							),
						),
					),
				),
			)
		);
	}


	/**
	 * Get instance of the class.
	 *
	 * @param array $args // Future use.
	 * @return RestAPIProvider
	 */
	public static function instance( $args = array() ): RestAPIProvider {
		return ( new self( $args ) );
	}
}
