/**
 * Perfection42 Init.
 */
jQuery( document ).ready(
	function ($) {
		'use strict';

		// ---------- Utilities ----------.
		function renderMessages(messages, type) {
			const $wrap = $( '.p42-messages' );
			if ( ! $wrap.length) {
				return;
			}
			// Clear previous messages to avoid stacking too much.
			$wrap.empty();
			(messages || []).forEach(
				function (msg) {
					const cls  = type === 'error' ? 'p42-message--error' : 'p42-message--success';
					const role = type === 'error' ? 'alert' : 'status';
					$( '<div/>', { class: 'p42-message ' + cls, role: role, text: msg } ).appendTo( $wrap );
				}
			);
			// Scroll to messages.
			if ((messages || []).length) {
				window.scrollTo( { top: Math.max( 0, $wrap.offset().top - 40 ), behavior: 'smooth' } );
			}
		}

		function setFormLoading($form, $button, labelWhileLoading) {
			$form.attr( 'aria-busy', 'true' ).addClass( 'is-loading' );
			$form.find( 'input, button, select, textarea' ).prop( 'disabled', true );
			if ($button && $button.length) {
				$button.data( 'orig-text', $button.val() || $button.text() );
				if ($button.is( 'input' )) {
					$button.val( labelWhileLoading );
				} else {
					$button.text( labelWhileLoading );
				}
			}
		}

		function clearFormLoading($form, $button) {
			$form.removeAttr( 'aria-busy' ).removeClass( 'is-loading' );
			$form.find( 'input, button, select, textarea' ).prop( 'disabled', false );
			if ($button && $button.length) {
				const orig = $button.data( 'orig-text' );
				if (orig) {
					if ($button.is( 'input' )) {
						$button.val( orig );
					} else {
						$button.text( orig );
					}
				}
			}
		}

		function setPanelLoading($panel, $button, labelWhileLoading) {
			const $body = $panel.find( '#p42-account-body' );
			$body.addClass( 'is-loading' );
			$panel.find( 'button' ).prop( 'disabled', true );
			const $loading = $panel.find( '.p42-account-loading' );
			$loading.text( labelWhileLoading || (P42Settings.i18n && P42Settings.i18n.loadingAcct) || 'Loading account…' ).show();
			$panel.find( '#p42-account-details' ).hide();
			if ($button && $button.length) {
				$button.data( 'orig-text', $button.text() );
				$button.text( labelWhileLoading || (P42Settings.i18n && P42Settings.i18n.refreshing) || 'Refreshing…' );
			}
		}

		function clearPanelLoading($panel, $button) {
			const $body = $panel.find( '#p42-account-body' );
			$body.removeClass( 'is-loading' );
			$panel.find( 'button' ).prop( 'disabled', false );
			if ($button && $button.length) {
				const orig = $button.data( 'orig-text' );
				if (orig) {
					$button.text( orig );
				}
			}
		}

		function renderAccountDetails(account) {
			const $details = $( '#p42-account-details' );
			if ( ! $details.length) {
				return;
			}
			$details.empty();
			function row(label, value) {
				$( '<div/>', { class: 'p42-k', text: label } ).appendTo( $details );
				$( '<div/>', { class: 'p42-v', text: value == null ? '' : String( value ) } ).appendTo( $details );
			}
			const i18n = (P42Settings && P42Settings.i18n) || {};
			// Show computed Status as ACTIVE on successful fetch.
			row( i18n.labelStatus || 'Status', 'ACTIVE' );
			// Render only the specified fields from the API response.
			row( i18n.labelId || 'ID', account.id || '' );
			row( i18n.labelEmail || 'Email', account.email || '' );
			row( i18n.labelName || 'Name', account.name || '' );
			row( i18n.labelPackageType || 'Package Type', account.package_type || '' );
			row( i18n.labelOrganization || 'Organization', account.organization_name || '' );
			// Explicitly exclude organization_description.
		}

		function showAccountPanel() {
			$( '#p42-sign-in-form' ).hide();
			$( '#p42-sign-up-form' ).hide();
			$( '#p42-account-panel' ).show();
		}

		function hideAccountPanel() {
			$( '#p42-account-panel' ).hide();
		}

		function fetchAccount($triggerBtn) {
			const $panel = $( '#p42-account-panel' );
			if ( ! $panel.length) {
				return;
			}
			setPanelLoading( $panel, $triggerBtn, (P42Settings.i18n && P42Settings.i18n.loadingAcct) || 'Loading account…' );
			$.post(
				P42Settings.ajaxUrl,
				{
					action: 'p42_get_account',
					nonce: P42Settings.nonce
				}
			).done(
				function (resp) {
					if (resp && resp.success && resp.data && resp.data.account) {
						const account = resp.data.account;
						renderAccountDetails( account );
						$( '#p42-account-details' ).show();
						$( '.p42-account-loading' ).hide();
					} else {
						const errs = (resp && (resp.data && resp.data.messages || resp.messages)) || [];
						renderMessages( errs.length ? errs : [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
					}
				}
			).fail(
				function () {
					renderMessages( [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
				}
			).always(
				function () {
					clearPanelLoading( $panel, $triggerBtn );
				}
			);
		}

		// If creds already present (server-rendered), show account once on load.
		(function bootstrapOnLoad() {
			const username = $( '#username' ).val();
			const password = $( '#password' ).val();
			if (username && password) {
				showAccountPanel();
				fetchAccount();
			}
		})();

		// ---------- Async Sign-In/Out ----------
		$( document ).on(
			'submit',
			'#p42-sign-in-form',
			function (e) {
				e.preventDefault();
				const $form  = $( this );
				const $btn   = $form.find( 'input[type="submit"].button-primary' );
				const action = $form.find( 'input[name="submitted-action"]' ).val();

				if (action === 'sign-out') {
					setFormLoading( $form, $btn, (P42Settings && P42Settings.i18n && P42Settings.i18n.signingOut) || 'Signing out…' );
					$.post(
						P42Settings.ajaxUrl,
						{
							action: 'p42_sign_out',
							nonce: P42Settings.nonce
						}
					).done(
						function (resp) {
							if (resp && resp.success) {
								renderMessages( resp.data && resp.data.messages ? resp.data.messages : resp.messages || [], 'success' );
								// Reset UI to signed-out state
								$( '#username' ).val( '' );
								$( '#password' ).val( '' );
								$form.find( '.connection-status' ).remove();
								$btn.val( (P42Settings.i18n && P42Settings.i18n.signIn) || 'Sign In' );
								$form.find( 'input[name="submitted-action"]' ).val( 'sign-in' );
								// Show sign-up switch if present.
								$( '.p42-form-switch' ).show();
								// Reset any UI tied to account state only.
							} else {
								renderMessages( (resp && (resp.data && resp.data.messages || resp.messages)) || [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
							}
						}
					).fail(
						function () {
							renderMessages( [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
						}
					).always(
						function () {
							clearFormLoading( $form, $btn );
						}
					);
					return;
				}

				// sign-in.
				const username = $( '#username' ).val();
				const password = $( '#password' ).val();
				if ( ! username || ! password) {
					renderMessages( [ wp.i18n.__( 'Email and password are required. Please fill in both fields.', 'edidev-ai-assistant-for-perfection42' ) ], 'error' );
					return;
				}

				setFormLoading( $form, $btn, (P42Settings && P42Settings.i18n && P42Settings.i18n.signingIn) || 'Signing in…' );
				$.post(
					P42Settings.ajaxUrl,
					{
						action: 'p42_sign_in',
						nonce: P42Settings.nonce,
						username: username,
						password: password
					}
				).done(
					function (resp) {
						if (resp && resp.success) {
							const data = resp.data && resp.data.data ? resp.data.data : resp.data;
							const msgs = resp.data && resp.data.messages ? resp.data.messages : resp.messages || [];
							renderMessages( msgs, 'success' );
							// Update UI to signed-in
							$( '#username' ).val( data && data.username ? data.username : username );
							$( '#password' ).val( data && data.password ? data.password : password );
							if ( ! $form.find( '.connection-status' ).length) {
								$( '<p class="description connection-status"/>' ).html(
									(wp.i18n.__( 'Connected as', 'edidev-ai-assistant-for-perfection42' ) + ' <strong>' + (data && data.username ? data.username : username) + '</strong>')
								).appendTo( $( '#username' ).closest( 'td' ) );
							}
							$btn.val( (P42Settings.i18n && P42Settings.i18n.signOut) || 'Sign Out' );
							$form.find( 'input[name="submitted-action"]' ).val( 'sign-out' );
							$( '.p42-form-switch' ).hide();
							// Show account panel and fetch account info.
							showAccountPanel();
							fetchAccount();
						} else {
							const errs = (resp && (resp.data && resp.data.messages || resp.messages)) || [];
							renderMessages( errs.length ? errs : [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
						}
					}
				).fail(
					function () {
						renderMessages( [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
					}
				).always(
					function () {
						clearFormLoading( $form, $btn );
					}
				);
			}
		);

		// ---------- Async Sign-Up ----------
		$( document ).on(
			'submit',
			'#p42-sign-up-form',
			function (e) {
				e.preventDefault();
				const $form = $( this );
				const $btn  = $form.find( 'input[type="submit"].button-primary' );
				const name  = $( '#signup_name' ).val();
				const email = $( '#signup_email' ).val();
				const pwd   = $( '#signup_password' ).val();
				const token = $( '#email_verified_token' ).val();
				if ( ! name || ! email || ! pwd) {
					renderMessages( [ wp.i18n.__( 'Please fill in all required fields.', 'edidev-ai-assistant-for-perfection42' ) ], 'error' );
					return;
				}
				setFormLoading( $form, $btn, (P42Settings && P42Settings.i18n && P42Settings.i18n.creatingAcc) || 'Creating account…' );
				$.post(
					P42Settings.ajaxUrl,
					{
						action: 'p42_sign_up',
						nonce: P42Settings.nonce,
						name: name,
						email: email,
						password: pwd,
						email_verified_token: token
					}
				).done(
					function (resp) {
						if (resp && resp.success) {
							const msgs = resp.data && resp.data.messages ? resp.data.messages : resp.messages || [];
							renderMessages( msgs, 'success' );
							// Switch to sign-in and prefill email.
							$( '#p42-sign-up-form' ).hide();
							$( '#p42-sign-in-form' ).show();
							$( '#username' ).val( email );
							$( '#password' ).val( '' );
							$( '#email_verified_token' ).val( '' );
						} else {
							const errs = (resp && (resp.data && resp.data.messages || resp.messages)) || [];
							renderMessages( errs.length ? errs : [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
						}
					}
				).fail(
					function () {
						renderMessages( [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
					}
				).always(
					function () {
						clearFormLoading( $form, $btn );
					}
				);
			}
		);

		// ---------- Account Panel actions ----------
		$( document ).on(
			'click',
			'#p42-check-status',
			function () {
				fetchAccount( $( this ) );
			}
		);

		$( document ).on(
			'click',
			'#p42-panel-sign-out',
			function () {
				const $btn  = $( this );
				const $form = $( '#p42-sign-in-form' );
				setPanelLoading( $( '#p42-account-panel' ), $btn, (P42Settings.i18n && P42Settings.i18n.signingOut) || 'Signing out…' );
				$.post(
					P42Settings.ajaxUrl,
					{
						action: 'p42_sign_out',
						nonce: P42Settings.nonce
					}
				).done(
					function (resp) {
						if (resp && resp.success) {
							renderMessages( resp.data && resp.data.messages ? resp.data.messages : resp.messages || [], 'success' );
							// Reset forms and show sign-in/sign-up.
							$( '#username' ).val( '' );
							$( '#password' ).val( '' );
							$form.find( '.connection-status' ).remove();
							$form.find( 'input[name="submitted-action"]' ).val( 'sign-in' );
							$form.find( 'input[type="submit"].button-primary' ).val( (P42Settings.i18n && P42Settings.i18n.signIn) || 'Sign In' );
							$( '.p42-form-switch' ).show();
							hideAccountPanel();
							$( '#p42-sign-in-form' ).show();
							// Reset any UI tied to account state only.
						} else {
							renderMessages( (resp && (resp.data && resp.data.messages || resp.messages)) || [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
						}
					}
				).fail(
					function () {
						renderMessages( [ (P42Settings.i18n && P42Settings.i18n.genericErr) || 'Something went wrong. Please try again.' ], 'error' );
					}
				).always(
					function () {
						clearPanelLoading( $( '#p42-account-panel' ), $btn );
					}
				);
			}
		);
	}
);
