<?php 

// user role
function icecomaf_affiliate_user_role() {

    $capabilities = [
        'read' => true, 
        'edit_posts' => false, 
        'delete_posts' => false, 
        'publish_posts' => false, 
        'upload_files' => false,
    ];

    add_role( 'affiliate_user', 'Affiliate User', $capabilities );
}
add_action( 'init', 'icecomaf_affiliate_user_role' );

// encryption
function icecomaf_encryption_key($user_id, $email) {
    $site_salt = wp_salt();
    return hash('sha256', $user_id . $email . $site_salt);
}


// register affiliate user
function icecomaf_affiliate_user_registration() {
    global $wpdb;

     // Verify Nonce
    if ( !isset($_POST['register_form_nonce']) || !wp_verify_nonce( sanitize_text_field(wp_unslash($_POST['register_form_nonce'])), 'register_form_nonce_action') ) {
        wp_send_json_error(['message' => esc_html__('Security check failed!', 'ecommerce-affiliate')]);
        return; 
    }

    parse_str($_POST['form_data'], $data);

    // Validate required fields
    if (empty($data['fullname'])){
        wp_send_json_error(['message' => esc_html__('Name is required.', 'ecommerce-affiliate')]);
    }
    if (empty($data['email'])){
        wp_send_json_error(['message' => esc_html__('Email is required.', 'ecommerce-affiliate')]);
    }
    if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        wp_send_json_error(['message' => esc_html__('Invalid email format.', 'ecommerce-affiliate')]);
    }
    if (isset($data['affiliateID']) && !empty($data['affiliateID'])) {
        $affiliateID = trim($data['affiliateID']); 
        if (!preg_match('/^\S+$/', $affiliateID)) { 
            wp_send_json_error(['message' => esc_html__('Affiliate ID must be a single word with no spaces.', 'ecommerce-affiliate')]);
        }
    }
    // affiliate id make unique
    $affiliate_id = sanitize_text_field($data['affiliateID']);

    $existing_affiliate = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT affiliate_id FROM {$wpdb->prefix}affiliate_user WHERE affiliate_id = %s",
            $affiliate_id
        )
    );

    if ($existing_affiliate) {
        wp_send_json_error([
            // 'message' => 'This affiliate ID '. '"'. $existing_affiliate . '"' .  ' is already in used.'
            'message' => sprintf(
                esc_html__('This affiliate ID "%s" is already in use.', 'ecommerce-affiliate'),
                esc_html($existing_affiliate)
            )
        ]);
    }

    if (empty($data['password'])){
        wp_send_json_error(['message' => 'Password is required.']);
    }
    if (strlen($data['password']) < 6) {
        wp_send_json_error(['message' => esc_html__('Password must be at least 6 characters long.', 'ecommerce-affiliate')]);
    }
    if (empty($data['cpassword'])){
        wp_send_json_error(['message' => esc_html__('Password confirm is required.', 'ecommerce-affiliate')]);
    }
    if ($data['password'] !== $data['cpassword']) {
        wp_send_json_error(['message' => esc_html__('Passwords do not match.', 'ecommerce-affiliate')]);
    }

    if (empty($data['paypal_email'])) {
        wp_send_json_error(['message' => esc_html__('PayPal Email is required.', 'ecommerce-affiliate')]);
    }

    if (!isset($data['consent']) || $data['consent'] !== 'on') {
        wp_send_json_error(['message' => esc_html__('You must accept the Terms and Privacy Policy.', 'ecommerce-affiliate')]);
    }

    // Create affiliate user
    // $username = $data['fullname']; 
    $username = sanitize_user(strtolower(str_replace(' ', '_', $data['fullname'])));
    $email = sanitize_email($data['email']);
    $password = sanitize_text_field($data['password']);

    $base_username = $username;
    $suffix = 1;

    while (username_exists($username)) {
        $username = $base_username . '_' . $suffix;
        $suffix++;
    }
    
    $users_table = esc_sql($wpdb->prefix . 'users');

    $sql = $wpdb->prepare(
        "SELECT ID FROM {$users_table} WHERE user_email = %s", 
        $email
    );
    
    $email_exists = $wpdb->get_var($sql);

    if ($email_exists) {
        wp_send_json_error(['message' => esc_html__('Email is already registered.', 'ecommerce-affiliate')]);
    }

    $user_id = wp_create_user($username, $password, $email);

    if (!is_wp_error($user_id)) {
        update_user_meta($user_id, 'fullname', $data['fullname']);
    }

    if (is_wp_error($user_id)) {
        wp_send_json_error(['message' => esc_html__('User registration failed.', 'ecommerce-affiliate')]);
    }

    $affiliate_user = new WP_User($user_id);
    $affiliate_user->set_role('affiliate_user');

    // Insert into custom table
    $table_name = esc_sql($wpdb->prefix . 'affiliate_user');

    $encryption_key = icecomaf_encryption_key($user_id, $data['email']);

    $payment_detail = wp_json_encode($data['paypal_email']);

    $insert_result = $wpdb->insert($table_name, [
        'user_id' => $user_id,
        'affiliate_id' => !empty($data['affiliateID']) ? $data['affiliateID'] : $user_id,
        // 'payment_type' => $data['payment_method'],
        'payment_detail' => $payment_detail,
        ]
    );

    if (false === $insert_result) {
        wp_send_json_error(['message' => esc_html__('Database insert failed.', 'ecommerce-affiliate')]);
    }

    // generate verification link and send to email for verification. 
    update_user_meta($user_id, 'email_verified', false);

    $verification_token = wp_hash($user_id . time());
    
    // Store token in user meta
    update_user_meta($user_id, 'email_verification_token', $verification_token);
    
    // Send verification email
    $verification_link = site_url() . '/verify-email?token=' . $verification_token;

    $email_subject = esc_html__('Email Verification', 'ecommerce-affiliate');

    $headerText = esc_html__("Email Verification", "ecommerce-affiliate");

    $bodyContent = "
        <h3>Email Verification</h3>
        <p><strong>Verification Link:</strong> {$verification_link}</p>
    ";

    $email_message = icecomaf_email_template($headerText, $bodyContent);

    $headers = array('Content-Type: text/html; charset=UTF-8');

    wp_mail($email, $email_subject, $email_message, $headers);

    // Send email to admin
    $admin_email = get_option('admin_email');
    $admin_message = 'A new affiliate user has registered. Details are as follows:' . PHP_EOL;
    $admin_message .= 'Name: ' . get_user_meta($user_id, 'fullname', true) . PHP_EOL;
    $admin_message .= 'Email: ' . $email . PHP_EOL;
    $admin_message .= 'Affiliate ID: ' . (!empty($data['affiliateID']) ? $data['affiliateID'] : 'N/A') . PHP_EOL;
    $admin_message .= 'Payment Method: ' . $data['payment_method'] . PHP_EOL;

    if ($data['payment_method'] === 'paypal') {
        $admin_message .= 'PayPal Email: ' . $data['paypal_email'] . PHP_EOL;
    } elseif ($data['payment_method'] === 'bank') {
        $admin_message .= 'Bank Details:' . PHP_EOL;
        $admin_message .= '- Account Number: ' . $data['account_number'] . PHP_EOL;
        $admin_message .= '- Branch Name: ' . $data['branch_name'] . PHP_EOL;
        $admin_message .= '- Routing Number: ' . $data['routing_number'] . PHP_EOL;
    }

    $email_subject = 'New Affiliate User Registration';

    $headerText = "New Affiliate User Registration";

    $admin_message_html = nl2br($admin_message);

    $bodyContent = "
        <h3>Affiliate user information during registration</h3>
        {$admin_message_html}
    ";

    $email_message = icecomaf_email_template($headerText, $bodyContent);

    $headers = array('Content-Type: text/html; charset=UTF-8');

    wp_mail($admin_email, $email_subject, $email_message, $headers);

    wp_send_json_success(['message' => esc_html__('Registration successful. Please check your email for verification.', 'ecommerce-affiliate')]);
}
add_action('wp_ajax_nopriv_register_affiliate_user', 'icecomaf_affiliate_user_registration');
add_action('wp_ajax_register_affiliate_user', 'icecomaf_affiliate_user_registration');

// email verification status
function icecomaf_email_verification() {
    if (isset($_GET['token'])) {
        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';

        if (!empty($token)) {
            global $wpdb;
            $user_id = $wpdb->get_var($wpdb->prepare(
                "SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key = 'email_verification_token' AND meta_value = %s",
                $token
            ));

            if ($user_id) {
                $user = get_userdata($user_id);

                if (in_array('affiliate_user', (array) $user->roles)) {
                    delete_user_meta($user_id, 'email_verification_token');
                    update_user_meta($user_id, 'email_verified', true);
                    wp_update_user(['ID' => $user_id, 'user_status' => 0]);

                    wp_redirect(site_url() . '/affiliate-login/');
                    exit;
                } else {
                    wp_redirect(site_url() . '/verification-success?status=failed');
                    exit;
                }
            } else {
                wp_redirect(site_url() . '/verification-success?status=failed');
                exit;
            }
        } else {
            wp_redirect(site_url() . '/verification-success?status=failed');
            exit;
        }
    }
}

add_action('init', 'icecomaf_email_verification');


function icecomaf_block_unverified_users($user) {
    if (in_array('affiliate_user', (array)$user->roles)) {
        if (get_user_meta($user->ID, 'email_verified', true) !== true) {
            return new WP_Error('email_not_verified', 'Your email address is not verified. Please check your email for the verification link.');
        }
    }

    return $user;
}
add_filter('wp_authenticate_user', 'icecomaf_block_unverified_users');

function icecomaf_show_login_error($error) {
    if (isset($_GET['email_not_verified'])) {
        $error = 'Your email address is not verified. Please check your email for the verification link.';
    }

    return $error;
}
add_filter('login_errors', 'icecomaf_show_login_error');


// login functionalities

function icecomaf_handle_affiliate_login_form() {
    // Check if the required POST variables are set
    if (isset($_POST['email']) && isset($_POST['login_form_nonce'])) {
        
        if ( !isset($_POST['login_form_nonce']) || !wp_verify_nonce( sanitize_text_field(wp_unslash($_POST['login_form_nonce'])), 'login_form_nonce_action') ) {
            wp_send_json_error(['message' => esc_html__('Nonce verification failed, Try again.', 'ecommerce-affiliate')]);
            return; 
        }
        

        // Sanitize the input fields
        $email = sanitize_email($_POST['email']);
        $password = isset($_POST['password']) ? sanitize_text_field($_POST['password']) : '';

        // Check if email or password is empty
        if (empty($email)) {
            wp_send_json_error(['message' => esc_html__('Email is required.', 'ecommerce-affiliate')]);
            return;
        }

        if (empty($password)) {
            wp_send_json_error(['message' => esc_html__('Password is required.', 'ecommerce-affiliate')]);
            return;
        }

        // Retrieve the user by email
        $user = get_user_by('email', $email);

        if ($user && wp_check_password($password, $user->user_pass, $user->ID)) {
            // Check if the email is verified
            $email_verified = get_user_meta($user->ID, 'email_verified', true);

            if (!$email_verified) {
                wp_send_json_error([
                    'message' => esc_html__('Your email is not verified. Please check your email and verify it before logging in.', 'ecommerce-affiliate')
                ]);
                return;
            }

            // Proceed with OTP generation for verified users
            if (in_array('affiliate_user', (array)$user->roles)) {
                $otp = wp_rand(100000, 999999);
                update_user_meta($user->ID, 'two_factor_otp', $otp);
                update_user_meta($user->ID, 'two_factor_otp_expiry', time() + 300);

                $email_subject = 'Two-Factor Authentication';

                $headerText = "Authentication";

                $bodyContent = "
                    <h3>Two-Factor Authentication Code</h3>
                    <p><strong>OTP:</strong> <span>{$otp}</span> It will expire in 5 minutes.</p>
                ";

                $email_message = icecomaf_email_template($headerText, $bodyContent);

                $headers = array('Content-Type: text/html; charset=UTF-8');

                wp_mail($user->user_email, $email_subject, $email_message, $headers);

                wp_send_json_success([
                    'message' => esc_html__('Login successful. Please check your email for the OTP.', 'ecommerce-affiliate'),
                    'redirect_url' => get_site_url() . '/verify-otp/?user_id=' . $user->ID
                ]);
                return;
            } else {
                wp_send_json_error(['message' => esc_html__('You are not an affiliate user.', 'ecommerce-affiliate')]);
                return;
            }
        } else {
            wp_send_json_error(['message' => esc_html__('Invalid login credentials.', 'ecommerce-affiliate')]);
        }
    } else {
        wp_send_json_error(['message' => esc_html__('Required fields are missing.', 'ecommerce-affiliate')]);
    }
    die();
}

add_action('wp_ajax_nopriv_affiliate_login', 'icecomaf_handle_affiliate_login_form');
add_action('wp_ajax_affiliate_login', 'icecomaf_handle_affiliate_login_form');



// logout ajax
function icecomaf_affiliate_logout() {
    if (is_user_logged_in() && in_array('affiliate_user', (array) wp_get_current_user()->roles)) {

        wp_logout();

        wp_send_json_success([
            'redirect_url' => home_url() . '/affiliate-login' 
        ]);
    } else {
        wp_send_json_error(['message' => esc_html__('Error logging out.', 'ecommerce-affiliate')]);
    }
}

add_action('wp_ajax_ic_affiliate_logout', 'icecomaf_affiliate_logout');
add_action('wp_ajax_nopriv_ic_affiliate_logout', 'icecomaf_affiliate_logout');

// forgot password ajax

function icecomaf_handle_forgot_password() {
    // Verify nonce for security
    if (!isset($_POST['forgot_password_nonce']) || !wp_verify_nonce($_POST['forgot_password_nonce'], 'forgot_password_nonce_action')) {
        wp_send_json_error(['message' => esc_html__('Invalid request. Please refresh the page and try again.', 'ecommerce-affiliate')]);

        return;
    }

    $email = sanitize_email($_POST['email']);

    if (!is_email($email)) {
        wp_send_json_error(['message' => esc_html__('Invalid email address.', 'ecommerce-affiliate')]);
    }

    $user = get_user_by('email', $email);
    if (!$user || !in_array('affiliate_user', $user->roles)) {
        wp_send_json_error(['message' => esc_html__('No affiliate user found with this email address.', 'ecommerce-affiliate')]);
    }

    $reset_key = get_password_reset_key($user);
    if (is_wp_error($reset_key)) {
        wp_send_json_error(['message' => esc_html__('Unable to generate reset link. Please try again later.', 'ecommerce-affiliate')]);
    }

    $reset_url = network_site_url("wp-login.php?action=rp&key=$reset_key&login=" . rawurlencode($user->user_login));

    $email_subject = esc_html__('Password Reset Request', 'ecommerce-affiliate'); 
    $headerText = esc_html__('Password Reset Request', 'ecommerce-affiliate'); 

    $bodyContent = "
        <h3>Password Reset</h3>
        <p>You requested a password reset. Click the link below to reset your password:</p>
        <p><a href='{$reset_url}'>{$reset_url}</a></p>
        <p>If you did not request this, please ignore this email.</p>
    ";

    // Generate the full email message with the template function
    $email_message = icecomaf_email_template($headerText, $bodyContent);

    // Email headers
    $headers = array('Content-Type: text/html; charset=UTF-8');

    // Send the email
    $email_sent = wp_mail($user->user_email, $email_subject, $email_message, $headers);
    
    if ($email_sent) {
        wp_send_json_success(['message' => esc_html__('Password reset email sent successfully. Please check your inbox.', 'ecommerce-affiliate')]);
    } else {
        wp_send_json_error(['message' => esc_html__('Failed to send email. Please try again later.', 'ecommerce-affiliate')]);
    }
}

add_action('wp_ajax_ic_forgot_password', 'icecomaf_handle_forgot_password');
add_action('wp_ajax_nopriv_ic_forgot_password', 'icecomaf_handle_forgot_password');

// 2FA otp ajax
function icecomaf_otp_verification_ajax() {

    if (!isset($_POST['affiliate_verify_otp_nonce']) || !wp_verify_nonce($_POST['affiliate_verify_otp_nonce'], 'affiliate_verify_otp_nonce_action')) {
        wp_send_json_error(['message' => esc_html__('Nonce verification failed.', 'ecommerce-affiliate')]);
        return;
    }
    
    if (isset($_POST['user_id']) && isset($_POST['otp'])) {
        $user_id = intval($_POST['user_id']);
        $user = get_user_by('id', $user_id);

        if (empty($_POST['otp'])) {
            wp_send_json_error(['message' => esc_html__('OTP is required.', 'ecommerce-affiliate')]);
            return;
        }

        if ($user) {
            $entered_otp = sanitize_text_field($_POST['otp']);
            $stored_otp = get_user_meta($user_id, 'two_factor_otp', true);
            $otp_expiry = get_user_meta($user_id, 'two_factor_otp_expiry', true);

            if ($entered_otp == $stored_otp && time() <= $otp_expiry) {

                wp_set_auth_cookie($user_id);
                wp_set_current_user($user_id);

                wp_send_json_success([
                    'message' => esc_html__('OTP verified successfully.', 'ecommerce-affiliate'),
                    'redirect_url' => home_url('/affiliate-profile/'),
                ]);
            } else {
                wp_send_json_error(['message' => esc_html__('Invalid or expired OTP. Please try again.', 'ecommerce-affiliate')]);
            }
        } else {
            wp_send_json_error(['message' => esc_html__('Invalid user.', 'ecommerce-affiliate')]);
        }
    } else {
        wp_send_json_error(['message' => esc_html__('Required fields are missing.', 'ecommerce-affiliate')]);
    }

    die();
}

add_action('wp_ajax_affiliate_verify_otp', 'icecomaf_otp_verification_ajax');
add_action('wp_ajax_nopriv_affiliate_verify_otp', 'icecomaf_otp_verification_ajax');

// resend otp ajax
function icecomaf_handle_resend_otp() {
    // Verify nonce for security
    if ( !isset($_POST['resend_nonce']) || !wp_verify_nonce( sanitize_text_field(wp_unslash($_POST['resend_nonce'])), 'resend_otp_nonce_action') ) {
        wp_send_json_error(['message' => esc_html__('Verification failed for resend OTP. Please try again.', 'ecommerce-affiliate')]);
        return;
    }

    // Get the user ID from the POST data
    $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : null;

    if (!$user_id) {
        wp_send_json_error(['message' => esc_html__('User ID is missing. Cannot resend OTP.', 'ecommerce-affiliate')]);
        return;
    }

    // Retrieve the user
    $user = get_user_by('id', $user_id);
    if (!$user) {
        wp_send_json_error(['message' => esc_html__('Invalid user.', 'ecommerce-affiliate')]);
        return;
    }

    // Generate a new OTP
    $otp = wp_rand(100000, 999999);

    // Update user meta with the new OTP and expiry time
    update_user_meta($user_id, 'two_factor_otp', $otp);
    update_user_meta($user_id, 'two_factor_otp_expiry', time() + 300);

    // Send the OTP to the user's email

    $email_subject = esc_html__('Your New Two-Factor Authentication Code', 'ecommerce-affiliate');

    $headerText = esc_html__('New OTP', 'ecommerce-affiliate');

    $bodyContent = "
        <h3>Your New Two-Factor Authentication Code</h3>
        <p><strong>OTP:</strong> <span>{$otp}</span> It will expire in 5 minutes.</p>
    ";

    $email_message = icecomaf_email_template($headerText, $bodyContent);

    $headers = array('Content-Type: text/html; charset=UTF-8');

    wp_mail($user->user_email, $email_subject, $email_message, $headers);

    wp_send_json_success(['message' => esc_html__('OTP has been resent successfully. Please check your email.', 'ecommerce-affiliate')]);
}

add_action('wp_ajax_nopriv_resend_otp', 'icecomaf_handle_resend_otp');
add_action('wp_ajax_resend_otp', 'icecomaf_handle_resend_otp');

// payment request
add_action('wp_ajax_ic_claim_payment', 'icecomaf_claim_payment_handler'); 
add_action('wp_ajax_nopriv_ic_claim_payment', 'icecomaf_claim_payment_handler'); 

function icecomaf_claim_payment_handler() {
    global $wpdb;

    // Verify nonce
    if ( !isset($_POST['payment_request_nonce']) || !wp_verify_nonce( sanitize_text_field(wp_unslash($_POST['payment_request_nonce'])), 'payment_request_nonce_action') ) {
        wp_send_json_error(['message' => esc_html__('Security check failed!', 'ecommerce-affiliate')]);
        return; 
    }
    

    parse_str($_POST['form_data'], $form_data);

    // Validate form data
    $payment_method = sanitize_text_field($form_data['payment_method']);

    $request_amount = floatval($form_data['request_amount']);
    $user_id = get_current_user_id();

    // if(empty($payment_method)) {
    //     wp_send_json_error(['message' => 'You should select a payment method.']);
    // }

    if(empty($request_amount)) {
        wp_send_json_error(['message' => esc_html__('You should enter request amount.', 'ecommerce-affiliate')]);
    }

    if (!$user_id) {
        wp_send_json_error(['message' => esc_html__('User is not logged in.', 'ecommerce-affiliate')]);
    }

    $user_data = get_userdata($user_id);

    // $user_name = $user_data->display_name;

    $user_name = get_user_meta($user_id, 'fullname', true) ? get_user_meta($user_id, 'fullname', true) : $user_data->display_name;

    $user_email = $user_data->user_email;

    if ($request_amount <= 0) {
        wp_send_json_error(['message' => esc_html__('Invalid request amount.', 'ecommerce-affiliate')]);
    }

    // Fetch data from affiliate_user table
    $affiliate_user_table = esc_sql($wpdb->prefix . 'affiliate_user'); 
    $payment_table = esc_sql($wpdb->prefix . 'affiliate_payments'); 

    $user_data = $wpdb->get_row($wpdb->prepare(
        "SELECT affiliate_id, due_balance, account_status  FROM $affiliate_user_table WHERE user_id = %d",
        $user_id
    ));

    $pending_count = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) 
             FROM $payment_table 
             WHERE user_id = %d 
               AND payment_status = %s",
            $user_id,
            'pending'
        )
    );

    $min_req_amount_options = get_option('settings_control_options');
    $withdraw_limit = $min_req_amount_options['min_withdraw_amount'];


    if($user_data->due_balance < $withdraw_limit) {
        wp_send_json_error([
            // 'message' => 'Insufficient balance, minimum should have $' . $withdraw_limit
            'message' => sprintf(
                esc_html__('Insufficient balance, minimum should have $%s', 'ecommerce-affiliate'),
                esc_html($withdraw_limit)
            )
        ]);
    }

    if (!$user_data) {
        wp_send_json_error(['message' => 'User data not found.']);
    }

    if ($user_data->account_status === 'pending') {
        wp_send_json_error(['message' => esc_html__('Your account status is pending. You cannot request for payout at this time.', 'ecommerce-affiliate')]);
    }

    // Check if request amount is less than or equal to Due balance

    if ($request_amount > $user_data->due_balance) {
        wp_send_json_error(['message' => esc_html__('The requested amount cannot exceed the due balance.', 'ecommerce-affiliate')]);
    }

    if($pending_count > 0) {
        wp_send_json_error(['message' => esc_html__('You have already pending request.', 'ecommerce-affiliate')]);
    }

    // Prepare data for affiliate_payments table
    $affiliate_payments_table = esc_sql($wpdb->prefix . 'affiliate_payments');

    $data = [
        'user_id'        => $user_id,
        'name'           => $user_name,
        'email'          => $user_email,
        'affiliate_id'   => $user_data->affiliate_id,
        'payment_type'   => $payment_method,
        'request_amount' => $request_amount,
        'due_balance'=> $user_data->due_balance,
        'payment_status' => 'pending',
        'created_at'     => current_time('mysql')
    ];

    // Insert data into affiliate_payments table
    $result = $wpdb->insert($affiliate_payments_table, $data);

    if ($result) {
        wp_send_json_success(['message' => esc_html__('Payment request submitted successfully.', 'ecommerce-affiliate')]);
    } else {
        wp_send_json_error(['message' => esc_html__('Failed to submit payment request.', 'ecommerce-affiliate')]);
    }
}