<?php 

function icecomaf_view_admin_user_payment_transition($status = []) {
    global $wpdb;

    if (!is_array($status)) {
        $status = [$status];
    }

    $user_id = isset($_GET['user_id']) ? sanitize_text_field($_GET['user_id']) : '';

    $table_payments = esc_sql($wpdb->prefix . 'affiliate_payments');

    $placeholders = implode(',', array_fill(0, count($status), '%s'));

    $results_payments_pending = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT user_id, name, email, affiliate_id, payment_type, request_amount, paid_amount, due_balance, payment_status, created_at, status_updated_at 
             FROM $table_payments 
             WHERE user_id = %d AND payment_status IN ($placeholders) ", 
             $user_id,
            ...$status,
        )
    );


    if($results_payments_pending):
        ?>
        <table class="ic_pay_tbl widefat fixed" cellspacing="0">
            <thead>
                <tr>
                    <th><?php echo esc_html__('Amount', 'ecommerce-affiliate') ?></th>
                    <th><?php echo esc_html__('Payment Type', 'ecommerce-affiliate') ?></th>
                    <th><?php echo esc_html__('Status', 'ecommerce-affiliate') ?></th>
                    <th><?php echo esc_html__('Date', 'ecommerce-affiliate') ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach($results_payments_pending as $item) { ?>
                    <tr>
                        <?php if($item->payment_status == 'cancel' || $item->payment_status == 'pending'): ?>
                            <td><?php echo esc_html($item->request_amount); ?></td>
                        <?php endif; ?>

                        <?php if($item->payment_status == 'full paid' || $item->payment_status == 'partial paid'): ?>
                            <td><?php echo esc_html($item->paid_amount); ?></td>
                        <?php endif; ?>

                        <td><?php echo esc_html($item->payment_type); ?></td>
                        
                        <td class="<?php
                            if($item->payment_status == 'pending') {
                                echo esc_attr('ic-warning');
                            } elseif($item->payment_status == 'cancel') {
                                echo esc_attr('ic-danger');
                            } elseif($item->payment_status == 'partial paid') {
                                echo esc_attr('ic-info');
                            } elseif($item->payment_status == 'full paid') {
                                echo esc_attr('ic-success');
                            }
                        ?>"><?php echo esc_html($item->payment_status); ?></td>

                        <?php if($item->payment_status == 'pending'): ?>
                            <td><?php echo esc_html($item->created_at); ?></td>
                        <?php endif; ?>
                        <?php if($item->payment_status == 'cancel' || $item->payment_status == 'full paid' || $item->payment_status == 'partial paid'): ?>
                            <td><?php echo esc_html($item->status_updated_at); ?></td>
                        <?php endif; ?>
                    </tr>
                <?php } ?>
            </tbody>
        </table>
        
        <?php else: ?>
            <p><?php echo esc_html__('Data not found', 'ecommerce-affiliate'); ?>.</p>
        <?php 
    endif;
}

function icecomaf_admin_user_get_paid_amount_sum($statuses = ['full paid', 'partial paid']) {
    global $wpdb;

    $table_payments = esc_sql($wpdb->prefix . 'affiliate_payments');
    $user_id = isset($_GET['user_id']) ? absint($_GET['user_id']) : 0;

    if (!is_array($statuses)) {
        $statuses = [$statuses];
    }
    $placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $query = $wpdb->prepare(
        "SELECT SUM(paid_amount) as total_paid 
         FROM $table_payments 
         WHERE user_id = %d AND payment_status IN ($placeholders)",
        $user_id,
        ...$statuses
    );

    $result = $wpdb->get_var($query);

    return $result ? $result : 0; // Return 0 if no matching records are found
}



function icecomaf_affiliates_program() {
    global $wpdb;

    $table_users = esc_sql($wpdb->prefix . 'affiliate_user');
    $table_payments = esc_sql($wpdb->prefix . 'affiliate_payments');

    $total_affiliate_users = $wpdb->get_var("SELECT COUNT(*) FROM $table_users where account_status = 'approved'");

    $total_affiliate_clicks = $wpdb->get_var("SELECT SUM(clicks_count) FROM $table_users where account_status = 'approved'");

    $total_affiliate_paid = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT SUM(paid_amount) FROM $table_payments WHERE payment_status IN (%s, %s, %s)",
            'full paid',
            'partial paid',
            'SUCCESS'
        )
    );

    $args = [
        'status' => ['wc-completed'], 
        'post_type' => 'shop_order',
        'limit' => -1, 
        'meta_query' => [
            [
                'key' => 'affiliate_id', 
                'compare' => 'EXISTS' 
            ]
        ]
    ];

    $all_orders = wc_get_orders($args);

    $total_amount = 0;

    foreach ($all_orders as $order) {
        if($order->get_meta('affiliate_id')) {
            $total_amount += $order->get_total();
        }
    }

    ?>
        <div class="ic_affiliate_overviews_sec ic-admin-sec">
            <h2>Affiliates Overview</h2>
            <!-- overview -->
            <div class="ic-affiliate-overview-wrapper">
                <div class="ic-single-aff-overview ic-aff-users">
                    <span><?php echo esc_html($total_affiliate_users ? $total_affiliate_users : 0); ?></span>
                    <h3><?php echo esc_html__('Affiliates', 'ecommerce-affiliate') ?></h3>
                </div>
                <div class="ic-single-aff-overview ic-aff-clicks">
                    <span><?php echo esc_html($total_affiliate_clicks ? $total_affiliate_clicks : 0); ?></span>
                    <h3><?php echo esc_html__('Clicks', 'ecommerce-affiliate') ?></h3>
                </div>
                <div class="ic-single-aff-overview ic-aff-earnings">
                    <span>$<?php echo esc_html(number_format($total_amount, 2)); ?></span>
                    <h3><?php echo esc_html__('Earning', 'ecommerce-affiliate') ?></h3>
                </div>
                <div class="ic-single-aff-overview ic-aff-paid">
                    <span>$<?php echo esc_html($total_affiliate_paid ? $total_affiliate_paid : 0); ?></span>
                    <h3><?php echo esc_html__('Paid', 'ecommerce-affiliate') ?></h3>
                </div>
            </div>
        </div>
    <?php
}

function icecomaf_users_list_callback() {
    ?>
        <div class="ic-admin-sec">
            <h2><?php echo esc_html__('Affiliate Program', 'ecommerce-affiliate'); ?></h2>
        </div>
        <div class="ic-aff-tabs">
            <a href="#pending-users" class="ic-nav-tab ic-nav-tab-active" data-tab="pending-users">
                <?php echo esc_html__('Pending Affiliate Users', 'ecommerce-affiliate') ?>
            </a>
            <a href="#all-users" class="ic-nav-tab" data-tab="all-users">
                <?php echo esc_html__('Approved All Affiliate Users', 'ecommerce-affiliate') ?>
            </a>
        </div>
        
        <div id="pending-users" class="ic-tab-content ic-active">
            <h2><?php echo esc_html__('Pending Affiliate Users', 'ecommerce-affiliate') ?></h2>
            <?php icecomaf_render_pending_users(); ?>
        </div>

        <div id="all-users" class="ic-tab-content">
            <h2><?php echo esc_html__('Approved All Affiliate Users', 'ecommerce-affiliate') ?></h2>
            <?php icecomaf_render_all_users(); ?>
        </div>


    <?php
}

function icecomaf_account_status_based($status) {
    global $wpdb;
    $table_name = esc_sql($wpdb->prefix . 'affiliate_user'); 

    $query = $wpdb->prepare("SELECT * FROM $table_name WHERE account_status = %s ORDER BY created_at DESC", $status);
    $pending_users = $wpdb->get_results($query);

    if($pending_users):

        ?>
            <table class="widefat fixed" cellspacing="0">
                <thead>
                    <tr>
                        <th scope="col"><?php echo esc_html__('Name', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Email', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Affiliate ID', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Due Balance', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Account Status', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Action', 'ecommerce-affiliate') ?></th>
                        <th scope="col"><?php echo esc_html__('Register Date', 'ecommerce-affiliate') ?></th>
                    </tr>
                </thead>
    
                <tbody>
                    <?php 
                        foreach($pending_users as $user): 
                            $user_id = $user->user_id;
                            $user_data = get_userdata($user_id);
                            $user_name = get_user_meta($user_id, 'fullname', true) ? get_user_meta($user_id, 'fullname', true) : $user_data->display_name;
                            $user_email = $user_data->user_email ? $user_data->user_email : '';
                            $payment_detail_json = $user->payment_detail;
                            $payment_details = json_decode($payment_detail_json, true);
                    ?>
                    <tr>
                        <?php if($user_name): ?>
                            <td><?php echo esc_html($user_name); ?></td>
                        <?php endif; ?>
    
                        <?php if($user_email): ?>
                            <td><?php echo esc_html($user_email); ?></td>
                        <?php endif; ?>
    
                        <?php if($user->affiliate_id): ?>
                            <td><?php echo esc_html($user->affiliate_id); ?></td>
                        <?php endif; ?>
    
                        <?php if($user->due_balance): ?>
                        <td><?php echo esc_html($user->due_balance); ?></td>
                        <?php endif; ?>
                        
                        <?php if($user->account_status): ?>
                            <td><?php echo esc_html($user->account_status); ?></td>
                        <?php endif; ?>
    
                        <td>        
                            <a href="<?php echo esc_url(admin_url('admin.php?page=user_details&user_id=' . esc_attr($user->user_id))); ?>">
                                <?php echo esc_html__('View More', 'ecommerce-affiliate'); ?>
                            </a>
                        </td>
                        <td>
                            <?php echo esc_html($user->created_at); ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php

else :
// If no data found
echo '<p>' . esc_html__('No data found yet.', 'ecommerce-affiliate') . '</p>';
    
        endif;

    return $pending_users;
}

function icecomaf_render_pending_users() {
   icecomaf_account_status_based('pending');
}

function icecomaf_render_all_users() {
    icecomaf_account_status_based('approved');
}

function icecomaf_render_user_details() {
    global $wpdb;
    
    $user_id = isset($_GET['user_id']) ? absint($_GET['user_id']) : 0;

    $get_user_data = get_userdata($user_id);
    $user_email = $get_user_data->user_email;

    if (!$user_id) {
        echo '<p>No user selected.</p>';
        return;
    }
    
    $user_data = get_userdata($user_id);

    $user_full_name = get_user_meta($user_id, 'fullname', true);

    $table_name = esc_sql($wpdb->prefix . 'affiliate_user');
    $query = $wpdb->prepare("SELECT * FROM $table_name WHERE user_id = %d", $user_id);
    $user = $wpdb->get_row($query);
    $payment_detail = json_decode($user->payment_detail);

    $encryption_key = icecomaf_encryption_key($user_id, $user_email);
    $enc_substr = substr($encryption_key, 0, 16);

    $encrypted_bank_name = $payment_detail->bank->branch_name;
    $encrypted_account_number = $payment_detail->bank->account_number;
    $encrypted_routing_number = $payment_detail->bank->routing_number;

    $encrypted_paypal_email = $payment_detail->paypal->paypal_email;

    $decrypted_bank_name = openssl_decrypt($encrypted_bank_name, 'aes-256-cbc', $encryption_key, 0, $enc_substr);
    $decrypted_account_number = openssl_decrypt($encrypted_account_number, 'aes-256-cbc', $encryption_key, 0, $enc_substr);
    $decrypted_routing_number = openssl_decrypt($encrypted_routing_number, 'aes-256-cbc', $encryption_key, 0, $enc_substr);

    $decrypted_paypal_email = openssl_decrypt($encrypted_paypal_email, 'aes-256-cbc', $encryption_key, 0, $enc_substr);

    $affiliate_id = $user->affiliate_id;

    $affiliate_args = [
        // 'status' => ['wc-completed', 'wc-processing'], 
        'status' => ['wc-completed'], 
        'post_type' => 'shop_order', 
        'limit' => -1, 
        'meta_query' => [
            [
                'key' => 'affiliate_id', 
                'value' => $affiliate_id, 
                'compare' => '='
            ]
        ]
    ];

    $orders = wc_get_orders($affiliate_args);

    // Generate the nonce
    $nonce = wp_create_nonce('approve_user_nonce');

    $options_commision = get_option('settings_control_options');
    $default_number = $options_commision['default_number'] ?? 3;

    ?>
        <div class="wrap">
            <div class="ic_admin_affiliates_user_intro">
                <div class="ic-admin-sec">
                    <h2><?php echo esc_html__('User Details', 'ecommerce-affiliate'); ?></h2>
                </div>
                
                <div class="ic_admin_aff_user_col">
                    <table class="widefat fixed">
                        <tr>
                            <th><?php echo esc_html__('User ID', 'ecommerce-affiliate'); ?>:</th>
                            <td><?php echo esc_html($user_data->ID); ?></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Name', 'ecommerce-affiliate'); ?>:</th>
                            <td><?php echo esc_html($user_full_name); ?></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Email', 'ecommerce-affiliate'); ?>:</th>
                            <td><?php echo esc_html($user_data->user_email); ?></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Affiliate ID', 'ecommerce-affiliate'); ?>:</th>
                            <td><?php echo esc_html($user->affiliate_id); ?></td>
                        </tr>
                        <?php
                             if ($user->account_status === 'approved'):
                            $affiliate_link = home_url('?ref=' . $user->affiliate_id);
                        ?>
                        <tr>
                            <th><?php echo esc_html__('Affiliate Link', 'ecommerce-affiliate'); ?>: </th>
                            <td><?php echo esc_html($affiliate_link); ?></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Link Clicks', 'ecommerce-affiliate'); ?>: </th>
                            <td><?php echo esc_html($user->clicks_count); ?></td>
                        </tr>
                        <?php endif; ?>
        
                        <tr>
                            <th><?php echo esc_html__('Account Status', 'ecommerce-affiliate'); ?>:</th>
                            <td>
                                <?php echo esc_html($user->account_status); ?>
        
                                <?php if ($user->account_status === 'pending') : ?>
                                    <button id="ic-approve-button" data-user-id="<?php echo esc_attr($user->user_id); ?>" data-nonce="<?php echo esc_attr($nonce); ?>">Approve</button>
                                <?php endif; ?>
                        </tr>
                    </table>
                    <div class="ic_admin_aff_user_pic">
                        <?php
                            if($user->profile_pic) {
                                ?>
                                    <img src="<?php echo esc_url($user->profile_pic); ?>" alt="<?php echo esc_attr($user_data->display_name); ?>">
                                <?php
                            } else {
                                ?><p><?php echo esc_html__('No profile Picture Set by', 'ecommerce-affiliate'); ?> <strong><?php echo esc_html($user_data->display_name); ?></strong></p><?php
                            }
                        ?>
                    </div>
                </div>
            </div>
            <div class="ic_admin_affiliates_payment_order_col">
                <div class="ic_admin_affiliates_payment ic_col_fad">
                    <h3><?php echo esc_html__('Payment Information', 'ecommerce-affiliate') ?></h3>                        
                    <div class="ic_admin_aff_paypal_info">
                        <p><strong><?php echo esc_html__('PayPal Email', 'ecommerce-affiliate') ?>: </strong> <?php echo esc_html($payment_detail); ?></p>
                    </div>                  
                </div>
                <div class="ic_admin_affiliates_order_history ic_col_fad">
                    <div class="ic_affiliate_sales_history">
                        <h3><?php echo esc_html__('Order history', 'ecommerce-affiliate'); ?></h3>
                        
                        <?php
                            $commission_percentage = icecomaf_affiliate_commission(); 

                            if ($orders && !empty($orders)) : 
                            $has_orders = false; 
                        ?>
                            <div class="ic_admin_order_inner_col_body">
                                <?php
                                    $options_commision = get_option('settings_control_options');
                                    $default_number = $options_commision['default_number'] ?? 3;
                                ?>
                                <?php foreach ($orders as $order) :
                                   
                                    $order_id = $order->get_id();
                                    $order_total = $order->get_total();

                                    

                                    $commission = $order_total * $commission_percentage;

                                    if ($order->get_meta('affiliate_id') == strtolower($affiliate_id)) {
                                        if (!$has_orders) : 
                                            // Render the header only once
                                            $has_orders = true; 
                                            ?>
                                            <div class="ic_admin_order_inner_col_head">
                                                <div><?php echo esc_html__('Order ID', 'ecommerce-affiliate'); ?></div>
                                                <div><?php echo esc_html__('Order Amount', 'ecommerce-affiliate'); ?></div>
                                                <div><?php echo esc_html__('Commission', 'ecommerce-affiliate'); echo '(' . esc_html($default_number).'%'. ')';?></div>
                                            </div>
                                            <?php
                                        endif;
                                        ?>
                                        <div class="ic_admin_order_inner_col_body_single">
                                            <div><?php echo esc_html($order_id); ?></div>
                                            <div><?php echo wp_kses_post(wc_price($order_total)); ?></div>
                                            <div><?php echo wp_kses_post(wc_price($commission)); ?></div>

                                        </div>
                                        <?php
                                    }
                                endforeach;

                                if (!$has_orders) :
                                    ?>
                                    <p style="text-align: center"><?php echo esc_html__('No order made yet!', 'ecommerce-affiliate'); ?></p>
                                    <?php
                                endif;
                                ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="ic_admin_affiliates_payment_order_col">
                <div class="ic_admin_aff_payment_history">
                    <h3><?php echo esc_html__('Payments history', 'ecommerce-affiliate'); ?></h3>

                    <div class="ic_col_fad" id="ic_admin_payments_payout_tabs">
                        <ul class="ic_payments_payout_nav" style="display: flex; gap: 10px;">
                            <li><a class="ic-admin-user-tab-link active" href="#ic_admin_payout_all"><?php echo esc_html__('All', 'ecommerce-affiliate') ?></a></li>
                            <li><a class="ic-admin-user-tab-link" href="#ic_admin_payout_pending"><?php echo esc_html__('Pending', 'ecommerce-affiliate') ?></a></li>
                            <li><a class="ic-admin-user-tab-link" href="#ic_admin_payout_cancel_request"><?php echo esc_html__('Cancel', 'ecommerce-affiliate') ?></a></li>
                            <li><a class="ic-admin-user-tab-link" href="#ic_admin_payout_success">Paid</a></li>
                        </ul>

                        <div class="ic_payments_payout_all ic-user-detail-tab-content active" id="ic_admin_payout_all">
                            <?php 
                                echo wp_kses_post( icecomaf_view_admin_user_payment_transition(['pending', 'cancel', 'full paid', 'partial paid'], 'paid_amount') );    
                            ?>
                        </div>

                        <div class="ic_payments_payout_pending ic-user-detail-tab-content" id="ic_admin_payout_pending">
                            <?php echo wp_kses_post(icecomaf_view_admin_user_payment_transition('pending', 'request_amount')); ?>
                        </div>

                        <div class="ic_payments_payout_pending ic-user-detail-tab-content" id="ic_admin_payout_cancel_request">
                            <?php echo wp_kses_post(icecomaf_view_admin_user_payment_transition('cancel', 'request_amount')); ?>
                        </div>

                        <div class="ic_payments_payout_success ic-user-detail-tab-content" id="ic_admin_payout_success">
                            <?php echo wp_kses_post(icecomaf_view_admin_user_payment_transition(['full paid', 'partial paid'], 'paid_amount')); ?>
                        </div>
                    </div>
                </div>
                
                <div class="ic_col_fad ic_admin_view_due_col">
                    <div class="ic_my_balance">
                        <h3><?php echo esc_html__('Payout Balance', 'ecommerce-affiliate') ?></h3>
                        <strong>$ <?php echo wp_kses_post(icecomaf_admin_user_get_paid_amount_sum()); ?></strong>
                    </div>
                    <div class="ic_due_balance">
                        <h3><?php echo esc_html__('Due Balance', 'ecommerce-affiliate') ?></h3>
                        <strong>$ <?php echo wp_kses_post($user->due_balance); ?></strong>
                    </div>
                </div>
            </div>
        </div> 
    <?php
}

function icecomaf_payment_options_list() {
    ?>
        <div class="wrap ic-admin-sec ic-payment-opt">
        <h2><?php echo esc_html__('Payment Options', 'ecommerce-affiliate') ?></h2>

        <div class="ic-payment-tabs">
            <a href="#payment-request" class="ic-pay-nav-tab ic-pay-nav-tab-active" data-tab="payment-request">
            <?php echo esc_html__('Payment Request', 'ecommerce-affiliate') ?>
            </a>
            <a href="#cancel-payment" class="ic-pay-nav-tab" data-tab="cancel-payment">
            <?php echo esc_html__('Cancel Payment', 'ecommerce-affiliate') ?>
            </a>
            <a href="#payment-history" class="ic-pay-nav-tab" data-tab="payment-history">
            <?php echo esc_html__('Payment History', 'ecommerce-affiliate') ?>
            </a>
        </div>

        <div id="payment-request" class="ic-pay-tab-content ic-pay-active">
            <h3><?php echo esc_html__('Payment Request', 'ecommerce-affiliate') ?></h3>
            <?php echo wp_kses_post(icecomaf_payment_detail('pending', 'Request')); ?>
        </div>

        <div id="cancel-payment" class="ic-pay-tab-content">
            <h3><?php echo esc_html__('Cancel Payment', 'ecommerce-affiliate') ?></h3>
            <?php echo wp_kses_post(icecomaf_payment_detail('cancel', 'Request')); ?>
        </div>

        <div id="payment-history" class="ic-pay-tab-content">
            <h3><?php echo esc_html__('Payment History', 'ecommerce-affiliate') ?></h3>
            <?php echo wp_kses_post(icecomaf_payment_detail(['full paid', 'partial paid', 'SUCCESS'], 'Paid')); ?>
        </div>
    </div>

    <?php
}

function icecomaf_payment_detail($status = [], $label) {
    global $wpdb;

    if (!is_array($status)) {
        $status = [$status];
    }

    // Prepare placeholders for the IN clause
    $placeholders = implode(',', array_fill(0, count($status), '%s'));

    $payment_table = esc_sql($wpdb->prefix . 'affiliate_payments');
    $users_table = esc_sql($wpdb->prefix . 'affiliate_user');

    $results = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT 
                p.id, 
                p.user_id, 
                p.name, 
                p.email, 
                p.affiliate_id, 
                p.payment_type, 
                p.request_amount, 
                p.paid_amount, 
                p.due_balance, 
                p.payment_status, 
                p.status_reason, 
                p.created_at, 
                u.payment_detail
             FROM $payment_table AS p
             LEFT JOIN $users_table AS u 
             ON p.user_id = u.user_id
             WHERE p.payment_status IN ($placeholders)", 
            ...$status

        )
    );

    ?>
        <div class="ic-payment-request-tbl" id="ic_payment_request">
            <table class="widefat fixed">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('User ID', 'ecommerce-affiliate') ?></th>
                        <th><?php echo esc_html__('Name', 'ecommerce-affiliate') ?></th>
                        <th><?php echo esc_html__('Email', 'ecommerce-affiliate') ?></th>
                        <th><?php echo esc_html__('Affiliate ID', 'ecommerce-affiliate') ?></th>
                        <?php if ($label === "Request"): ?>
                            <th><?php echo esc_html__('Request Amount', 'ecommerce-affiliate') ?></th>
                        <?php elseif ($label === "Paid"): ?>
                            <th><?php echo esc_html__('Paid Amount', 'ecommerce-affiliate') ?></th>
                        <?php endif; ?>
                        <th><?php echo esc_html__('Due Balance', 'ecommerce-affiliate') ?></th>
                        <th><?php echo esc_html__('Status', 'ecommerce-affiliate') ?></th>
                        <?php if(!in_array("pending", $status, true)): ?>
                        <th><?php echo esc_html__('Status reason', 'ecommerce-affiliate') ?></th>
                        <?php endif; ?>
                        <th><?php echo esc_html__('Request date', 'ecommerce-affiliate') ?></th>
                        <?php if(in_array("pending", $status, true)): ?>
                        <th><?php echo esc_html__('Action', 'ecommerce-affiliate') ?></th>
                        <?php endif; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($results && count($results) > 0): ?>
                    <?php foreach ($results as $row): ?>
                        <?php
                            $payment_detail = json_decode($row->payment_detail, true);
                            $paypal_mail_enc = "";
                            $bank_name_enc = "";
                            $account_number_enc = 0;
                        ?>
                    <tr>
                        <td><?php echo esc_html($row->user_id); ?></td>
                        <td><?php echo esc_html($row->name); ?></td>
                        <td><?php echo esc_html($row->email); ?></td>
                        <td><?php echo esc_html($row->affiliate_id); ?></td>

                        <?php if ($label === "Request"): ?>
                            <td><?php echo esc_html(number_format($row->request_amount, 2)); ?></td>
                        <?php elseif ($label === "Paid"): ?>
                            <td><?php echo esc_html(number_format($row->paid_amount, 2)); ?></td>
                        <?php endif; ?>

                        <td><?php echo esc_html(number_format($row->due_balance, 2)); ?></td>
                        <td><?php echo esc_html($row->payment_status); ?></td>
                        <?php if(!in_array("pending", $status, true)): ?>
                        <td><?php echo esc_html($row->status_reason); ?></td>
                        <?php endif; ?>
                        <td><?php echo esc_html($row->created_at); ?></td>

                        <?php if (in_array("pending", $status, true)): ?>
                        <td class="ic_payment_actions">
                           
                            <?php
                                if($row->payment_type == "paypal" && $payment_detail['paypal']) {
                                    $paypal_mail_enc = $payment_detail['paypal']['paypal_email'];

                                    $user_id = $row->user_id;
                                    $user_email = $row->email;
                                    $user_name = get_user_meta($user_id, 'fullname', true);

                                    $encryption_key = icecomaf_encryption_key($user_id, $user_email);
                                    $enc_substr = substr($encryption_key, 0, 16);

                                    $decrypted_paypal_email = openssl_decrypt($paypal_mail_enc, 'aes-256-cbc', $encryption_key, 0, $enc_substr);


                                    ?>
                                    <button 
                                        class="ic-make-payment-btn" 
                                        data-user-id="<?php echo esc_attr($row->user_id); ?>"
                                        data-request-id="<?php echo esc_attr($row->id); ?>"
                                        data-payment-type="<?php echo esc_attr($row->payment_type); ?>" 
                                        data-email="<?php echo esc_attr($decrypted_paypal_email); ?>" 
                                        data-request-amount="<?php echo esc_attr($row->request_amount); ?>" 
                                        data-nonce="<?php echo esc_attr(wp_create_nonce('ic_payment_action_nonce')); ?>">
                                        <?php echo esc_html__('Make Payment', 'ecommerce-affiliate') ?>
                                    </button>
                                    <?php
                                }
                                
                            ?>

                            <button 
                                class="ic-make-payment-btn" 
                                data-user-id="<?php echo esc_attr($row->user_id); ?>"
                                data-request-id="<?php echo esc_attr($row->id); ?>"
                                data-payment-type="<?php echo esc_attr($row->payment_type); ?>" 
                                data-email="<?php echo esc_attr($decrypted_paypal_email); ?>" 
                                data-request-amount="<?php echo esc_attr($row->request_amount); ?>" 
                                data-nonce="<?php echo esc_attr(wp_create_nonce('ic_payment_action_nonce')); ?>">
                                <?php echo esc_html__('Make Payment', 'ecommerce-affiliate') ?>
                            </button>

                            <button 
                                class="ic-cancel-payment-btn" 
                                data-request-id="<?php echo esc_attr($row->id); ?>"
                                data-user-id="<?php echo esc_attr($row->user_id); ?>" 
                                data-nonce="<?php echo esc_attr(wp_create_nonce('ic_payment_action_nonce')); ?>">
                                <?php echo esc_html__('Cancel Payment', 'ecommerce-affiliate') ?>
                            </button>
                        </td>
                        <?php endif; ?>
                    </tr>
                    <?php endforeach; ?>
                    <?php else: ?>
                        <tr><td colspan="10" style="text-align: center; font-size: 18px"><?php echo esc_html__('No', 'ecommerce-affiliate') ?> <?php echo esc_html($status[0]); ?> <?php echo esc_html__('payments found', 'ecommerce-affiliate') ?>.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
            <div id="paypalPaymentPopup" class="ic_paypal_modal" style="display:none;">
                <div class="ic_paypal_modal_content">
                    <h3><?php echo esc_html__('Pay Affiliate User', 'ecommerce-affiliate') ?>: <span id="userNameDisplay"></span></h3>
                    <form id="paypalPaymentForm">
                        <label for="req_amount"><?php echo esc_html__('Amount', 'ecommerce-affiliate'); ?></label>
                        <input type="number" id="req_amount" name="req_amount" readonly>
                        
                        <input type="hidden" id="userName" name="userName">
                        <input type="hidden" id="userEmail" name="userEmail">
                        <input type="hidden" id="userID" name="userID">
                        <input type="hidden" id="request_id" name="request_id">
                        <div class="ic_paypal_modal_btns">
                            <button type="submit"><?php echo esc_html__('Pay Now', 'ecommerce-affiliate') ?></button>
                            <button class="ic_paypal_modal_cancel" type="button"><?php echo esc_html__('Cancel', 'ecommerce-affiliate') ?></button>
                        </div>
                    </form>
                    <div id="ic_papal_msg"></div>
                    <button id="ic_paypal_backtohome" style="display: none;" type="button"><?php echo esc_html__('Back to home', 'ecommerce-affiliate') ?></button>
                </div>
            </div>
        </div>
    <?php
}

// settings control

function icecomaf_settings_control() {
    ?>
        <div class="wrap">
            <div class="ic-admin-sec">
                <h2><?php echo esc_html__('Affiliates Settings Control', 'ecommerce-affiliate') ?></h2>
            </div>
            <form method="post" action="options.php">
                <?php
                    settings_fields('settings_control_group');
                    
                    do_settings_sections('settings-control');
                    
                    submit_button('Save Changes');
                ?>
            </form>
        </div>
    <?php
}

// settings control fields
add_action('admin_init', 'icecomaf_custom_settings_control_init');

function icecomaf_custom_settings_control_init() {
    // Register settings group
    register_setting(
        'settings_control_group', 
        'settings_control_options',
        array(
            'type' => 'array',
            'sanitize_callback' => 'icecomaf_sanitize_settings_control_options', // Custom sanitization callback
        )
    );
    
    // Add section
    add_settings_section(
        'settings_control_section', 
        'Settings Control Options', 
        null,                       
        'settings-control'          
    );
    
    // Input fields
    add_settings_field(
        'default_number',           
        'Commission',           
        'icecomaf_render_default_number',    
        'settings-control',         
        'settings_control_section'  
    );
    
    add_settings_field(
        'cookie_time',              
        'Cookie Time (in days)',    
        'icecomaf_render_cookie_time',       
        'settings-control',         
        'settings_control_section'  
    );
}

// Custom sanitization function
function icecomaf_sanitize_settings_control_options($input) {
    $sanitized_input = array();
    $sanitized_input['default_number'] = isset($input['default_number']) ? absint($input['default_number']) : 0;
    $sanitized_input['cookie_time'] = isset($input['cookie_time']) ? absint($input['cookie_time']) : 0;

    return $sanitized_input;
}

// Callback functions for each field
function icecomaf_render_default_number() {
    $options = get_option('settings_control_options');
    ?>
    <input type="number" name="settings_control_options[default_number]" min="1" value="<?php echo esc_attr($options['default_number'] ?? 3); ?>" /> %
    <?php
}

function icecomaf_render_cookie_time() {
    $options = get_option('settings_control_options');
    ?>
    <input type="number" name="settings_control_options[cookie_time]" min="1" value="<?php echo esc_attr($options['cookie_time'] ?? 30); ?>" /> <?php echo esc_html__('days', 'ecommerce-affiliate') ?>
    <?php
}