<?php
/**
 * Admin functionality for EchoDash.
 *
 * @package EchoDash
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Class EchoDash_Admin
 *
 * Handles the admin functionality.
 *
 * @since 1.0.0
 */
class EchoDash_Admin {

	/**
	 * Stores any data that needs to be passed to the admin scripts.
	 *
	 * @var array $localize_data
	 * @since 1.0.0
	 */
	public $localize_data = array();

	/**
	 * Constructs a new instance.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		add_action( 'admin_footer', array( $this, 'admin_scripts' ) );
		add_action( 'admin_init', array( $this, 'save_echodash_callback' ) );

		add_action( 'wp_ajax_echodash_send_test', array( $this, 'send_event_test' ) );
	}

	/**
	 * Send a test event using the current user data.
	 *
	 * @return void
	 */
	public function send_event_test() {

		check_ajax_referer( 'echodash_ajax_nonce', '_ajax_nonce' );

		if ( ! isset( $_POST['data'] ) ) {
			wp_send_json_error( 'echodash_empty_data' );
		}

		$data = echodash_clean( wp_unslash( $_POST['data'] ) );

		if ( empty( $data['event_name'] ) || empty( $data['integration'] ) ) {
			wp_send_json_error( 'echodash_missing_required_fields' );
		}

		$event_name = sanitize_text_field( $data['event_name'] );
		$source     = sanitize_text_field( $data['integration'] );
		$trigger    = sanitize_text_field( $data['trigger'] );
		$event_data = array();

		if ( ! empty( $data['event_keys_values'] ) ) {
			foreach ( $data['event_keys_values'] as $event ) {
				if ( ! isset( $event['key'] ) || ! isset( $event['value'] ) ) {
					continue;
				}
				$key                = sanitize_text_field( $event['key'] );
				$value              = sanitize_text_field( $event['value'] );
				$event_data[ $key ] = $value;
			}
		}

		$source_name  = echodash()->integration( $source )->name;
		$trigger_name = echodash()->integration( $source )->get_trigger( $trigger )['name'];

		echodash_track_event( $event_name, $event_data, $source_name, $trigger_name );

		wp_send_json_success( 'success' );
	}

	/**
	 * Used by the plugin integrations to pass data to the admin scripts.
	 *
	 * @since 1.1.3
	 *
	 * @param string $integration The integration slug.
	 * @param string $trigger     The trigger ID.
	 * @param array  $options     The options.
	 */
	public function localize( $integration, $trigger, $options ) {

		if ( ! isset( $this->localize_data['triggers'] ) ) {
			$this->localize_data = array( 'triggers' => array() );
		}

		if ( ! isset( $this->localize_data['triggers'][ $integration ] ) ) {
			$this->localize_data['triggers'][ $integration ] = array();
		}

		if ( ! isset( $this->localize_data['triggers'][ $integration ][ $trigger ] ) ) {
			$this->localize_data['triggers'][ $integration ][ $trigger ] = array();
		}

		$this->localize_data['triggers'][ $integration ][ $trigger ]['options'] = $options;
	}

	/**
	 * Get the connect URL to EchoDash.
	 *
	 * @since 0.0.3
	 *
	 * @return string
	 */
	public function get_connect_url() {

		$args = array(
			'a'            => 'plugin_install',
			'source'       => __( 'WordPress', 'echodash' ),
			'site_name'    => __( 'WordPress', 'echodash' ) . ' - ' . sanitize_text_field( get_bloginfo( 'name' ) ),
			'redirect_uri' => esc_url( admin_url( 'options-general.php?page=echodash' ) ),
			'state'        => wp_create_nonce( 'echodash_connect' ),
		);

		// Local testing.
		if ( str_ends_with( home_url(), '.local' ) ) {
			$url = 'http://127.0.0.1:3000/endpoints/new';
		} else {
			$url = 'https://echodash.com/endpoints/new';
		}

		$url = add_query_arg( $args, $url );

		return apply_filters( 'echodash_connect_url', $url, $args );
	}

	/**
	 * Saves the callback URL generated by EchoDash.
	 *
	 * @since 0.0.3
	 */
	public function save_echodash_callback() {
		if ( ! isset( $_GET['endpoint_url'] ) || ! isset( $_GET['state'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( sanitize_key( wp_unslash( $_GET['state'] ) ), 'echodash_connect' ) ) {
			return;
		}

		$endpoint_url = esc_url_raw( wp_unslash( $_GET['endpoint_url'] ) );

		update_option( 'echodash_endpoint', $endpoint_url, true );

		$this->initialize_default_settings();

		wp_safe_redirect( admin_url( 'options-general.php?page=echodash' ) );
		exit;
	}

	/**
	 * Register scripts and styles.
	 *
	 * @since 1.0.0
	 */
	public function admin_scripts() {

		wp_register_style( 'select4', ECHODASH_DIR_URL . 'assets/dist/select4.min.css', array(), '4.0.13' );
		wp_register_script( 'select4', ECHODASH_DIR_URL . 'assets/dist/select4.min.js', array( 'jquery' ), '4.0.13', true );

		wp_register_script( 'echodash-jquery-repeater', ECHODASH_DIR_URL . 'assets/dist/jquery.repeater.min.js', array( 'jquery' ), '1.2.2', true );

		wp_register_script( 'echodash-admin', ECHODASH_DIR_URL . 'assets/echodash-admin.js', array( 'jquery', 'jquery-ui-sortable', 'select4' ), ECHODASH_VERSION, true );
		wp_register_style( 'echodash-admin', ECHODASH_DIR_URL . 'assets/echodash-admin.css', array(), ECHODASH_VERSION );

		// Integrations set $this->localize_data based on the fields specific to the integration.
		// We only want to enqueue the scripts and styles if the current page has settings on it.

		if ( ! empty( $this->localize_data ) ) {

			$this->localize_data['ajaxurl'] = admin_url( 'admin-ajax.php' );
			$this->localize_data['nonce']   = wp_create_nonce( 'echodash_ajax_nonce' );

			wp_enqueue_style( 'select4' );
			wp_enqueue_script( 'select4' );
			wp_enqueue_script( 'echodash-jquery-repeater' );
			wp_enqueue_style( 'echodash-admin' );
			wp_enqueue_script( 'echodash-admin' );
			wp_localize_script( 'echodash-admin', 'ecdEventData', $this->localize_data );

		}
	}

	/**
	 * Initialize default settings for all integrations.
	 *
	 * @since 1.2.0
	 */
	public function initialize_default_settings() {
		$settings = get_option( 'echodash_options', array() );
		$modified = false;

		foreach ( echodash()->integrations as $integration ) {
			// Skip if this integration already has settings.
			if ( ! empty( $settings[ $integration->slug ] ) ) {
				continue;
			}

			$default_settings = array();

			foreach ( $integration->get_triggers() as $trigger => $config ) {
				if ( ! empty( $config['enabled_by_default'] ) ) {
					$default_event = $integration->get_defaults( $trigger );

					if ( $default_event ) {
						$default_settings[] = array(
							'trigger' => $trigger,
							'name'    => $default_event['name'],
							'value'   => array_map(
								function ( $key, $value ) {
									return array(
										'key'   => $key,
										'value' => $value,
									);
								},
								array_keys( $default_event['mappings'] ),
								$default_event['mappings']
							),
						);
					}
				}
			}

			if ( ! empty( $default_settings ) ) {
				$settings[ $integration->slug ] = $default_settings;
				$modified                       = true;
			}
		}

		if ( $modified ) {
			update_option( 'echodash_options', $settings, false );
		}
	}
}
