import React, { useState, useEffect, useRef } from 'react';
import { registerPlugin } from '@wordpress/plugins';
import { NoticeComponent } from './NoticeComponent';

const { dispatch } = wp.data;
const { VALIDATION_STORE_KEY } = window.wc.wcBlocksData;
const { setValidationErrors } = dispatch(VALIDATION_STORE_KEY);
const { __ } = window.wp.i18n;
const { ExperimentalOrderShippingPackages } = window.wc.blocksCheckout;

const TestDivComponent = () => {
    // Check if FFL map should be shown (admin setting)
    const showFflMap = window.eCheckpointParams?.show_ffl_map !== false;

    const [containerClass, setContainerClass] = useState("map-container-hide"); // Start hidden, show when FFL required
    const [selectedTradeType, setSelectedTradeType] = useState(null);
    const [clientMessage, setClientMessage] = useState(getCookieValue("client_message"));
    const [cacheVersion, setCacheVersion] = useState(0); // Tracks localStorage cache changes

    // If FFL map is disabled in admin, don't render anything
    if (!showFflMap) {
        return null;
    }

    function getCookieValue(name) {
        const cookie = document.cookie
            .split("; ")
            .find((row) => row.startsWith(`${name}=`));
        return cookie ? decodeURIComponent(cookie.split("=")[1]) : null;
    }

    // Helper function to get effective trade type with b2C fallback (same as App component)
    const getEffectiveTradeType = (businessTradeTypes, selectedType) => {
        if (businessTradeTypes[selectedType]?.eligiblePurchaser) {
            return selectedType;
        }
        // Fallback to b2C if selected type not eligible
        if (selectedType !== 'b2C' && businessTradeTypes['b2C']?.eligiblePurchaser) {
            return 'b2C';
        }
        return null; // Not eligible for any type
    };

    // Evaluate FFL requirements from localStorage cache (same logic as App component)
    const evaluateFflFromCache = (customerType) => {
        const cachedResponse = localStorage.getItem('echeckpoint_api_response');
        if (!cachedResponse) {
            setContainerClass("map-container-hide");
            return false;
        }

        try {
            const response = JSON.parse(cachedResponse);
            const items = response?.modules?.regionalRestrictionsCheck?.items || [];

            let fflRequiredProducts = [];
            let blockedProducts = [];

            items.forEach(item => {
                const businessTradeTypes = item.businessTradeTypes;
                const productId = item.response?.productID;
                const isRegulated = item.regulatedProduct === true;

                // Skip non-regulated products
                if (!isRegulated || !businessTradeTypes) {
                    return;
                }

                const effectiveType = getEffectiveTradeType(businessTradeTypes, customerType);

                // Check if product is blocked (not eligible for any customer type)
                if (effectiveType === null) {
                    blockedProducts.push(productId);
                    return;
                }

                // Get FFL requirement for the effective trade type - must be explicitly true
                const fflRequired = businessTradeTypes[effectiveType]?.fflRequired === true;

                if (fflRequired) {
                    fflRequiredProducts.push(productId);
                }
            });

            // Hide map when there are blocked products - customer needs to fix cart first
            if (blockedProducts.length > 0) {
                setContainerClass("map-container-hide");
                return false;
            }

            // Show map if any products require FFL for this customer type
            if (fflRequiredProducts.length > 0) {
                setContainerClass("map-container");

                // Re-render markers when map becomes visible
                // Get FFL locations from cached response and update markers
                if (response?.modules?.regionalRestrictionsCheck?.licensing?.[0]?.availableLicenses) {
                    const licensingLocations = response.modules.regionalRestrictionsCheck.licensing[0].availableLicenses;
                    // Small delay to ensure map container is visible before updating markers
                    setTimeout(() => {
                        if (window.updateMarkers && licensingLocations.length > 0) {
                            window.updateMarkers(licensingLocations);
                        }
                    }, 100);
                }

                return true;
            } else {
                setContainerClass("map-container-hide");
                return false;
            }
        } catch (error) {
            console.error('Error evaluating FFL from cache:', error);
            setContainerClass("map-container-hide");
            return false;
        }
    };

    // Poll the "client_message" cookie every second and update state when it changes
    useEffect(() => {
        const interval = setInterval(() => {
            const newClientMessage = getCookieValue("client_message");
            if (newClientMessage !== clientMessage) {
                setClientMessage(newClientMessage);
            }
        }, 1000);
        return () => clearInterval(interval);
    }, [clientMessage]);

    // Listen for cache update events (dispatched by App component when localStorage updates)
    // This triggers re-evaluation of FFL map visibility without polling
    useEffect(() => {
        const handleCacheUpdate = () => {
            // Get current customer type from cookie or state
            const consumerTradeTypeCookie = getCookieValue("consumerTradeType");
            let currentCustomerType = 'b2C';
            if (consumerTradeTypeCookie) {
                try {
                    const parsed = JSON.parse(consumerTradeTypeCookie);
                    currentCustomerType = parsed.selectedTradeType || 'b2C';
                } catch (e) {}
            }

            // Immediately re-evaluate FFL map visibility with current customer type
            evaluateFflFromCache(currentCustomerType);

            // Also trigger state update for any other dependent effects
            setCacheVersion(v => v + 1);
        };

        window.addEventListener('echeckpoint-cache-updated', handleCacheUpdate);
        return () => window.removeEventListener('echeckpoint-cache-updated', handleCacheUpdate);
    }, []);

    const loadGoogleMapsScript = () => {
        // Check if Google Maps is already loaded or being loaded
        if (window.google && window.google.maps) {
            // Google Maps already loaded, just initialize our map
            if (typeof window.initMap === 'function') {
                window.initMap();
            }
            return;
        }

        // Check if another script is already loading Google Maps
        const existingScript = document.querySelector('script[src*="maps.googleapis.com"]');
        if (existingScript) {
            // Wait for existing script to load
            existingScript.addEventListener('load', () => {
                if (typeof window.initMap === 'function') {
                    window.initMap();
                }
            });
            return;
        }

        // Load Google Maps script
        const script = document.createElement("script");
        script.src = `https://maps.googleapis.com/maps/api/js?key=${window.eCheckpointParams?.google_maps_api_key || ''}&callback=initMap`;
        script.async = true;
        script.defer = true;
        document.body.appendChild(script);
    };

    window.initMap = () => {
        window.mapInstance = new google.maps.Map(document.getElementById("map"), {
            zoom: 12,
            center: { lat: 34.21721, lng: -119.04726 },
        });
    };

    window.updateMarkers = (locations) => {
        if (!window.mapInstance) {
            console.error("Map is not initialized yet.");
            return;
        }
        if (window.markersArray && window.markersArray.length > 0) {
            window.markersArray.forEach(marker => marker.setMap(null));
        }
        window.markersArray = [];

        locations.forEach(location => {
            const lat = parseFloat(location.premiseLat);
            const lng = parseFloat(location.premiseLon);
            if (isNaN(lat) || isNaN(lng)) {
                console.error("Invalid lat or lng for location:", location);
                return;
            }
            let marker = new google.maps.Marker({
                position: { lat, lng },
                map: window.mapInstance,
                title: location.licenseName,
            });

            let infoWindow = new google.maps.InfoWindow({
                content: `<strong>${location.licenseName}</strong><br>
						  ${location.premiseStreet}, ${location.premiseCity}, ${location.premiseState} ${location.premiseZipCode}`,
            });

            marker.addListener("click", () => {
                infoWindow.open(window.mapInstance, marker);

                const shippingAddress = {
                    address_1: location.premiseStreet || '',
                    address_2: '',
                    city: location.premiseCity || '',
                    company: location.businessName || location.licenseName || '',
                    state: location.premiseState || '',
                    postcode: location.premiseZipCode ? location.premiseZipCode.toString().slice(0, 5) : '',
                };

                const fieldMappings = {
                    'shipping-address_1': 'premiseStreet',
                    'shipping-address_2': '',
                    'shipping-city': 'premiseCity',
                    'shipping-state': 'premiseState',
                    'shipping-postcode': 'premiseZipCode',
                };

                Object.entries(fieldMappings).forEach(([fieldId, locationKey]) => {
                    const field = document.getElementById(fieldId);
                    if (field) {
                        field.value = locationKey ? location[locationKey] : '';
                        field.dispatchEvent(new Event('input', { bubbles: true }));
                    }
                });

                // **Updating the Company Field Properly**
                setTimeout(() => {
                    const companyField = document.getElementById("shipping-namespace-select-company");

                    if (companyField) {
                        companyField.value = shippingAddress.company;
                        companyField.dispatchEvent(new Event('input', { bubbles: true }));
                        companyField.dispatchEvent(new Event('change', { bubbles: true }));

                        //console.log("Updated company field to:", companyField.value);
                    } else {
                        console.error("Company field not found in the DOM.");
                    }

                    try {
                        const { dispatch } = window.wp.data;
                        if (window.wc && window.wc.blocksCheckout) {
                            const { setShippingAddress } = dispatch('wc/store/cart');
                            if (typeof setShippingAddress === 'function') {
                                setShippingAddress(shippingAddress);
                               // console.log("Updated WooCommerce store with company:", shippingAddress.company);
                            }
                        }
                    } catch (error) {
                        console.error("Error updating WooCommerce store:", error);
                    }

                    document.dispatchEvent(new CustomEvent('wc-shipping-address-update', {
                        detail: shippingAddress,
                        bubbles: true,
                    }));
                }, 500); // Add a slight delay to prevent React overwriting

            });

            window.markersArray.push(marker);
        });

        if (window.shippingAddressData) {
            const lat = parseFloat(window.shippingAddressData.addressLat);
            const lng = parseFloat(window.shippingAddressData.addressLng);
            if (!isNaN(lat) && !isNaN(lng)) {
                window.mapInstance.setCenter({ lat, lng });
                //console.log("Map center updated to shipping address:", lat, lng);
            } else {
                console.error("Invalid shipping address coordinates.");
            }
        }
    };

    // New useEffect to update selectedTradeType from the consumerTradeType cookie on mount
    useEffect(() => {
        function getCookieValue(name) {
            const cookie = document.cookie
                .split("; ")
                .find((row) => row.startsWith(`${name}=`));
            return cookie ? decodeURIComponent(cookie.split("=")[1]) : null;
        }
        const consumerTradeTypeCookie = getCookieValue("consumerTradeType");
        if (consumerTradeTypeCookie) {
            try {
                const consumerTradeType = JSON.parse(consumerTradeTypeCookie);
                setSelectedTradeType(consumerTradeType.selectedTradeType);
            } catch (error) {
                console.error("Error parsing consumerTradeType cookie:", error);
            }
        }
    }, []);

    // New polling useEffect to update selectedTradeType when the cookie changes
    useEffect(() => {
        const interval = setInterval(() => {
            function getCookieValue(name) {
                const cookie = document.cookie
                    .split("; ")
                    .find((row) => row.startsWith(`${name}=`));
                return cookie ? decodeURIComponent(cookie.split("=")[1]) : null;
            }
            const consumerTradeTypeCookie = getCookieValue("consumerTradeType");
            if (consumerTradeTypeCookie) {
                try {
                    const consumerTradeType = JSON.parse(consumerTradeTypeCookie);
                    if (consumerTradeType.selectedTradeType !== selectedTradeType) {
                        //console.log("consumerTradeType updated:", consumerTradeType.selectedTradeType);
                        setSelectedTradeType(consumerTradeType.selectedTradeType);
                    }
                } catch (error) {
                    console.error("Error parsing consumerTradeType cookie during poll:", error);
                }
            }
        }, 1000);
        return () => clearInterval(interval);
    }, [selectedTradeType]);

    useEffect(() => {
        loadGoogleMapsScript();

        // Evaluate FFL map visibility from localStorage cache based on current customer type
        // This uses the same logic as the App component for consistency
        const currentCustomerType = selectedTradeType || 'b2C';
        evaluateFflFromCache(currentCustomerType);

        const handleAddressUpdate = (event) => {
            try {
                const { detail } = event;
                if (detail && typeof detail === 'object') {
                    if (window.wc && window.wc.blocksCheckout && window.wp.data.dispatch('wc/store/cart')) {
                        const { setShippingAddress } = window.wp.data.dispatch('wc/store/cart');
                        if (typeof setShippingAddress === 'function') {
                            setShippingAddress(detail);
                        }
                    } else if (window.wp.data.dispatch('wc/store')) {
                        const { updateShippingAddress } = window.wp.data.dispatch('wc/store');
                        if (typeof updateShippingAddress === 'function') {
                            updateShippingAddress(detail);
                        }
                    }
                }
            } catch (error) {
                console.error('Error handling address update event:', error);
            }
        };

        document.addEventListener('wc-shipping-address-update', handleAddressUpdate);
        return () => {
            document.removeEventListener('wc-shipping-address-update', handleAddressUpdate);
        };
    }, [selectedTradeType, clientMessage, cacheVersion]); // cacheVersion triggers re-evaluation when cart changes

    return (
        <div id="map-container" className={containerClass} style={{ padding: "20px", backgroundColor: "#f0f0f0", textAlign: "center" }}>
            <h4>FFL Shipping Address Required</h4>
            <p>Please select a marker to change the shipping address to an active FFL address.</p>
            <div id="map" style={{ width: "100%", height: "300px" }}></div>
        </div>
    );
};



const App = () => {
    const [noticeData, setNoticeData] = useState(null);
    const [currentMessage, setCurrentMessage] = useState('');
    const [fflNoticeData, setFflNoticeData] = useState(null);
    const [currentFFLMessage, setCurrentFFLMessage] = useState('');
    const [selectedCustomerType, setSelectedCustomerType] = useState('b2C'); // Track selected customer type
    const isRecheckingCompliance = useRef(false); // Use ref for immediate update (state updates are async)
    const previousShippingAddress = useRef(null); // Track previous shipping address to detect changes

    const getCookie = (name) => {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
    };

    const setCookie = (name, value, days) => {
        const expires = new Date(Date.now() + days * 24 * 60 * 60 * 1000).toUTCString();
        document.cookie = `${name}=${encodeURIComponent(value)}; expires=${expires}; path=/`;
    };

    // Helper function to get effective trade type with b2C fallback
    const getEffectiveTradeType = (businessTradeTypes, selectedType) => {
        if (businessTradeTypes[selectedType]?.eligiblePurchaser) {
            return selectedType;
        }
        // Fallback to b2C if selected type not eligible
        if (selectedType !== 'b2C' && businessTradeTypes['b2C']?.eligiblePurchaser) {
            return 'b2C';
        }
        return null; // Not eligible for any type
    };

    // Helper to get current cart SKUs
    const getCartSkus = () => {
        try {
            const cartStore = wp.data.select('wc/store/cart');
            const cartData = cartStore?.getCartData();
            const cartItems = cartData?.items || [];
            const skus = cartItems.map(item => item.sku || item.id?.toString()).filter(Boolean).sort();
            return skus;
        } catch (e) {
            return [];
        }
    };

    // Check if cart has changed since cache was stored
    const isCartChanged = () => {
        const cachedResponse = localStorage.getItem('echeckpoint_api_response');
        const cachedSkus = localStorage.getItem('echeckpoint_cart_skus');

        // If no cached response, this is first load - not a cart change
        if (!cachedResponse || !cachedSkus) {
            return false;
        }

        const currentSkus = getCartSkus();
        const storedSkus = JSON.parse(cachedSkus);

        // If current cart is empty but we can't get data yet, wait
        if (currentSkus.length === 0 && storedSkus.length > 0) {
            // Check if cart store is actually loaded
            try {
                const cartStore = wp.data.select('wc/store/cart');
                const isResolving = cartStore?.isResolving?.('getCartData') || false;
                if (isResolving) {
                    return false;
                }
            } catch (e) {}
        }

        if (currentSkus.length !== storedSkus.length) {
            return true;
        }

        return currentSkus.some((sku, i) => sku !== storedSkus[i]);
    };

    // Clear cache if cart changed and trigger new compliance check
    const checkAndClearCacheIfCartChanged = async () => {
        // Skip if already rechecking
        if (isRecheckingCompliance.current) {
            return false;
        }

        if (isCartChanged()) {
            isRecheckingCompliance.current = true;

            localStorage.removeItem('echeckpoint_api_response');
            localStorage.removeItem('echeckpoint_cart_skus');

            // Clear old notices since they're for a different cart
            setNoticeData({ message: 'N/A', type: 'N/A' });
            setFflNoticeData({ message: 'N/A', type: 'N/A' });

            // Trigger PHP to make a new API call
            try {
                const formData = new URLSearchParams();
                formData.append('action', 'echeckpoint_recheck_compliance');
                formData.append('nonce', eCheckpointParams.nonce);

                await fetch(eCheckpointParams.ajax_url, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
                    },
                    body: formData.toString()
                });
            } catch (error) {
                console.error('Error triggering compliance recheck:', error);
            } finally {
                // Reset flag after a delay to allow new response to be fetched
                setTimeout(() => { isRecheckingCompliance.current = false; }, 3000);
            }

            return true;
        }
        return false;
    };

    // Helper to get product name from cart by SKU
    const getProductNameFromCart = (productId) => {
        try {
            const cartStore = wp.data.select('wc/store/cart');
            const cartItems = cartStore?.getCartData()?.items || [];
            for (const item of cartItems) {
                if (item.sku === productId || item.id?.toString() === productId) {
                    return item.name;
                }
            }
        } catch (e) {
            // Silently fail - cart data not available yet
        }
        return null;
    };

    // Format product for display - use name if available, otherwise fall back to ID
    const formatProduct = (productId) => {
        const name = getProductNameFromCart(productId);
        return name || productId;
    };

    // Evaluate messages based on cached API response and customer type
    const evaluateComplianceFromCache = (customerType) => {
        const cachedResponse = localStorage.getItem('echeckpoint_api_response');
        if (!cachedResponse) {
            return;
        }

        try {
            const response = JSON.parse(cachedResponse);
            const items = response?.modules?.regionalRestrictionsCheck?.items || [];

            let failedProducts = [];
            let conditionalProducts = [];
            let fflRequiredProducts = [];

            items.forEach(item => {
                const businessTradeTypes = item.businessTradeTypes;
                const productId = item.response?.productID;
                const result = item.response?.result;
                const isRegulated = item.regulatedProduct === true; // Explicitly check for true

                // Skip non-regulated products - they have no restrictions
                if (!isRegulated) {
                    return;
                }

                // Skip if no business trade types data
                if (!businessTradeTypes) {
                    return;
                }

                const effectiveType = getEffectiveTradeType(businessTradeTypes, customerType);

                // Get FFL requirement for the effective trade type - must be explicitly true
                const fflRequired = effectiveType && businessTradeTypes[effectiveType]?.fflRequired === true;

                if (effectiveType === null) {
                    // Not eligible for any customer type - product is blocked
                    failedProducts.push(formatProduct(productId));
                } else if (result === 3) {
                    // Result 3 = Additional requirements needed
                    conditionalProducts.push(formatProduct(productId));
                    // Only add to FFL list if fflRequired is explicitly true
                    if (fflRequired) {
                        fflRequiredProducts.push(formatProduct(productId));
                    }
                }
                // Result 1 = Passed - no action needed
            });

            // Update client_message based on evaluation
            if (failedProducts.length > 0) {
                const productList = failedProducts.join(', ');
                const blockedMessage = window.eCheckpointParams?.blocked_products_message || 'The following items cannot be shipped to your location due to federal, state, or local regulations. Please remove these items from your cart to proceed with checkout:';
                const message = blockedMessage + ' ' + productList;
                setNoticeData({ message, type: 'error' });
                setValidationErrors({
                    'billing-first-name': { message: 'Please resolve compliance message.', hidden: true },
                    'billing-last-name': { message: 'Please resolve compliance message.', hidden: true },
                });
            } else if (conditionalProducts.length > 0) {
                const message = window.eCheckpointParams?.additional_info_message || 'To complete this purchase, additional information will be required after checkout.';
                setNoticeData({ message, type: 'info' });
                const store = dispatch(VALIDATION_STORE_KEY);
                store.clearValidationErrors(['billing-first-name', 'billing-last-name']);
            } else {
                setNoticeData({ message: 'N/A', type: 'N/A' });
                const store = dispatch(VALIDATION_STORE_KEY);
                store.clearValidationErrors(['billing-first-name', 'billing-last-name']);
            }

            // Update FFL message based on evaluation
            // Hide FFL notice when there are blocked products - customer needs to fix cart first
            if (failedProducts.length > 0) {
                setFflNoticeData({ message: 'N/A', type: 'N/A' });
            } else if (fflRequiredProducts.length > 0) {
                const productList = fflRequiredProducts.join(', ');
                const fflBaseMessage = window.eCheckpointParams?.ffl_required_message || 'The following product(s) must be shipped to a Licensed Federal Firearms (FFL) dealer. Please select an FFL from the map below:';
                const fflMessage = {
                    message: fflBaseMessage + ' ' + productList,
                    type: 'error'
                };
                setFflNoticeData(fflMessage);
            } else {
                setFflNoticeData({ message: 'N/A', type: 'N/A' });
            }

        } catch (error) {
            console.error('Error evaluating cached response:', error);
        }
    };

    // Listen for select field change - evaluate from localStorage, no PHP call needed
    const handleCustomerTypeChange = (event) => {
        const newCustomerType = event.target.value;

        // Update React state to preserve selection across re-renders
        setSelectedCustomerType(newCustomerType);

        // Update the consumerTradeType cookie for PHP to use on next address change
        const cookieValue = JSON.stringify({ selectedTradeType: newCustomerType });
        document.cookie = `consumerTradeType=${encodeURIComponent(cookieValue)}; path=/; max-age=3600`;

        // Evaluate messages from cached API response - no PHP call needed
        evaluateComplianceFromCache(newCustomerType);
    };

    // Sync dropdown value with React state after each render
    useEffect(() => {
        const customerTypeSelect = document.getElementById("contact-namespace-select-tradetype");
        if (customerTypeSelect && customerTypeSelect.value !== selectedCustomerType) {
            customerTypeSelect.value = selectedCustomerType;
        }
    }, [selectedCustomerType]);

    // Check for cart changes immediately on mount
    useEffect(() => {
        checkAndClearCacheIfCartChanged();
    }, []);

    // Subscribe to WooCommerce store for shipping address changes
    useEffect(() => {
        // Debounce timer for address changes
        let addressDebounceTimer = null;
        let unsubscribe = null;

        try {
            const { subscribe, select } = wp.data;

            unsubscribe = subscribe(() => {
                try {
                    const cartStore = select('wc/store/cart');
                    if (!cartStore) return;

                    const cartData = cartStore.getCartData?.();
                    if (!cartData) return;

                    const shippingAddress = cartData.shippingAddress;
                    if (!shippingAddress) return;

                    // Build address string for comparison
                    const currentAddressKey = [
                        shippingAddress.address_1,
                        shippingAddress.city,
                        shippingAddress.state,
                        shippingAddress.postcode
                    ].join('|');

                    // Skip if address hasn't changed
                    if (previousShippingAddress.current === currentAddressKey) return;

                    // Skip initial load (when previousShippingAddress is null)
                    if (previousShippingAddress.current === null) {
                        previousShippingAddress.current = currentAddressKey;
                        return;
                    }

                    // Address changed - update ref and trigger compliance check
                    previousShippingAddress.current = currentAddressKey;

                    // Debounce the compliance check (1000ms)
                    clearTimeout(addressDebounceTimer);
                    addressDebounceTimer = setTimeout(() => {
                        // Only trigger if we have enough address data
                        if (shippingAddress.address_1 && shippingAddress.city &&
                            shippingAddress.state && shippingAddress.postcode) {

                            // Trigger compliance recheck via AJAX
                            const formData = new URLSearchParams();
                            formData.append('action', 'echeckpoint_recheck_compliance');
                            formData.append('nonce', window.eCheckpointParams?.nonce || '');
                            formData.append('customer_type', selectedCustomerType);

                            fetch(window.eCheckpointParams?.ajax_url || '/wp-admin/admin-ajax.php', {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                                body: formData.toString()
                            });
                        }
                    }, 1000);
                } catch (e) {
                    // Silently ignore errors in subscription callback
                }
            });
        } catch (e) {
            // wp.data not available yet
        }

        return () => {
            if (unsubscribe) unsubscribe();
            clearTimeout(addressDebounceTimer);
        };
    }, [selectedCustomerType]);

    useEffect(() => {
        const customerTypeSelect = document.getElementById("contact-namespace-select-tradetype");

        if (customerTypeSelect) {
            customerTypeSelect.addEventListener("change", handleCustomerTypeChange);
        }

        // Only use cookies for initial load BEFORE we have localStorage cache
        // Once localStorage is populated, all messaging comes from there
        const checkCookieChange = () => {
            // Skip if we're rechecking compliance (cart changed)
            if (isRecheckingCompliance.current) {
                return;
            }

            // Skip cookie-based messaging if we have localStorage cache
            // localStorage is the source of truth once populated
            const cachedResponse = localStorage.getItem('echeckpoint_api_response');
            if (cachedResponse) {
                return; // Don't overwrite localStorage-based messaging with cookie values
            }

            const clientMessageCookie = getCookie('client_message');
            if (clientMessageCookie && clientMessageCookie !== currentMessage) {
                try {
                    const decodedMessage = decodeURIComponent(clientMessageCookie);
                    const parsedMessage = JSON.parse(decodedMessage);

                    setNoticeData({ ...parsedMessage });
                    setCurrentMessage(clientMessageCookie);

                    if (parsedMessage.type === 'error') {
                        setValidationErrors({
                            'billing-first-name': { message: 'Please resolve compliance message.', hidden: true },
                            'billing-last-name': { message: 'Please resolve compliance message.', hidden: true },
                        });
                    } else {
                        const store = dispatch(VALIDATION_STORE_KEY);
                        store.clearValidationErrors(['billing-first-name', 'billing-last-name']);
                    }
                } catch (error) {
                    console.error('Error parsing client_message cookie:', error);
                }
            }

            const fflMessageCookie = getCookie('ffl_message');
            if (fflMessageCookie && fflMessageCookie !== currentFFLMessage) {
                try {
                    const decodedFFLMessage = decodeURIComponent(fflMessageCookie);
                    const parsedFFLMessage = JSON.parse(decodedFFLMessage);

                    setFflNoticeData({ ...parsedFFLMessage });
                    setCurrentFFLMessage(fflMessageCookie);
                } catch (error) {
                    console.error('Error parsing ffl_message cookie:', error);
                }
            }
        };

        const interval = setInterval(() => {
            // Check if cart has changed - if so, clear cache to trigger new API call
            checkAndClearCacheIfCartChanged();

            // Fetch API response and cache in localStorage
            // Cookie check only runs if localStorage not yet populated
            checkCookieChange();
            fetchFFLResponse();
        }, 1000);

        let previousCookieValue = null;
        const fetchFFLResponse = async () => {
            // Skip fetching if we're rechecking compliance (cart changed, waiting for new data)
            if (isRecheckingCompliance.current) {
                return;
            }

            try {
                // Replace 'currentCookie' with the appropriate cookie name if needed.
                const currentCookie = getCookie('ffl_response_key');
                if (currentCookie !== previousCookieValue) {
                    previousCookieValue = currentCookie;
                  //  console.log("ffl_response_key cookie changed:", currentCookie);
                    if (!currentCookie) {
                        return;
                    } else {
                        const formData = new URLSearchParams();
                        formData.append('action', 'get_ffl_response'); // Must match your AJAX handler
                        formData.append('key', currentCookie);
                        formData.append('nonce', eCheckpointParams.nonce); // eCheckpointParams must be defined globally

                        const response = await fetch(eCheckpointParams.ajax_url, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
                            },
                            body: formData.toString()
                        });

                        const data = await response.json();

                        if (data.success) {
                            // console.log("Retrieved FFL response via AJAX:", data.data);

                            // Cache the API response and current cart SKUs in localStorage
                            localStorage.setItem('echeckpoint_api_response', JSON.stringify(data.data));
                            localStorage.setItem('echeckpoint_cart_skus', JSON.stringify(getCartSkus()));

                            // Notify TestDivComponent that cache has been updated
                            window.dispatchEvent(new CustomEvent('echeckpoint-cache-updated'));

                            // Evaluate initial compliance based on current customer type
                            const customerTypeSelect = document.getElementById("contact-namespace-select-tradetype");
                            const currentCustomerType = customerTypeSelect?.value || 'b2C';
                            evaluateComplianceFromCache(currentCustomerType);

                            window.shippingAddressData =
                                data.data.modules.addressValidationCheck.items[1].response.address;
                            const licensingLocations = processMap(data.data);
                            if (licensingLocations.length > 0 && window.updateMarkers) {
                                window.updateMarkers(licensingLocations);
                            }
                        } else {
                            console.error("Error retrieving FFL response:", data.data);
                        }
                    }
                }

            } catch (error) {
                console.error("AJAX error while retrieving FFL response:", error);
            }
        };

        const processMap = (data) => {
            if (
                data.modules &&
                data.modules.regionalRestrictionsCheck &&
                Array.isArray(data.modules.regionalRestrictionsCheck.licensing) &&
                data.modules.regionalRestrictionsCheck.licensing.length > 0 &&
                Array.isArray(data.modules.regionalRestrictionsCheck.licensing[0].availableLicenses)
            ) {
                return data.modules.regionalRestrictionsCheck.licensing[0].availableLicenses;
            } else {
                console.error("Data structure is not as expected:", data);
                return [];
            }
        };

        return () => {
            clearInterval(interval);
            if (customerTypeSelect) {
                customerTypeSelect.removeEventListener("change", handleCustomerTypeChange);
            }
        };
    }, [currentMessage, currentFFLMessage]);

    return (
        <div>
            {noticeData && <NoticeComponent data={noticeData} noticeId="client-message-id" />}
            {fflNoticeData && <NoticeComponent data={fflNoticeData} noticeId="ffl-notice-id" />}
        </div>
    );
};

// Wrap the test component inside the ExperimentalOrderMeta slot.
const renderTestDiv = () => {
    if (!window.location.pathname.includes('/checkout')) return null;

    return (
        <ExperimentalOrderShippingPackages>
            <TestDivComponent />
        </ExperimentalOrderShippingPackages>
    );
};
registerPlugin('echeckpoint', {
    render: App,
    scope: 'woocommerce-checkout',
});

registerPlugin('test-div-slot', {
    render: renderTestDiv,
    scope: 'woocommerce-checkout',
});