///////////////////////////////////////////
// PRE-ORDER CHECK JS
// echeckpoint_pre-order-check.js
///////////////////////////////////////////

// Debug mode - enable with ?echeckpoint_debug=1 in URL or localStorage.setItem('echeckpoint_debug', '1')
// Defined globally so all functions can access it
var echeckpointDebugMode = (new URLSearchParams(window.location.search).get('echeckpoint_debug') === '1') ||
                           (localStorage.getItem('echeckpoint_debug') === '1');

// Debug logging helper - only logs when debug mode is enabled
// Defined globally so all functions can access it
function debugLog() {
    if (echeckpointDebugMode) {
        console.log.apply(console, arguments);
    }
}

jQuery(function ($) {
    var debounceTimer;
    var isUpdating = false;
    var complianceCheckTriggered = false;
    var freshNonceFetched = false;

    // Ensure eCheckpointParams exists (handles aggressive page caching scenarios)
    // This creates a fallback if wp_localize_script didn't run or was cached incorrectly
    if (typeof window.eCheckpointParams === 'undefined') {
        // Determine ajax_url from the page - WordPress always has this available
        var ajaxUrl = '/wp-admin/admin-ajax.php';

        // Try to get the site URL from known WordPress elements
        var siteUrl = '';
        var homeLink = document.querySelector('link[rel="home"]');
        if (homeLink) {
            siteUrl = homeLink.href.replace(/\/$/, '');
        } else {
            // Fallback: extract from current page URL
            siteUrl = window.location.protocol + '//' + window.location.host;
        }

        window.eCheckpointParams = {
            ajax_url: siteUrl + ajaxUrl,
            nonce: '', // Will be fetched via AJAX
            google_maps_api_key: ''
        };
        debugLog('eCheckpoint: Initialized params from fallback, will fetch nonce via AJAX');
    }


    // Store previous values for comparison
    var previousValues = {};

    // Function to compare field values
    function hasFieldValueChanged(field) {
        var fieldName = field.attr('name');
        var currentValue = field.val();

        if (previousValues[fieldName] !== currentValue) {
            previousValues[fieldName] = currentValue; // Update the stored value
            return true; // Value has changed
        }
        return false; // Value is the same, no need to trigger compliance check
    }

    // Fetch a fresh nonce via AJAX (handles cached pages and fallback initialization)
    function fetchFreshNonce(callback) {
        // Skip if already fetched AND we have a valid nonce
        if (freshNonceFetched && eCheckpointParams.nonce) {
            if (callback) callback();
            return;
        }

        $.ajax({
            url: eCheckpointParams.ajax_url,
            type: 'POST',
            data: {
                action: 'echeckpoint_get_fresh_nonce'
            },
            success: function (response) {
                if (response.success && response.data.nonce) {
                    // Update the global nonce value
                    eCheckpointParams.nonce = response.data.nonce;
                    freshNonceFetched = true;
                    debugLog('eCheckpoint: Fresh nonce obtained');

                    // Update existing hidden field if present
                    if ($('input[name="echeckpoint_nonce"]').length) {
                        $('input[name="echeckpoint_nonce"]').val(response.data.nonce);
                    }

                    if (callback) callback();
                } else {
                    console.error('eCheckpoint: Failed to get nonce from response');
                    if (callback) callback();
                }
            },
            error: function (xhr) {
                console.error('eCheckpoint: Error fetching fresh nonce:', xhr.responseText);
                if (callback) callback(); // Continue anyway
            }
        });
    }

    function addNonceToCheckoutForm() {
        if ($('form.checkout').length > 0) {
            // Add or update nonce as hidden input
            if (!$('input[name="echeckpoint_nonce"]').length) {
                $('<input>').attr({
                    type: 'hidden',
                    id: 'echeckpoint_nonce',
                    name: 'echeckpoint_nonce',
                    value: eCheckpointParams.nonce
                }).appendTo('form.checkout');
            } else {
                // Update existing field with current nonce value
                $('input[name="echeckpoint_nonce"]').val(eCheckpointParams.nonce);
            }
        }
    }

    // Function to trigger the compliance check
    function triggerUpdateCheckout() {
        if (!isUpdating && !complianceCheckTriggered) {
            clearTimeout(debounceTimer);
            debounceTimer = setTimeout(function () {
                // console.log('Checkout fields changed, triggering update_checkout');
                addNonceToCheckoutForm(); // Ensure nonce is added before the update_checkout event
                isUpdating = true;
                complianceCheckTriggered = true; // Mark as triggered
                $('body').trigger('update_checkout');
            }, 1000);
        }
    }

    // Handle multiple form field changes at once
    function handleFormChange() {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(function () {
            // console.log('Form fields changed, triggering single update_checkout');
            triggerUpdateCheckout();
        }, 1000); // Debounce for 1000ms
    }

    // Listen for changes on billing and shipping fields in the WooCommerce form
    $('form.checkout').on('change', 'input[name*="billing"], input[name*="shipping"], select[name*="billing"], select[name*="shipping"], textarea[name*="billing"], textarea[name*="shipping"]', function () {
        var fieldChanged = hasFieldValueChanged($(this));
        if (fieldChanged) {
            // User is actively changing address - JS can take over notice control again
            if (phpOwnsNotices) {
                debugLog('eCheckpoint: User changed address, JS resuming notice control');
                phpOwnsNotices = false;
            }
            handleFormChange();
        }
    });

    // When Place Order is clicked, PHP takes over all notice generation
    // JS will not display any notices until user actively changes an address field
    $('form.checkout').on('checkout_place_order', function() {
        debugLog('eCheckpoint: Place Order clicked, PHP now owns notices');
        phpOwnsNotices = true;
        usingLocalStorageCache = false;

        // Clear all eCheckpoint notices - PHP will generate new ones
        $('.echeckpoint-notice').remove();

        // Keep localStorage cache intact - phpOwnsNotices flag prevents it from being used
        // This allows instant customer type switching after Place Order fails without new API call

        // Clear cookies so polling doesn't re-trigger JS notices
        document.cookie = 'client_message=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
        document.cookie = 'ffl_response_key=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
    });

    // Fetch fresh nonce and add to form on initial page load
    $(document).ready(function () {
        fetchFreshNonce(function() {
            addNonceToCheckoutForm();
        });
    });

    // Reset the isUpdating flag after the checkout is updated
    $(document.body).on('updated_checkout', function () {
        isUpdating = false;
        complianceCheckTriggered = false;
        addNonceToCheckoutForm();
    });

});

addEventListener("beforeunload", () => {
    // Clear all eCheckpoint intervals to prevent memory leaks
    if (typeof echeckpointIntervals !== 'undefined') {
        echeckpointIntervals.forEach(id => clearInterval(id));
    }

    // Clear eCheckpoint cookies
    document.cookie = 'client_message=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
    document.cookie = 'ffl_message=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
    document.cookie = 'checkout_block=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
    document.cookie = 'ffl_requirements=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;'
    document.cookie = 'consumerTradeType=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
    document.cookie = 'ffl_response_key=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';

    // Clear eCheckpoint localStorage and reset flags
    localStorage.removeItem("validTradeTypes");
    localStorage.removeItem("echeckpoint_api_response");
    localStorage.removeItem("echeckpoint_cart_skus");
    usingLocalStorageCache = false;
    phpOwnsNotices = false;
});


var previousTradeTypeVal = "b2C"; //Default B2C

// Store interval IDs for cleanup
var echeckpointIntervals = [];

// Flag to prevent cookie polling from overriding client-side cache evaluation
// When true, cookie polling is disabled because we're using localStorage cache
var usingLocalStorageCache = false;

// When true, PHP owns all notices via wc_add_notice() - JS will not display any
// This is set when Place Order is clicked, cleared when user changes address
var phpOwnsNotices = false;

/**
 * Display a notice banner before the checkout form.
 * Consolidated function used by all notice systems.
 *
 * @param {string} message  The notice text.
 * @param {string} [type]   The type of notice: 'info', 'error', or 'success'.
 */
function displayNotice(message, type) {
    // If PHP owns notices (after Place Order), don't display JS notices
    if (phpOwnsNotices) {
        debugLog('eCheckpoint: PHP owns notices, skipping displayNotice');
        return;
    }

    type = type || 'info';
    // Remove only eCheckpoint-generated notices to avoid removing WooCommerce system notices
    jQuery('.echeckpoint-notice').not('.echeckpoint-ffl-notice').remove();

    // Define the notice class based on the type - add echeckpoint-notice for identification
    var noticeClass = 'wc-block-components-notice-banner is-' + type + ' echeckpoint-notice';

    // WooCommerce info icon SVG (circle with "i")
    var iconSVG = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false"><path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z"></path></svg>';

    // Create the notice HTML with proper WooCommerce Blocks structure
    var noticeHTML = '<div class="' + noticeClass + '" tabindex="-1">' +
        iconSVG +
        '<div class="wc-block-components-notice-banner__content">' + message + '</div>' +
        '</div>';

    // Inject the notice before the checkout form
    jQuery('form.checkout').before(noticeHTML);
}

/**
 * Helper function to get effective trade type with b2C fallback.
 * If selected type is not eligible, falls back to b2C rules.
 *
 * @param {Object} businessTradeTypes - The trade types object from API
 * @param {string} selectedType - The currently selected customer type
 * @returns {string|null} The effective trade type, or null if not eligible
 */
function getEffectiveTradeType(businessTradeTypes, selectedType) {
    if (businessTradeTypes[selectedType] && businessTradeTypes[selectedType].eligiblePurchaser) {
        return selectedType;
    }
    // Fallback to b2C if selected type not eligible
    if (selectedType !== 'b2C' && businessTradeTypes['b2C'] && businessTradeTypes['b2C'].eligiblePurchaser) {
        return 'b2C';
    }
    return null; // Not eligible for any type
}

/**
 * Build a map of SKU -> Product Name from WooCommerce checkout data.
 */
var productSkuToNameMap = null;

function buildProductSkuMap() {
    if (productSkuToNameMap !== null) {
        return productSkuToNameMap;
    }

    productSkuToNameMap = {};

    // Method 1: Check WooCommerce cart data if available
    if (typeof wc_cart_params !== 'undefined' && wc_cart_params.cart_contents) {
        try {
            var cartContents = JSON.parse(wc_cart_params.cart_contents);
            for (var key in cartContents) {
                var item = cartContents[key];
                if (item.sku && item.name) {
                    productSkuToNameMap[item.sku] = item.name;
                }
            }
        } catch (e) {}
    }

    // Method 2: Check echeckpoint localized data
    if (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.cart_products) {
        var products = eCheckpointParams.cart_products;
        for (var i = 0; i < products.length; i++) {
            if (products[i].sku && products[i].name) {
                productSkuToNameMap[products[i].sku] = products[i].name;
            }
        }
    }

    // Method 3: Parse from checkout table with all possible selectors
    jQuery('.woocommerce-checkout-review-order-table .cart_item, .shop_table .cart_item').each(function() {
        var $row = jQuery(this);
        var $nameCell = $row.find('.product-name, td.product-name');

        var fullText = $nameCell.text().trim();
        var productName = fullText.split('×')[0].trim();

        // Try multiple SKU locations
        var sku = $row.attr('data-product_sku') ||
                  $row.attr('data-sku') ||
                  $row.find('[data-product_sku]').attr('data-product_sku') ||
                  $row.find('[data-sku]').attr('data-sku') ||
                  $row.find('.sku').text().trim() ||
                  $row.find('.product-sku').text().trim();

        if (sku && productName) {
            productSkuToNameMap[sku] = productName;
        }
    });

    debugLog('eCheckpoint: Built product SKU map:', productSkuToNameMap);
    return productSkuToNameMap;
}

/**
 * Format product ID (SKU) for display.
 * Looks up product name from SKU map, falls back to SKU.
 *
 * @param {string} productId - The SKU/product ID
 * @returns {string} Product name or SKU
 */
function formatProductForDisplay(productId) {
    var map = buildProductSkuMap();

    if (map[productId]) {
        return map[productId];
    }

    // Fallback: return the SKU
    return productId;
}

/**
 * Evaluate compliance from cached API response in localStorage.
 * Updates notices without making a server call.
 *
 * @param {string} customerType - The customer type to evaluate (b2C, b2B, b2G)
 */
function evaluateComplianceFromCache(customerType) {
    // If PHP owns notices (after Place Order), don't evaluate cache for JS notices
    if (phpOwnsNotices) {
        debugLog('eCheckpoint: PHP owns notices, skipping cache evaluation');
        return;
    }

    var cachedResponse = localStorage.getItem('echeckpoint_api_response');
    if (!cachedResponse) {
        debugLog('eCheckpoint: No cached API response found');
        return;
    }

    try {
        var response = JSON.parse(cachedResponse);
        var items = (response.modules && response.modules.regionalRestrictionsCheck &&
                     response.modules.regionalRestrictionsCheck.items) || [];

        // Check if FFL licensing requirement has been satisfied (FFL dealer selected)
        var fflProvided = false;
        if (response.modules && response.modules.regionalRestrictionsCheck &&
            response.modules.regionalRestrictionsCheck.licensing) {
            var licensing = response.modules.regionalRestrictionsCheck.licensing;
            licensing.forEach(function(license) {
                if (license.name === 'FFL' && license.provided === true) {
                    fflProvided = true;
                }
            });
        }
        debugLog('eCheckpoint: FFL provided (satisfied):', fflProvided);

        var failedProducts = [];
        var conditionalProducts = [];
        var fflRequiredProducts = [];

        items.forEach(function(item) {
            var businessTradeTypes = item.businessTradeTypes;
            var productId = item.response && item.response.productID;
            var result = item.response && item.response.result;
            var isRegulated = item.regulatedProduct === true;

            // Skip non-regulated products
            if (!isRegulated) {
                return;
            }

            // Skip if no business trade types data
            if (!businessTradeTypes) {
                return;
            }

            var effectiveType = getEffectiveTradeType(businessTradeTypes, customerType);
            var fflRequired = effectiveType && businessTradeTypes[effectiveType] &&
                              businessTradeTypes[effectiveType].fflRequired === true;

            // Debug logging
            debugLog('eCheckpoint: Product', productId,
                        '- customerType:', customerType,
                        '- effectiveType:', effectiveType,
                        '- result:', result,
                        '- fflRequired:', fflRequired,
                        '- fflProvided:', fflProvided,
                        '- businessTradeTypes:', JSON.stringify(businessTradeTypes));

            if (effectiveType === null) {
                // Not eligible for any customer type - product is blocked
                failedProducts.push(formatProductForDisplay(productId));
            } else if (result === 3) {
                // Result 3 = Additional requirements needed
                conditionalProducts.push(formatProductForDisplay(productId));
                // Only show FFL requirement if required AND not yet provided
                if (fflRequired && !fflProvided) {
                    fflRequiredProducts.push(formatProductForDisplay(productId));
                }
            }
        });

        debugLog('eCheckpoint: Cache evaluation for', customerType, '- Failed:', failedProducts,
                    'Conditional:', conditionalProducts,
                    'FFL Required:', fflRequiredProducts);

        // Update notices based on evaluation
        updateNoticesFromCacheEvaluation(failedProducts, conditionalProducts, fflRequiredProducts);

        // Enable localStorage mode - this disables cookie polling entirely
        usingLocalStorageCache = true;
        debugLog('eCheckpoint: Set usingLocalStorageCache = true');

    } catch (error) {
        console.error('eCheckpoint: Error evaluating cached response:', error);
    }
}

/**
 * Update Classic checkout notices based on cache evaluation results.
 *
 * @param {Array} failedProducts - Products that cannot be purchased
 * @param {Array} conditionalProducts - Products with conditional requirements
 * @param {Array} fflRequiredProducts - Products requiring FFL
 */
function updateNoticesFromCacheEvaluation(failedProducts, conditionalProducts, fflRequiredProducts) {
    // Remove eCheckpoint-generated JS notices before adding new ones
    jQuery('.echeckpoint-notice').not('.echeckpoint-ffl-notice').remove();

    // WooCommerce info icon SVG (circle with "i")
    var iconSVG = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false"><path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z"></path></svg>';

    var message = '';
    var type = 'N/A';

    if (failedProducts.length > 0) {
        var productList = failedProducts.join(', ');
        var blockedMessage = (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.blocked_products_message) ? eCheckpointParams.blocked_products_message : 'The following items cannot be shipped to your location due to federal, state, or local regulations. Please remove these items from your cart to proceed with checkout:';
        message = blockedMessage + ' ' + productList;
        type = 'error';
        var noticeHTML = '<div class="wc-block-components-notice-banner is-error echeckpoint-notice" tabindex="-1">' +
            iconSVG + '<div class="wc-block-components-notice-banner__content">' + message + '</div></div>';
        jQuery('form.checkout').before(noticeHTML);
    } else if (conditionalProducts.length > 0) {
        message = (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.additional_info_message) ? eCheckpointParams.additional_info_message : 'To complete this purchase, additional information will be required after checkout.';
        type = 'info';
        var noticeHTML = '<div class="wc-block-components-notice-banner is-info echeckpoint-notice" tabindex="-1">' +
            iconSVG + '<div class="wc-block-components-notice-banner__content">' + message + '</div></div>';
        jQuery('form.checkout').before(noticeHTML);
    }

    // Update the client_message cookie to match our evaluation
    // This prevents the polling interval from overwriting with stale data
    var cookieValue = JSON.stringify({ message: message, type: type });
    document.cookie = 'client_message=' + encodeURIComponent(cookieValue) + '; path=/; max-age=1200';

    // Handle FFL requirement notice and map display (only if FFL map is enabled)
    var showFflMap = (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.show_ffl_map !== false);
    var mapContainer = document.getElementById('map-container');

    if (fflRequiredProducts.length > 0 && showFflMap) {
        // Show FFL requirement message with comma-separated list
        var fflProductList = fflRequiredProducts.join(', ');
        var fflBaseMessage = (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.ffl_required_message) ? eCheckpointParams.ffl_required_message : 'The following product(s) must be shipped to a Licensed Federal Firearms (FFL) dealer. Please select an FFL from the map below:';
        var fflMessage = fflBaseMessage + ' ' + fflProductList;
        var fflNoticeHTML = '<div class="wc-block-components-notice-banner is-error echeckpoint-ffl-notice" tabindex="-1">' +
            iconSVG + '<div class="wc-block-components-notice-banner__content">' + fflMessage + '</div></div>';

        // Remove any existing FFL notices (both JS and PHP generated)
        jQuery('.echeckpoint-ffl-notice').remove();
        // Also remove PHP-generated FFL notices (contain "FFL" text)
        jQuery('.woocommerce-message, .woocommerce-info, .woocommerce-error, .wc-block-components-notice-banner').each(function() {
            var text = jQuery(this).text();
            if (text.indexOf('FFL') !== -1 && text.indexOf('Federal Firearms') !== -1) {
                jQuery(this).remove();
            }
        });
        jQuery('form.checkout').before(fflNoticeHTML);

        // Update ffl_message cookie for consistency
        var fflCookieValue = JSON.stringify({ message: fflMessage, type: 'error' });
        document.cookie = 'ffl_message=' + encodeURIComponent(fflCookieValue) + '; path=/; max-age=1200';

        if (mapContainer) {
            mapContainer.style.display = 'block';
        }
        var shipCheckbox = document.getElementById('ship-to-different-address-checkbox');
        if (shipCheckbox) {
            shipCheckbox.checked = true;
        }
        var shippingFields = document.getElementsByClassName('shipping_address')[0];
        if (shippingFields) {
            shippingFields.style.display = 'block';
        }
    } else {
        // Remove FFL notice and hide map
        jQuery('.echeckpoint-ffl-notice').remove();
        document.cookie = 'ffl_message=' + encodeURIComponent(JSON.stringify({ message: '', type: 'N/A' })) + '; path=/; max-age=1200';

        if (mapContainer) {
            mapContainer.style.display = 'none';
        }
    }
}

/**
 * Helper function to check if an element has a child with a class containing a specific substring.
 * @param {Element} element  The parent element to search within.
 * @param {string} text      The substring to look for in class names.
 * @returns {boolean}        True if a matching child is found, false otherwise.
 */
function hasChildWithClassContaining(element, text) {
    if (!element) return false;
    return Array.from(element.querySelectorAll('*')).some(child => {
        return typeof child.className === 'string' && child.className.includes(text);
    });
}

echeckpointIntervals.push(setInterval(() => {
    const noticeCheckoutElement = document.querySelector('.woocommerce-NoticeGroup.woocommerce-NoticeGroup-checkout');
    const noticeUpdateElement = document.querySelector('.woocommerce-NoticeGroup.woocommerce-NoticeGroup-updateOrderReview');

    // Ensure tradeTypeVal handles null, empty string, or undefined properly
    var selectElement = document.getElementById('contact-namespace-select-tradetype') ?? document.getElementById('namespace_select_tradetype');

    // Validate trade type against known values
    var validTradeTypes = ['b2C', 'b2B', 'b2G'];
    var rawTradeTypeVal = (selectElement && selectElement.value) ? selectElement.value : 'b2C';
    var tradeTypeVal = validTradeTypes.includes(rawTradeTypeVal) ? rawTradeTypeVal : 'b2C';

    if (tradeTypeVal !== previousTradeTypeVal) {
        previousTradeTypeVal = tradeTypeVal;
        updateTradeType(tradeTypeVal);
    }

    // Check if the "checkout_block" cookie exists and is false
    const checkoutBlockCookie = getCookie('checkout_block');
    if (checkoutBlockCookie === 'false') {
        // Define your elements
        const noticeCheckoutElement = document.querySelector('.notice-checkout'); // Adjust the selector
        const noticeUpdateElement = document.querySelector('.notice-update'); // Adjust the selector

        // Check if the elements have any child elements with class names containing "error" or "info"
        // Uses the global hasChildWithClassContaining function defined above
        const hasErrorChildCheckout = hasChildWithClassContaining(noticeCheckoutElement, 'error');
        const hasErrorChildUpdate = hasChildWithClassContaining(noticeUpdateElement, 'error');
        const hasInfoChildCheckout = hasChildWithClassContaining(noticeCheckoutElement, 'info');
        const hasInfoChildUpdate = hasChildWithClassContaining(noticeUpdateElement, 'info');

        // Case: Both have an error, and both have an info - remove is-info child from noticeUpdateElement
        if (hasErrorChildCheckout && hasErrorChildUpdate && hasInfoChildCheckout && hasInfoChildUpdate && noticeUpdateElement) {
            const infoChild = Array.from(noticeUpdateElement.querySelectorAll('*')).find(child => child.className.includes('info'));
            if (infoChild) {
               // console.log('Removing is-info child from noticeUpdateElement');
                infoChild.remove();
            }
        } else if (hasErrorChildUpdate) {
            if (noticeCheckoutElement) {
             //   console.log('Removing noticeCheckoutElement because noticeUpdateElement has an error child');
                noticeCheckoutElement.remove();
            }
        } else if (hasErrorChildCheckout) {
            if (noticeUpdateElement) {
             //   console.log('Removing noticeUpdateElement because noticeCheckoutElement has an error child');
                noticeUpdateElement.remove();
            }
        } else if (noticeCheckoutElement && noticeUpdateElement && !hasErrorChildUpdate && !hasErrorChildCheckout) {
           // console.log('Removing noticeCheckoutElement because neither element has an error child');
            noticeCheckoutElement.remove();
        }
    }
}, 1000)); // Check every 1000ms (1 second)


/**
 * Function to get a cookie's value by name.
 * This returns the raw, decoded string stored for that cookie, or null if not found.
 *
 * NOTE: If the cookie value is JSON-encoded (as set by `setCookie` above),
 * you can parse it like:
 *   const raw = getCookie('someCookie');
 *   const asObj = raw ? JSON.parse(raw) : null;
 *
 * @param {string} name  The name of the cookie to retrieve.
 * @returns {string|null} Returns the decoded string if found, else null.
 */
function getCookie(name) {
    try {
        const cookies = document.cookie.split('; ').reduce((acc, cookie) => {
            const [key, val] = cookie.split('=');
            // decodeURIComponent safely decodes the stored string
            acc[key] = val ? decodeURIComponent(val) : '';
            return acc;
        }, {});
        return cookies[name] ? cookies[name] : null;
    } catch (e) {
        console.warn('eCheckpoint: Error reading cookie "' + name + '":', e.message);
        return null;
    }
}

/**
 * Function to set a cookie.
 * Stores the given `value` as JSON,
 * then URL-encodes it for safe storage.
 *
 * @param {string} name  The name of the cookie.
 * @param {any}    value The value to store (will be JSON-stringified).
 * @param {number} days  Number of days until the cookie expires.
 */
function setCookie(name, value, days) {
    try {
        const expires = new Date(Date.now() + days * 24 * 60 * 60 * 1000).toUTCString();
        // Always JSON-stringify the value, then encode
        document.cookie = name + '=' + encodeURIComponent(JSON.stringify(value)) + '; expires=' + expires + '; path=/';
    } catch (e) {
        console.warn('eCheckpoint: Error setting cookie "' + name + '":', e.message);
    }
}

/***************************************************
 * Trigger FFL Map and Requirement
 ***************************************************/

/***************************************************
 * Initialize the consumerTradeType cookie on page load
 ***************************************************/
document.addEventListener('DOMContentLoaded', () => {
    // Check if the cookie already exists
    if (!getCookie('consumerTradeType')) {
        // If not, set it with the default value
        setCookie('consumerTradeType', { selectedTradeType: 'b2C' }, 1); // Expires in 1 day; deletes on page redirect
        //console.log('Cookie initialized:', getCookie('consumerTradeType'));  // Logs the raw JSON string
    }

    // Check if we have a cached API response from previous compliance check
    // If so, evaluate immediately (flag will be set inside evaluateComplianceFromCache if successful)
    var cachedResponse = localStorage.getItem('echeckpoint_api_response');
    if (cachedResponse) {
        debugLog('eCheckpoint: Found cached API response, evaluating on page load');

        // Get current customer type and evaluate
        var currentCustomerType = 'b2C';
        var selectElement = document.getElementById('contact-namespace-select-tradetype') ||
                           document.getElementById('namespace_select_tradetype');
        if (selectElement && selectElement.value) {
            currentCustomerType = selectElement.value;
        }
        // This will set usingLocalStorageCache = true if evaluation succeeds
        evaluateComplianceFromCache(currentCustomerType);
    }
});

/***************************************************
 * Function to update the consumerTradeType cookie
 * Note: Map display is now handled by evaluateComplianceFromCache()
 ***************************************************/
function updateTradeType(newTradeType) {
    // Build object and store it as JSON
    const updatedValue = { selectedTradeType: newTradeType };
    setCookie('consumerTradeType', updatedValue, 7); // Update cookie with new value (7 days)

    // Set flag to tell PHP to skip API call - only customer type changed, not address/cart
    // The cached API response already contains all businessTradeTypes data
    document.cookie = 'echeckpoint_skip_api=1; path=/; max-age=10';
    debugLog('eCheckpoint: Set echeckpoint_skip_api cookie before update_checkout');

    // Small delay to ensure cookie is set before AJAX request is made
    setTimeout(function() {
        // Trigger an update of the checkout (WooCommerce, etc.)
        document.body.dispatchEvent(new CustomEvent('update_checkout', { bubbles: true }));
    }, 50);
}

/***************************************************
 * jQuery block: Notice handling, AJAX, Google Maps
 ***************************************************/
jQuery(function ($) {

    // Note: displayNotice is now a global function defined at the top of this file

    /**
     * Fetch a custom notice from the server (via AJAX).
     * The server returns JSON data with { success, data: { message, type } }
     * which we then pass to displayNotice().
     */
    function fetchNoticeFromServer(tradeType) {
        $.ajax({
            url: eCheckpointParams.ajax_url,
            type: 'POST',
            data: {
                action: 'get_custom_notice',
                trade_type: tradeType,
                nonce: eCheckpointParams.nonce, // Include nonce for security
            },
            success: function (response) {
                if (response.success) {
                    // Remove only eCheckpoint-generated error banners (preserve WooCommerce system notices)
                    const noticeElement = document.querySelector('.echeckpoint-notice.is-error');
                    if (noticeElement) {
                        noticeElement.remove();
                    }
                    // Display the notice if type != 'N/A'
                    if (response.data.type != 'N/A') {
                        displayNotice(response.data.message, response.data.type);
                    }
                }
            },
            error: function (xhr) {
                console.error('Error fetching notice:', xhr.responseText);
            },
        });
    }

    // Listen for changes in the trade type dropdown
    // Uses localStorage cache for instant evaluation - no server round-trip needed
    $('#contact-namespace-select-tradetype, #namespace_select_tradetype').on('change', function () {
        const selectedTradeType = $(this).val();
        debugLog('eCheckpoint: Customer type changed to:', selectedTradeType);
        debugLog('eCheckpoint: phpOwnsNotices =', phpOwnsNotices, ', usingLocalStorageCache =', usingLocalStorageCache);
        debugLog('eCheckpoint: Cache exists =', !!localStorage.getItem('echeckpoint_api_response'));

        // User interaction - JS can take over notice control
        if (phpOwnsNotices) {
            debugLog('eCheckpoint: Customer type changed, JS resuming notice control');
            phpOwnsNotices = false;
        }

        // Always clear existing notices when customer type changes
        // They will be regenerated by cache evaluation
        // Remove JS notices, classic WC notices, AND Blocks-style notices (hybrid setups)
        var jsNoticeCount = jQuery('.echeckpoint-notice').not('.echeckpoint-ffl-notice').length;
        var wcClassicCount = jQuery('.woocommerce-error').length;
        var wcBlocksCount = jQuery('.woocommerce-NoticeGroup-checkout').length;
        jQuery('.echeckpoint-notice').not('.echeckpoint-ffl-notice').remove();
        jQuery('.woocommerce-error').remove();
        jQuery('.woocommerce-NoticeGroup-checkout').remove();
        debugLog('eCheckpoint: Removed', jsNoticeCount, 'JS,', wcClassicCount, 'WC classic,', wcBlocksCount, 'WC blocks notice(s)');

        // Clear the client_message cookie so polling doesn't re-add old notice
        document.cookie = 'client_message=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';

        // Evaluate compliance from cached API response (instant - no server call)
        evaluateComplianceFromCache(selectedTradeType);

        // Update the cookie for PHP to use on next address change
        updateTradeType(selectedTradeType);
    });

    /***************************************************
     * Google Maps Integration
     ***************************************************/
    let map;
    let markers = [];

    // Define the initMap function for the Google Maps API callback
    window.initMap = function () {
        var mapElement = document.getElementById("map");
        if (!mapElement) {
            console.warn('eCheckpoint: Map element not found, skipping initialization');
            return;
        }
        map = new google.maps.Map(mapElement, {
            zoom: 12,
            center: { lat: 34.21721, lng: -119.04726 } // Default center
        });

        window.currentInfoWindow = null;

        // Process any pending marker updates that arrived before map was ready
        if (window.pendingMarkerLocations) {
            updateMarkers(window.pendingMarkerLocations);
            window.pendingMarkerLocations = null;
        }
    }

    /**
     * Update the Google Map markers based on an array of location objects.
     * Each location object should include premiseLat, premiseLon, licenseName, premiseStreet, etc.
     */
    function updateMarkers(locations) {
        if (!map) {
            console.warn("eCheckpoint: Map not initialized, queuing marker update");
            window.pendingMarkerLocations = locations;
            return;
        }
        // Clear old markers
        markers.forEach(marker => marker.setMap(null));
        markers = [];

        // Add new markers for each location
        locations.forEach(location => {
            try {
                const lat = parseFloat(location.premiseLat);
                const lng = parseFloat(location.premiseLon);
                if (isNaN(lat) || isNaN(lng)) {
                    console.error("Invalid lat or lng value for location:", location);
                    return; // Skip this location
                }

                let marker = new google.maps.Marker({
                    position: { lat: lat, lng: lng },
                    map: map,
                    title: location.licenseName
                });

                let infoWindow = new google.maps.InfoWindow({
                    content: `<strong>${location.licenseName}</strong><br>
                              ${location.premiseStreet}, ${location.premiseCity}, 
                              ${location.premiseState} ${location.premiseZipCode}`
                });

                marker.addListener("click", () => {
                    if (window.currentInfoWindow) {
                        window.currentInfoWindow.close();
                    }
                    // Open the new info window and store its reference globally
                    infoWindow.open(map, marker);
                    window.currentInfoWindow = infoWindow;

                    const fieldMappings = {
                        // We leave these keys empty because we'll retrieve the values directly from the billing fields
                        'shipping_first_name': '',
                        'shipping_last_name': '',
                        'shipping_address_1': 'premiseStreet',
                        'shipping_address_2': '',
                        'shipping_company': '',  // Handled specially below
                        'shipping_city': 'premiseCity',
                        'shipping_state': 'premiseState',
                        'shipping_postcode': 'premiseZipCode',
                    };

                    // Loop through the mappings and update the fields
                    Object.entries(fieldMappings).forEach(([fieldId, locationKey]) => {
                        const field = document.getElementById(fieldId);
                        if (field) {
                            if (fieldId === 'shipping_first_name') {
                                // Pull the billing first name from its DOM element
                                const billingFirst = document.getElementById('billing_first_name');
                                field.value = billingFirst ? billingFirst.value : '';
                            } else if (fieldId === 'shipping_last_name') {
                                // Pull the billing last name from its DOM element
                                const billingLast = document.getElementById('billing_last_name');
                                field.value = billingLast ? billingLast.value : '';
                            } else if (fieldId === 'shipping_company') {
                                // If businessName is empty or whitespace, use licenseName
                                field.value = location['businessName']?.trim() ? location['businessName'] : location['licenseName'];
                            } else if (fieldId === 'shipping_postcode' && locationKey) {
                                // Ensure ZIP code is only 5 digits
                                field.value = location[locationKey] ? location[locationKey].toString().slice(0, 5) : '';
                            } else if (fieldId === 'shipping_state' && locationKey) {
                                // State field is a select dropdown - set value and trigger change
                                var stateValue = location[locationKey] || '';
                                field.value = stateValue;
                                // Also try jQuery to trigger WooCommerce's state change handlers
                                jQuery(field).val(stateValue).trigger('change');
                            } else {
                                field.value = locationKey ? location[locationKey] : '';
                            }
                            // Dispatch appropriate event based on field type
                            if (field.tagName === 'SELECT') {
                                field.dispatchEvent(new Event('change', { bubbles: true }));
                            } else {
                                field.dispatchEvent(new Event('input', { bubbles: true }));
                            }
                        }
                    });

                    document.body.dispatchEvent(new CustomEvent('update_checkout', { bubbles: true }));
                });

                markers.push(marker);
            } catch (error) {
                console.error("Error processing marker for location:", location, error);
            }
        });

        // Insert the snippet here to update the map center based on shipping data
        if (window.shippingAddressData) {
            const lat = parseFloat(window.shippingAddressData.addressLat);
            const lng = parseFloat(window.shippingAddressData.addressLng);
            if (!isNaN(lat) && !isNaN(lng)) {
                map.setCenter({ lat, lng });
                //console.log("Map center updated to shipping address:", lat, lng);
            } else {
                console.error("Invalid shipping address coordinates.");
            }
        }
    }

    // Expose updateMarkers globally so other places can call it
    window.updateMarkers = updateMarkers;

    // Dynamically load the Google Maps API
    function loadGoogleMapsApi() {
        // Use API key from eCheckpointParams (passed securely from PHP)
        var apiKey = eCheckpointParams.google_maps_api_key || '';
        if (!apiKey) {
            console.warn('eCheckpoint: Google Maps API key not configured, map will not load');
            return;
        }

        const script = document.createElement("script");
        script.src = "https://maps.googleapis.com/maps/api/js?key=" + encodeURIComponent(apiKey) + "&callback=initMap";
        script.async = true;
        script.defer = true;
        document.body.appendChild(script);
    }

    // Load the Google Maps API when the document is ready
    $(document).ready(function () {
        loadGoogleMapsApi();
    });
});


/***************************************************
 * Polling for ffl_response_key cookie changes
 * (Used to trigger a map update via AJAX)
 ***************************************************/
let previousCookieValue = null;

// Check the cookie every second for changes
echeckpointIntervals.push(setInterval(() => {
    const currentCookie = getCookie('ffl_response_key'); // raw cookie string
    if (currentCookie !== previousCookieValue) {
        previousCookieValue = currentCookie;
      //  console.log("ffl_response_key cookie changed:", currentCookie);

        // If a key exists, trigger the AJAX call to get the full data
        if (currentCookie) {
            jQuery.ajax({
                url: eCheckpointParams.ajax_url, // e.g., admin-ajax.php URL
                method: 'POST',
                data: {
                    action: 'get_ffl_response', // Must match your AJAX handler
                    key: currentCookie,
                    nonce: eCheckpointParams.nonce  // Security nonce
                },
                success: function (data) {
                    if (data.success) {
                        //console.log("Retrieved FFL response via AJAX:", data.data);

                        // Cache the API response in localStorage for instant customer type switching
                        try {
                            localStorage.setItem('echeckpoint_api_response', JSON.stringify(data.data));
                            debugLog('eCheckpoint: Cached API response in localStorage');
                        } catch (e) {
                            console.warn('eCheckpoint: Could not cache API response:', e);
                        }

                        // Get current customer type and evaluate if FFL is actually required
                        var currentCustomerType = 'b2C';
                        var selectElement = document.getElementById('contact-namespace-select-tradetype') ||
                                           document.getElementById('namespace_select_tradetype');
                        if (selectElement && selectElement.value) {
                            currentCustomerType = selectElement.value;
                        }

                        // Check if FFL licensing requirement has been satisfied
                        var fflProvided = false;
                        if (data.data.modules && data.data.modules.regionalRestrictionsCheck &&
                            data.data.modules.regionalRestrictionsCheck.licensing) {
                            var licensing = data.data.modules.regionalRestrictionsCheck.licensing;
                            licensing.forEach(function(license) {
                                if (license.name === 'FFL' && license.provided === true) {
                                    fflProvided = true;
                                }
                            });
                        }

                        // Check if FFL is required for current customer type (must be eligible AND fflRequired)
                        var fflRequiredForCurrentType = false;
                        var items = (data.data.modules && data.data.modules.regionalRestrictionsCheck &&
                                    data.data.modules.regionalRestrictionsCheck.items) || [];

                        items.forEach(function(item) {
                            if (!item.regulatedProduct || !item.businessTradeTypes) return;

                            var effectiveType = getEffectiveTradeType(item.businessTradeTypes, currentCustomerType);
                            // Only show FFL map if customer IS eligible AND FFL is required
                            if (effectiveType !== null &&
                                item.businessTradeTypes[effectiveType] &&
                                item.businessTradeTypes[effectiveType].fflRequired === true) {
                                fflRequiredForCurrentType = true;
                            }
                        });

                        debugLog('eCheckpoint: FFL required for', currentCustomerType, ':', fflRequiredForCurrentType, '- FFL provided:', fflProvided);

                        // Only show map if FFL is required AND not yet provided (satisfied) AND FFL map is enabled
                        var showFflMap = (typeof eCheckpointParams !== 'undefined' && eCheckpointParams.show_ffl_map !== false);
                        var mapContainer = document.getElementById('map-container');
                        if (fflRequiredForCurrentType && !fflProvided && showFflMap) {
                            if (mapContainer) {
                                mapContainer.style.display = "block";
                            }
                            var shipToDifferentAddressCheckBox = document.getElementById('ship-to-different-address-checkbox');
                            if (shipToDifferentAddressCheckBox) {
                                shipToDifferentAddressCheckBox.checked = true;
                            }
                            var shippingFields = document.getElementsByClassName('shipping_address')[0];
                            if (shippingFields) {
                                shippingFields.style.display = 'block';
                            }
                        } else {
                            if (mapContainer) {
                                mapContainer.style.display = "none";
                            }
                        }

                        // Evaluate compliance for current customer type (will show error if not eligible)
                        evaluateComplianceFromCache(currentCustomerType);

                        // Extract and set the shipping address data from the response
                        if (data.data.modules &&
                            data.data.modules.addressValidationCheck &&
                            Array.isArray(data.data.modules.addressValidationCheck.items)) {
                            const shippingItem = data.data.modules.addressValidationCheck.items.find(item => item.type === "Shipping Address");
                            if (shippingItem && shippingItem.response && shippingItem.response.address) {
                                window.shippingAddressData = shippingItem.response.address;
                                // For debugging, you might log the shipping address data:
                              //  console.log("Shipping address data set:", window.shippingAddressData);
                            }
                        }

                        // Process the response to extract licensing (assumed to contain location data)
                        let licensingLocations = processMap(data.data);
                        if (licensingLocations.length > 0) {
                            updateMarkers(licensingLocations);
                        }

                    } else {
                        console.error("Error retrieving FFL response:", data.data);
                    }
                },
                error: function (xhr) {
                    console.error("AJAX error while retrieving FFL response:", xhr.responseText);
                }
            });
        }
    }
}, 1000));

/**
 * Process the map data from the AJAX response
 * and return an array of availableLicenses (marker locations).
 */
function processMap(data) {
    //console.log("Processing data:", data);
    if (
        data.modules &&
        data.modules.regionalRestrictionsCheck &&
        Array.isArray(data.modules.regionalRestrictionsCheck.licensing) &&
        data.modules.regionalRestrictionsCheck.licensing.length > 0 &&
        Array.isArray(data.modules.regionalRestrictionsCheck.licensing[0].availableLicenses)
    ) {
        return data.modules.regionalRestrictionsCheck.licensing[0].availableLicenses;
    } else {
        // No FFL licensing data available - this is normal when FFL is not required
        return [];
    }
}

/***************************************************
 * jQuery block for client_message cookie polling
 * (2nd notice system)
 ***************************************************/
jQuery(function ($) {

    // Note: displayNotice is now a global function defined at the top of this file

    /**
     * Decodes HTML entities in a string.
     * @param {string} html
     * @returns {string}
     */
    function decodeHtmlEntities(html) {
        let textarea = document.createElement('textarea');
        textarea.innerHTML = html;
        return textarea.value;
    }

    /**
     * Check for the `client_message` cookie and display or remove a notice accordingly.
     * NOTE: This is disabled when PHP owns notices or using localStorage cache mode.
     */
    function checkAndUpdateBanner() {
        // Skip if PHP owns notices (after Place Order) or using localStorage cache mode
        if (phpOwnsNotices || usingLocalStorageCache) {
            return;
        }

        let cookieValue = getCookie('client_message'); // raw string
        if (cookieValue) {
            try {
                // The cookie may be URL-encoded, so decode it first before parsing JSON
                let decodedValue = decodeURIComponent(cookieValue);
                let clientMessage = JSON.parse(decodedValue);

                // If the cookie indicates a notice type != 'N/A', display/refresh the banner
                if (clientMessage.type != 'N/A') {
                    // Find existing non-FFL notices for comparison
                    const existingNotice = $('.wc-block-components-notice-banner').not('.echeckpoint-ffl-notice').first();
                    let existingHtml = existingNotice.length && existingNotice.html() ? decodeHtmlEntities(existingNotice.html().trim()) : '';
                    let cookieMessage = decodeHtmlEntities(clientMessage.message.trim());

                    // Compare existing vs. new (only update if different)
                    if (existingNotice.length === 0 || existingHtml !== cookieMessage) {
                        displayNotice(clientMessage.message, clientMessage.type);
                    }
                }
            } catch (e) {
                console.error('eCheckpoint: Error parsing client_message cookie:', e);
            }
        } else {
            // Remove only eCheckpoint notices if the cookie is no longer present
            // Preserve WooCommerce system notices and FFL notices
            $('.echeckpoint-notice').not('.echeckpoint-ffl-notice').remove();
        }
    }

    // Run the check every second (add to echeckpointIntervals for cleanup)
    echeckpointIntervals.push(setInterval(checkAndUpdateBanner, 1000));
});








