<?php

///////////////////////////////////////////
// PRE-ORDER CHECK
// echeckpoint_pre-order-check.php
///////////////////////////////////////////
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

require plugin_dir_path(__FILE__) . 'echeckpoint_compliance-fee.php';

if (!class_exists('eCheckpoint_Pre_Order_Checks')) {
	class eCheckpoint_Pre_Order_Checks
	{
		private static $initialized = false;
		public static $clientMessage = []; //Cookie val to display notifications on checkout page.
		public static $total_fee = 0; //cookie val for compliance fee total

		/**
		 * Get user-specific transient key for server-side state storage
		 *
		 * @param string $key The state key
		 * @return string The user-specific transient key
		 */
		private static function get_user_transient_key($key) {
			$user_id = get_current_user_id();
			$session_id = $user_id ? $user_id : (WC()->session ? WC()->session->get_customer_id() : '');
			return 'echeckpoint_' . $key . '_' . $session_id;
		}

		/**
		 * Set state with WC session and transient fallback
		 *
		 * @param string $key The state key
		 * @param mixed $value The value to store
		 * @param int $expiration Transient expiration in seconds (default 1 hour)
		 */
		private static function set_state($key, $value, $expiration = 3600) {
			// Try WC session first
			if (WC()->session) {
				WC()->session->set('echeckpoint_' . $key, $value);
			}
			// Also set transient as fallback
			set_transient(self::get_user_transient_key($key), $value, $expiration);
		}

		/**
		 * Get state with WC session and transient fallback
		 *
		 * @param string $key The state key
		 * @return mixed The stored value or false if not found
		 */
		private static function get_state($key) {
			// Try WC session first
			if (WC()->session) {
				$value = WC()->session->get('echeckpoint_' . $key);
				if ($value !== null) {
					return $value;
				}
			}
			// Fallback to transient
			return get_transient(self::get_user_transient_key($key));
		}

		/**
		 * Get the blocked products message from settings
		 *
		 * @return string The message template
		 */
		public static function get_blocked_products_message() {
			$pre_settings = get_option('eCheckpoint_pre_settings');
			return isset($pre_settings['blocked_products_message'])
				? $pre_settings['blocked_products_message']
				: __('The following items cannot be shipped to your location due to federal, state, or local regulations. Please remove these items from your cart to proceed with checkout:', 'echeckpoint');
		}

		/**
		 * Get the FFL required message from settings
		 *
		 * @return string The message template
		 */
		public static function get_ffl_required_message() {
			$pre_settings = get_option('eCheckpoint_pre_settings');
			return isset($pre_settings['ffl_required_message'])
				? $pre_settings['ffl_required_message']
				: __('The following product(s) must be shipped to a Licensed Federal Firearms (FFL) dealer. Please select an FFL from the map below:', 'echeckpoint');
		}

		/**
		 * Get the additional information required message from settings
		 *
		 * @return string The message template
		 */
		public static function get_additional_info_message() {
			$pre_settings = get_option('eCheckpoint_pre_settings');
			return isset($pre_settings['additional_info_message'])
				? $pre_settings['additional_info_message']
				: __('To complete this purchase, additional information will be required after checkout.', 'echeckpoint');
		}

		/**
		 * Check if the order is shipping to the United States
		 * eCheckpoint only covers US orders - returns false for international
		 *
		 * @return bool True if US order, false otherwise
		 */
		public static function is_us_order() {
			$shipping_country = '';
			$billing_country = '';

			if (WC()->customer) {
				$shipping_country = WC()->customer->get_shipping_country();
				$billing_country = WC()->customer->get_billing_country();
			}

			// Use shipping country if set, otherwise billing
			$country = !empty($shipping_country) ? $shipping_country : $billing_country;

			// Return true only if explicitly US
			return $country === 'US';
		}

		public static function init()
		{

			if (self::$initialized) {
				return;
			}
			self::$initialized = true;

			add_action('init', function () {

				if (!function_exists('is_checkout') || !is_checkout()) {
					return; // Exit early if not on the checkout page or if the function isn't available.
				}

				if (!class_exists('WC_Blocks_Utils')) {
					return;
				}

				if (!method_exists('WC_Blocks_Utils', 'has_block_in_page')) {
					return;
				}

				$result = WC_Blocks_Utils::has_block_in_page(wc_get_page_id('checkout'), 'woocommerce/checkout');

				if ($result) {
					// Block checkout - set cookie to 'true' and clear WC notices
					setcookie('checkout_block', 'true', time() + 3600, COOKIEPATH, COOKIE_DOMAIN);
					wc_clear_notices();
					if (WC()->session) {
						WC()->session->set('wc_notices', array());
					}
				} else {
					// Classic checkout - set cookie to 'false'
					setcookie('checkout_block', 'false', time() + 3600, COOKIEPATH, COOKIE_DOMAIN);
				}
			}, 1);


			// Run on the checkout process when Place Order button is clicked - Classic (Shortcode) Checkout
			add_action('woocommerce_checkout_process', [__CLASS__, 'check_required_fields_and_run_compliance']);

			// Enqueue scripts
			add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_scripts']);

			// Run on the checkout page when order review is updated - Classic (Shortcode) Checkout
			add_action('woocommerce_checkout_update_order_review', [__CLASS__, 'check_required_fields_and_run_compliance']);

			// Hook into WooCommerce thank you page to clear session data
			add_action('woocommerce_thankyou', [__CLASS__, 'clear_wc_session_data_after_order'], 99);

			//re-adds fee if removed on place order for Classic (Shortcode) Checkout
			add_action('woocommerce_store_api_checkout_order_processed', [__CLASS__, 'verify_fee_added_at_checkout'], 10);

			// REMOVED: validate_blocks_checkout was causing duplicate notices
			// The React notice via cookies is sufficient - this hook was not in working v2.1.1
			// add_action('woocommerce_store_api_checkout_update_order_from_request', [__CLASS__, 'validate_blocks_checkout'], 10, 2);

			// Set default customer type (b2C) in session for Block checkout
			add_filter('woocommerce_get_default_value_for_namespace/select-tradetype', function($value, $group, $wc_object) {
				return 'b2C';
			}, 10, 3);

			//adds ffl notice to classic checkout 
			add_action('wp_ajax_get_custom_notice', [__CLASS__, 'handle_custom_notice_request']);
			add_action('wp_ajax_nopriv_get_custom_notice', [__CLASS__, 'handle_custom_notice_request']);

			// Add the field for Classic (Shortcode) Checkout (conditionally based on setting)
			add_filter('woocommerce_checkout_fields', function ($fields) {
				$pre_settings = get_option('eCheckpoint_pre_settings');
				$show_customer_type = isset($pre_settings['show_customer_type_dropdown']) ? ($pre_settings['show_customer_type_dropdown'] === 'on') : true;

				if ($show_customer_type) {
					$fields['billing']['namespace_select_tradetype'] = array(
						'type' => 'select',
						'label' => 'Customer Type',
						'required' => false,
						'priority' => 5,
						'options' => array(
							'b2C' => 'Individual',
							'b2B' => 'Authorized Retailer',
							'b2G' => 'Government / Law Enforcement / Military',
						)
					);
				}

				return $fields;
			});

			add_filter('woocommerce_checkout_fields', function ($fields) {
				// Add the "Company" field before "Shipping Address 1"
				$fields['shipping']['shipping_company'] = array(
					'type' => 'text',
					'label' => 'Company',
					'placeholder' => 'Enter your company name',
					'required' => false,
					'priority' => 30,
				);

				return $fields;
			});
			//adds company field to checkout form
			add_action('woocommerce_checkout_update_order_review', function ($post_data) {
				parse_str($post_data, $post_data_array);

				if (!empty($post_data_array['shipping_company'])) {
					WC()->session->set('shipping_company', sanitize_text_field($post_data_array['shipping_company']));
				}
			});

			// 3️⃣ Save "Company" field to WooCommerce order meta
			add_action('woocommerce_checkout_update_order_meta', function ($order_id) {

				if (
					!isset($_POST['echeckpoint_nonce']) ||
					!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['echeckpoint_nonce'])), 'echeckpoint_nonce')
				) {
					return; // Nonce verification failed.
				}

				if (!empty($_POST['shipping_company'])) {
					$shipping_company = sanitize_text_field(wp_unslash($_POST['shipping_company'])); // Unslash and sanitize
					update_post_meta($order_id, 'shipping_company', $shipping_company);
				}
			});

			// 4️⃣ Retrieve and display "Company" field in the WooCommerce Admin Order Details
			add_action('woocommerce_admin_order_data_after_shipping_address', function ($order) {
				$shipping_company = get_post_meta($order->get_id(), 'shipping_company', true);
				if (!empty($shipping_company)) {
					echo '<p><strong>Company Name:</strong> ' . esc_html($shipping_company) . '</p>';
				}
			});

			add_action('woocommerce_before_checkout_shipping_form', [__CLASS__, 'echeckpoint_add_map_in_shipping']);

			add_action('wp_ajax_get_ffl_response', [__CLASS__, 'handle_get_ffl_response']);
			add_action('wp_ajax_nopriv_get_ffl_response', [__CLASS__, 'handle_get_ffl_response']);

			// WooCommerce Blocks specific hooks

			//Fires when checkout is updated.
			add_action('woocommerce_store_api_cart_update_customer_from_request', [__CLASS__, 'handle_blocks_customer_update'], 10, 2);

			// Clear notices AFTER compliance check runs (priority 99 > 10)
			add_action('woocommerce_store_api_cart_update_customer_from_request', [__CLASS__, 'clear_notices_for_blocks'], 99);

			//adds fee for woocommerce block checkout
			add_action('woocommerce_blocks_enqueue_checkout_block_scripts_before', [__CLASS__, 'add_fee'], 10);

			// REMOVED: add_filter('woocommerce_use_block_notices_in_classic_theme', '__return_true');
			// This was forcing Block-style notices in Classic Checkout, causing conflicts with other plugins

			// Clear WC notices on ALL Store API requests - Block checkout uses React notices via cookies
			add_action('woocommerce_store_api_cart_update_order_from_request', [__CLASS__, 'clear_notices_for_blocks'], 1);

			// Hook into multiple Store API actions to ensure notices are cleared
			add_action('woocommerce_store_api_checkout_update_order_from_request', [__CLASS__, 'clear_notices_for_blocks'], 1);
			add_action('woocommerce_store_api_cart_update_customer_from_request', [__CLASS__, 'clear_notices_for_blocks'], 1);

			// Filter out WC notices for Block checkout when they're retrieved for display
			// Block checkout uses React notices via cookies, not wc_add_notice()
			add_filter('woocommerce_session_get_customer', function ($value) {
				// Clear notices during REST API / Block checkout requests
				if (defined('REST_REQUEST') && REST_REQUEST) {
					if (WC()->session) {
						WC()->session->set('wc_notices', array());
					}
				}
				return $value;
			});

			// Also clear notices on every Store API request type
			add_action('woocommerce_store_api_cart_select_shipping_rate', [__CLASS__, 'clear_notices_for_blocks'], 1);
			add_action('woocommerce_store_api_cart_apply_coupon', [__CLASS__, 'clear_notices_for_blocks'], 1);
			add_action('woocommerce_store_api_cart_remove_coupon', [__CLASS__, 'clear_notices_for_blocks'], 1);
			add_action('woocommerce_store_api_cart_extensions', [__CLASS__, 'clear_notices_for_blocks'], 1);

			// Final safety net: Clear notices right before Store API response is sent
			add_filter('rest_pre_echo_response', function($result, $server, $request) {
				// Only for WooCommerce Store API requests
				if (strpos($request->get_route(), '/wc/store/') === 0) {
					wc_clear_notices();
					if (WC()->session) {
						WC()->session->set('wc_notices', array());
					}
				}
				return $result;
			}, 999, 3);

			// Add the customer type field for Block Checkout (only if enabled in settings)
			add_action(
				'woocommerce_init',
				function () {
					$pre_settings = get_option('eCheckpoint_pre_settings');
					$show_customer_type = isset($pre_settings['show_customer_type_dropdown']) ? ($pre_settings['show_customer_type_dropdown'] === 'on') : true;

					if ($show_customer_type) {
						woocommerce_register_additional_checkout_field(
							array(
								'id' => 'namespace/select-tradetype',
								'label' => 'Customer Type',
								'location' => 'contact',
								'required' => false,
								'type' => 'select',
								'default' => 'b2C', // Default to Individual
								'options' => [
									[
										'value' => 'b2C',
										'label' => 'Individual',
									],
									[
										'value' => 'b2B',
										'label' => 'Authorized Retailer'
									],
									[
										'value' => 'b2G',
										'label' => 'Government / Law Enforcement / Military'
									]
								]
							)
						);
					}
				}
			);

			// Add the field for Checkout
			add_action(
				'woocommerce_init',
				function () {
					woocommerce_register_additional_checkout_field(
						array(
							'id' => 'namespace/select-company',
							'label' => 'Company',
							'location' => 'address',
							'required' => false,
							'type' => 'text',
						)
					);
				}
			);

			add_action('wp_ajax_update_customer_type', [__CLASS__, 'update_customer_type_callback']);
			add_action('wp_ajax_nopriv_update_customer_type', [__CLASS__, 'update_customer_type_callback']);

			// AJAX handler for cart change recheck
			add_action('wp_ajax_echeckpoint_recheck_compliance', [__CLASS__, 'handle_recheck_compliance']);
			add_action('wp_ajax_nopriv_echeckpoint_recheck_compliance', [__CLASS__, 'handle_recheck_compliance']);

			// AJAX handler to get fresh nonce (for cached pages)
			add_action('wp_ajax_echeckpoint_get_fresh_nonce', [__CLASS__, 'handle_get_fresh_nonce']);
			add_action('wp_ajax_nopriv_echeckpoint_get_fresh_nonce', [__CLASS__, 'handle_get_fresh_nonce']);

			// Output eCheckpointParams in head for blocks checkout
			add_action('wp_head', function () {
				if (!function_exists('is_checkout') || !is_checkout()) {
					return;
				}
				// Only output if not already defined (classic checkout uses wp_localize_script)
				$params = array(
					'ajax_url' => admin_url('admin-ajax.php'),
					'nonce' => wp_create_nonce('echeckpoint_nonce'),
					'google_maps_api_key' => defined('ECHECKPOINT_GOOGLE_MAPS_API_KEY') ? ECHECKPOINT_GOOGLE_MAPS_API_KEY : ''
				);
				echo '<script>if(typeof window.eCheckpointParams === "undefined") { window.eCheckpointParams = ' . wp_json_encode($params) . '; }</script>';
			}, 1);
		}

		public static function update_customer_type_callback()
		{
			check_ajax_referer('echeckpoint_nonce', 'nonce');

			// Get customer type from the request
			$customer_type = isset($_POST['customer_type']) ? sanitize_text_field(wp_unslash($_POST['customer_type'])) : '';

			// Validate customer_type
			if (empty($customer_type)) {
				wp_send_json_error(['message' => 'Customer type is required']);
			}

			// Ensure customer_type is one of the expected values (b2C, b2B, b2G)
			$valid_types = ['b2C', 'b2B', 'b2G'];
			if (!in_array($customer_type, $valid_types, true)) {
				wp_send_json_error(['message' => 'Invalid customer type']);
			}

			// Create the JSON-encoded value
			$cookie_value = json_encode(['selectedTradeType' => $customer_type]);

			// Set the cookie (valid for 1 hour, available site-wide)
			setcookie('consumerTradeType', $cookie_value, time() + 3600, COOKIEPATH, COOKIE_DOMAIN);

			// Trigger WooCommerce hook to re-evaluate compliance with the new customer type
			// This calls handle_blocks_customer_update which runs the compliance check
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Calling existing WooCommerce hook, not creating a new one
			do_action('woocommerce_store_api_cart_update_customer_from_request', WC()->customer, new WP_REST_Request());

			// Return success response
			wp_send_json_success(['message' => 'Customer type updated', 'customer_type' => $customer_type]);
		}

		/**
		 * AJAX handler to trigger compliance recheck when cart changes
		 */
		public static function handle_recheck_compliance()
		{
			check_ajax_referer('echeckpoint_nonce', 'nonce');

			// Clear cached compliance response to force a new API call
			self::clear_cached_compliance_response();

			// Delete the ffl_response_key cookie to signal JS that new data is needed
			setcookie('ffl_response_key', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN);

			// Trigger the compliance check using the current cart and stored address
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Calling existing WooCommerce hook, not creating a new one
			do_action('woocommerce_store_api_cart_update_customer_from_request', WC()->customer, new WP_REST_Request());

			wp_send_json_success(['message' => 'Compliance recheck triggered']);
		}

		/**
		 * AJAX handler to get a fresh nonce (for cached pages)
		 * This allows the JS to fetch a valid nonce even if the page HTML is cached
		 */
		public static function handle_get_fresh_nonce()
		{
			wp_send_json_success([
				'nonce' => wp_create_nonce('echeckpoint_nonce')
			]);
		}


		//AJAX Call that retrieves a specific key from client. If valid, the key return the GetResults call response to the browser to process
		public static function handle_get_ffl_response()
		{
			// Verify the nonce for security
			check_ajax_referer('echeckpoint_nonce', 'nonce');

			// Get and sanitize the unique key sent via AJAX
			$key = isset($_POST['key']) ? sanitize_text_field(wp_unslash($_POST['key'])) : '';


			// Retrieve the stored response using the key
			$response = get_transient($key);

			if (!$response) {
				// If the data isn’t found or has expired, return an error
				wp_send_json_error('No response found or it has expired.');
			}

			// Return the response as JSON
			wp_send_json_success($response);
		}


		//Adds ffl map to classic shortcode checkout page
		public static function echeckpoint_add_map_in_shipping()
		{
			// Check if FFL map is enabled in settings
			$pre_settings = get_option('eCheckpoint_pre_settings');
			$show_ffl_map = isset($pre_settings['show_ffl_map']) ? ($pre_settings['show_ffl_map'] === 'on') : true;

			if (!$show_ffl_map) {
				return; // FFL map is disabled in settings
			}

			// 1. Output HTML markup for the map container
			?>
			<div id="map-container" style="display: none;">
				<h4>FFL Shipping Address Required</h4>
				<p>Please select a marker to change the shipping address to an active FFL address.</p>
				<div id="map" style="width: 100%; height: 300px;"></div>
			</div>
			<?php

			// 2. Build the JavaScript as a normal string (no <<<)
			//    Use either single or double quotes and escape as needed.

			// Define initMap and map variable, then load Google Maps on demand
			$inline_js = 'var map;' . "\n";
			$inline_js .= 'var googleMapsLoaded = false;' . "\n\n";

			// Define initMap callback (required by Google Maps API)
			$inline_js .= '// Google Maps callback - initializes after API loads' . "\n";
			$inline_js .= 'function initMap() {' . "\n";
			$inline_js .= '    // Map will be initialized when FFL data is available' . "\n";
			$inline_js .= '}' . "\n\n";

			$inline_js .= '// Dynamically load the Google Maps API script (only when needed)' . "\n";
			$inline_js .= 'function loadGoogleMapsApi() {' . "\n";
			$inline_js .= '    if (googleMapsLoaded || typeof google !== "undefined") return;' . "\n";
			$inline_js .= '    googleMapsLoaded = true;' . "\n";
			$inline_js .= '    const script = document.createElement("script");' . "\n";
			$inline_js .= '    script.src = "https://maps.googleapis.com/maps/api/js?key=' . esc_js(defined('ECHECKPOINT_GOOGLE_MAPS_API_KEY') ? ECHECKPOINT_GOOGLE_MAPS_API_KEY : '') . '&callback=initMap&loading=async";' . "\n";
			$inline_js .= '    script.async = true;' . "\n";
			$inline_js .= '    script.defer = true;' . "\n";
			$inline_js .= '    document.body.appendChild(script);' . "\n";
			$inline_js .= '}' . "\n";

			// 3. Register a "dummy" script handle (no real file)
			wp_register_script(
				'echeckpoint-inline-map',
				false,       // no file path
				array(),     // no dependencies
				'1.0.0',
				true         // load in footer
			);

			// 4. Attach your concatenated JS to that handle
			wp_add_inline_script('echeckpoint-inline-map', $inline_js);

			// 5. Enqueue the handle so WP outputs your inline JS
			wp_enqueue_script('echeckpoint-inline-map');

			// Note: Google Maps API is now loaded on-demand via loadGoogleMapsApi()
			// to avoid conflicts with other plugins and unnecessary loading
		}


		public static function handle_updated_ffl_markers($response)
		{
			check_ajax_referer('echeckpoint_nonce', 'nonce');

			$trade_type = isset($_POST['trade_type']) ? sanitize_text_field(wp_unslash($_POST['trade_type'])) : '';

			if (!$trade_type) {
				wp_send_json_error(['message' => 'Trade type is required.']);
			}

			// Retrieve and decode the ffl_requirements cookie
			$products_cookie = isset($_COOKIE['ffl_requirements']) ? sanitize_text_field(wp_unslash($_COOKIE['ffl_requirements'])) : '';

			if (!$products_cookie) {
				wp_send_json_error(['message' => 'No products found in the cookie.']);
			}

			$products = json_decode(stripslashes($products_cookie), true);

			if (!$products) {
				wp_send_json_error(['message' => 'Invalid product data in the cookie.']);
			}

			// Initialize the message
			$message = self::get_ffl_required_message() . '<br>';
			$type = 'error'; // Default message type

			// Filter products based on the trade type and build the list
			$selected_products = [];
			foreach ($products as $product) {
				foreach ($product['tradeTypes'] as $trade) {
					// Add product only if the trade type matches and the purchaser is eligible
					if ($trade['type'] === $trade_type && $trade['eligiblePurchaser']) {
						$selected_products[] = $product['productName'];
						break; // No need to check other trade types for this product
					}
				}
			}

			if (!empty($selected_products)) {
				foreach ($selected_products as $product) {
					$message .= "&emsp;&nbsp;" . $product . '<br>';
				}
			} else {
				$message .= "N/A";
				$type = 'N/A';
				wp_send_json_success(['message' => $message, 'type' => $type]);
			}

			$message .= "<br>Please update the shipping address to an active FFL address.";
			$type = 'error';
			wp_send_json_success(['message' => $message, 'type' => $type]);
		}


		private static function ffl_required_in_cart()
		{
			return !empty(WC()->session->get('ffl_requirements'));
		}


		public static function handle_custom_notice_request()
		{
			check_ajax_referer('echeckpoint_nonce', 'nonce');

			$trade_type = isset($_POST['trade_type']) ? sanitize_text_field(wp_unslash($_POST['trade_type'])) : '';

			if (!$trade_type) {
				wp_send_json_error(['message' => 'Trade type is required.']);
			}

			// Retrieve and decode the ffl_requirements cookie
			$products_cookie = isset($_COOKIE['ffl_requirements']) ? sanitize_text_field(wp_unslash($_COOKIE['ffl_requirements'])) : '';

			if (!$products_cookie) {
				wp_send_json_error(['message' => 'No products found in the cookie.']);
			}

			$products = json_decode(stripslashes($products_cookie), true);

			if (!$products) {
				wp_send_json_error(['message' => 'Invalid product data in the cookie.']);
			}

			// Initialize the message
			$message = self::get_ffl_required_message() . '<br>';
			$type = 'error'; // Default message type

			// Filter products based on the trade type and build the list
			$selected_products = [];
			foreach ($products as $product) {
				foreach ($product['tradeTypes'] as $trade) {
					// Add product only if the trade type matches and the purchaser is eligible
					if ($trade['type'] === $trade_type && $trade['eligiblePurchaser']) {
						$selected_products[] = $product['productName'];
						break; // No need to check other trade types for this product
					}
				}
			}

			if (!empty($selected_products)) {
				foreach ($selected_products as $product) {
					$message .= "&emsp;&nbsp;" . $product . '<br>';
				}
			} else {
				$message .= "N/A";
				$type = 'N/A';
				wp_send_json_success(['message' => $message, 'type' => $type]);
			}

			$message .= "<br>Please update the shipping address to an active FFL address.";
			$type = 'error';
			wp_send_json_success(['message' => $message, 'type' => $type]);
		}


		public static function verify_fee_added_at_checkout($order)
		{
			try {
				// Check if the client_message cookie is set
				if (!isset($_COOKIE['client_message'])) {
					return; // Return immediately to process the order as normal
				}

				// Sanitize the raw input from the cookie
				$raw_client_message = sanitize_text_field(wp_unslash($_COOKIE['client_message']));

				// Decode the JSON from the sanitized input
				$client_message = json_decode($raw_client_message, true);

				// Optionally, log and sanitize the client_message data if it's an array
				if (is_array($client_message)) {
					// Sanitize each value in the array for further use
					$client_message = array_map('sanitize_text_field', $client_message);
				}

				// Check if total_fee is available and valid in the decoded message
				if (isset($client_message['total_fee']) && $client_message['total_fee'] > 0) {
					$total_fee = floatval($client_message['total_fee']);

					// Check if the fee with the name '50-State Compliance Fee' already exists in the order
					$existing_fees = $order->get_items('fee');
					foreach ($existing_fees as $existing_fee) {
						if ($existing_fee->get_name() === '50-State Compliance Fee') {
							return; // Fee already exists, do not add a duplicate
						}
					}

					// Add the fee to the order if it does not already exist
					$fee = new WC_Order_Item_Fee();
					$compliance_fee_message = get_option('eCheckpoint_compliance_fee_message', '50-State Compliance Fee');
					$fee->set_name($compliance_fee_message); // Set the fee name
					$fee->set_amount($total_fee); // Set the fee amount
					$fee->set_total($total_fee); // Set the total fee

					// Add the fee to the order
					$order->add_item($fee);

					// Save the order to ensure the fee is included
					$order->calculate_totals();
					$order->save();
				} else {
					return; // Return immediately to process the order as normal
				}

			} catch (Exception $e) {
				return; // Return immediately to process the order as normal
			}

			$order->save(); // Ensure the order updates are saved.
		}

		/**
		 * Validate blocks checkout before order is created
		 * Throws exception to block order if compliance check fails
		 * ONLY runs during actual order placement, NOT during form updates
		 */
		public static function validate_blocks_checkout($order, $request)
		{
			// Clear any WooCommerce notices - Block checkout uses React notices via cookies
			wc_clear_notices();

			// Detect if this is actual order placement vs form update
			// Order placement requests have payment_method in the request body
			$request_body = $request->get_json_params();
			$has_payment_method = !empty($request_body['payment_method']);

			// Also check if request method is POST and route ends with checkout (not just updates)
			$request_method = isset($_SERVER['REQUEST_METHOD']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD'])) : '';
			$is_checkout_submission = $request_method === 'POST' && $has_payment_method;

			// Only validate during actual order placement, not form updates
			// Form updates don't have payment_method - they're just updating customer/shipping data
			if (!$is_checkout_submission) {
				return; // Don't validate during form updates, only during place order
			}

			// Check client_message cookie for errors
			// The React component displays the detailed error message and sets validation errors
			// We only need to block here if the cookie has an error
			if (isset($_COOKIE['client_message'])) {
				// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Value is JSON decoded and validated, not directly output
				$raw_cookie = wp_unslash($_COOKIE['client_message']);
				$decoded_cookie = urldecode($raw_cookie);
				$client_message = json_decode($decoded_cookie, true);

				if (is_array($client_message) && isset($client_message['type']) && $client_message['type'] === 'error') {
					// Get the error message from the cookie to display
					$error_message = isset($client_message['message']) ? wp_strip_all_tags($client_message['message']) : __('Compliance check failed. Please review your order.', 'echeckpoint');
					throw new \Automattic\WooCommerce\StoreApi\Exceptions\RouteException(
						'echeckpoint_compliance_failed',
						esc_html( $error_message ),
						400
					);
				}
			}

			// Fallback: Check session for failed products (in case cookie is not set)
			$failed_products = WC()->session->get('echeckpoint_failed_products', []);
			if (!empty($failed_products)) {
				$product_names = array_map(function($product) {
					return is_array($product) && isset($product['name']) ? $product['name'] : $product;
				}, $failed_products);
				// Build UL list of products
				$product_list_html = '<ul><li>' . implode('</li><li>', array_map('esc_html', $product_names)) . '</li></ul>';
				throw new \Automattic\WooCommerce\StoreApi\Exceptions\RouteException(
					'echeckpoint_compliance_failed',
					wp_kses_post( self::get_blocked_products_message() . $product_list_html ),
					400
				);
			}
		}

		public static function clear_wc_session_data_after_order($order_id)
		{
			if (is_wc_endpoint_url('order-received')) {
				// Clear WooCommerce session data related to notices
				WC()->session->__unset('wc_notices');
				// Clear eCheckpoint session data
				WC()->session->__unset('echeckpoint_failed_products');
			}
		}

		/**
		 * Clear WC notices for Block checkout
		 * Block checkout uses React notices via cookies, not wc_add_notice()
		 */
		public static function clear_notices_for_blocks()
		{
			// Always clear notices during Store API requests - Block checkout doesn't use them
			wc_clear_notices();

			// Also clear from session to prevent persistence
			if (WC()->session) {
				WC()->session->set('wc_notices', array());
			}
		}


		public static function add_custom_fields_to_session($posted_data)
		{
			// Check if $posted_data is an array or a string, and handle accordingly
			if (is_array($posted_data)) {
				// Directly use the array if it's already parsed
				$checkout_data = $posted_data;
			} else {
				// If it's a string, parse it into an array
				parse_str($posted_data, $checkout_data);
			}

			// Update session data
			$existing_customer = WC()->session->get('customer');
			WC()->session->set('customer', array_merge($existing_customer ? $existing_customer : array(), array(
				'billing_first_name' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
				'billing_last_name' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
				'billing_address_1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
				'billing_address_2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
				'billing_city' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
				'billing_state' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
				'billing_postcode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
				'ship_to_different_address' => !empty($checkout_data['ship_to_different_address']) ? '1' : '0',
				'shipping_first_name' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
				'shipping_last_name' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
				'shipping_company' => sanitize_text_field($checkout_data['shipping_company'] ?? ''),
				'shipping_address_1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
				'shipping_address_2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
				'shipping_city' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
				'shipping_state' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
				'shipping_postcode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
			)));
		}

		public static function debug_woocommerce_session()
		{
			// Retrieve WooCommerce session data
			$session_data = WC()->session->get('customer');
		}


		public static function enqueue_scripts()
		{

			$version = time(); // or use a static version number if preferred
			// Enqueue CSS (use wp_enqueue_style for CSS files)
			wp_enqueue_style('echeckpoint-admin-styles', plugin_dir_url(__FILE__) . 'src/css/echeckpoint-admin-styles.css', array(), $version);

			// Check if WooCommerce is active before using is_checkout()
			// Don't load checkout scripts on order-received (thank you) page
			if (class_exists('WooCommerce') && is_checkout() && !is_wc_endpoint_url('order-received')) {
				// Defensive check for WC_Blocks_Utils - default to classic checkout if not available
				$is_blocks_checkout = false;
				if (class_exists('WC_Blocks_Utils') && method_exists('WC_Blocks_Utils', 'has_block_in_page')) {
					$is_blocks_checkout = WC_Blocks_Utils::has_block_in_page(wc_get_page_id('checkout'), 'woocommerce/checkout');
				}

				// DEBUG: Log checkout type detection



				if (!$is_blocks_checkout) {

					wp_enqueue_script('echeckpoint_pre-order-check', plugin_dir_url(__FILE__) . 'src/js/echeckpoint_pre-order-check.js', array('jquery'), $version, true);

					// Build cart products array with SKU and Name for client-side display
					$cart_products = array();
					if (WC()->cart) {
						foreach (WC()->cart->get_cart() as $cart_item) {
							$product = $cart_item['data'];
							$sku = $product->get_sku();
							$name = $product->get_name();
							if ($sku && $name) {
								$cart_products[] = array(
									'sku' => sanitize_text_field($sku),
									'name' => sanitize_text_field($name)
								);
							}
						}
					}

					// Get display settings
					$pre_settings = get_option('eCheckpoint_pre_settings');
					$show_customer_type = isset($pre_settings['show_customer_type_dropdown']) ? ($pre_settings['show_customer_type_dropdown'] === 'on') : true;
					$show_ffl_map = isset($pre_settings['show_ffl_map']) ? ($pre_settings['show_ffl_map'] === 'on') : true;

					wp_localize_script('echeckpoint_pre-order-check', 'eCheckpointParams', array(
						'ajax_url' => admin_url('admin-ajax.php'),
						'nonce' => wp_create_nonce('echeckpoint_nonce'),
						'google_maps_api_key' => defined('ECHECKPOINT_GOOGLE_MAPS_API_KEY') ? ECHECKPOINT_GOOGLE_MAPS_API_KEY : '',
						'cart_products' => $cart_products,
						'show_customer_type_dropdown' => $show_customer_type,
						'show_ffl_map' => $show_ffl_map,
						'blocked_products_message' => self::get_blocked_products_message(),
						'ffl_required_message' => self::get_ffl_required_message(),
						'additional_info_message' => self::get_additional_info_message()
					));
					// Localize the script with parameters, including the nonce
				} else {
					wp_enqueue_script('block-checkout', plugin_dir_url(__FILE__) . 'src/js/block-checkout.js', array('jquery'), $version, true);
					wp_localize_script('block-checkout', 'eCheckpointParams', array(
						'ajax_url' => admin_url('admin-ajax.php'),
						'nonce' => wp_create_nonce('echeckpoint_nonce'),
						'google_maps_api_key' => defined('ECHECKPOINT_GOOGLE_MAPS_API_KEY') ? ECHECKPOINT_GOOGLE_MAPS_API_KEY : ''
					));
				}

			}
		}

		public static function check_required_fields_and_run_compliance($posted_data)
		{
			// DEBUG: Log function entry


			// This function is for CLASSIC checkout only
			// Block checkout uses handle_blocks_customer_update and check_required_fields_and_run_compliance_blocks
			if (self::is_block_checkout_request()) {

				return;
			}

			// Skip compliance for non-US orders - eCheckpoint only covers United States
			if (!self::is_us_order()) {
				return;
			}

			// Set cookie to indicate classic checkout
			setcookie('checkout_block', 'false', time() + 3600, COOKIEPATH, COOKIE_DOMAIN); // 1 hour expiration

			// Proceed with compliance check on normal request or place order action
			// Use filter_input to safely get the post_data from the POST request
			// Use FILTER_SANITIZE_STRING to properly sanitize the data instead of FILTER_DEFAULT
			$post_data = filter_input(INPUT_POST, 'post_data', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

			// Initialize $nonce as null
			$nonce = null;

			if ($post_data) {
				$post_data = html_entity_decode($post_data);
				parse_str($post_data, $parsed_data);

				// Look for echeckpoint_nonce within the parsed data and sanitize it
				$nonce = isset($parsed_data['echeckpoint_nonce']) ? sanitize_text_field(wp_unslash($parsed_data['echeckpoint_nonce'])) : null;
			}

			// If nonce is not found in parsed post_data, check the main POST array and sanitize
			if (empty($nonce)) {
				$nonce = isset($_POST['echeckpoint_nonce']) ? sanitize_text_field(wp_unslash($_POST['echeckpoint_nonce'])) : '';
			}

			// Check if the nonce is set
			// For Place Order (woocommerce_checkout_process), block the order if nonce fails
			// For AJAX updates (woocommerce_checkout_update_order_review), fail gracefully for caching compatibility
			$current_hook = current_filter();
			$is_place_order = ($current_hook === 'woocommerce_checkout_process');

			// DEBUG: Log nonce status


			if (empty($nonce)) {

				if ($is_place_order) {
					wc_add_notice(__('Security verification failed. Please refresh the page and try again.', 'echeckpoint'), 'error');
				}
				return;
			}

			// Verify the nonce using wp_verify_nonce with sanitized value
			if (!wp_verify_nonce($nonce, 'echeckpoint_nonce')) {

				if ($is_place_order) {
					wc_add_notice(__('Security verification failed. Please refresh the page and try again.', 'echeckpoint'), 'error');
				}
				return;
			}



			// Check if $post_data is an array or a string, and handle accordingly
			if (is_array($post_data) || empty($post_data)) {
				// Avoid using the entire $_POST array and instead process only required fields
				// phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified above
				$checkout_data = array(
					'billing_first_name' => sanitize_text_field(wp_unslash($_POST['billing_first_name'] ?? '')),
					'billing_last_name' => sanitize_text_field(wp_unslash($_POST['billing_last_name'] ?? '')),
					'billing_address_1' => sanitize_text_field(wp_unslash($_POST['billing_address_1'] ?? '')),
					'billing_address_2' => sanitize_text_field(wp_unslash($_POST['billing_address_2'] ?? '')),
					'billing_city' => sanitize_text_field(wp_unslash($_POST['billing_city'] ?? '')),
					'billing_state' => sanitize_text_field(wp_unslash($_POST['billing_state'] ?? '')),
					'billing_postcode' => sanitize_text_field(wp_unslash($_POST['billing_postcode'] ?? '')),
					'billing_email' => sanitize_email(wp_unslash($_POST['billing_email'] ?? '')),
					'ship_to_different_address' => sanitize_text_field(wp_unslash($_POST['ship_to_different_address'] ?? '')),
					'shipping_first_name' => sanitize_text_field(wp_unslash($_POST['shipping_first_name'] ?? '')),
					'shipping_last_name' => sanitize_text_field(wp_unslash($_POST['shipping_last_name'] ?? '')),
					'shipping_company' => sanitize_text_field(wp_unslash($_POST['shipping_company'] ?? '')),
					'shipping_address_1' => sanitize_text_field(wp_unslash($_POST['shipping_address_1'] ?? '')),
					'shipping_address_2' => sanitize_text_field(wp_unslash($_POST['shipping_address_2'] ?? '')),
					'shipping_city' => sanitize_text_field(wp_unslash($_POST['shipping_city'] ?? '')),
					'shipping_state' => sanitize_text_field(wp_unslash($_POST['shipping_state'] ?? '')),
					'shipping_postcode' => sanitize_text_field(wp_unslash($_POST['shipping_postcode'] ?? ''))
				);
				// phpcs:enable WordPress.Security.NonceVerification.Missing
			} else {
				parse_str($post_data, $checkout_data);
			}

			// Add custom fields to session and prepare data
			self::add_custom_fields_to_session($posted_data);

			// Use the new helper function to check if address fields have been updated
			$address_update_status = self::is_address_update($checkout_data);

			// For Place Order, always validate - either re-run check or validate cached result
			if (!$address_update_status) {
				if ($is_place_order) {
					// On Place Order, check if we have a cached failed compliance status
					$failed_products = WC()->session ? WC()->session->get('echeckpoint_failed_products') : null;
					if (!empty($failed_products)) {
						$failed_product_list = "<ul>" . implode('', $failed_products) . "</ul>";
						wc_add_notice(esc_html(self::get_blocked_products_message()) . ' ' . $failed_product_list, 'error');
					}
				}
				return; // Exit the function early if no critical fields have been updated
			}

			// Prepare data to check for compliance with sanitized values
			$data_to_check = array(
				'billing_first_name' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
				'billing_last_name' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
				'billing_address_1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
				'billing_address_2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
				'billing_city' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
				'billing_state' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
				'billing_postcode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
				'billing_email' => sanitize_email($checkout_data['billing_email'] ?? ''),
				'ship_to_different_address' => !empty($checkout_data['ship_to_different_address']) ? '1' : '0',
				'shipping_first_name' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
				'shipping_last_name' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
				'shipping_company' => sanitize_text_field($checkout_data['shipping_company'] ?? ''),
				'shipping_address_1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
				'shipping_address_2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
				'shipping_city' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
				'shipping_state' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
				'shipping_postcode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
			);

			// Check if this is just a customer type change (skip API cookie set by JS)
			$skip_api_cookie = isset($_COOKIE['echeckpoint_skip_api']) && $_COOKIE['echeckpoint_skip_api'] === '1';


			if ($skip_api_cookie) {
				// Clear the cookie after reading
				setcookie('echeckpoint_skip_api', '', time() - 3600, '/');

				// JS handles the cache evaluation, no API call needed
				return;
			}

			if (self::are_required_fields_completed($data_to_check)) {
				// Run the compliance check
				self::trigger_compliance_check($data_to_check);
			}
		}



		// New helper function to check if the address fields are being updated
		private static function is_address_update($checkout_data)
		{
			// Check if customer type is being updated via AJAX
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Called from WooCommerce hooks which handle nonce verification
			$ajax_action = isset($_REQUEST['action']) ? sanitize_text_field(wp_unslash($_REQUEST['action'])) : '';
			$is_customer_type_change = ($ajax_action === 'update_customer_type');

			// Critical fields that should trigger NEW API calls
			$critical_fields = [
				'billing_address_1',
				'billing_address_2',
				'billing_city',
				'billing_state',
				'billing_email',
				'billing_postcode',
				'shipping_address_1',
				'shipping_address_2',
				'shipping_city',
				'shipping_state',
				'shipping_postcode'
			];

			// Get previous data from session
			$previous_data = WC()->session->get('customer', []);

			// Check if any address fields changed
			$address_changed = false;
			foreach ($critical_fields as $field) {
				if (isset($checkout_data[$field]) && $checkout_data[$field] !== ($previous_data[$field] ?? '')) {
					$address_changed = true;
					break;
				}
			}

			if ($address_changed) {
				return 'address_change'; // Address changed, need new API call
			}

			if ($is_customer_type_change) {
				return 'customer_type_change'; // Customer type changed, use cached response
			}

			// Check for initial load with saved address: no cached response but fields are complete
			// This handles logged-in customers with saved addresses on first checkout load
			$cached_response = self::get_cached_compliance_response();
			if (!$cached_response) {
				// Check if minimum required fields are present (billing address or shipping address)
				$has_billing = !empty($checkout_data['billing_address_1']) &&
							   !empty($checkout_data['billing_city']) &&
							   !empty($checkout_data['billing_state']) &&
							   !empty($checkout_data['billing_postcode']);

				$has_shipping = !empty($checkout_data['shipping_address_1']) &&
								!empty($checkout_data['shipping_city']) &&
								!empty($checkout_data['shipping_state']) &&
								!empty($checkout_data['shipping_postcode']);

				if ($has_billing || $has_shipping) {

					return 'initial_load'; // First load with saved address, need API call
				}
			}

			// No critical fields were updated
			return false;
		}



		private static function trigger_compliance_check($data_to_check, $force_block_checkout = false)
		{
			// Implement your compliance check logic here
			self::initiate_compliance_check($data_to_check, $force_block_checkout);
		}




		public static function are_required_fields_completed($checkout_data)
		{

			if (self::is_block_checkout_request()) {
				$checkbox_checked = true;
			} else {
				$checkbox_checked = isset($checkout_data['ship_to_different_address']) ? (bool) $checkout_data['ship_to_different_address'] : false;
			}

			// Check if billing fields are completed
			$billing_complete = self::is_billing_complete($checkout_data);

			// Check if shipping fields are completed
			$shipping_complete = !$checkbox_checked || self::is_shipping_complete($checkout_data);

			// Check if billing fields are completed
			$billing_complete = self::is_billing_complete($checkout_data);

			// Check if shipping fields are completed
			$shipping_complete = !$checkbox_checked || self::is_shipping_complete($checkout_data);

			$result = $billing_complete && $shipping_complete;
			return $result;
		}

		private static function is_billing_complete($checkout_data)
		{
			$billing_first_name = $checkout_data['billing_first_name'] ?? '';
			$billing_last_name = $checkout_data['billing_last_name'] ?? '';
			$billing_address_1 = $checkout_data['billing_address_1'] ?? '';
			$billing_city = $checkout_data['billing_city'] ?? '';
			$billing_state = $checkout_data['billing_state'] ?? '';
			$billing_postcode = $checkout_data['billing_postcode'] ?? '';
			$billing_email = $checkout_data['billing_email'] ?? '';

			// Check if billing fields are completed
			$complete = !empty($billing_first_name) &&
				!empty($billing_last_name) &&
				!empty($billing_address_1) &&
				!empty($billing_city) &&
				!empty($billing_state) &&
				!empty($billing_postcode) &&
				!empty($billing_email);

			return $complete;
		}

		private static function is_shipping_complete($checkout_data)
		{
			$shipping_first_name = $checkout_data['shipping_first_name'] ?? '';
			$shipping_last_name = $checkout_data['shipping_last_name'] ?? '';
			$shipping_address_1 = $checkout_data['shipping_address_1'] ?? '';
			$shipping_city = $checkout_data['shipping_city'] ?? '';
			$shipping_state = $checkout_data['shipping_state'] ?? '';
			$shipping_postcode = $checkout_data['shipping_postcode'] ?? '';

			// Check if shipping fields are completed
			$complete = !empty($shipping_first_name) &&
				!empty($shipping_last_name) &&
				!empty($shipping_address_1) &&
				!empty($shipping_city) &&
				!empty($shipping_state) &&
				!empty($shipping_postcode);

			return $complete;
		}


		public static function initiate_compliance_check($checkout_data, $force_block_checkout = false)
		{
			try {
				// Use checkout_data for compliance check
				if (self::pre_order_active() && !self::is_state_excluded($checkout_data)) {
					self::custom_compliance_check($checkout_data, $force_block_checkout);
				}
			} catch (Exception $e) {
			}
		}

		public static function pre_order_active()
		{
			$options = get_option('eCheckpoint_pre_settings');
			return isset($options['checkbox']) && $options['checkbox'] === 'on';
		}

		public static function is_state_excluded($checkout_data)
		{
			$excluded_states = get_option('eCheckpoint_excluded_states', []);
			if (!is_array($excluded_states)) {
				$excluded_states = [];
			}

			$shipping_state = $checkout_data['shipping_state'] ?? '';
			$billing_state = $checkout_data['billing_state'] ?? '';
			$checkbox_checked = isset($checkout_data['ship_to_different_address']) ? (bool) $checkout_data['ship_to_different_address'] : false;
			$shipping_complete = self::is_shipping_complete($checkout_data);

			// If shipping is complete, use the shipping state, otherwise use the billing state
			if ($shipping_complete && $checkbox_checked) {
				$is_excluded = in_array($shipping_state, $excluded_states);
				return $is_excluded;
			} else {
				$is_excluded = in_array($billing_state, $excluded_states);
				return $is_excluded;
			}
		}


		public static function custom_compliance_check($checkout_data, $force_block_checkout = false)
		{

			$options = get_option('eCheckpoint_pre_settings');
			$show_conditional_fail = isset($options['conditional_fail_checkbox']) && $options['conditional_fail_checkbox'] === 'on';
			$regulatedProducts = [];

			// Detect if we're in Block checkout context
			// Uses cookie-based detection which is reliable during AJAX/REST requests
			$hook_name = current_filter();
			$isCheckoutBlock = $force_block_checkout || self::is_block_checkout_request();

			// For Block checkout, clear any WC notices to prevent duplicates
			// Block checkout uses React notices via cookies, not wc_add_notice()
			if ($isCheckoutBlock) {
				wc_clear_notices();
			}

			$isECheckpointPassWithoutConditions = true;
			$response = self::api_call_during_checkout($checkout_data);

			// Cache the successful API response for re-use when customer type changes
			if ($response) {
				self::cache_compliance_response($response);

				// Always cache API response for JavaScript localStorage
				// This enables customer type switching without new API calls
				$unique_key = 'ffl_response_' . wp_generate_password(12, false);
				set_transient($unique_key, $response, 60 * 60);
				setcookie(
					'ffl_response_key',
					$unique_key,
					time() + 3600,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);

			}

			$place_order_hook = 'woocommerce_checkout_process';
			if (!$response) {
				// For Block checkout OR non-place-order hooks, use cookies
				if ($isCheckoutBlock || $hook_name !== $place_order_hook) {
					// Create the client message as a variable
					$client_message_content = __('Error: Could not perform compliance check. Please contact Customer Service.', 'echeckpoint');
					$client_message_type = 'error';

					// Set the client message
					self::$clientMessage = [
						$client_message_content,
						$client_message_type
					];

					// Set the cookie with the client message
					setcookie(
						'client_message',
						wp_json_encode([
							'message' => $client_message_content,
							'type' => $client_message_type
						]),
						time() + 1200,  // Cookie expires in 20 minutes
						COOKIEPATH,
						COOKIE_DOMAIN,
						is_ssl(),  // Secure cookie if using HTTPS
						false      // Not HTTPOnly to allow JavaScript access
					);
					$isECheckpointPassWithoutConditions = false;
				} else {
					// Classic checkout place order - use WC notice
					wc_add_notice(__('Error: Could not perform compliance check. Please contact Customer Service.', 'echeckpoint'), 'error');
				}

				return;
			}

			$compliance_passed = true;
			$failed_product_names = [];
			$conditional_fail_messages = []; // Array to collect conditional fail messages
			$failed_product_list = "";
			// Catalog Integrity Check
			if (isset($response['modules']['catalogIntegrityCheck']) && isset($response['modules']['catalogIntegrityCheck']['items']) && is_array($response['modules']['catalogIntegrityCheck']['items'])) {
				foreach ($response['modules']['catalogIntegrityCheck']['items'] as $item) {
					if (isset($item['response']['result']) && $item['response']['result'] == 2) {
						$product_id = wc_get_product_id_by_sku($item['response']['productID']);
						$product = wc_get_product($product_id);
						if ($product) {
							$failed_product_names[] = $product->get_name();
						} else {
							$failed_product_names[] = "SKU " . $item['response']['productID'];
						}
					}
				}

				if (!empty($failed_product_names)) {
					// Store failed products in session for blocks checkout validation
					WC()->session->set('echeckpoint_failed_products', $failed_product_names);

					$failed_product_list = "<ul><li>" . implode('</li><li>', $failed_product_names) . "</li></ul>";
					/* translators: %s: failed product list */
					if ($isCheckoutBlock) {
						// Create the client message as a variable
						$message_template = __('Please contact Customer Service. An issue with some of the products in your order is preventing checkout:<br>', 'echeckpoint');
						$client_message_content = $message_template . $failed_product_list;
						$client_message_type = 'error';

						// Set the client message
						self::$clientMessage = [
							$client_message_content,
							$client_message_type
						];

						// Set the cookie with the client message
						setcookie(
							'client_message',
							wp_json_encode([
								'message' => $client_message_content,
								'type' => $client_message_type
							]),
							time() + 1200,  // Cookie expires in 20 minutes
							COOKIEPATH,
							COOKIE_DOMAIN,
							is_ssl(),  // Secure cookie if using HTTPS
							false      // Not HTTPOnly to allow JavaScript access
						);
						$isECheckpointPassWithoutConditions = false;
					} else {
						$message_template = __('Please contact Customer Service. An issue with some of the products in your order is preventing checkout:<br>', 'echeckpoint');
						$full_message = $message_template . $failed_product_list;
						wc_add_notice($full_message, 'error');
					}

					$compliance_passed = false;
				}
			}

			// Address Validation Check
			if (isset($response['modules']['addressValidationCheck']) && isset($response['modules']['addressValidationCheck']['items']) && is_array($response['modules']['addressValidationCheck']['items'])) {
				$items = $response['modules']['addressValidationCheck']['items'];

				// Ensure items is an array and contains at least 2 elements
				if (count($items) >= 2) {
					$billingResponse = isset($items[0]['response']['result']) ? $items[0]['response']['result'] : null;
					$shippingResponse = isset($items[1]['response']['result']) ? $items[1]['response']['result'] : null;

					// Handling Billing Address Validation
					if ($billingResponse == 2) { // Validation Failed

						// For Block checkout OR non-place-order hooks, use cookies
						if ($isCheckoutBlock || $hook_name !== $place_order_hook) {
							// Create the client message as a variable
							$client_message_content = __('Your billing address is not valid. To ensure the fastest service for your order, we verify your billing address with the U.S. Postal Service. Please ensure it is correct and try again.', 'echeckpoint');
							$client_message_type = 'error';

							// Set the client message
							self::$clientMessage = [
								$client_message_content,
								$client_message_type
							];

							// Set the cookie with the client message
							setcookie(
								'client_message',
								wp_json_encode([
									'message' => $client_message_content,
									'type' => $client_message_type
								]),
								time() + 1200,  // Cookie expires in 20 minutes
								COOKIEPATH,
								COOKIE_DOMAIN,
								is_ssl(),  // Secure cookie if using HTTPS
								false      // Not HTTPOnly to allow JavaScript access
							);
							$isECheckpointPassWithoutConditions = false;
							$compliance_passed = false;
						} else {
							// Classic checkout place order - use WC notice
							$message_template = __('Your billing address is not valid. To ensure the fastest service for your order, we verify your billing address with the U.S. Postal Service. Please ensure it is correct and try again. If you need assistance locating your official billing address, please look it up here: https://tools.usps.com/zip-code-lookup.htm?byaddress', 'echeckpoint');
							wc_add_notice($message_template, 'error');
							$compliance_passed = false;
						}

					}

					// Handling Shipping Address Validation
					if ($shippingResponse == 2) { // Validation Failed

						// For Block checkout OR non-place-order hooks, use cookies
						if ($isCheckoutBlock || $hook_name !== $place_order_hook) {
							// Create the client message as a variable
							$client_message_content = __('Your shipping address is not valid. To ensure the fastest service for your order, we verify your shipping address with the U.S. Postal Service. Please ensure it is correct and try again.', 'echeckpoint');
							$client_message_type = 'error';

							// Set the client message
							self::$clientMessage = [
								$client_message_content,
								$client_message_type
							];

							// Set the cookie with the client message
							setcookie(
								'client_message',
								wp_json_encode([
									'message' => $client_message_content,
									'type' => $client_message_type
								]),
								time() + 1200,  // Cookie expires in 20 minutes
								COOKIEPATH,
								COOKIE_DOMAIN,
								is_ssl(),  // Secure cookie if using HTTPS
								false      // Not HTTPOnly to allow JavaScript access
							);
							$isECheckpointPassWithoutConditions = false;
							$compliance_passed = false;
						} else {
							// Classic checkout place order - use WC notice
							$message_template = __('Your shipping address is not valid. To ensure the fastest service for your order, we verify your shipping address with the U.S. Postal Service. Please ensure it is correct and try again. If you need assistance locating your official shipping address, please look it up here: https://tools.usps.com/zip-code-lookup.htm?byaddress', 'echeckpoint');
							wc_add_notice($message_template, 'error');
							$compliance_passed = false;
						}
					}
				}
			}

			$eligible_tradetypes = null;

			// Get the current customer type for eligibility checks
			$current_customer_type = 'b2C'; // Default
			if (isset($_REQUEST['customer_type']) && !empty($_REQUEST['customer_type'])) {
				$current_customer_type = sanitize_text_field(wp_unslash($_REQUEST['customer_type']));
			} else if (isset($_COOKIE['consumerTradeType'])) {
				$cookie_value = sanitize_text_field(wp_unslash($_COOKIE['consumerTradeType']));
				$decodedValue = json_decode($cookie_value, true);
				if (json_last_error() === JSON_ERROR_NONE && isset($decodedValue['selectedTradeType'])) {
					$current_customer_type = sanitize_text_field($decodedValue['selectedTradeType']);
				}
			}


			// Regional Restrictions Check
			if (isset($response['modules']['regionalRestrictionsCheck']) && isset($response['modules']['regionalRestrictionsCheck']['items']) && is_array($response['modules']['regionalRestrictionsCheck']['items'])) {
				$failed_product_names = [];
				foreach ($response['modules']['regionalRestrictionsCheck']['items'] as $item) {
					// Check eligiblePurchaser for the current customer type
					$is_eligible = true;
					if (isset($item['businessTradeTypes'])) {
						$effective_type = self::get_effective_trade_type($item['businessTradeTypes'], $current_customer_type);
						if ($effective_type === null) {
							// Not eligible for this customer type (and no b2C fallback available)
							$is_eligible = false;

						}
					}

					// Mark as failed if result == 2 OR if not eligible for customer type
					if ((isset($item['response']['result']) && $item['response']['result'] == 2) || !$is_eligible) {
						$product_id = wc_get_product_id_by_sku($item['response']['productID']);
						$product = wc_get_product($product_id);
						if ($product) {
							$failed_product_names[] = "<li>" . $product->get_name() . "</li>";
						} else {
							$failed_product_names[] = "<li>SKU " . $item['response']['productID'] . "</li>";
						}
					} elseif (isset($item['response']['result']) && $item['response']['result'] == 3 && $show_conditional_fail) {
						$product_id = wc_get_product_id_by_sku($item['response']['productID']);
						$product = wc_get_product($product_id);
						if (isset($item['businessTradeTypes'])) {
							$business_trade_types = $item['businessTradeTypes'];
							// Log the businessTradeTypes to ensure it is being received correctly
							$eligible_tradetypes = self::get_eligible_tradetypes($business_trade_types);

							if ($product) {
								/* translators: 1: product name */
								$message = sprintf(__('To complete the purchase of %1$s, additional information will be required after checkout.', 'echeckpoint'), $product->get_name());
								$conditional_fail_messages[] = $message;
							} else {
								/* translators: 1: SKU */
								$message = sprintf(__('To complete the purchase of this SKU, %1$s, additional information will be required after checkout.', 'echeckpoint'), $item['response']['productID']);
								$conditional_fail_messages[] = $message;
							}
						}
					} elseif (isset($item['response']['result']) && $item['response']['result'] == 3 && !$show_conditional_fail && $isCheckoutBlock) {
						$client_message_content = 'show_conditional_fail disabled';
						$client_message_type = 'N/A';

						// Set the client message
						self::$clientMessage = [
							$client_message_content,
							$client_message_type
						];

						// Set the cookie with the client message
						setcookie(
							'client_message',
							wp_json_encode([
								'message' => $client_message_content,
								'type' => $client_message_type
							]),
							time() + 1200,  // Cookie expires in 20 minutes
							COOKIEPATH,
							COOKIE_DOMAIN,
							is_ssl(),  // Secure cookie if using HTTPS
							false      // Not HTTPOnly to allow JavaScript access
						);
						$isECheckpointPassWithoutConditions = false;
					}
					if (isset($item['regulatedProduct']) && $item['regulatedProduct'] === true) {
						$product_id = wc_get_product_id_by_sku($item['response']['productID']);

						$product = wc_get_product($product_id);
						array_push($regulatedProducts, $product->get_name());
					}
				}

				if (count($regulatedProducts) > 0) {
					// Define the fee calculation action
					$calculate_fees_action = function () use ($regulatedProducts) {
						// Calculate and log the fee
						self::$total_fee = eCheckpoint_Compliance_Fee::ComplianceFee_calculate($regulatedProducts);

					};

					// Attach the fee calculation action to the WooCommerce hook
					add_action('woocommerce_cart_calculate_fees', $calculate_fees_action, 20);

					// Recalculate cart totals to trigger the fee calculation
					WC()->cart->calculate_totals();
				}



				if (!empty($failed_product_names)) {
					// Store failed products in session for blocks checkout validation
					WC()->session->set('echeckpoint_failed_products', $failed_product_names);

					$failed_product_list = "<ul>" . implode('', $failed_product_names) . "</ul>";
					/* translators: %s: failed product list */

					// For Block checkout OR non-place-order hooks, use cookies
					if ($isCheckoutBlock || $hook_name !== $place_order_hook) {

						// Create the client message as a variable
						$client_message_content = esc_html(self::get_blocked_products_message()) . '<br>' . $failed_product_list;


						$client_message_type = 'error';

						// Set the client message
						self::$clientMessage = [
							$client_message_content,
							$client_message_type
						];

						// Set the cookie with the client message
						setcookie(
							'client_message',
							wp_json_encode([
								'message' => $client_message_content,
								'type' => $client_message_type
							]),
							time() + 1200,  // Cookie expires in 20 minutes
							COOKIEPATH,
							COOKIE_DOMAIN,
							is_ssl(),  // Secure cookie if using HTTPS
							false      // Not HTTPOnly to allow JavaScript access
						);
						$isECheckpointPassWithoutConditions = false;
						return;
					} else {
						// Classic checkout place order - use WC notice
						$notice_message = esc_html(self::get_blocked_products_message()) . ' ' . $failed_product_list;

						// Add the notice using the variable
						wc_add_notice($notice_message, 'error');
					}
					$compliance_passed = false;
				}
			}


			// Display all conditional fail messages as a single notice
			if (!empty($conditional_fail_messages)) {
				//$conditional_fail_messages[] = __('If you are not eligible to purchase these products, please remove them from your shopping cart.', 'echeckpoint');

				$consumerTradeType = self::get_customer_type_from_session(); // Default fallback

				$ajax_action = '';
				$customer_type = '';

				if (defined('DOING_AJAX') && DOING_AJAX) {
					if (isset($_REQUEST['post_data'])) {
						// Ensure we get a proper string
						$post_data = sanitize_text_field(wp_unslash($_REQUEST['post_data']));

						// Parse the query string into an associative array
						parse_str($post_data, $post_data_array);

						// Check if 'echeckpoint_nonce' exists
						if (isset($post_data_array['echeckpoint_nonce'])) {
							// Sanitize the extracted nonce
							$echeckpoint_nonce = sanitize_text_field($post_data_array['echeckpoint_nonce']);
							if (!wp_verify_nonce($echeckpoint_nonce, 'echeckpoint_nonce')) {
								wp_send_json_error('Invalid nonce');
							} 
						}
					}
					$ajax_action = isset($_REQUEST['action']) ? sanitize_text_field(wp_unslash($_REQUEST['action'])) : '';
					$customer_type = isset($_REQUEST['customer_type']) ? sanitize_text_field(wp_unslash($_REQUEST['customer_type'])) : '';
				}

				if ($ajax_action === 'update_customer_type' && $customer_type !== '') {
					$consumerTradeType = $customer_type;
				} else if (isset($_COOKIE['consumerTradeType'])) {
					$cookie_value = sanitize_text_field(wp_unslash($_COOKIE['consumerTradeType']));
					// Decode the JSON value
					$decodedValue = json_decode($cookie_value, true); // Decode with stripslashes to handle escaped JSON
					if (json_last_error() === JSON_ERROR_NONE && isset($decodedValue['selectedTradeType'])) {
						$consumerTradeType = sanitize_text_field($decodedValue['selectedTradeType']);
					} else {
						$consumerTradeType = self::get_customer_type_from_session(); // Default fallback

					}
				} else {
					$consumerTradeType = self::get_customer_type_from_session(); // Default fallback
				}

				if (!empty($eligible_tradetypes) && is_array($eligible_tradetypes)) {
					if (in_array($consumerTradeType, $eligible_tradetypes)) {

						// Create the client message as a variable
						$client_message_content = implode('<br>', $conditional_fail_messages);
						$client_message_type = 'info';
						$client_message_total_fee = self::$total_fee;

						// Set the client message
						self::$clientMessage = [
							$client_message_content,
							$client_message_type,
							$client_message_total_fee
						];

						// Set the cookie with the client message
						setcookie(
							'client_message',
							wp_json_encode([
								'message' => $client_message_content,
								'type' => $client_message_type,
								'total_fee' => $client_message_total_fee
							]),
							time() + 1200,  // Cookie expires in 20 minutes
							COOKIEPATH,
							COOKIE_DOMAIN,
							is_ssl(),  // Secure cookie if using HTTPS
							false      // Not HTTPOnly to allow JavaScript access
						);
						$isECheckpointPassWithoutConditions = false;
						self::FFLCheck($response, $isCheckoutBlock, $regulatedProducts);
					} else {

						// If $failed_product_list is empty, try to build it from $regulatedProducts.
						if (empty($failed_product_list)) {
							if (!empty($regulatedProducts) && is_array($regulatedProducts)) {
								$failed_product_list = "<ul>";
								foreach ($regulatedProducts as $productName) {
									// Assuming $regulatedProducts is an array of product names.
									$failed_product_list .= "<li>" . esc_html($productName) . "</li>";
								}
								$failed_product_list .= "</ul>";
							}
						}

						// For Block checkout OR non-place-order hooks, use cookies
						if ($isCheckoutBlock || $hook_name !== $place_order_hook) {
							// Create the client message as a variable
							$client_message_content = esc_html(self::get_blocked_products_message()) . '<br>' . $failed_product_list;
							$client_message_type = 'error';

							// Set the client message
							self::$clientMessage = [
								$client_message_content,
								$client_message_type
							];

							// Set the cookie with the client message
							setcookie(
								'client_message',
								wp_json_encode([
									'message' => $client_message_content,
									'type' => $client_message_type
								]),
								time() + 1200,  // Cookie expires in 20 minutes
								COOKIEPATH,
								COOKIE_DOMAIN,
								is_ssl(),  // Secure cookie if using HTTPS
								false      // Not HTTPOnly to allow JavaScript access
							);
							$isECheckpointPassWithoutConditions = false;
							return;
						} else {
							// Classic checkout place order - use WC notice
							$notice_message = esc_html(self::get_blocked_products_message()) . ' ' . $failed_product_list;
							wc_add_notice($notice_message, 'error');
						}

					}
				}

				if (!$compliance_passed) {
					return;
				}

				if ($isECheckpointPassWithoutConditions) {
					// Clear any previously cached failed products since compliance now passes
					if (WC()->session) {
						WC()->session->__unset('echeckpoint_failed_products');
					}

					// Create the client message as a variable
					$client_message_content = 'compliance passed.';
					$client_message_type = 'N/A';

					// Set the client message
					self::$clientMessage = [
						$client_message_content,
						$client_message_type
					];

					// Set the cookie with the client message
					setcookie(
						'client_message',
						wp_json_encode([
							'message' => $client_message_content,
							'type' => $client_message_type
						]),
						time() + 1200,  // Cookie expires in 20 minutes
						COOKIEPATH,
						COOKIE_DOMAIN,
						is_ssl(),  // Secure cookie if using HTTPS
						false      // Not HTTPOnly to allow JavaScript access
					);
				}

				self::log_cart_details_on_checkout();
			}
		}

		public static function FFLCheck($response, $isCheckoutBlock, $regulatedProducts)
		{
			// Additional licensing checks
			$resultArray = []; // Initialize the array to hold the results

			// Get the selected customer type - check multiple sources in order of priority
			$selected_customer_type = 'b2C'; // Ultimate fallback

			// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing -- Called from WooCommerce hooks which handle nonce verification
			// 1. First priority: Direct request parameter (from AJAX call)
			if (isset($_REQUEST['customer_type']) && !empty($_REQUEST['customer_type'])) {
				$selected_customer_type = sanitize_text_field(wp_unslash($_REQUEST['customer_type']));
			}
			// 2. Second priority: POST data (backup)
			else if (isset($_POST['customer_type']) && !empty($_POST['customer_type'])) {
				$selected_customer_type = sanitize_text_field(wp_unslash($_POST['customer_type']));
			}
			// phpcs:enable WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
			// 3. Third priority: Cookie
			else if (isset($_COOKIE['consumerTradeType'])) {
				$cookie_value = sanitize_text_field(wp_unslash($_COOKIE['consumerTradeType']));
				$decodedValue = json_decode($cookie_value, true);
				if (json_last_error() === JSON_ERROR_NONE && isset($decodedValue['selectedTradeType'])) {
					$selected_customer_type = sanitize_text_field($decodedValue['selectedTradeType']);
				}
			}
			// 4. Fourth priority: Session
			else {
				$selected_customer_type = self::get_customer_type_from_session();
			}



			if (isset($response['modules']['regionalRestrictionsCheck']['licensing'])) {
				$licensing = $response['modules']['regionalRestrictionsCheck']['licensing'];

				foreach ($licensing as $license) {
					if ($license['name'] === 'FFL') {

						// Check if FFL has NOT been provided yet
						if (!$license['provided']) {
							$items = $response['modules']['regionalRestrictionsCheck']['items'];
							foreach ($items as $item) {
								$tradeTypes = $item['businessTradeTypes'];
								$sku = $item['response']['productID'];
								$product_id = wc_get_product_id_by_sku($sku);
								$product = wc_get_product($product_id);
								$product_name = $product->get_name();
								$productData = [
									'productID' => $sku,
									'productName' => $product_name
								];

								// Use effective trade type logic - get the type to use for this item
								$effective_type = self::get_effective_trade_type($tradeTypes, $selected_customer_type);


								// Log the trade type data for debugging
								if ($effective_type !== null && isset($tradeTypes[$effective_type])) {

									$ffl_required = isset($tradeTypes[$effective_type]['fflRequired']) ? $tradeTypes[$effective_type]['fflRequired'] : 'not set';

								}

								// Only add to result if effective type exists and requires FFL
								// Check the PRODUCT-level fflRequired from businessTradeTypes, not license-level
								if ($effective_type !== null &&
									isset($tradeTypes[$effective_type]['fflRequired']) &&
									$tradeTypes[$effective_type]['fflRequired']) {

									$productData['tradeTypes'][] = [
										'type' => $effective_type,
										'eligiblePurchaser' => $tradeTypes[$effective_type]['eligiblePurchaser'],
										'fflRequired' => $tradeTypes[$effective_type]['fflRequired']
									];
								} else {

								}

								// Add product data to the results array if it has tradeTypes
								if (!empty($productData['tradeTypes'])) {
									$resultArray[] = $productData;
								}
							}
						}
					}
				}
			}



			setcookie(
				'ffl_requirements',
				wp_json_encode($resultArray), // Encode $resultArray directly as JSON
				time() + 4800,
				COOKIEPATH,                  // Path for the cookie
				COOKIE_DOMAIN,               // Domain for the cookie
				is_ssl(),                    // Secure cookie if using HTTPS
				false                        // Not HTTPOnly to allow JavaScript access
			);

			// Use the selected customer type we already determined at the start of this function
			$consumerTradeType = $selected_customer_type;

			// Set consumerTradeType cookie for FFL map visibility in Block checkout
			// This ensures the cookie exists on initial page load (not just when customer type changes)
			$consumer_trade_type_value = json_encode(['selectedTradeType' => $selected_customer_type]);
			setcookie(
				'consumerTradeType',
				$consumer_trade_type_value,
				time() + 3600,
				COOKIEPATH,
				COOKIE_DOMAIN,
				is_ssl(),
				false
			);

			// Define the base notice content
			$client_message_content = self::get_ffl_required_message() . '<br>';

			// Get the WooCommerce cart items
			$cart = WC()->cart->get_cart();

			// Initialize the product list
			$product_list = '';

			foreach (WC()->cart->get_cart() as $cart_item) {
				// Get the product ID and product details
				$product_id = $cart_item['variation_id'] ?: $cart_item['product_id'];
				$product = wc_get_product($product_id);
				$product_sku = sanitize_text_field($product->get_sku());

				// Iterate through the resultArray to find matching products
				// Since resultArray now only contains the effective trade type (which accounts for
				// b2C fallback), we just check if the product is in the array - no need to compare
				// against consumerTradeType since we already filtered to the effective type
				foreach ($resultArray as $result) {
					if ($result['productID'] === $product_sku && !empty($result['tradeTypes'])) {
						// If a match is found, add the product name to the product_list
						$product_list .= "&emsp;&nbsp;" . $product->get_name() . "<br>";
						break; // Exit after a match
					}
				}
			}

			if ($isCheckoutBlock) {
				// If there are regulated products, append them to the message
				if (!empty($product_list)) {
					$client_message_content .= $product_list;
					$client_message_content .= "<br>Please update the shipping address to an active FFL address.";
					// Set the message type
					$client_message_type = 'error';
					// Set the cookie with the updated client message
					setcookie(
						'ffl_message',
						wp_json_encode([
							'message' => $client_message_content,
							'type' => $client_message_type,
							'products_ffl_required' => $regulatedProducts
						]),
						time() + 1200,  // Cookie expires in 20 minutes
						COOKIEPATH,
						COOKIE_DOMAIN,
						is_ssl(),  // Secure cookie if using HTTPS
						false      // Not HTTPOnly to allow JavaScript access
					);

					// 1. Generate a unique key (for example, using wp_generate_password)
					$unique_key = 'ffl_response_' . wp_generate_password(12, false);

					// 2. Store the large response on the server using a transient (expires in 1 hour)
					set_transient($unique_key, $response, 60 * 60);

					// 3. Set a cookie with just the unique key
					setcookie(
						'ffl_response_key',   // Cookie name
						$unique_key,          // Cookie value (the unique key)
						time() + 3600,        // Expires in 1 hour
						COOKIEPATH,           // Path (WordPress constant)
						COOKIE_DOMAIN,        // Domain (WordPress constant)
						is_ssl(),
						false           // Secure if using HTTPS
					);

					return;
				} else {
					$client_message_content = 'N/A';
					// Set the message type
					$client_message_type = 'N/A';
					// Set the cookie with the updated client message
					setcookie(
						'ffl_message',
						wp_json_encode([
							'message' => $client_message_content,
							'type' => $client_message_type,
							'products_ffl_required' => $regulatedProducts
						]),
						time() + 1200,  // Cookie expires in 20 minutes
						COOKIEPATH,
						COOKIE_DOMAIN,
						is_ssl(),  // Secure cookie if using HTTPS
						false      // Not HTTPOnly to allow JavaScript access
					);

					// Always cache the API response for JavaScript to use
					// This enables customer type switching without new API calls
					$unique_key = 'ffl_response_' . wp_generate_password(12, false);
					set_transient($unique_key, $response, 60 * 60);
					setcookie(
						'ffl_response_key',
						$unique_key,
						time() + 3600,
						COOKIEPATH,
						COOKIE_DOMAIN,
						is_ssl(),
						false
					);
				}

			} else if (!self::is_block_checkout_request()) { //classic shortcode checkout notice
				// Classic Shortcode Checkout Notice - safety check to never add WC notices during Block checkout

				if (!empty($product_list)) {
					$client_message_content .= $product_list . "<br>Please update the shipping address to an active FFL address.";
					wc_add_notice($client_message_content, 'error');

					// 1. Generate a unique key (for example, using wp_generate_password)
					$unique_key = 'ffl_response_' . wp_generate_password(12, false);

					// 2. Store the large response on the server using a transient (expires in 1 hour)
					set_transient($unique_key, $response, 60 * 60);

					// 3. Set a cookie with just the unique key
					setcookie(
						'ffl_response_key',   // Cookie name
						$unique_key,          // Cookie value (the unique key)
						time() + 3600,        // Expires in 1 hour
						COOKIEPATH,           // Path (WordPress constant)
						COOKIE_DOMAIN,        // Domain (WordPress constant)
						is_ssl(),
						false           // Secure if using HTTPS
					);

				}
			}

		}

		public static function api_call_during_checkout($checkout_data)
		{

			if (!self::pre_order_active())
				return;

			if (is_array($checkout_data) && isset($checkout_data['billing_email'])) {
				$customer_email = sanitize_email($checkout_data['billing_email']);
			} else {
				$customer_email = ''; // or handle it differently based on your logic
			}
			$order_id = gmdate('Ymd_His', microtime(true)); // Use email if available, otherwise use timestamp

			// Fetch cart products and handle errors
			$cart_products = self::get_cart_products();
			if (is_wp_error($cart_products)) {
				return false;
			}

			// Fetch billing address and handle errors
			$billing_address = self::get_billing_address($checkout_data);
			if ($billing_address == false) {
				return;
			}
			if (is_wp_error($billing_address)) {
				return false;
			}

			// Fetch shipping address and handle errors
			$shipping_address = self::get_shipping_address($checkout_data);
			if ($shipping_address == false) {
				return;
			}
			if (is_wp_error($shipping_address)) {
				return false;
			}

			$json_data = [
				'OrderID' => sanitize_text_field($order_id),
				'Products' => $cart_products,
				'BillingAddress' => $billing_address,
				'ShippingAddress' => $shipping_address,
				'CustomerEmail' => $customer_email // Added customer email
			];
			$json_data_encoded = wp_json_encode($json_data);

			//PROD
			$api_url = 'https://api.echeckpoint.com/api/compliancecheck/getresults';
			//DEV PROD DOCKER
			//$api_url = 'http://host.docker.internal:5000/api/compliancecheck/getresults';

			//DEV PROD
			//$api_url = 'http://tetra-accurate-nationally.ngrok-free.app/api/compliancecheck/getresults';

			$api_key = sanitize_text_field(get_option('eCheckpoint_pre_settings')['eCheckpoint_pre_order_apikey'] ?? '');

			$response = wp_remote_post($api_url, [
				'body' => $json_data_encoded,
				'headers' => [
					'Content-Type' => 'application/json',
					'Authorization' => 'Bearer ' . $api_key,
				],
				'timeout' => 30,
			]);

			if (is_wp_error($response)) {
				return false;
			}

			$response_body = wp_remote_retrieve_body($response);

			return json_decode($response_body, true);
		}

		public static function get_billing_address($checkout_data)
		{
			$billing_complete = self::is_billing_complete($checkout_data);
			$shipping_complete = self::is_shipping_complete($checkout_data);

			if ($billing_complete) {
				// Extract billing address from checkout data
				return [
					'CustomerFirstName' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
					'CustomerLastName' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
					'Address1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
					'Address2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
					'City' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
					'State' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
					'PostalCode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
					'Email' => sanitize_text_field($checkout_data['billing_email'] ?? '')
				];
			} else if ($shipping_complete) {
				// Extract shipping address if billing isn't complete
				return [
					'CustomerFirstName' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
					'CustomerLastName' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
					'Address1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
					'Address2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
					'City' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
					'State' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
					'PostalCode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
				];
			} else {
				return false;
			}
		}

		public static function get_shipping_address($checkout_data)
		{
			$billing_complete = self::is_billing_complete($checkout_data);
			$shipping_complete = self::is_shipping_complete($checkout_data);

			if (self::is_block_checkout_request()) {
				$checkbox_checked = true;
			} else {
				$checkbox_checked = isset($checkout_data['ship_to_different_address']) ? (bool) $checkout_data['ship_to_different_address'] : false;
			}

			if ($checkbox_checked && $shipping_complete) {
				// Extract shipping address from checkout data
				return [
					'CustomerFirstName' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
					'CustomerLastName' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
					'Address1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
					'Address2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
					'Company' => sanitize_text_field($checkout_data['shipping_company'] ?? ''),
					'City' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
					'State' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
					'PostalCode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
				];
			} else if ($billing_complete) {
				// Extract billing address if shipping isn't complete
				return [
					'CustomerFirstName' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
					'CustomerLastName' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
					'Address1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
					'Address2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
					'City' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
					'State' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
					'PostalCode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
					'Email' => sanitize_text_field($checkout_data['billing_email'] ?? '')
				];
			} else {
				return false;
			}
		}

		public static function get_cart_products()
		{
			$product_skus = [];
			foreach (WC()->cart->get_cart() as $cart_item) {
				$product = $cart_item['data'];
				$sku = sanitize_text_field($product->get_sku());

				if (empty($sku)) {
					$product_name = sanitize_text_field($product->get_name());
					return new WP_Error
					(
						'empty_sku',
						sprintf('Product SKU is missing for product "%s".', $product_name),
						['product_name' => $product_name, 'cart_item_key' => $cart_item['key']]
					);
				}

				$product_skus[] = [
					'ID' => $sku,
					'Name' => sanitize_text_field($product->get_name())
				];
			}
			return $product_skus;
		}

		/**
		 * Get the effective customer type with b2C fallback logic.
		 *
		 * Per the eCheckpoint API matrix:
		 * - If selected type is eligible → use selected type
		 * - If selected type is NOT eligible AND b2C IS eligible → fall back to b2C
		 * - If neither is eligible → return null (blocked)
		 *
		 * @param array $business_trade_types The businessTradeTypes from API response
		 * @param string $selected_type The customer-selected trade type (b2C, b2B, or b2G)
		 * @return string|null The effective trade type to use, or null if blocked
		 */
		private static function get_effective_trade_type($business_trade_types, $selected_type) {
			// Check if selected type is eligible
			if (isset($business_trade_types[$selected_type]['eligiblePurchaser']) &&
				$business_trade_types[$selected_type]['eligiblePurchaser']) {
				return $selected_type;
			}

			// Selected type not eligible - try b2C fallback (unless already b2C)
			if ($selected_type !== 'b2C') {
				if (isset($business_trade_types['b2C']['eligiblePurchaser']) &&
					$business_trade_types['b2C']['eligiblePurchaser']) {
					return 'b2C'; // Fallback to consumer rules
				}
			}

			// Neither selected type nor b2C is eligible - blocked
			return null;
		}

		private static function get_eligible_tradetypes($business_trade_types)
		{
			$requirements = [];

			// Get customer type - same priority as FFLCheck
			$customer_type = 'b2C'; // Default
			// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing -- Called from WooCommerce hooks which handle nonce verification
			if (isset($_REQUEST['customer_type']) && !empty($_REQUEST['customer_type'])) {
				$customer_type = sanitize_text_field(wp_unslash($_REQUEST['customer_type']));
			} else if (isset($_POST['customer_type']) && !empty($_POST['customer_type'])) {
				$customer_type = sanitize_text_field(wp_unslash($_POST['customer_type']));
			// phpcs:enable WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
			} else if (isset($_COOKIE['consumerTradeType'])) {
				$cookie_value = sanitize_text_field(wp_unslash($_COOKIE['consumerTradeType']));
				$decodedValue = json_decode($cookie_value, true);
				if (json_last_error() === JSON_ERROR_NONE && isset($decodedValue['selectedTradeType'])) {
					$customer_type = sanitize_text_field($decodedValue['selectedTradeType']);
				}
			}

			// Use effective trade type logic with b2C fallback
			$effective_type = self::get_effective_trade_type($business_trade_types, $customer_type);

			if ($effective_type !== null) {
				$requirements[] = $effective_type;
			}
			// If null, requirements stays empty (blocked)

			return $requirements;
		}

		/***************************************************************************************************** */
		// WOOCOMMERCE CHECKOUT BLOCK
/***************************************************************************************************** */
		public static function add_fee()
		{
			if (!self::is_block_checkout_request()) {
				return;
			}

			// Clear any WooCommerce notices - Block checkout uses React notices via cookies
			wc_clear_notices();

			// Run compliance check; adapt this call if parameters are needed
			eCheckpoint_Pre_Order_Checks::check_required_fields_and_run_compliance_blocks();
		}

		public static function log_cart_details_on_checkout()
		{
			if (!WC()->cart) {
				return;
			}

			$cart_items = WC()->cart->get_cart();
			$log_data = [];

			// Log details for each product in the cart
			foreach ($cart_items as $cart_item_key => $cart_item) {
				$product = $cart_item['data'];

				$log_data['Products'][] = [
					'Product Name' => $product->get_name(),
					'Product ID' => $product->get_id(),
					'SKU' => $product->get_sku(),
					'Quantity' => $cart_item['quantity'],
					'Price' => wc_price($product->get_price()),
					'Subtotal' => wc_price($cart_item['line_subtotal']),
					'Total' => wc_price($cart_item['line_total'])
				];
			}

			// Log any fees applied in the cart
			$fees = WC()->cart->get_fees();
			if (!empty($fees)) {
				foreach ($fees as $fee) {
					$log_data['Fees'][] = [
						'Fee Name' => $fee->name,
						'Amount' => wc_price($fee->amount)
					];
				}
			} else {
				$log_data['Fees'] = 'No fees applied';
			}

			// Log cart totals
			$log_data['Cart Totals'] = [
				'Cart Total (excluding tax)' => wc_price(WC()->cart->cart_contents_total),
				'Total Tax' => wc_price(WC()->cart->get_cart_contents_tax() + WC()->cart->get_shipping_tax()),
				'Fees Total' => wc_price(WC()->cart->get_fee_total()),
				'Shipping Total' => wc_price(WC()->cart->get_shipping_total()),
				'Grand Total' => wc_price(WC()->cart->get_total(''))
			];
		}


		public static function run_compliance_check()
		{
			//Verify nonce if needed for security
			if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'echeckpoint_nonce')) {
				wp_send_json_error('Invalid nonce');
			}


			if (!self::is_block_checkout_request()) {
				return;
			}

			// Clear any WooCommerce notices - Block checkout uses React notices via cookies
			wc_clear_notices();

			// Run compliance check; adapt this call if parameters are needed
			$result = eCheckpoint_Pre_Order_Checks::check_required_fields_and_run_compliance_blocks();
			//If the compliance check passes, set the cookie
			if (!empty(self::$clientMessage)) {
				$data = array(
					'messages' => isset(self::$clientMessage[0]) ? self::$clientMessage[0] : '', // Default to an empty string if not set
					'notice' => isset(self::$clientMessage[1]) ? self::$clientMessage[1] : 'info', // Default notice type
					'total_fee' => isset(self::$clientMessage[2]) ? self::$clientMessage[2] : 0 // Default to 0 if not set
				);
				wp_send_json_success($data);
			} else {
				wp_send_json_error('Compliance check failed, cookie not set.');
			}
		}

		// Handle customer update for WooCommerce Blocks

		public static function handle_blocks_customer_update($customer, $request)
		{
			// Note: This hook is ONLY called from WooCommerce Blocks Store API
			// so we don't need to check is_checkout_block() here

			// Clear any existing WooCommerce notices to prevent duplicates
			// Block checkout uses React notices via cookies, not wc_add_notice()
			wc_clear_notices();

			// Skip compliance for non-US orders - eCheckpoint only covers United States
			$shipping_country = $customer->get_shipping_country();
			$billing_country = $customer->get_billing_country();
			$country = !empty($shipping_country) ? $shipping_country : $billing_country;
			if ($country !== 'US') {
				return;
			}

			// Check if this is a customer type update via AJAX or JS cookie flag
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Called from WooCommerce Store API hook which handles its own verification
			$ajax_action = isset($_REQUEST['action']) ? sanitize_text_field(wp_unslash($_REQUEST['action'])) : '';
			$skip_api_cookie = isset($_COOKIE['echeckpoint_skip_api']) && $_COOKIE['echeckpoint_skip_api'] === '1';
			$is_customer_type_update = ($ajax_action === 'update_customer_type') || $skip_api_cookie;

			// Clear the skip API cookie after reading it
			if ($skip_api_cookie) {
				setcookie('echeckpoint_skip_api', '', time() - 3600, '/');
			}

			// Get billing and shipping addresses from the customer object directly
			$billing_address = array(
				'first_name' => $customer->get_billing_first_name(),
				'last_name' => $customer->get_billing_last_name(),
				'address_1' => $customer->get_billing_address_1(),
				'address_2' => $customer->get_billing_address_2(),
				'city' => $customer->get_billing_city(),
				'state' => $customer->get_billing_state(),
				'postcode' => $customer->get_billing_postcode(),
				'country' => $customer->get_billing_country(),
				'phone' => $customer->get_billing_phone(),
				'email' => $customer->get_billing_email()
			);

			$shipping_address = array(
				'first_name' => $customer->get_shipping_first_name(),
				'last_name' => $customer->get_shipping_last_name(),
				'address_1' => $customer->get_shipping_address_1(),
				'address_2' => $customer->get_shipping_address_2(),
				'city' => $customer->get_shipping_city(),
				'state' => $customer->get_shipping_state(),
				'postcode' => $customer->get_shipping_postcode(),
				'country' => $customer->get_shipping_country()
			);

			// For customer type updates, fall back to session data if customer object is empty
			if ($is_customer_type_update && empty($billing_address['address_1']) && WC()->session) {
				$session_billing = WC()->session->get('billing_address', array());
				$session_shipping = WC()->session->get('shipping_address', array());

				if (!empty($session_billing)) {
					$billing_address = array_merge($billing_address, $session_billing);
				}
				if (!empty($session_shipping)) {
					$shipping_address = array_merge($shipping_address, $session_shipping);
				}
			}

			// Store the complete billing and shipping addresses in session
			if (!empty($billing_address['address_1'])) {
				WC()->session->set('billing_address', $billing_address);
			}
			if (!empty($shipping_address['address_1'])) {
				WC()->session->set('shipping_address', $shipping_address);
			}

			// Create checkout_data array from the customer object
			$checkout_data = array(
				'billing_first_name' => $billing_address['first_name'],
				'billing_last_name' => $billing_address['last_name'],
				'billing_address_1' => $billing_address['address_1'],
				'billing_address_2' => $billing_address['address_2'],
				'billing_city' => $billing_address['city'],
				'billing_state' => $billing_address['state'],
				'billing_postcode' => $billing_address['postcode'],
				'billing_country' => $billing_address['country'],
				'billing_phone' => $billing_address['phone'],
				'billing_email' => $billing_address['email'],
				'shipping_first_name' => $shipping_address['first_name'],
				'shipping_last_name' => $shipping_address['last_name'],
				'shipping_address_1' => $shipping_address['address_1'],
				'shipping_address_2' => $shipping_address['address_2'],
				'shipping_city' => $shipping_address['city'],
				'shipping_state' => $shipping_address['state'],
				'shipping_postcode' => $shipping_address['postcode'],
				'shipping_country' => $shipping_address['country']
			);


			// Use the new helper function to check if address fields have been updated
			$update_type = self::is_address_update($checkout_data);

			if ($update_type === false) {
				return; // Exit the function early if no critical fields have been updated
			}

			// Prepare data to check for compliance
			$data_to_check = array(
				'billing_first_name' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
				'billing_last_name' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
				'billing_address_1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
				'billing_address_2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
				'billing_city' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
				'billing_state' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
				'billing_postcode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
				'billing_email' => sanitize_email($checkout_data['billing_email'] ?? ''),
				'ship_to_different_address' => !empty($checkout_data['ship_to_different_address']) ? '1' : '0',
				'shipping_first_name' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
				'shipping_last_name' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
				'shipping_address_1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
				'shipping_address_2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
				'shipping_city' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
				'shipping_state' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
				'shipping_postcode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
			);

			// Check if we have a cached response and this is just a customer type change
			$cached_response = self::get_cached_compliance_response();

			if ($is_customer_type_update && $cached_response) {
				// Customer type changed but address is the same - use cached response
				// Just re-evaluate the messages with the new customer type

				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Called from WooCommerce Store API hook which handles its own verification
				$customer_type = isset($_REQUEST['customer_type']) ? sanitize_text_field(wp_unslash($_REQUEST['customer_type'])) : 'b2C';

				self::reevaluate_cached_response($cached_response, $data_to_check);
			} else if (self::are_required_fields_completed($data_to_check)) {
				// New address or no cache - make full API call

				// Pass true to force Block checkout mode (cookies only, no wc_add_notice)
				self::trigger_compliance_check($data_to_check, true);
			}

			// Final cleanup - ensure no WC notices leak to Block checkout
			wc_clear_notices();
			if (WC()->session) {
				WC()->session->set('wc_notices', array());
			}
		}

		public static function check_required_fields_and_run_compliance_blocks()
		{

			if (!self::is_block_checkout_request()) {
				return;
			}

			// Clear any WooCommerce notices - Block checkout uses React notices via cookies
			wc_clear_notices();

			// Log the 'customer' session data
			$customer_data = WC()->session->get('customer', []);

			$checkout_data = array(
				'billing_first_name' => $customer_data['first_name'] ?? '',
				'billing_last_name' => $customer_data['last_name'] ?? '',
				'billing_address_1' => $customer_data['address_1'] ?? '',
				'billing_city' => $customer_data['city'] ?? '',
				'billing_state' => $customer_data['state'] ?? '',
				'billing_email' => $customer_data['email'] ?? '',
				'billing_postcode' => $customer_data['postcode'] ?? '',
				'shipping_first_name' => $customer_data['shipping_first_name'] ?? '',
				'shipping_last_name' => $customer_data['shipping_last_name'] ?? '',
				'shipping_address_1' => $customer_data['shipping_address_1'] ?? '',
				'shipping_company' => $customer_data['shipping_company'] ?? '',
				'shipping_city' => $customer_data['shipping_city'] ?? '',
				'shipping_state' => $customer_data['shipping_state'] ?? '',
				'shipping_postcode' => $customer_data['shipping_postcode'] ?? ''
			);

			// Use the new helper function to check if address fields have been updated
			if (!self::is_address_update($checkout_data)) {
				return; // Exit the function early if no critical fields have been updated
			}

			// Prepare data to check for compliance
			$data_to_check = array(
				'billing_first_name' => sanitize_text_field($checkout_data['billing_first_name'] ?? ''),
				'billing_last_name' => sanitize_text_field($checkout_data['billing_last_name'] ?? ''),
				'billing_address_1' => sanitize_text_field($checkout_data['billing_address_1'] ?? ''),
				'billing_address_2' => sanitize_text_field($checkout_data['billing_address_2'] ?? ''),
				'billing_city' => sanitize_text_field($checkout_data['billing_city'] ?? ''),
				'billing_state' => sanitize_text_field($checkout_data['billing_state'] ?? ''),
				'billing_postcode' => sanitize_text_field($checkout_data['billing_postcode'] ?? ''),
				'billing_email' => sanitize_email($checkout_data['billing_email'] ?? ''),
				'ship_to_different_address' => !empty($checkout_data['ship_to_different_address']) ? '1' : '0',
				'shipping_first_name' => sanitize_text_field($checkout_data['shipping_first_name'] ?? ''),
				'shipping_last_name' => sanitize_text_field($checkout_data['shipping_last_name'] ?? ''),
				'shipping_company' => sanitize_text_field($checkout_data['shipping_company'] ?? ''),
				'shipping_address_1' => sanitize_text_field($checkout_data['shipping_address_1'] ?? ''),
				'shipping_address_2' => sanitize_text_field($checkout_data['shipping_address_2'] ?? ''),
				'shipping_city' => sanitize_text_field($checkout_data['shipping_city'] ?? ''),
				'shipping_state' => sanitize_text_field($checkout_data['shipping_state'] ?? ''),
				'shipping_postcode' => sanitize_text_field($checkout_data['shipping_postcode'] ?? '')
			);

			$result = self::are_required_fields_completed($data_to_check);

			if ($result) {
				// Run the compliance check - pass true for block checkout mode
				self::trigger_compliance_check($data_to_check, true);
			}

			// Final cleanup - ensure no WC notices leak to Block checkout
			wc_clear_notices();
			if (WC()->session) {
				WC()->session->set('wc_notices', array());
			}
		}

		public static function is_checkout_block()
		{
			$result = WC_Blocks_Utils::has_block_in_page(wc_get_page_id('checkout'), 'woocommerce/checkout');
			return $result;
		}

		/**
		 * Detect if current request is from Block checkout
		 * More reliable than is_checkout_block() during AJAX/REST requests
		 */
		public static function is_block_checkout_request()
		{
			// 1. Check cookie set on page load (most reliable)
			if (isset($_COOKIE['checkout_block']) && $_COOKIE['checkout_block'] === 'true') {
				return true;
			}

			// 2. Check REST_REQUEST (Store API calls)
			if (defined('REST_REQUEST') && REST_REQUEST) {
				return true;
			}

			// 3. Check for Store API nonce header
			if (isset($_SERVER['HTTP_X_WC_STORE_API_NONCE'])) {
				return true;
			}

			// 4. Fallback to page check for initial load only
			if (!defined('DOING_AJAX') && !defined('REST_REQUEST')) {
				return self::is_checkout_block();
			}

			return false;
		}

		private static function get_customer_type_from_session()
		{
			// Check if WooCommerce session exists
			$customer_data = WC()->session->get('customer');

			// 1️⃣ Check if `meta_data` exists and contains the customer type
			if (!empty($customer_data['meta_data'])) {
				foreach ($customer_data['meta_data'] as $meta) {
					if ($meta['key'] === '_wc_other/namespace/select-tradetype' && !empty($meta['value'])) {
						return $meta['value']; // Return stored customer type
					}
				}
			}

			// 2️⃣ If logged in, check user meta (for returning customers)
			if (is_user_logged_in()) {
				$user_id = get_current_user_id();
				$meta_type = get_user_meta($user_id, 'customer_type', true);
				if (!empty($meta_type)) {
					return $meta_type; // Return stored meta value
				}
			}

			// 3️⃣ Default to 'b2C' for new customers or guests
			return 'b2C';
		}

		/**
		 * Cache the compliance API response for re-use when only customer type changes
		 */
		private static function cache_compliance_response($response) {
			if (WC()->session) {
				WC()->session->set('echeckpoint_cached_response', $response);
				WC()->session->set('echeckpoint_cache_time', time());
			}
		}

		/**
		 * Get the cached compliance API response
		 * Cache expires after 30 minutes
		 */
		private static function get_cached_compliance_response() {
			if (!WC()->session) {
				return null;
			}

			$cached_response = WC()->session->get('echeckpoint_cached_response');
			$cache_time = WC()->session->get('echeckpoint_cache_time', 0);

			// Cache expires after 30 minutes
			if ($cached_response && (time() - $cache_time) < 1800) {
				return $cached_response;
			}

			return null;
		}

		/**
		 * Clear the cached compliance response (called when address changes)
		 */
		private static function clear_cached_compliance_response() {
			if (WC()->session) {
				WC()->session->set('echeckpoint_cached_response', null);
				WC()->session->set('echeckpoint_cache_time', 0);
			}
		}

		/**
		 * Re-evaluate cached API response with new customer type
		 * This avoids making a new API call when only customer type changes
		 */
		private static function reevaluate_cached_response($response, $checkout_data) {
			$options = get_option('eCheckpoint_pre_settings');
			$regulatedProducts = [];
			$isCheckoutBlock = true; // We're always in Block checkout context here

			// Get the customer type from the AJAX request
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Called from WooCommerce AJAX hooks which handle nonce verification
			$customer_type = isset($_REQUEST['customer_type']) ? sanitize_text_field(wp_unslash($_REQUEST['customer_type'])) : 'b2C';



			// Clear any WC notices
			wc_clear_notices();

			// Extract regulated products from the cached response
			if (isset($response['modules']['regionalRestrictionsCheck']['items'])) {
				foreach ($response['modules']['regionalRestrictionsCheck']['items'] as $item) {
					if (isset($item['regulatedProduct']) && $item['regulatedProduct'] === true) {
						$product_id = wc_get_product_id_by_sku($item['response']['productID']);
						$product = wc_get_product($product_id);
						if ($product) {
							array_push($regulatedProducts, $product->get_name());
						}
					}
				}
			}

			// Re-run FFLCheck with the cached response and current customer type
			self::FFLCheck_cached($response, $isCheckoutBlock, $regulatedProducts, $customer_type);

			// Re-evaluate eligibility messages based on new customer type
			self::reevaluate_eligibility_messages($response, $checkout_data, $regulatedProducts, $customer_type);
		}

		/**
		 * FFLCheck for cached responses - takes explicit customer type
		 */
		private static function FFLCheck_cached($response, $isCheckoutBlock, $regulatedProducts, $customer_type) {
			$resultArray = [];



			if (isset($response['modules']['regionalRestrictionsCheck']['licensing'])) {
				$licensing = $response['modules']['regionalRestrictionsCheck']['licensing'];

				foreach ($licensing as $license) {
					if ($license['name'] === 'FFL') {

						if (!$license['provided']) {
							$items = $response['modules']['regionalRestrictionsCheck']['items'];
							foreach ($items as $item) {
								$tradeTypes = $item['businessTradeTypes'];
								$sku = $item['response']['productID'];
								$product_id = wc_get_product_id_by_sku($sku);
								$product = wc_get_product($product_id);
								$product_name = $product ? $product->get_name() : $sku;
								$productData = [
									'productID' => $sku,
									'productName' => $product_name
								];

								// Use effective trade type logic with explicit customer type
								$effective_type = self::get_effective_trade_type($tradeTypes, $customer_type);


								// Log the trade type data for debugging
								if ($effective_type !== null && isset($tradeTypes[$effective_type])) {

									$ffl_required = isset($tradeTypes[$effective_type]['fflRequired']) ? $tradeTypes[$effective_type]['fflRequired'] : 'not set';

								}

								// Only add to result if effective type exists and requires FFL
								if ($effective_type !== null &&
									isset($tradeTypes[$effective_type]['fflRequired']) &&
									$tradeTypes[$effective_type]['fflRequired']) {

									$productData['tradeTypes'][] = [
										'type' => $effective_type,
										'eligiblePurchaser' => $tradeTypes[$effective_type]['eligiblePurchaser'],
										'fflRequired' => $tradeTypes[$effective_type]['fflRequired']
									];
								} else {

								}

								if (!empty($productData['tradeTypes'])) {
									$resultArray[] = $productData;
								}
							}
						}
					}
				}
			}



			// Set ffl_requirements cookie
			setcookie(
				'ffl_requirements',
				wp_json_encode($resultArray),
				time() + 4800,
				COOKIEPATH,
				COOKIE_DOMAIN,
				is_ssl(),
				false
			);

			// Set consumerTradeType cookie for FFL map visibility in Block checkout
			$consumer_trade_type_value = json_encode(['selectedTradeType' => $customer_type]);
			setcookie(
				'consumerTradeType',
				$consumer_trade_type_value,
				time() + 3600,
				COOKIEPATH,
				COOKIE_DOMAIN,
				is_ssl(),
				false
			);

			// Build product list for FFL message
			$client_message_content = self::get_ffl_required_message() . '<br>';
			$product_list = '';

			foreach (WC()->cart->get_cart() as $cart_item) {
				$product_id = $cart_item['variation_id'] ?: $cart_item['product_id'];
				$product = wc_get_product($product_id);
				$product_sku = sanitize_text_field($product->get_sku());

				foreach ($resultArray as $result) {
					if ($result['productID'] === $product_sku && !empty($result['tradeTypes'])) {
						$product_list .= "&emsp;&nbsp;" . $product->get_name() . "<br>";
						break;
					}
				}
			}

			// Set ffl_message cookie based on results
			if (!empty($product_list)) {
				$client_message_content .= $product_list;
				$client_message_content .= "<br>Please update the shipping address to an active FFL address.";
				setcookie(
					'ffl_message',
					wp_json_encode([
						'message' => $client_message_content,
						'type' => 'error',
						'products_ffl_required' => $regulatedProducts
					]),
					time() + 1200,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);
			} else {
				// No FFL required - clear the message
				setcookie(
					'ffl_message',
					wp_json_encode([
						'message' => 'N/A',
						'type' => 'N/A',
						'products_ffl_required' => $regulatedProducts
					]),
					time() + 1200,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);
			}
		}

		/**
		 * Re-evaluate eligibility messages for the cached response
		 */
		private static function reevaluate_eligibility_messages($response, $checkout_data, $regulatedProducts, $customer_type = 'b2C') {
			$options = get_option('eCheckpoint_pre_settings');
			$show_conditional_fail = isset($options['conditional_fail_checkbox']) && $options['conditional_fail_checkbox'] === 'on';
			$conditional_fail_messages = [];
			$failed_product_names = [];

			if (isset($response['modules']['regionalRestrictionsCheck']['items'])) {
				foreach ($response['modules']['regionalRestrictionsCheck']['items'] as $item) {
					$product_id = wc_get_product_id_by_sku($item['response']['productID']);
					$product = wc_get_product($product_id);

					// Check eligibility for the specific customer type
					if (isset($item['businessTradeTypes'])) {
						$business_trade_types = $item['businessTradeTypes'];
						$effective_type = self::get_effective_trade_type($business_trade_types, $customer_type);

						// If no effective type (blocked for this customer type)
						if ($effective_type === null) {
							// Check result code - result 2 = hard fail
							if (isset($item['response']['result']) && $item['response']['result'] == 2) {
								if ($product) {
									$failed_product_names[] = "<li>" . $product->get_name() . "</li>";
								}
							}
						} else if (isset($item['response']['result']) && $item['response']['result'] == 3 && $show_conditional_fail) {
							// Conditional result - additional info required
							if ($product) {
								/* translators: %1$s: product name */
								$message = sprintf(__('To complete the purchase of %1$s, additional information will be required after checkout.', 'echeckpoint'), $product->get_name());
								$conditional_fail_messages[] = $message;
							}
						}
					}
				}
			}

			// Set the appropriate client_message cookie based on results
			if (!empty($failed_product_names)) {
				// Products that cannot be shipped
				// Store failed products in session for Block checkout validation
				WC()->session->set('echeckpoint_failed_products', $failed_product_names);


				$failed_product_list = "<ul>" . implode('', $failed_product_names) . "</ul>";
				$client_message_content = esc_html(self::get_blocked_products_message()) . '<br>' . $failed_product_list;
				$client_message_type = 'error';

				setcookie(
					'client_message',
					wp_json_encode([
						'message' => $client_message_content,
						'type' => $client_message_type
					]),
					time() + 1200,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);
			} else {
				// Clear failed products from session if none failed
				WC()->session->set('echeckpoint_failed_products', []);
			}

			if (!empty($failed_product_names)) {
				// Already handled above
			} else if (!empty($conditional_fail_messages)) {
				// Conditional pass - additional info required
				$client_message_content = implode('<br>', $conditional_fail_messages);
				$client_message_type = 'info';

				setcookie(
					'client_message',
					wp_json_encode([
						'message' => $client_message_content,
						'type' => $client_message_type,
						'total_fee' => self::$total_fee
					]),
					time() + 1200,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);
			} else {
				// No issues - compliance passed
				setcookie(
					'client_message',
					wp_json_encode([
						'message' => 'compliance passed.',
						'type' => 'N/A'
					]),
					time() + 1200,
					COOKIEPATH,
					COOKIE_DOMAIN,
					is_ssl(),
					false
				);
			}
		}


	}
}

// Initialize the pre-order checks
eCheckpoint_Pre_Order_Checks::init();

?>
