<?php

///////////////////////////////////////////
// POST-ORDER CHECK
// echeckpoint_post-order-check.php
///////////////////////////////////////////
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('eCheckpoint_Post_Order_Checks')) {
    class eCheckpoint_Post_Order_Checks {

        public static function api_call($order_id) {
            // Get The Order Object
            $order = wc_get_order($order_id);

            // Prevent duplicate API calls on page refresh/revisit
            if ($order->get_meta('_echeckpoint_post_check_completed')) {
                return;
            }
            $order->update_meta_data('_echeckpoint_post_check_completed', current_time('mysql'));
            $order->save();

            // Check if the state is excluded before running the compliance check
            if (self::is_state_excluded($order)) {
                return;
            }

            // Skip compliance for non-US orders - eCheckpoint only covers United States
            $shipping_country = $order->get_shipping_country();
            $billing_country = $order->get_billing_country();
            $country = !empty($shipping_country) ? $shipping_country : $billing_country;
            if ($country !== 'US') {
                $order->add_order_note(__('eCheckpoint: Compliance check bypassed - customer is not in the United States.', 'echeckpoint'));
                return;
            }

            // Initialization To Avoid Duplication
            $billing_first_name = sanitize_text_field($order->get_billing_first_name());
            $billing_last_name = sanitize_text_field($order->get_billing_last_name());
            $billing_company = sanitize_text_field($order->get_billing_company());
            $billing_email = sanitize_email($order->get_billing_email());

            // Get The Full Billing Address
            $billing_address = array(
                'CustomerFirstName' => $billing_first_name,
                'CustomerLastName' => $billing_last_name,
                'Address1' => sanitize_text_field($order->get_billing_address_1()),
                'Address2' => sanitize_text_field($order->get_billing_address_2()),
                'City' => sanitize_text_field($order->get_billing_city()),
                'State' => sanitize_text_field($order->get_billing_state()),
                'PostalCode' => sanitize_text_field($order->get_billing_postcode())
            );

            // Get The Full Shipping Address
            $shipping_address = array(
                'CustomerFirstName' => $billing_first_name,
                'CustomerLastName' => $billing_last_name,
                'Address1' => sanitize_text_field($order->get_shipping_address_1()),
                'Address2' => sanitize_text_field($order->get_shipping_address_2()),
                'City' => sanitize_text_field($order->get_shipping_city()),
                'State' => sanitize_text_field($order->get_shipping_state()),
                'PostalCode' => sanitize_text_field($order->get_shipping_postcode())
            );

            // Gets All The Ordered Products
            $items = $order->get_items();

            // Initialize An Array To Store The Product Data
            $product_skus = array();

            // Loops Through Each Item & Stores Their Product Skus
            foreach ($items as $item) {
                $product_id = $item->get_product_id();
                $variation_id = $item->get_variation_id();
                // Determine if it's a variation
                $product_exist = $variation_id ? wc_get_product($variation_id) : wc_get_product($product_id);
                $product_skus[] = $product_exist ? array('ID' => sanitize_text_field($product_exist->get_sku()), 'Name' => sanitize_text_field($product_exist->get_name())) : ' ';
            }

            // Constructing The JSON Data
            $json_data = array(
                'OrderID' => (string) $order_id,
                'Products' => $product_skus,
                'BillingAddress' => $billing_address,
                'ShippingAddress' => $shipping_address,
                'CustomerEmail' => $billing_email, // Added customer email
                'Company' => $billing_company
            );

            // Converting The Array Into JSON Format
            $json_data_converted = wp_json_encode($json_data);

            // API Endpoint
            //PROD
            $api_url = 'https://api.echeckpoint.com/api/compliancecheck/getresults';
            //DEV PROD DOCKER
            //$api_url = 'http://host.docker.internal:5000/api/compliancecheck/getresults';
            //DEV PROD
            //$api_url = 'http://tetra-accurate-nationally.ngrok-free.app/api/compliancecheck/getresults';
            // API Key
            $api_key = sanitize_text_field(get_option('eCheckpoint_API_Key_Value'));

            // Making the request using wp_remote_post
            $response = wp_remote_post($api_url, array(
                'body' => $json_data_converted,
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $api_key
                ),
                'data_format' => 'body'
            ));

            if (is_wp_error($response)) {
                return;
            }

            $decoded_response = json_decode(wp_remote_retrieve_body($response), true);

            if ($decoded_response === null) {
                return;
            }

            // Save verificationLink to order meta if it exists
            if (isset($decoded_response['modules']['regionalRestrictionsCheck']['verificationLink'])) {
                $verification_link = esc_url($decoded_response['modules']['regionalRestrictionsCheck']['verificationLink']);
                $order->update_meta_data('_echeckpoint_verification_link', $verification_link);
                $order->save();
            }

            // Handle different checks based on the API response
            self::handle_checks($order, $decoded_response);  // Consolidated all checks into a single method to manage order hold state
        }

        private static function is_state_excluded($order) {
            $excluded_states = get_option('eCheckpoint_excluded_states', []);
            if (!is_array($excluded_states)) {
                $excluded_states = [];
            }

            $shipping_state = $order->get_shipping_state();

            // Get all states from WooCommerce
            $all_states = WC()->countries->get_states();

            // Check if shipping state is in the excluded states list
            if (in_array($shipping_state, $excluded_states)) {
                // Find the full state name using WooCommerce's list of states
                $country_code = $order->get_shipping_country();
                $full_state_name = isset($all_states[$country_code][$shipping_state]) ? $all_states[$country_code][$shipping_state] : $shipping_state;

                /* translators: %s: Full state name that has been excluded */
                $order->add_order_note(sprintf(__('eCheckpoint: A compliance check was not run as %s has been excluded in your "Excluded States" settings.', 'echeckpoint'), $full_state_name));
                return true;
            }

            return false;
        }

        private static function handle_checks($order, $decoded_response) {
            //////////////////////////////////////////////////////////
            // CONSOLIDATED CHECK HANDLING
            //////////////////////////////////////////////////////////

            // Set Variable Values
            $shouldAddNotes = get_option('eCheckpoint_Message_Checkbox_Value') == 1;
            $shouldHoldOrder = get_option('eCheckpoint_OnHold_Checkbox_Value') == 1;
            $orderOnHold = false; // Flag to track order hold status

            // Catalog Integrity Check
            $catalogIntegrityCheck_NumberOfItems = count($decoded_response['modules']['catalogIntegrityCheck']['items']);

            for ($i = 0; $i < $catalogIntegrityCheck_NumberOfItems; $i++) {
                $catalogIntegrityCheck_ItemResponse = $decoded_response['modules']['catalogIntegrityCheck']['items'][$i]['response']['result'];

                if ($catalogIntegrityCheck_ItemResponse == 2) {
                    if ($shouldHoldOrder && !$orderOnHold) {
                        $order->update_status('on-hold', __('eCheckpoint: Order placed on hold.', 'echeckpoint'));
                        $orderOnHold = true;
                    }

                    if ($shouldAddNotes) {
                        $noteContent = "eCheckpoint: " . $decoded_response['modules']['catalogIntegrityCheck']['items'][$i]['response']['description'];
                        $order->add_order_note($noteContent);
                    }
                }
            }

            // Address Validation Check
            $addressValidation_BillingAddress_Response = $decoded_response['modules']['addressValidationCheck']['items'][0]['response']['result'];
            $addressValidation_ShippingAddress_Response = $decoded_response['modules']['addressValidationCheck']['items'][1]['response']['result'];

            if ($addressValidation_BillingAddress_Response == 2 || $addressValidation_ShippingAddress_Response == 2) {
                if ($shouldHoldOrder && !$orderOnHold) {
                    $order->update_status('on-hold', __('eCheckpoint: Order placed on hold.', 'echeckpoint'));
                    $orderOnHold = true;
                }

                if ($shouldAddNotes) {
                    if ($addressValidation_BillingAddress_Response == 2) {
                        $description = "eCheckpoint: " . $decoded_response['modules']['addressValidationCheck']['items'][0]['response']['description'];
                        $order->add_order_note($description);
                    }

                    if ($addressValidation_ShippingAddress_Response == 2) {
                        $description = "eCheckpoint: " . $decoded_response['modules']['addressValidationCheck']['items'][1]['response']['description'];
                        $order->add_order_note($description);
                    }
                }
            }

            // Regional Restrictions Check
            $items = $decoded_response['modules']['regionalRestrictionsCheck']['items'];

            foreach ($items as $item) {
                $result = $item['response']['result'];
                $description = isset($item['response']['description']) ? $item['response']['description'] : '';

                $noteContent = "eCheckpoint: ";

                switch ($result) {
                    case 1: // Pass
                        $noteContent .= $description;
                        break;
                    case 2: // Fail
                        $noteContent .= $description;
                        if ($shouldHoldOrder && !$orderOnHold) {
                            $order->update_status('on-hold', __('eCheckpoint: Order placed on hold.', 'echeckpoint'));
                            $orderOnHold = true;
                        }
                        break;
                    case 3: // Conditional Fail
                        $noteContent .= $description;
                        if ($shouldHoldOrder && !$orderOnHold) {
                            $order->update_status('on-hold', __('eCheckpoint: Order placed on hold.', 'echeckpoint'));
                            $orderOnHold = true;
                        }
                        break;
                }

                if ($shouldAddNotes && !empty($description)) {
                    $order->add_order_note($noteContent);
                }
            }
        }
    }
}
