<?php
if (!defined('ABSPATH')) {
    exit;
}

add_action('wp_ajax_submit_rating_form', 'wpra_handle_rating_form');
add_action('wp_ajax_nopriv_submit_rating_form', 'wpra_handle_rating_form');

function wpra_handle_rating_form() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'wpra_nonce')) {
        wp_send_json_error(esc_html__('Security check failed', 'easy-rating-assessor'));
    }

    if (!isset($_POST['data'])) {
        wp_send_json_error(esc_html__('Invalid data', 'easy-rating-assessor'));
    }

    parse_str(sanitize_text_field(wp_unslash($_POST['data'])), $data);
    $name = isset($data['first_name']) ? sanitize_text_field($data['first_name']) : '';
    $email = isset($data['email']) ? sanitize_email($data['email']) : '';
    $mobile = isset($data['mobile']) ? sanitize_text_field($data['mobile']) : '';
    $ratings = isset($data['rating']) ? $data['rating'] : array();

    if (empty($name) || empty($email) || empty($ratings)) {
        wp_send_json_error(esc_html__('Required fields are missing', 'easy-rating-assessor'));
    }

    $result_html = '<div style="max-width:700px; margin:0 auto; font-family:Arial,sans-serif;">';
    $total_score = 0;
    $total_questions = 0;
    $summary_data = array();

    foreach ($ratings as $set_id => $set_ratings) {
        $set_id = absint($set_id);
        $set_title = get_the_title($set_id);
        
        $set_score = 0;
        $question_count = 0;
        
        foreach ($set_ratings as $qid => $rating) {
            $qid = absint($qid);
            $rating = floatval($rating);
            $set_score += $rating;
            $question_count++;
        }
        
        $total_questions += $question_count;
        $max_score = $question_count * 5;
        $level = wpra_get_exact_level($set_score);
        
        $result_html .= wpra_build_exact_gauge_chart(
            esc_html($set_title),
            $set_score,
            $max_score,
            $level
        );
        
        $summary_data[] = array(
            'name' => esc_html($set_title),
            'score' => $set_score,
            'level' => $level
        );
        
        $total_score += $set_score;

        // Detailed question ratings
        $result_html .= '<div style="margin-top:20px; background:#f8f9fa; padding:20px; border-radius:8px;">';
        $result_html .= '<h3 style="text-align:center; border-bottom:1px solid #ddd; padding-bottom:10px; margin-bottom:20px;">' . esc_html__('Detailed Ratings:', 'easy-rating-assessor') . ' ' . esc_html($set_title) . '</h3>';
        
        foreach ($set_ratings as $qid => $rating) {
            $qid = absint($qid);
            $question = get_post($qid);
            $rating = floatval($rating);
            $normalized_rating = ($rating / 5) * 100;
            
            $result_html .= '
            <div style="margin-bottom:25px;">
                <div style="margin-bottom:5px; font-weight:bold; font-size:15px;">' . esc_html($question->post_title) . ' <span style="font-weight:normal; color:#666;">' . number_format($rating, 2) . ' / 5</span></div>
                <div style="height:20px; background:#f0f0f0; border-radius:10px; overflow:hidden; position:relative;">
                    <div style="height:100%; width:' . esc_attr($normalized_rating) . '%; background:#3498db;"></div>
                </div>
                <div style="display:flex; justify-content:space-between; font-size:12px; color:#666; margin-top:5px;">
                    <span>1</span>
                    <span>2</span>
                    <span>3</span>
                    <span>4</span>
                    <span>5</span>
                </div>
            </div>';
        }
        
        $result_html .= '</div>';
    }

    $max_possible = $total_questions * 5;
    $overall_level = wpra_get_overall_level($total_score, $max_possible);
    
    // Summary table
    $result_html .= '<div style="background:#f8f9fa; padding:20px; border-radius:8px; margin:20px 0;">';
    $result_html .= '<h3 style="text-align:center; border-bottom:1px solid #ddd; padding-bottom:10px; margin-bottom:20px;">' . esc_html__('Summary Table', 'easy-rating-assessor') . '</h3>';
    $result_html .= '<table style="width:100%; border-collapse:collapse;">';
    $result_html .= '<tr style="background:#e9ecef;"><th style="padding:10px; text-align:left;">' . esc_html__('Section', 'easy-rating-assessor') . '</th><th style="padding:10px; text-align:center;">' . esc_html__('Score', 'easy-rating-assessor') . '</th><th style="padding:10px; text-align:center;">' . esc_html__('Level', 'easy-rating-assessor') . '</th></tr>';
    
    foreach ($summary_data as $item) {
        $result_html .= '<tr style="border-bottom:1px solid #ddd;">';
        $result_html .= '<td style="padding:10px;">' . esc_html($item['name']) . '</td>';
        $result_html .= '<td style="padding:10px; text-align:center;">' . esc_html($item['score']) . '</td>';
        $result_html .= '<td style="padding:10px; text-align:center;">' . esc_html($item['level']) . '</td>';
        $result_html .= '</tr>';
    }
    
    $result_html .= '<tr style="background:#e9ecef; font-weight:bold;">';
    $result_html .= '<td style="padding:10px;">' . esc_html__('Overall Score', 'easy-rating-assessor') . '</td>';
    $result_html .= '<td style="padding:10px; text-align:center;">' . esc_html($total_score) . '</td>';
    $result_html .= '<td style="padding:10px; text-align:center;">' . esc_html($overall_level) . '</td>';
    $result_html .= '</tr>';
    $result_html .= '</table></div>';

    $result_html .= wpra_build_overall_gauge_chart(
        esc_html__('Overall Score', 'easy-rating-assessor'),
        $total_score,
        $max_possible,
        $overall_level
    );

    $result_html .= '</div>';

    // Email content
    $email_subject = esc_html__('Assessment Results for', 'easy-rating-assessor') . ' ' . esc_html($name);
    $headers = array('Content-Type: text/html; charset=UTF-8');
    $email_content = $result_html;

    wp_mail($email, $email_subject, $email_content, $headers);
    wp_mail(get_option('admin_email'), esc_html__('New Rating Submission from', 'easy-rating-assessor') . ' ' . esc_html($name), $email_content, $headers);

    wp_send_json_success($result_html);
}



function wpra_build_exact_gauge_chart($title, $score, $max, $level) {
    $percent = min(100, round(($score / $max) * 100));
    $scale_numbers = [5, 10, 13.75, 17.5, 21.25, 25];

    $level_ranges = [
        'Beginner' => [5, 10],
        'Intermediate' => [10.25, 13.75],
        'Mid-Level' => [14, 17.5],
        'Advance' => [17.75, 21.25],
        'Elite' => [21.5, 25]
    ];

    $level_colors = [
        'Beginner' => '#e74c3c',
        'Intermediate' => '#f39c12',
        'Mid-Level' => '#f1c40f',
        'Advance' => '#2ecc71',
        'Elite' => '#3498db'
    ];

    $level_indicators = '';
    foreach ($level_ranges as $lvl => $range) {
        $color = $level_colors[$lvl];
        $level_indicators .= "<div style='text-align:center; font-size:12px; color:{$color};'>";
        $level_indicators .= "<strong>{$lvl}</strong><br>({$range[0]}-{$range[1]})";
        $level_indicators .= "</div>";
    }

    return "
    <div style='background:white; padding:25px; margin-bottom:30px; border-radius:8px; box-shadow:0 2px 10px rgba(0,0,0,0.05);'>
        <h3 style='font-size:18px; color:#333; text-align:center; margin:0 0 20px; font-weight:600; border-bottom:1px solid #eee; padding-bottom:15px;'>{$title} - {$score}/{$max} ({$level})</h3>
        
        <div style='height:16px; background:{$level_colors[$level]}; width:{$percent}%; border-radius:8px; margin-bottom:8px; border:2px solid #2980b9;'></div>
        <div style='text-align:center; font-weight:bold; margin:0 0 25px; font-size:14px; color:{$level_colors[$level]};'>
            👤 Your Score: {$score} ({$level})
        </div>

        <div style='display:flex; height:12px; background:#f0f0f0; border-radius:6px; overflow:hidden; margin-bottom:25px;'>
            <div style='width:20%; background:#e74c3c;'></div>
            <div style='width:15%; background:#f39c12;'></div>
            <div style='width:15%; background:#f1c40f;'></div>
            <div style='width:25%; background:#2ecc71;'></div>
            <div style='width:25%; background:#3498db;'></div>
        </div>

        <div style='display:flex; justify-content:space-between; margin:15px 0 20px;'>
            {$level_indicators}
        </div>

        <div style='display:flex; justify-content:space-between; font-size:11px; color:#777; margin-top:5px;'>
            " . implode('', array_map(function($num) { return "<span>{$num}</span>"; }, $scale_numbers)) . "
        </div>
    </div>";
}

function wpra_build_overall_gauge_chart($title, $score, $max, $level) {
    $percent = min(100, round(($score / $max) * 100));
    $scale_numbers = [
        round($max * 0.2),
        round($max * 0.4),
        round($max * 0.6),
        round($max * 0.8),
        $max
    ];
    
    $level_ranges = [
        'Beginner' => [0, round($max * 0.2)],
        'Intermediate' => [round($max * 0.2) + 0.1, round($max * 0.4)],
        'Mid-Level' => [round($max * 0.4) + 0.1, round($max * 0.6)],
        'Advance' => [round($max * 0.6) + 0.1, round($max * 0.8)],
        'Elite' => [round($max * 0.8) + 0.1, $max]
    ];

    $level_colors = [
        'Beginner' => '#e74c3c',
        'Intermediate' => '#f39c12',
        'Mid-Level' => '#f1c40f',
        'Advance' => '#2ecc71',
        'Elite' => '#3498db'
    ];

    $level_indicators = '';
    foreach ($level_ranges as $lvl => $range) {
        $color = $level_colors[$lvl];
        $level_indicators .= "<div style='text-align:center; font-size:12px; color:{$color};'>";
        $level_indicators .= "<strong>{$lvl}</strong><br>({$range[0]}-{$range[1]})";
        $level_indicators .= "</div>";
    }

    return "
    <div style='background:#f8f9fa; padding:25px; margin-bottom:30px; border-radius:8px; border:1px solid #e0e0e0;'>
        <h3 style='font-size:18px; color:#333; text-align:center; margin:0 0 20px; font-weight:600; border-bottom:1px solid #eee; padding-bottom:15px;'>{$title} - {$score}/{$max} ({$level})</h3>
        
        <div style='height:16px; background:{$level_colors[$level]}; width:{$percent}%; border-radius:8px; margin-bottom:8px; border:2px solid #2980b9;'></div>
        <div style='text-align:center; font-weight:bold; margin:0 0 25px; font-size:14px; color:{$level_colors[$level]};'>
            👤 Your Score: {$score} ({$level})
        </div>

        <div style='display:flex; height:12px; background:#f0f0f0; border-radius:6px; overflow:hidden; margin-bottom:25px;'>
            <div style='width:20%; background:#e74c3c;'></div>
            <div style='width:20%; background:#f39c12;'></div>
            <div style='width:20%; background:#f1c40f;'></div>
            <div style='width:20%; background:#2ecc71;'></div>
            <div style='width:20%; background:#3498db;'></div>
        </div>

        <div style='display:flex; justify-content:space-between; margin:15px 0 20px;'>
            {$level_indicators}
        </div>

        <div style='display:flex; justify-content:space-between; font-size:11px; color:#777; margin-top:5px;'>
            " . implode('', array_map(function($num) { return "<span>{$num}</span>"; }, $scale_numbers)) . "
        </div>
    </div>";
}