<?php
/**
 * Plugin Name: Easy Product Tabs For WooCommerce
 * Description: Built for WooCommerce store owners, make it easy to adding custom tabs showing for all products.
 * Version: 1.0.2
 * Author: Leo Q
 */

use EPTFW\AdminPage;
use EPTFW\Constants;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Define plugin constants
define( 'EPTFW_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'EPTFW_PLUGIN_DIRNAME', __FILE__ );
define( 'EPTFW_PLUGIN_VERSION', '1.0.2' );

// Require plugin autoload
require_once( EPTFW_PLUGIN_DIR . 'vendor/autoload.php' );

if ( ! class_exists( 'EasyProductTabsForWoocommerce' ) ):

    class EasyProductTabsForWoocommerce {
        /**
         * @var AdminPage $admin_page;
         *
         * @since 1.0.0
         */
        private $admin_page;

        /**
         * @var array
         *
         * @since 1.0.0
         */
        private static $tab_names = array();

        /**
         * The single instance of the class
         *
         * @since 1.0.0
         */
        private static $_instance = null;

        /**
         * Get instance
         *
         * @return EasyProductTabsForWoocommerce
         * @since 1.0.0
         */
        public static function instance()
        {
            if ( is_null( self::$_instance ) ) {
                self::$_instance = new self();
            }

            return self::$_instance;
        }

        /**
         * Constructor
         *
         * @since 1.0.0
         */
        public function __construct()
        {
            // Plugin hooks
            $this->hooks();
        }

        /**
         * Hooks
         *
         * @since 1.0.0
         */
        private function hooks()
        {
            // Initialize plugin parts
            add_action( 'plugins_loaded', array( $this, 'init' ) );

            // Add tabs
            add_action( 'wp', array( $this, 'add_tabs' ) );

            // Plugin updates
            add_action( 'admin_init', array( $this, 'admin_init' ) );

            if ( is_admin() ) {
                // Plugin activation
                add_action( 'activated_plugin', array( $this, 'plugin_activation' ) );
            }
        }

        /**
         * Add product tabs
         *
         * @since 1.0.0
         */
        public function add_tabs()
        {
            if ( function_exists( 'is_product' ) && is_product() ) {
                // Add custom tabs
                $tab_names = get_option( Constants::APP_SETTING );
                $tab_names = json_decode( $tab_names, true );
                if ( $tab_names ) {
                    self::$tab_names = $tab_names;
                    add_filter( 'woocommerce_product_tabs', array( $this, 'new_product_tabs' ) );
                }
            }
        }

        /**
         * Add product tabs
         *
         * @param $tabs
         * @return mixed
         * @since 1.0.0
         */
        public function new_product_tabs( $tabs ) {
            // Add new tabs
            foreach ( self::$tab_names as $tab_name ) {
                $tab = get_option( $tab_name );
                $tab = json_decode( $tab, true );
                if ( $tab && isset( $tab['id'], $tab['priority'], $tab['name'], $tab['content'] ) && $tab['name'] && $tab['content'] ) {
                    if ( isset( $tab['categories'] ) ) {
                        global $product;
                        $hasTerm = false;
                        foreach ($tab['categories'] as $categoryId) {
                            if( has_term( $categoryId, 'product_cat', $product->get_id() ) ) {
                                $hasTerm = true;
                            }
                        }

                        if (!$hasTerm) {
                            continue;
                        }
                    }

                    $tabs[$tab['id']] = array(
                        'title' => $tab['name'],
                        'priority' => $tab['priority'],
                        'callback' => function () use ( $tab ) {
                            echo $tab['content'];
                        },
                    );
                }
            }

            return $tabs;
        }

        /**
         * Init
         *
         * @since 1.0.0
         */
        public function init()
        {
            if ( is_admin() ) {
                $this->admin_page = new AdminPage();
            }
        }

        /**
         * Admin init
         *
         * @since 1.0.0
         */
        public function admin_init()
        {
            // Listen ajax
            $this->ajax();

            // Add the plugin page Settings and Docs links
            add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_links' ) );

            // Enqueue scripts
            add_action( 'admin_enqueue_scripts', array( $this, 'register_script' ) );
        }

        /**
         * Enqueue and localize js
         *
         * @since 1.0.0
         * @param $hook
         */
        public function register_script( $hook )
        {
            // Load only on plugin page
            if ( $hook != 'toplevel_page_' . Constants::ADMIN_PATH ) {
                return;
            }

            // Enqueue style
            wp_register_style( 'eptfw_app_style', plugins_url( 'dist/static/css/app.css', __FILE__ ), array(), EPTFW_PLUGIN_VERSION, 'all' );
            wp_enqueue_style( 'eptfw_app_style' );

            // Enqueue script
            wp_enqueue_media();
            wp_enqueue_editor();
            wp_register_script( 'eptfw_app_script', plugins_url( 'dist/static/js/app.js', __FILE__ ) , array(), EPTFW_PLUGIN_VERSION, false );
            wp_enqueue_script( 'eptfw_app_script' );

            $tab_names = get_option( Constants::APP_SETTING );
            $tab_names = json_decode( $tab_names, true );
            $tabs = array();
            if ($tab_names) {
                foreach ($tab_names as $tab_name) {
                    $tab = get_option($tab_name);
                    $tab = json_decode($tab, true);
                    if ($tab) {
                        $tabs[] = $tab;
                    }
                }
            }

            wp_localize_script( 'eptfw_app_script', 'eptfw_app_vars', array(
                'tabs' => $tabs,
            ));
        }

        /**
         * Ajax
         *
         * @since 1.0.0
         */
        public function ajax()
        {
            add_action( 'wp_ajax_eptfw_save_tabs', array( $this, 'save_tabs_callback' ) );
            add_action( 'wp_ajax_eptfw_get_categories', array( $this, 'get_categories_callback' ) );
        }

        /**
         * get categories callback
         *
         * @since 1.0.1
         */
        public function get_categories_callback() {
            $title = isset( $_GET['name'] ) ? sanitize_text_field( $_GET['name'] ) : '';
            $ids = isset( $_GET['ids'] ) ? $_GET['ids'] : array();
            $limit = isset( $_GET['limit'] ) ? sanitize_text_field( $_GET['limit'] ) : '';

            $args = array(
                'hide_empty' => false,
            );

            // Limit
            if ( $limit ) {
                $args['number'] = $limit;
            }

            // Title
            if ( $title ) {
                $args['search'] = $title;
            }

            // ids
            if ( $ids ) {
                $args['include'] = $ids;
            }

            $result = array();
            // Get terms
            $terms = get_terms( 'product_cat', $args );

            // Traverse all terms
            if ( !is_wp_error( $result ) ) {
                foreach ( $terms as $term ) {
                    $result[] = array(
                        'id' => $term->term_id,
                        'title' => $term->name,
                    );
                }
            }

            wp_send_json_success($result);
            wp_die();
        }

        /**
         * Save tabs callback
         *
         * @since 1.0.0
         */
        public function save_tabs_callback()
        {
            $tabs = isset( $_POST['tabs'] ) ? $_POST['tabs'] : array();
            if ( !$tabs ) {
                wp_send_json_error();
                wp_die();
            }

            $current_tab_names = get_option( Constants::APP_SETTING );
            $current_tab_names = json_decode( $current_tab_names, true );

            // Save tabs setting
            $tab_names = array();
            foreach ( $tabs as $tab ) {
                if ( isset( $tab['id'], $tab['priority'], $tab['name'], $tab['content'] ) && $tab['id'] && ( $tab['name'] || $tab['content'] ) ) {
                    $tab['id'] = sanitize_text_field( $tab['id'] );
                    $tab_name = sprintf( 'eptfw_product_tab_%s', $tab['id'] );
                    $tab_names[] = $tab_name;
                    $tab['priority'] = (int) $tab['priority'];
                    $tab['name'] = sanitize_text_field( $tab['name'] );
                    $tab['content'] = stripslashes( $tab['content'] );
                    $tab['categories'] = array_filter( $tab['categories'] );
                    update_option( $tab_name, wp_json_encode( $tab ) );
                }
            }

            // Save app setting
            update_option( Constants::APP_SETTING, wp_json_encode( $tab_names ) );

            // Delete removed tabs
            if ( $current_tab_names ) {
                foreach ( $current_tab_names as $current_tab_name ) {
                    if ( !in_array( $current_tab_name, $tab_names ) ) {
                        delete_option( $current_tab_name );
                    }
                }
            }

            wp_send_json_success();
            wp_die();
        }

        /**
         * Plugin links
         *
         * @param $links
         * @return array
         * @since 1.0.0
         */
        public function plugin_links( $links )
        {
            $more_links = array();
            $more_links['settings'] = '<a href="' . admin_url( 'admin.php?page=' . Constants::ADMIN_PATH) . '">' . __( 'Settings', 'eptfw' ) . '</a>';

            return array_merge( $more_links, $links );
        }

        /**
         * Plugin activation
         *
         * @param $plugin
         * @since 1.0.0
         */
        public function plugin_activation( $plugin )
        {
            if ( $plugin == plugin_basename( __FILE__ ) ) {
                exit( wp_redirect( admin_url( 'admin.php?page=' . Constants::ADMIN_PATH ) ) );
            }
        }

        /**
         * Plugin uninstall
         *
         * @since 1.0.0
         */
        public function plugin_uninstall()
        {
            // Delete tabs setting
            $tab_names = get_option( Constants::APP_SETTING );
            $tab_names = json_decode( $tab_names, true );
            foreach ( $tab_names as $tab_name ) {
                delete_option( $tab_name );
            }

            // Delete app setting
            delete_option( Constants::APP_SETTING );
        }
    }

    // Run plugin
    new EasyProductTabsForWoocommerce();

endif;