<?php
// easy-mobile-poster/frontend/shortcode-form.php

if (!defined('ABSPATH')) exit;

function easymopo_shortcode_handler() {
    easymopo_handle_post_requests();
    ob_start();

    $transient_key = 'easymopo_message_' . md5( $_SERVER['REMOTE_ADDR'] );
    $message_data = get_transient( $transient_key );

    if ( $message_data ) {
        echo '<p class="' . esc_attr( $message_data['type'] ) . '">' . esc_html( $message_data['message'] ) . '</p>';
        delete_transient( $transient_key );
    }

    if ( easymopo_is_logged_in() ) {
        easymopo_display_post_form();
    } else {
        easymopo_display_login_form();
    }
    return ob_get_clean();
}

function easymopo_get_cookie_name() {
    return 'easymopo_auth_' . COOKIEHASH;
}

function easymopo_is_logged_in() {
    $cookie_name = easymopo_get_cookie_name();
    if (!isset($_COOKIE[$cookie_name])) {
        return false;
    }

    $options = get_option('easymopo_settings');
    $expected_value = hash('sha256', 'easymopo_logged_in' . $options['password_hash'] . $_SERVER['REMOTE_ADDR'] . $_SERVER['HTTP_USER_AGENT']);
    
    return hash_equals($expected_value, $_COOKIE[$cookie_name]);
}

function easymopo_handle_post_requests() {
    if (isset($_GET['easymopo_logout']) && $_GET['easymopo_logout'] == '1') {
        setcookie(easymopo_get_cookie_name(), '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN);
        wp_redirect(get_permalink());
        exit;
    }
    if (isset($_POST['easymopo_login_submit'])) {
        easymopo_handle_login();
    }
    if (isset($_POST['easymopo_post_submit'])) {
        easymopo_handle_post_submission();
    }
}

function easymopo_set_flash_message( $message, $type ) {
    $transient_key = 'easymopo_message_' . md5( $_SERVER['REMOTE_ADDR'] );
    $message_data = ['message' => $message, 'type' => $type];
    set_transient( $transient_key, $message_data, 5 * MINUTE_IN_SECONDS );
}

function easymopo_handle_login() {
    if (!isset($_POST['easymopo_username'], $_POST['easymopo_password'], $_POST['easymopo_nonce']) || !wp_verify_nonce($_POST['easymopo_nonce'], 'easymopo_login')) {
        easymopo_set_flash_message(esc_html__('Invalid request.', 'easy-mobile-poster'), 'easymopo-error');
        return;
    }
    $options = get_option('easymopo_settings');
    if (empty($options['username']) || empty($options['password_hash'])) {
        easymopo_set_flash_message(esc_html__('Login details are not configured.', 'easy-mobile-poster'), 'easymopo-error');
        return;
    }
    if ($_POST['easymopo_username'] === $options['username'] && password_verify($_POST['easymopo_password'], $options['password_hash'])) {
        $cookie_name = easymopo_get_cookie_name();
        $cookie_value = hash('sha256', 'easymopo_logged_in' . $options['password_hash'] . $_SERVER['REMOTE_ADDR'] . $_SERVER['HTTP_USER_AGENT']);
        $expiration = time() + (2 * HOUR_IN_SECONDS);
        setcookie($cookie_name, $cookie_value, $expiration, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
        wp_redirect(get_permalink());
        exit;
    } else {
        easymopo_set_flash_message(esc_html__('Incorrect username or password.', 'easy-mobile-poster'), 'easymopo-error');
    }
}

function easymopo_handle_post_submission() {
    if ( !easymopo_is_logged_in() || !isset($_POST['easymopo_nonce']) || !wp_verify_nonce($_POST['easymopo_nonce'], 'easymopo_post')) {
        easymopo_set_flash_message(esc_html__('Invalid request. Please log in again.', 'easy-mobile-poster'), 'easymopo-error');
        return;
    }
    if (empty($_POST['easymopo_title'])) {
        easymopo_set_flash_message(esc_html__('Title is required.', 'easy-mobile-poster'), 'easymopo-error');
        return;
    }

    $post_data = [
        'post_title'   => sanitize_text_field($_POST['easymopo_title']), 
        'post_content' => wp_kses_post($_POST['easymopo_content']), 
        'post_status'  => (isset($_POST['easymopo_status']) && $_POST['easymopo_status'] === 'publish') ? 'publish' : 'draft', 
        'post_author'  => 1,
        'post_category' => isset($_POST['post_category']) ? [intval($_POST['post_category'])] : []
    ];
    $post_id = wp_insert_post($post_data, true);

    if (is_wp_error($post_id)) {
        easymopo_set_flash_message(esc_html__('Failed to create post:', 'easy-mobile-poster') . ' ' . $post_id->get_error_message(), 'easymopo-error');
        return;
    }

    do_action('easymopo_handle_images', $post_id);

    if (!did_action('easymopo_handle_images')) {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        if (!empty($_FILES['easymopo_image']['name'])) {
            $attachment_id = media_handle_upload('easymopo_image', $post_id);
            if (!is_wp_error($attachment_id)) {
                set_post_thumbnail($post_id, $attachment_id);
            }
        }

        // Handle additional image 1
        if (!empty($_FILES['easymopo_image2']['name'])) {
            $attachment_id_2 = media_handle_upload('easymopo_image2', $post_id);
            if (!is_wp_error($attachment_id_2)) {
                add_post_meta($post_id, '_easymopo_additional_image_2', $attachment_id_2);
            }
        }

        // Handle additional image 2
        if (!empty($_FILES['easymopo_image3']['name'])) {
            $attachment_id_3 = media_handle_upload('easymopo_image3', $post_id);
            if (!is_wp_error($attachment_id_3)) {
                add_post_meta($post_id, '_easymopo_additional_image_3', $attachment_id_3);
            }
        }
    }

    $message = ($post_data['post_status'] === 'publish') ? esc_html__('Post published successfully.', 'easy-mobile-poster') : esc_html__('Post saved as a draft.', 'easy-mobile-poster');
    easymopo_set_flash_message( $message, 'easymopo-success' );
    wp_redirect(get_permalink());
    exit;
}

function easymopo_display_login_form() {
    ?>
    <div class="easymopo-form-container">
        <h3><?php esc_html_e('Login', 'easy-mobile-poster'); ?></h3>
        <form method="post">
            <?php wp_nonce_field('easymopo_login', 'easymopo_nonce'); ?>
            <p><label for="easymopo_username"><?php esc_html_e('Username', 'easy-mobile-poster'); ?></label><input type="text" id="easymopo_username" name="easymopo_username" required></p>
            <p><label for="easymopo_password"><?php esc_html_e('Password', 'easy-mobile-poster'); ?></label><input type="password" id="easymopo_password" name="easymopo_password" required></p>
            <p><input type="submit" name="easymopo_login_submit" value="<?php esc_attr_e('Login', 'easy-mobile-poster'); ?>"></p>
            <p><button type="button" id="easymopo_clear_fields" class="easymopo-clear-btn"><?php esc_html_e('Clear Fields', 'easy-mobile-poster'); ?></button></p>
        </form>
    </div>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const clearBtn = document.getElementById('easymopo_clear_fields');
        if (clearBtn) {
            clearBtn.addEventListener('click', function() {
                document.getElementById('easymopo_username').value = '';
                document.getElementById('easymopo_password').value = '';
                document.getElementById('easymopo_username').focus();
            });
        }
    });
    </script>
    <?php
}

function easymopo_display_post_form() {
    ?>
    <div class="easymopo-form-container">
        <h3><?php esc_html_e('New Post', 'easy-mobile-poster'); ?></h3>
        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field('easymopo_post', 'easymopo_nonce'); ?>
            <p><label for="easymopo_title"><?php esc_html_e('Title', 'easy-mobile-poster'); ?></label><input type="text" id="easymopo_title" name="easymopo_title" required></p>
            <p><label for="easymopo_content"><?php esc_html_e('Content', 'easy-mobile-poster'); ?></label><textarea id="easymopo_content" name="easymopo_content" rows="10"></textarea></p>
            
            <p>
                <label for="post_category"><?php esc_html_e('Category', 'easy-mobile-poster'); ?></label>
                <?php wp_dropdown_categories(['show_option_none' => esc_html__('Select Category', 'easy-mobile-poster'), 'taxonomy' => 'category', 'name' => 'post_category', 'id' => 'post_category', 'hierarchical' => true, 'required' => true]); ?>
            </p>

            <p>
                <label for="easymopo_image"><?php esc_html_e('Featured Image', 'easy-mobile-poster'); ?></label>
                <input type="file" id="easymopo_image" name="easymopo_image" accept="image/*">
            </p>
            <p>
                <label for="easymopo_image2"><?php esc_html_e('Additional Image 1', 'easy-mobile-poster'); ?></label>
                <input type="file" id="easymopo_image2" name="easymopo_image2" accept="image/*">
            </p>
            <p>
                <label for="easymopo_image3"><?php esc_html_e('Additional Image 2', 'easy-mobile-poster'); ?></label>
                <input type="file" id="easymopo_image3" name="easymopo_image3" accept="image/*">
            </p>

            <p><label><?php esc_html_e('Post Status', 'easy-mobile-poster'); ?></label><label><input type="radio" name="easymopo_status" value="draft" checked> <?php esc_html_e('Draft', 'easy-mobile-poster'); ?></label> <label><input type="radio" name="easymopo_status" value="publish"> <?php esc_html_e('Publish', 'easy-mobile-poster'); ?></label></p>
            <hr>
            <p><input type="submit" name="easymopo_post_submit" value="<?php esc_attr_e('Submit Post', 'easy-mobile-poster'); ?>"></p>
        </form>
        <p class="easymopo-logout-link"><a href="?easymopo_logout=1"><?php esc_html_e('Logout', 'easy-mobile-poster'); ?></a></p>
    </div>
    <?php
}
?>