<?php
/**
 * Plugin settings.
 *
 * @package EasyFooterMenu
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add settings menu.
 */
function easy_footer_menu_add_settings_page() {
	add_options_page(
		'Easy Footer Menu Settings',
		'Easy Footer Menu',
		'manage_options',
		'easy-footer-menu-settings',
		'easy_footer_menu_render_settings_page'
	);
}

add_action( 'admin_menu', 'easy_footer_menu_add_settings_page' );

/**
 * Register settings.
 */
function easy_footer_menu_register_settings() {
	register_setting( 'easy_footer_menu_settings', 'easy_footer_menu_items', 'easy_footer_menu_sanitize_items' );
}

add_action( 'admin_init', 'easy_footer_menu_register_settings' );

/**
 * Add settings link to plugin page.
 *
 * @param array $links Plugin action links.
 */
function easy_footer_menu_add_settings_link( $links ) {
	$settings_link = '<a href="options-general.php?page=easy-footer-menu-settings">' . __( 'Settings', 'easy-footer-menu' ) . '</a>';
	array_unshift( $links, $settings_link );
	return $links;
}

add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'easy_footer_menu_add_settings_link' );

/**
 * Get list of Feather icons.
 */
function easy_footer_menu_get_feather_icons() {
	return array(
		'home',
		'user',
		'message-circle',
		'info',
		'settings',
		'mail',
		'heart',
		'shopping-cart',
		'calendar',
		'camera',
		'music',
		'map',
		'book',
		'phone',
		'award',
		'globe',
		'activity',
		'send',
		'coffee',
		// Add more icons as needed.
	);
}

/**
 * Enqueue admin scripts and styles.
 *
 * @param string $hook The current admin page.
 *
 * @since 1.0.0
 */
function easy_footer_menu_admin_enqueue_scripts( $hook ) {
	if ( 'settings_page_easy-footer-menu-settings' !== $hook ) {
		return;
	}
	wp_enqueue_style( 'easy-footer-menu-admin-style', plugins_url( 'assets/css/admin-style.css', EASY_FOOTER_MENU_PLUGIN_FILE ), array(), EASY_FOOTER_MENU_VERSION );
	wp_enqueue_script( 'feather-icons', plugins_url( 'assets/js/feather.min.js', EASY_FOOTER_MENU_PLUGIN_FILE ), array( 'jquery' ), EASY_FOOTER_MENU_VERSION, true );
	wp_enqueue_script( 'easy-footer-menu-admin-js', plugins_url( 'assets/js/admin-script.js', EASY_FOOTER_MENU_PLUGIN_FILE ), array( 'jquery' ), EASY_FOOTER_MENU_VERSION, true );
	wp_enqueue_script( 'easy-footer-menu-admin-js', plugins_url( 'assets/js/admin.js', EASY_FOOTER_MENU_PLUGIN_FILE ), array( 'jquery' ), EASY_FOOTER_MENU_VERSION, true );
	wp_localize_script( 'easy-footer-menu-admin-js', 'featherIcons', easy_footer_menu_get_feather_icons() );

	// Localize script to pass PHP variables to JavaScript.
	$menu_items = get_option( 'easy_footer_menu_items', array() );
	$index      = ! empty( $menu_items ) ? max( array_keys( $menu_items ) ) + 1 : 0;

	wp_localize_script(
		'easy-footer-menu-admin-js',
		'easyFooterMenuData',
		array(
			'index'        => $index,
			'featherIcons' => easy_footer_menu_get_feather_icons(),
		)
	);
}

add_action( 'admin_enqueue_scripts', 'easy_footer_menu_admin_enqueue_scripts' );

/**
 * Sanitize and validate form input.
 *
 * @param array $input Input array.
 *
 * @since 1.0.0
 *
 * @return array
 */
function easy_footer_menu_sanitize_items( $input ) {
	$new_input = array();

	if ( is_array( $input ) ) {
		foreach ( $input as $key => $item ) {
			if ( ! empty( $item['name'] ) || ! empty( $item['link'] ) || ! empty( $item['feather_icon'] ) ) {
				$new_input[ $key ]['name']         = sanitize_text_field( $item['name'] );
				$new_input[ $key ]['link']         = esc_url_raw( $item['link'] );
				$new_input[ $key ]['feather_icon'] = sanitize_text_field( $item['feather_icon'] );
			}
		}
	}

	return $new_input;
}

/**
 * Render settings page.
 *
 * @since 1.0.0
 */
function easy_footer_menu_render_settings_page() {
	$menu_items = get_option( 'easy_footer_menu_items', array() );
	?>
	<div class="wrap">
		<h1>Easy Footer Menu Custonization</h1>
		<form method="post" action="options.php">
			<?php settings_fields( 'easy_footer_menu_settings' ); ?>
			<table class="form-table" id="menu-items-table">
				<thead>
					<tr>
						<th>Name</th>
						<th>Link</th>
						<th>Icon</th>
						<th>Action</th>
					</tr>
				</thead>
				<tbody>
					<?php
					if ( ! empty( $menu_items ) ) {
						foreach ( $menu_items as $index => $item ) {
							easy_footer_menu_render_item_fields( $index, $item );
						}
					}
					?>
				</tbody>
			</table>
			<button type="button" id="add-menu-item" class="button">Add Menu Item</button>
			<?php submit_button(); ?>
		</form>
	</div>


	<?php
}

/**
 * Render fields for a single menu item.
 *
 * @param int   $index The index of the menu item in the array.
 * @param array $item The menu item data.
 */
function easy_footer_menu_render_item_fields( $index, $item ) {
	$feather_icons = easy_footer_menu_get_feather_icons();
	?>
	<tr>
		<td>
			<input type="text" name="easy_footer_menu_items[<?php echo esc_attr( $index ); ?>][name]" value="<?php echo esc_attr( $item['name'] ); ?>" />
		</td>
		<td>
			<input type="text" name="easy_footer_menu_items[<?php echo esc_attr( $index ); ?>][link]" value="<?php echo esc_url( $item['link'] ); ?>" />
		</td>
		<td>
			<!-- Feather Icon Select -->
			<div class="custom-select-wrapper">
				<select name="easy_footer_menu_items[<?php echo esc_attr( $index ); ?>][feather_icon]" class="icon-select" style="display: none;">
					<?php foreach ( $feather_icons as $ficon ) : ?>
						<option value="<?php echo esc_attr( $ficon ); ?>" <?php selected( $item['feather_icon'] ?? '', $ficon ); ?>>
							<?php echo esc_html( $ficon ); ?>
						</option>
					<?php endforeach; ?>
				</select>
				<div class="custom-select">
					<div class="custom-select-trigger">
						<i data-feather="<?php echo esc_attr( $item['feather_icon'] ?? '' ); ?>" style="font-size: 24px;"></i>
						<span><?php echo esc_html( $item['feather_icon'] ?? '' ); ?></span>
					</div>
					<div class="custom-options">    
						<?php foreach ( $feather_icons as $ficon ) : ?>  
							<span class="custom-option <?php echo ( isset( $item['feather_icon'] ) && $item['feather_icon'] === $ficon ) ? 'selected' : ''; ?>" data-value="<?php echo esc_attr( $ficon ); ?>">
								<i data-feather="<?php echo esc_attr( $ficon ); ?>" style="font-size: 24px;"></i>    
								<span><?php echo esc_html( $ficon ); ?></span>
							</span>
						<?php endforeach; ?>
					</div>
				</div>
			</div>
		</td>
		<td>
			<button type="button" class="remove-item button">Remove</button>
		</td>
	</tr>
	<?php
}