<?php
namespace EASYERLO\Inc;

defined('ABSPATH') || die('Hey, what are you doing here? You silly human!');

/**
 * Easy Error Log Base Class.
 *
 * This class handles all the base functionality for the Easy Error Log plugin.
 *
 * @package EASYERLO\Inc
 */
class EASYERLO_Base {

	/**
	 * The constructor of the class.
	 *
	 * @return void
	 */
	public function __construct() {
		$this->includes();

		add_action('admin_menu', array( $this, 'add_error_page' ));
		add_action('admin_enqueue_scripts', [ $this, 'admin_enqueue' ]);
		add_action('wp_enqueue_scripts', [ $this, 'fe_scripts' ]);
		add_action('admin_bar_menu', array( $this, 'add_my_page_to_admin_bar' ), 100);
		add_action('init', array( $this, 'system_info' ));
		add_action('wp_footer', [ $this, 'display_error_floating_widget' ], 99);

		add_action('wp_ajax_toggle_admin_widget', array( $this, 'toggle_admin_widget' ));
		add_action('wp_ajax_toggle_fatal_error_safer', array( $this, 'toggle_fatal_error_safer' ));
		add_action('admin_footer', array( $this, 'display_admin_error_widget' ));

		add_action('init', array( $this, 'trigger_test_error' ));

		// Add the action to handle the AJAX request.
		$this->init_ajax_handlers();
		$this->init_fatal_handlers();
	}

	/**
	 * Include the necessary files.
	 *
	 * @return void
	 */
	private function init_ajax_handlers() {
		$ajax = new EASYERLO_Ajax();
		$ajax->init();
	}

	private function init_fatal_handlers() {
		$fatal = new EASYERLO_Fatal();
		$fatal->init();
	}

	/**
	 * Include the necessary files.
	 *
	 * @return void
	 */
	public function includes() {
		add_action('plugins_loaded', array( $this, 'easyerlo_load' ));
	}

	/**
	 * Include the necessary files.
	 *
	 * @return void
	 */
	public function easyerlo_load() {
		load_plugin_textdomain('easy-error-log', false, __DIR__ . 'languages');
	}

	/**
	 * Include the necessary files.
	 */
	public function get_admin_widget_status() {
		return get_option('easyerlo_admin_widget_enabled', 'active');
	}
	/**
	 *
	 */
	public function get_fatal_error_safer() {
		return get_option('easyerlo_fatal_error_safer_enabled', 'active');
	}

	/**
	 * Include the necessary files.
	 *
	 * @return void
	 */
	public function toggle_admin_widget() {
		check_ajax_referer('easyerlo_admin_nonce', 'nonce');

		$current_status = $this->get_admin_widget_status();
		$new_status = ( 'active' === $current_status ) ? 'inactive' : 'active';

		update_option('easyerlo_admin_widget_enabled', $new_status);
		wp_send_json_success(array( 'status' => $new_status ));
	}

	/**
	 *
	 */
	public function toggle_fatal_error_safer() {
		check_ajax_referer('easyerlo_admin_nonce', 'nonce');

		$current_status = $this->get_fatal_error_safer();
		$new_status = ( 'active' === $current_status ) ? 'inactive' : 'active';

		update_option('easyerlo_fatal_error_safer_enabled', $new_status);
		wp_send_json_success(array( 'status' => $new_status ));
	}

	/**
	 * Add error page.
	 */
	public function add_error_page() {
		$debug_error_mode_enabled = get_option('easyerlo_debug_mode_enabled', 0);
		if ( empty($debug_error_mode_enabled) ) {
			$config_path = ABSPATH . 'wp-config.php';
			if ( file_exists($config_path) ) {
				$config_contents = file_get_contents($config_path);

				// First remove any existing duplicate constants.
				$config_contents = preg_replace('/define\s*\(\s*[\'"]WP_DEBUG[\'"]\s*,\s*(?:true|false)\s*\);\s*\n?/i', '', $config_contents);
				$config_contents = preg_replace('/define\s*\(\s*[\'"]WP_DEBUG_LOG[\'"]\s*,\s*(?:true|false)\s*\);\s*\n?/i', '', $config_contents);

				// Add new constants.
				$constants_to_add = "define('WP_DEBUG', false);\ndefine('WP_DEBUG_LOG', false);\n";

				// Find position to insert.
				$position_to_insert = strpos($config_contents, '/* That\'s all, stop editing! Happy publishing. */');

				if ( false !== $position_to_insert ) {
					$config_contents = substr_replace($config_contents, $constants_to_add, $position_to_insert, 0);
					file_put_contents($config_path, $config_contents);
					update_option('easyerlo_debug_mode_enabled', 1);
				}
			}
		}

		add_management_page('Error Log', 'Error Log', 'manage_options', 'easy_error_log', array( $this, 'display_errors' ));
	}


	/**
	 * Enqueue plugin files.
	 *
	 * @param string $screen   use to get the current page screen.
	 */
	public function admin_enqueue( $screen ) {
		if ( 'tools_page_easy_error_log' === $screen ) {
			remove_all_actions( 'admin_notices' );
			remove_all_actions( 'all_admin_notices' );

			wp_enqueue_style(
				'easyerlo-admin-css',
				EASY_ERROR_LOG_DIR_URL . 'assets/easy-errors.css',
				'',
				time(),
				'all'
			);

			wp_enqueue_script(
				'easyerlo-admin-js',
				EASY_ERROR_LOG_DIR_URL . 'assets/easy-errors.js',
				[ 'jquery' ],
				time(),
				true
			);

			// Enqueue CodeMirror dependencies
			$this->enqueue_codemirror();

			// Localize the script with new data.
			$ajax_url = admin_url( 'admin-ajax.php' );
			wp_localize_script('easyerlo-admin-js', 'easyerlo_ajax_object', array(
				'ajax_url' => admin_url('admin-ajax.php'),
				'admin_widget_status' => $this->get_admin_widget_status(),
				'fatal_error_safer' => $this->get_fatal_error_safer(),
				'nonce' => wp_create_nonce('easyerlo_admin_nonce'),
			));
		} else {
			wp_enqueue_script('jquery-ui-draggable');
			wp_enqueue_style('dashicons');
			wp_enqueue_style(
				'easyerlo-admin-css',
				EASY_ERROR_LOG_DIR_URL . 'assets/admin-easy-errors.css',
				'',
				time(),
				'all'
			);

			wp_enqueue_script(
				'easyerlo-admin-js',
				EASY_ERROR_LOG_DIR_URL . 'assets/admin-easy-errors.js',
				[ 'jquery' ],
				time(),
				true
			);

			// Localize the script with new data.
			$ajax_url = admin_url( 'admin-ajax.php' );
			wp_localize_script('easyerlo-admin-js', 'easyerlo_ajax_object', array(
				'ajax_url' => admin_url('admin-ajax.php'),
				'admin_widget_status' => $this->get_admin_widget_status(),
				'fatal_error_safer' => $this->get_fatal_error_safer(),
				'nonce' => wp_create_nonce('easyerlo_admin_nonce'),
			));

		}
	}


	/**
	 * Display errors files.
	 */
	public function display_errors() {
		$mode = '';
		$status = '';
		$widgets_mode = '';
		$admin_widget_status = $this->get_admin_widget_status();
		$fatal_error_safer = $this->get_fatal_error_safer();

		?>
			<br>

			<div class="nav-tab-wrapper">
				<a href="#debugger" class="nav-tab nav-tab-active"><?php echo esc_html__( 'Debugger', 'easy-error-log' ); ?></a>
				<a href="#error" class="nav-tab"><?php echo esc_html__( 'Error Test', 'easy-error-log' ); ?></a>
				<a href="#system-info" class="nav-tab"><?php echo esc_html__( 'System Info', 'easy-error-log' ); ?></a>
				<a href="#theme-plugins" class="nav-tab"><?php echo esc_html__( 'Theme & Plugins', 'easy-error-log' ); ?></a>
				<a href="#user-info" class="nav-tab"><?php echo esc_html__( 'User Info', 'easy-error-log' ); ?></a>
				<a href="#about" class="nav-tab"><?php echo esc_html__( 'More+', 'easy-error-log' ); ?></a>
			</div>

			<div class="tab-content">
				<!-- Debugger Tab -->
				<div id="debugger" class="debugger-section tab-pane active">

					<div class="sidebar">
							<div class="header-status">
							<button id="toggle-controller" class="button">
								<span id="toggle-icon"><?php echo esc_html__( '▼', 'easy-error-log' ); ?></span> <?php echo esc_html__( 'Controls', 'easy-error-log' ); ?>
							</button>
							<div class="status">
							<!-- WP_DEBUG and WP_DEBUG_LOG is the Conatant  -->
								<h4><?php echo esc_html__( 'WP_DEBUG:', 'easy-error-log' ); ?> <span class="constant-status wp-debug" style="color: green;"><?php echo esc_html__( 'Found', 'easy-error-log' ); ?></span></h4>
								<h4><?php echo esc_html__( 'WP_DEBUG_LOG:', 'easy-error-log' ); ?> <span class="constant-status wp-debug-log" style="color: green;"><?php echo esc_html__( 'Found', 'easy-error-log' ); ?></span></h4>
							</div>
						</div>

						<div class="wpel-buttons" style="display: flex; flex-wrap: wrap; gap: 16px; margin: 20px 0; align-items: center;">
					
							<div class="debug-toggle-container" style="display: flex; align-items: center; gap: 8px;">
								<label class="switch">
									<input type="checkbox" id="toggle-debug-mode">
									<span class="slider"></span>
								</label>
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-1 17.93c-3.94-.49-7-3.85-7-7.93 0-.62.08-1.21.21-1.79L9 15v1c0 1.1.9 2 2 2v1.93zm6.9-2.54c-.26-.81-1-1.39-1.9-1.39h-1v-3c0-.55-.45-1-1-1H8v-2h2c.55 0 1-.45 1-1V7h2c1.1 0 2-.9 2-2v-.41c2.93 1.19 5 4.06 5 7.41 0 2.08-.8 3.97-2.1 5.39z"/></svg>
								<span class="toggle-label"><?php echo esc_html__('Toggle Debug Mode:', 'easy-error-log'); ?></span>
								<span id="debug-mode-status" style="margin-left: 10px;"></span>
							</div>

							<button id="clean-debug-log" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/></svg>
								<?php echo esc_html__('Clean', 'easy-error-log'); ?>
							</button>

							<button id="refresh-debug-log" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M17.65 6.35C16.2 4.9 14.21 4 12 4c-4.42 0-7.99 3.58-7.99 8s3.57 8 7.99 8c3.73 0 6.84-2.55 7.73-6h-2.08c-.82 2.33-3.04 4-5.65 4-3.31 0-6-2.69-6-6s2.69-6 6-6c1.66 0 3.14.69 4.22 1.78L13 11h7V4l-2.35 2.35z"/></svg>
								<?php echo esc_html__('Refresh', 'easy-error-log'); ?>
							</button>

							<form id="download-debug-log" method="post" action="" style="margin: 0;">
								<?php wp_nonce_field('download_debug_log_nonce', 'download_debug_log_nonce'); ?>
								<input type="hidden" name="action" value="download_debug_log">
								<button type="submit" class="button" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M19 9h-4V3H9v6H5l7 7 7-7zM5 18v2h14v-2H5z"/></svg>
									<?php echo esc_html__('Download', 'easy-error-log'); ?>
								</button>
							</form>

							<button id="reset-constant" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M13 3c-4.97 0-9 4.03-9 9H1l3.89 3.89.07.14L9 12H6c0-3.87 3.13-7 7-7s7 3.13 7 7-3.13 7-7 7c-1.93 0-3.68-.79-4.94-2.06l-1.42 1.42C8.27 19.99 10.51 21 13 21c4.97 0 9-4.03 9-9s-4.03-9-9-9zm-1 5v5l4.28 2.54.72-1.21-3.5-2.08V8H12z"/></svg>
								<?php echo esc_html__('Reset Debug Constant', 'easy-error-log'); ?>
							</button>

							<button id="toggle-admin-widget" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M13 13h8v8h-8v-8zM3 21h8v-8H3v8zM3 3v8h8V3H3zm9.5-1.5v3h3v-3h-3zm4.5 4.5h8v8h-8v-8z"/></svg>
								<?php echo esc_html__('Admin widget:', 'easy-error-log'); ?>
								<span id="admin-widget-status" style="color: <?php echo esc_attr('active' === $admin_widget_status ? '#ffee00' : 'red'); ?>">
									<?php echo esc_html('active' === $admin_widget_status ? __('ON', 'easy-error-log') : __('OFF', 'easy-error-log')); ?>
								</span>
							</button>
							
							<button id="toggle-fe-mode" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
								<?php echo esc_html__('FE widgets:', 'easy-error-log'); ?>
								<span id="debug-fe-status" style="color: <?php echo esc_attr('active' === $widgets_mode ? 'red' : '#ffee00'); ?>">
									<?php echo esc_html($status); ?>
								</span>
							</button>

							<button id="toggle-fatal-error-safer" class="button" style="display: flex; align-items: center; gap: 8px;">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>
								<?php echo esc_html__('Error protection:', 'easy-error-log'); ?>
								<span id="fatal-error-safer" style="color: <?php echo esc_attr('active' === $fatal_error_safer ? '#00ff86' : 'red'); ?>">
									<?php echo esc_html('active' === $fatal_error_safer ? __('ON', 'easy-error-log') : __('OFF', 'easy-error-log')); ?>
								</span>
							</button>

						</div>
						
						<div class="debug-constant">
							<div class="code-wrapper">
								<code contenteditable="true" id="code1"><?php echo esc_html__( "error_log( 'Log: ' . print_r( \$, true ) );", 'easy-error-log' ); ?></code>
								<button class="copy-btn" data-target="#code1" title="Copy to Clipboard">
									&#x1F4CB;
								</button>
							</div>
							<div class="code-wrapper">
								<code contenteditable="true" id="code2"><?php echo esc_html__( "error_log( 'Log:-  ' );", 'easy-error-log' ); ?></code>
								<button class="copy-btn" data-target="#code2" title="Copy to Clipboard">
									&#x1F4CB;
								</button>
							</div>
							<br>
						</div>


					</div>
					<!-- Display error and other status  -->
					<div class="debugger-panel">
						<table class="wp-list-table widefat fixed striped">
						<thead class="wp-error-head">
							<tr class="wp-error-row">
								<th class="wp-error-table-header"><?php echo esc_html__( 'Error Message: Duplicate errors removed from the panel', 'easy-error-log' ); ?></th>
							</tr>
						</thead>
						<tbody class="wp-error-body">
							<tr class="wp-error-body-row">
								<td class="wp-error-body-data"><p id="error-log-container" class="error-log-scrollable"></p></td>
							</tr>
						</tbody>
					</table>
					</div>
				</div>


				<!-- Fatal Error Test  -->
				 <div id="error" class="tab-pane" style="display: none;">
					<h3><?php echo esc_html__( 'Error Tester', 'easy-error-log' ); ?></h3>

					<table class="form-table" role="presentation">
						<tbody>
							<!-- Basic Fatal Errors -->
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>
									<?php echo esc_html__('Fatal Error (E_ERROR)', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=fatal'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary" 
									onclick="return confirm('This will cause a fatal error! Continue?');">
										<?php echo esc_html__('Trigger Fatal Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Calls undefined function', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M8 12h8v2H8zm2 8l4-4v3h8v2h-8v3l-4-4zM2 4v16h2V4h16V2H4c-1.1 0-2 .9-2 2z"/></svg>
									<?php echo esc_html__('Parse Error', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=parse'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a parse error! Continue?');">
										<?php echo esc_html__('Trigger Parse Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Creates invalid PHP syntax', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M15 9H9v6h6V9zm-2 4h-2v-2h2v2zm8-2V9h-2V7c0-1.1-.9-2-2-2h-2V3h-2v2H9V3H7v2H5c-1.1 0-2 .9-2 2v2H1v2h2v2H1v2h2v2c0 1.1.9 2 2 2h2v2h2v-2h6v2h2v-2h2c1.1 0 2-.9 2-2v-2h2v-2h-2v-2h2zm-4 8H5V5h14v14z"/></svg>
									<?php echo esc_html__('Memory Error', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=memory'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a memory error! Continue?');">
										<?php echo esc_html__('Trigger Memory Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Allocates excessive memory', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2zm0 7.5L9.5 12 12 14.5 14.5 12 12 9.5z"/></svg>
									<?php echo esc_html__('Class Not Found', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=class'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a class error! Continue?');">
										<?php echo esc_html__('Trigger Class Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Instantiates non-existent class', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<!-- New Advanced Test Cases -->
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm-1-5v-4h-2v4h2zm2-8h-2v2h2V7z"/></svg>
									<?php echo esc_html__('Stack Overflow (Recursion)', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=recursion'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a stack overflow! Continue?');">
										<?php echo esc_html__('Trigger Stack Overflow', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Infinite recursion causing stack overflow', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M15 1H9v2h6V1zm-4 13h2V8h-2v6zm8.03-6.61l1.42-1.42c-.43-.51-.9-.99-1.41-1.41l-1.42 1.42C16.07 4.74 14.12 4 12 4c-4.97 0-9 4.03-9 9s4.02 9 9 9 9-4.03 9-9c0-2.12-.74-4.07-1.97-5.61zM12 20c-3.87 0-7-3.13-7-7s3.13-7 7-7 7 3.13 7 7-3.13 7-7 7z"/></svg>
									<?php echo esc_html__('Execution Timeout', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=timeout'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a timeout error! Continue?');">
										<?php echo esc_html__('Trigger Timeout Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Exceeds maximum execution time', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>
									<?php echo esc_html__('Type Error', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=type'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a type error! Continue?');">
										<?php echo esc_html__('Trigger Type Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Wrong argument type passed to function', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"/></svg>
									<?php echo esc_html__('Division by Zero', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=division'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a division by zero error! Continue?');">
										<?php echo esc_html__('Trigger Division Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Attempts division by zero', 'easy-error-log'); ?></p>
								</td>
							</tr>
							
							<tr>
								<th scope="row" style="display: flex; align-items: center; gap: 8px;">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>
									<?php echo esc_html__('Null Pointer Error', 'easy-error-log'); ?>
								</th>
								<td>
									<a href="<?php echo esc_url(wp_nonce_url(admin_url('tools.php?page=easy_error_log&test=null'), 'fatal_error_test', 'test_nonce')); ?>" 
									class="button button-primary"
									onclick="return confirm('This will cause a null pointer error! Continue?');">
										<?php echo esc_html__('Trigger Null Error', 'easy-error-log'); ?>
									</a>
									<p class="description"><?php echo esc_html__('Attempts to call method on null object', 'easy-error-log'); ?></p>
								</td>
							</tr>
						</tbody>
					</table>
			
					<div style="margin-top: 30px; padding: 15px; background: #fff3cd; border: 1px solid #ffeaa7;">
						
						<?php echo esc_html__('Frontend Test URLs:', 'easy-error-log'); ?></h4>
						<p><?php echo esc_html__('You can also trigger errors directly via frontend URLs:', 'easy-error-log'); ?></p>
						<ul>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=fatal')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=parse')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=memory')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=class')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=recursion')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=timeout')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=type')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=division')); ?></code></li>
							<li><code><?php echo esc_url(home_url('/?easy_error_fatal_test=null')); ?></code></li>
						</ul>
					</div>

				</div>

				<!-- System Info Tab -->

			<?php
				// Get the system information.
				$system_info = $this->system_info();
			?>

				<div id="system-info" class="tab-pane" style="display: none;">
					<h3><?php echo esc_html__( 'System Information', 'easy-error-log' ); ?></h3>
					
					<h4 class="eel-title"><?php echo esc_html__( 'WordPress Environment Information', 'easy-error-log' ); ?></h4>
					<!-- General System Information -->
					<table class="wp-list-table widefat fixed striped">
						<tbody>
							<tr>
								<th><?php echo esc_html__( 'Home URL:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['home_url'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'Site URL:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['site_url'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Site Description:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['site_description'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Timezone:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['timezone'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Date Format:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['date_format'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Time Format:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['time_format'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Post per page:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['posts_per_page'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Permalink Structure:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['permalink_structure'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Active Theme:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['active_theme'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Child Theme:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['child_theme'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'WP Content Path:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['wp_content_path'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'WP Path:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['wp_path'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'WP Version:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['wp_version'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'Multisite:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['multisite'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'Memory Limit:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['memory_limit'] ); ?></td>
							</tr>

							<tr>
								<th><?php echo esc_html__( 'Database table:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( implode( ', ', $system_info['get_database_tables'] ) ); ?></td>
							</tr>
							
							
							
							<tr>
								<th><?php echo esc_html__( 'WP Debug:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['wp_debug'] ); ?></td>
							</tr>
							<tr>
								<th><?php echo esc_html__( 'Language:', 'easy-error-log' ); ?></th>
								<td><?php echo esc_html( $system_info['language'] ); ?></td>
							</tr>
						</tbody>
					</table>
					
					<!-- Server Information -->
					<h4 class="eel-title"><?php echo esc_html__( 'Server Information', 'easy-error-log' ); ?></h4>
					<table class="wp-list-table widefat fixed striped">
						<thead>
							<tr>
								<th><?php echo esc_html__( 'Parameter', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Value', 'easy-error-log' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<tr>
								<td><?php echo esc_html__( 'Operating System:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['os'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Server Info:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['server_info'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'PHP Version:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['php_version'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Post Max Size:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['post_max_size'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Time Limit:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['time_limit'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'MySQL Version:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['mysql_version'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Max Upload Size:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['max_upload_size'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'MBString:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['mbstring'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'XML:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['xml'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'DOM:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['dom'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'LibXML:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['libxml'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'PDO:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['pdo'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Zip:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['zip'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'cURL:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['curl'] ); ?></td>
							</tr>
							
							<tr>
								<td><?php echo esc_html__( 'Apache Status:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['apache_status'] ); ?></td>
							</tr>

							<tr>
								<td><?php echo esc_html__( 'Server IP:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['server_ip'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Server Protocol:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['server_protocol'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'https:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['https'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'PHP Architecture:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['php_architecture'] ); ?></td>
							</tr>
							
							<tr>
								<td><?php echo esc_html__( 'Max Execution Time:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['max_execution_time'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Max Input Vars:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['max_input_vars'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Aapache Modules:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['apache_modules'] ); ?></td>
							</tr>
							<tr>
								<td><?php echo esc_html__( 'Apache Version:', 'easy-error-log' ); ?></td>
								<td><?php echo esc_html( $system_info['apache_version'] ); ?></td>
							</tr>
							
						</tbody>
					</table>

					
				</div>


				<!-- Theme & Plugins Tab -->
				<div id="theme-plugins" class="tab-pane" style="display: none;">
					<h3><?php echo esc_html__( 'Theme & Plugins Information', 'easy-error-log' ); ?></h3>

					<h4 class="eel-title"><?php echo esc_html__( 'Active Theme Information', 'easy-error-log' ); ?></h4>
					<table class="wp-list-table widefat fixed striped theme-info-table">
						<thead>
							<tr>
								<th><?php echo esc_html__( 'Name', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Version', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Author', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Author URI', 'easy-error-log' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<tr>
								<td><?php echo esc_html( $system_info['theme_info']['Name'] ); ?></td>
								<td><?php echo esc_html( $system_info['theme_info']['Version'] ); ?></td>
								<td><?php echo esc_html( $system_info['theme_info']['Author'] ); ?></td>
								<td><?php echo esc_html( $system_info['theme_info']['AuthorURI'] ); ?></td>
							</tr>
						</tbody>
					</table>

					<h4 class="eel-title"><?php echo esc_html__( 'Active Plugins Information', 'easy-error-log' ); ?></h4>
					<table class="wp-list-table widefat fixed striped plugins-info-table">
						<thead>
							<tr>
								<th><?php echo esc_html__( 'Name', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Version', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Author', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Description', 'easy-error-log' ); ?></th>
							</tr>
						</thead>
						<tbody>
						<?php
						foreach ( $system_info['plugins_info'] as $plugin ) {
							?>
							<tr>
								<td>
									<?php
										echo wp_kses(
											$plugin['Name'],
											array(
												'a' => array(
													'href' => true,
													'title' => true,
													'target' => true,
												),
											)
										);
									?>
									</td>

								<td><?php echo esc_html( $plugin['Version'] ); ?></td>
								<td><?php echo esc_html( $plugin['Author'] ); ?></td>
								<td><?php echo wp_kses_post( html_entity_decode( $plugin['Description'] ) ); ?></td>
							</tr>
							<?php } ?>
						</tbody>
					</table>
				</div>


				<!-- User info Tab -->
				<div id="user-info" class="tab-pane" style="display: none;">
					<h3><?php echo esc_html__( 'User Basic Information', 'easy-error-log' ); ?></h3>

					<h4 class="eel-title"><?php echo esc_html__( 'Active user', 'easy-error-log' ); ?></h4>
					<table class="wp-list-table widefat fixed striped theme-info-table">
						<thead>
							<tr>
								<th><?php echo esc_html__( 'Name', 'easy-error-log' ); ?></th>
								<th><?php echo esc_html__( 'Value', 'easy-error-log' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ( $system_info['basic_user_info'] as $key => $value ) { ?>
							<tr>
								<td><?php echo esc_html( $key ); ?></td>
								<td><?php echo esc_html( $value ); ?></td>
							</tr>
							<?php } ?>
						</tbody>
					</table>
				</div>

				<!-- About  -->
				<div id="about" class="tab-pane tab-content" style="display: none;">
					<div class="about-info">
						<h2><?php echo esc_html__('WC Bulk Product & Order Generator', 'easy-error-log'); ?></h2>
						<p><?php echo esc_html__('Generates bulk orders/products for WooCommerce with optimized batch processing', 'easy-error-log'); ?></p>
					</div>

					<div class="plugins-section-header">
						<h2 class="plugins-section-title"><?php echo esc_html__('Get More Free Plugins', 'easy-error-log'); ?></h2>
					</div>

					<div class="plugin-cards-container">
						<?php
						$plugins = [
							[
								'icon' => 'forms',
								'name' => 'Simple Form',
								'description' => 'Simple Form – Quick Form Builder for Contact Forms, Multi Step Columns based Forms',
								'tags' => [ 'Lead Management', 'Form builder' ],
								'url' => 'https://wordpress.org/plugins/simple-form/',
							],
							[
								'icon' => 'shield',
								'name' => 'Activity Guard',
								'description' => 'Real Time Notifier to Slack for System & User Activity Logs, Forum Tracker and Security',
								'tags' => [ 'Security', 'Tracker', 'Notifier' ],
								'url' => 'https://wordpress.org/plugins/notifier-to-slack/',
							],
							[
								'icon' => 'warning',
								'name' => 'WC Bulk Order Generator',
								'description' => 'WC Bulk Generator to create realistic WooCommerce test data quickly. Generate thousands of WooCommerce products and orders with just a few clicks.',
								'tags' => [ 'Generates', 'Export', 'Import', 'orders-products' ],
								'url' => 'https://wordpress.org/plugins/easy-error-log',
							],
						];

						foreach ( $plugins as $plugin ) {
							?>
							<div class="plugin-card">
								<div class="plugin-content">
									<div class="plugin-header">
										<div class="plugin-icon">
											<span class="dashicons dashicons-<?php echo esc_attr($plugin['icon']); ?>"></span>
										</div>
										<h3><?php echo esc_html($plugin['name']); ?></h3>
									</div>
									<p><?php echo esc_html($plugin['description']); ?></p>
									<div class="plugin-features">
										<?php foreach ( $plugin['tags'] as $tag ) { ?>
											<span class="feature-tag"><?php echo esc_html($tag); ?></span>
										<?php } ?>
									</div>
									<a href="<?php echo esc_url($plugin['url']); ?>" 
									class="plugin-button" 
									target="_blank" 
									rel="noopener noreferrer">
										<?php echo esc_html__('Learn More', 'easy-error-log'); ?>
										<span class="dashicons dashicons-external"></span>
									</a>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>

			</div>
			
			<?php
	}

	/**
	 * Display error floating widget.
	 */
	public function display_admin_error_widget() {
		$screen = get_current_screen();
		if ( $screen && 'tools_page_easy_error_log' === $screen->base ) {
			return;
		}

		if ( ! current_user_can('manage_options') ) {
			return;
		}

		if ( $this->get_admin_widget_status() !== 'active' ) {
			return;
		}

		?>
			<div id="admin-error-widget" class="admin-error-widget">
				<div class="error-widget-header">
					<span class="error-widget-title">
						<span class="dashicons dashicons-warning"></span>
					<?php echo esc_html__('Error Log Monitor', 'easy-error-log'); ?>
					</span>
					<div class="error-widget-actions">
						<button id="refresh-widget-log" class="widget-action-btn" title="Refresh Log">
							<span class="dashicons dashicons-update-alt"></span>
						</button>
						<button id="clean-widget-log" class="widget-action-btn" title="Clean Log">
							<span class="dashicons dashicons-trash"></span>
						</button>
						<button id="toggle-widget-view" class="widget-action-btn" title="Toggle View">
							<span class="dashicons dashicons-arrow-up-alt2"></span>
						</button>
					</div>
				</div>
				<div class="error-widget-content">
					<div class="error-stats">
						<span class="error-count"><?php echo esc_html__('Errors: ', 'easy-error-log'); ?><span id="widget-error-count"><?php echo esc_html__('0', 'easy-error-log'); ?></span></span>
						<span class="last-updated"><?php echo esc_html__('Last updated:', 'easy-error-log'); ?> <span id="widget-last-updated"><?php echo esc_html__('Just now', 'easy-error-log'); ?></span></span>
					</div>
					<div id="widget-error-container" class="widget-error-container">
						<div class="loading-errors"><?php echo esc_html__('Loading error logs...', 'easy-error-log'); ?></div>
					</div>
				</div>
			</div>
			<?php
	}

	/**
	 * Enqueue front-end scripts and styles.
	 */
	public function fe_scripts() {

		wp_enqueue_script(
		'easyerlo-fe-js',
		EASY_ERROR_LOG_DIR_URL . 'assets/fe-easy-errors.js',
		[ 'jquery' ],
		time(),
		true
		);

		wp_enqueue_style(
		'easyerlo-fe-css',
		EASY_ERROR_LOG_DIR_URL . 'assets/fe-error-style.css',
		'',
		time(),
			'all'
		);

		// Enqueue CodeMirror dependencies
		$this->enqueue_codemirror();

		// Localize the script with new data.
		$ajax_url = admin_url( 'admin-ajax.php' );
		wp_localize_script( 'easyerlo-fe-js', 'easyerlo_ajax_fe_object', array( 'ajax_url' => $ajax_url ) );
	}


	/**
	 * Function to add error page in the admin bar.
	 *
	 * @param string $wp_admin_bar   use to add error page in the admin bar.
	 */
	public function add_my_page_to_admin_bar( $wp_admin_bar ) {
		$debug_log_paths = array(
			WP_CONTENT_DIR . '/debug.log',
			ABSPATH . 'debug.log',
		);

		$debug_log_path = '';
		foreach ( $debug_log_paths as $path ) {
			if ( file_exists($path) ) {
				$debug_log_path = $path;
				break;
			}
		}

		$error_count = 0;
		if ( file_exists($debug_log_path) ) {
			$debug_log_entries = file( $debug_log_path, FILE_IGNORE_NEW_LINES );
			$error_count = count($debug_log_entries);
		}

		$wp_admin_bar->add_node(array(
			'id'    => 'my-errors-page',
			'title' => "Errors-<span style='color:red;font-weight:bold;' class='update-plugins count-$error_count'><span class='update-count'>$error_count</span></span>",
			'href'  => admin_url('tools.php?page=easy_error_log'),
		));
	}

	/**
	 * Function to get the admin widget status.
	 *
	 * @return string
	 */
	public function system_info() {
		global $wpdb;

		// Ensure the get_plugin_data function is available.
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$home_url = esc_url_raw(home_url());
		$site_url = esc_url_raw(site_url());
		$wp_content_path = defined('WP_CONTENT_DIR') ? esc_html(WP_CONTENT_DIR) : esc_html__('N/A', 'easy-error-log');
		$wp_path = defined('ABSPATH') ? esc_html(ABSPATH) : esc_html__('N/A', 'easy-error-log');
		$wp_version = get_bloginfo('version');
		$multisite = is_multisite() ? 'Yes' : 'No';
		$memory = ini_get('memory_limit');
		$memory = ! $memory || -1 === $memory ? wp_convert_hr_to_bytes(WP_MEMORY_LIMIT) : wp_convert_hr_to_bytes($memory);
		$memory = is_numeric($memory) ? size_format($memory) : 'N/A';
		$wp_debug = defined('WP_DEBUG') && WP_DEBUG ? 'Active' : 'Inactive';
		$language = get_locale();

		// Server Information.
		$os = defined('PHP_OS') ? esc_html(PHP_OS) : esc_html__('N/A', 'easy-error-log');
		$server_info = isset($_SERVER['SERVER_SOFTWARE']) ? esc_html(sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE']))) : esc_html__('Unknown', 'easy-error-log');
		$php_version = phpversion();
		$post_max_size = size_format(wp_convert_hr_to_bytes(ini_get('post_max_size')));
		$time_limit = ini_get('max_execution_time');
		$mysql_version = $wpdb->db_version();
		$max_upload_size = size_format(wp_max_upload_size());
		$mbstring = extension_loaded('mbstring') ? 'Installed' : 'Not installed';
		$xml = extension_loaded('xml') ? 'Installed' : 'Not installed';
		$dom = extension_loaded('dom') ? 'Installed' : 'Not installed';

		$libxml = extension_loaded('libxml') ? ( defined('LIBXML_VERSION') && LIBXML_VERSION > 20760 ? 'Installed - Version: ' . LIBXML_DOTTED_VERSION : 'Lower version than required' ) : 'Not installed';
		$pdo = extension_loaded('pdo') ? 'Installed - PDO Drivers: ' . implode(', ', pdo_drivers()) : 'Not installed';
		$zip = class_exists('ZipArchive') ? 'Installed' : 'Not installed';
		$curl = extension_loaded('curl') ? 'Installed - Version: ' . curl_version()['version'] : 'Not installed';

		// New.
		$server_software = isset($_SERVER['SERVER_SOFTWARE']) ? esc_html(sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE']))) : esc_html__('Unknown', 'easy-error-log');
		$server_ip = isset($_SERVER['SERVER_ADDR']) ? esc_html(sanitize_text_field(wp_unslash($_SERVER['SERVER_ADDR']))) : esc_html__('N/A', 'easy-error-log');
		$server_protocol = isset($_SERVER['SERVER_PROTOCOL']) ? esc_html(sanitize_text_field(wp_unslash($_SERVER['SERVER_PROTOCOL']))) : esc_html__('N/A', 'easy-error-log');
		$https = isset($_SERVER['HTTPS']) ? 'On' : 'Off';
		$php_architecture = PHP_INT_SIZE * 8 . 'bit';
		$php_sapi = php_sapi_name();
		$php_extensions = get_loaded_extensions();

		$max_execution_time = ini_get('max_execution_time');
		$max_input_vars = ini_get('max_input_vars');

		$apache_modules = function_exists('apache_get_modules') ? implode(', ', apache_get_modules()) : esc_html__('N/A', 'easy-error-log');
		$apache_version = function_exists('apache_get_version') ? esc_html(apache_get_version()) : esc_html__('N/A', 'easy-error-log');

		// Theme information.
		$themeObject = wp_get_theme();
		$theme_info = array(
			'Name' => esc_html($themeObject->get('Name')),
			'Version' => esc_html($themeObject->get('Version')),
			'Author' => esc_html($themeObject->get('Author')),
			'AuthorURI' => esc_html($themeObject->get('AuthorURI')),
		);

		// Active plugins information.
		$active_plugins = (array) get_option('active_plugins', array());
		if ( is_multisite() ) {
			$active_plugins = array_merge($active_plugins, array_keys(get_site_option('active_sitewide_plugins', array())));
		}
		$plugins_info = array();
		foreach ( $active_plugins as $plugin ) {
			$plugin_data = @get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
			if ( ! empty($plugin_data['Name']) ) {
				$plugins_info[] = array(
					'Name' => ! empty($plugin_data['PluginURI']) ? '<a href="' . esc_url($plugin_data['PluginURI']) . '" title="' . esc_attr__('Visit plugin homepage', 'easy-error-log') . '" target="_blank">' . esc_html($plugin_data['Name']) . '</a>' : esc_html($plugin_data['Name']),
					'Author' => esc_html($plugin_data['AuthorName']),
					'Version' => esc_html($plugin_data['Version']),
					'Description' => esc_html($plugin_data['Description']),
				);
			}
		}

		// New Information Sections.
		$apache_status = function_exists('apache_get_version') ? esc_html(apache_get_version()) : esc_html__('N/A', 'easy-error-log');
		$database_name = $wpdb->dbname;
		$database_charset = $wpdb->charset;
		$database_collate = $wpdb->collate;

		$current_user = wp_get_current_user();
		$basic_user_info = array(
			'ID' => esc_html($current_user->ID),
			'user_login' => esc_html($current_user->user_login),
			'user_pass' => esc_html($current_user->user_pass),
			'user_nicename' => esc_html($current_user->user_nicename),
			'user_email' => esc_html($current_user->user_email),
			'user_url' => esc_html($current_user->user_url),
			'user_registered' => esc_html($current_user->user_registered),
			'user_activation_key' => esc_html($current_user->user_activation_key),
			'user_status' => esc_html($current_user->user_status),
			'user_firstname' => esc_html($current_user->user_firstname),
			'user_lastname' => esc_html($current_user->user_lastname),
			'display_name' => esc_html($current_user->display_name),
			'roles' => implode(', ', $current_user->roles),
			'user_email_verified' => 'N/A',
			'user_locale' => get_user_meta($current_user->ID, 'locale', true),
		);

		// Return the information as an associative array.
		return array(
			'home_url' => $home_url,
			'site_url' => $site_url,

			'site_description' => get_bloginfo('description'),
			'admin_email' => get_bloginfo('admin_email'),
			'timezone' => get_option('timezone_string') ? get_option('timezone_string') : 'UTC' . get_option('gmt_offset'),
			'date_format' => get_option('date_format'),
			'time_format' => get_option('time_format'),
			'posts_per_page' => get_option('posts_per_page'),
			'permalink_structure' => get_option('permalink_structure'),
			'active_theme' => wp_get_theme()->get('Name'),
			'child_theme' => is_child_theme() ? 'Yes' : 'No',

			'wp_content_path' => $wp_content_path,
			'wp_path' => $wp_path,
			'wp_version' => $wp_version,
			'multisite' => $multisite,
			'memory_limit' => $memory,

			'get_database_tables' => $this->get_database_tables(),

			'wp_debug' => $wp_debug,
			'language' => $language,
			'os' => $os,
			'server_info' => $server_info,
			'php_version' => $php_version,
			'post_max_size' => $post_max_size,
			'time_limit' => $time_limit,
			'mysql_version' => $mysql_version,
			'max_upload_size' => $max_upload_size,
			'mbstring' => $mbstring,
			'xml' => $xml,
			'dom' => $dom,
			'libxml' => $libxml,
			'pdo' => $pdo,
			'zip' => $zip,
			'curl' => $curl,

			'server_ip' => $server_ip,
			'server_protocol' => $server_protocol,
			'https' => $https,
			'php_architecture' => $php_architecture,
			'php_sapi' => $php_sapi,
			'max_execution_time' => $max_execution_time,
			'max_input_vars' => $max_input_vars,
			'apache_modules' => $apache_modules,
			'apache_version' => $apache_version,

			'apache_status' => $apache_status,
			'theme_info' => $theme_info,
			'plugins_info' => $plugins_info,

			'basic_user_info' => $basic_user_info,
		);
	}

	/**
	 * Function to get the database tables.
	 *
	 * @return array
	 */
	private function get_database_tables() {
		global $wpdb;

		// Use wp_cache functions instead of transients.
		$cache_key = 'eel_database_tables';
		$cache_group = 'easy_error_log';

		// Try to get from cache first.
		$tables = wp_cache_get($cache_key, $cache_group);

		if ( false === $tables ) {
			// Disable PHPCS warning for this necessary direct query.
			// phpcs:ignore
			$tables = $wpdb->get_results('SHOW TABLES', ARRAY_N);
			$tables = array_map(function ( $table ) {
				return $table[0];
			}, $tables);

			// Cache the resultsss.
			wp_cache_set($cache_key, $tables, $cache_group, HOUR_IN_SECONDS);

		}

		return $tables;
	}


	public function trigger_test_error() {
		// Check for admin test with nonce verification
		if ( is_admin() && isset($_GET['page']) && $_GET['page'] === 'easy_error_log' && isset($_GET['test']) ) {
			// Verify nonce for security
			if ( ! isset($_GET['test_nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['test_nonce'])), 'fatal_error_test') ) {
				wp_die(esc_html__('Security check failed.', 'easy-error-log'));
			}

			// Verify user capabilities
			if ( ! current_user_can('manage_options') ) {
				wp_die(esc_html__('You do not have sufficient permissions to perform this action.', 'easy-error-log'));
			}

			$test_type = sanitize_text_field(wp_unslash($_GET['test']));
			$this->trigger_error($test_type);
		}

		// Check for frontend test (less secure, for testing purposes only)
		if ( isset($_GET['easy_error_fatal_test']) ) {
			$test_type = sanitize_text_field(wp_unslash($_GET['easy_error_fatal_test']));
			$this->trigger_error($test_type);
		}
	}

	private function trigger_error( $type ) {
		// Sanitize the error type
		$type = sanitize_text_field($type);

		// Log the test attempt
		error_log(sprintf(
			/* translators: 1: error type, 2: file path */
			esc_html__('WPNTS Test: About to trigger %1$s error from %2$s', 'easy-error-log'),
			$type,
			__FILE__
		));

		switch ( $type ) {
			case 'fatal':
				// This will cause E_ERROR Fatal error: Uncaught Error:
				this_function_does_not_exist();
				break;

			case 'parse':
				// This will cause a parse error by creating invalid syntax
				// We'll use eval to create a parse error at runtime
				eval('this is invalid php syntax without semicolon');
				break;

			case 'memory':
				// This will cause a memory exhaustion error
				$array = array();
				while ( true ) {
					$array[] = str_repeat('x', 1024 * 1024); // 1MB strings
				}
				break;

			case 'class':
				// This will cause a fatal error - class not found
				$obj = new NonExistentClass();
				break;

			case 'recursion':
				// This will cause a stack overflow due to infinite recursion
				$this->infinite_recursion();
				break;

			case 'timeout':
				// This will cause a timeout error
				set_time_limit(1); // Set a very short time limit
				sleep(5); // Sleep longer than the time limit
				break;

			case 'type':
				// This will cause a type error
				$this->test_type_error(null);
				break;

			case 'division':
				// This attempts division by zero (may trigger error depending on PHP version)
				$result = 10 / 0;
				echo esc_html($result);
				break;

			case 'null':
				// This will cause a fatal error - attempting to call method on null
				$null_object = null;
				$null_object->some_method();
				break;

			default:
				wp_die(
					sprintf(
						/* translators: %s: invalid test type */
						esc_html__('Invalid test type: %s', 'easy-error-log'),
						esc_html($type)
					)
				);
		}
	}

	private function infinite_recursion( $depth = 0 ) {
		// Log every 1000 iterations to show progress
		if ( $depth % 1000 === 0 ) {
			error_log(sprintf(
				/* translators: %d: recursion depth */
				esc_html__('Recursion depth: %d', 'easy-error-log'),
				$depth
			));
		}

		// Infinite recursion - will eventually cause stack overflow
		return $this->infinite_recursion($depth + 1);
	}

	private function test_type_error( string $required_string ) {
		// This function requires a string but we'll pass null from trigger_error
		return strlen($required_string);
	}


	/**
	 * Function to add error page in the admin bar.
	 */
	public function display_error_floating_widget() {
		$mode = get_option( 'fe_widgets_mode', 'false' );
		if ( 'true' === $mode ) {
			?>
				<div id="error-log-container" class="error-log-container">
					<div class="error-log-header">
						<span><?php echo esc_html__( 'Error Log', 'easy-error-log' ); ?></span>
						<button id="error-log-toggle" class="error-log-toggle"><?php echo esc_html__( '+', 'easy-error-log' ); ?></button>
					</div>
					<div id="error-log-content" class="error-log-content"></div>
				</div>
				<?php
		}
	}

	/**
	 * Enqueue CodeMirror dependencies
	 */
	private function enqueue_codemirror() {
		// Core CodeMirror
		wp_enqueue_style('codemirror', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/codemirror.min.css', array(), '6.65.7');
		wp_enqueue_script('codemirror', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/codemirror.min.js', array(), '6.65.7', true);

		// Themes
		wp_enqueue_style('codemirror-monokai', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/monokai.min.css', array( 'codemirror' ), '6.65.7');
		wp_enqueue_style('codemirror-eclipse', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/eclipse.min.css', array( 'codemirror' ), '6.65.7');

		// Modes
		wp_enqueue_script('codemirror-xml', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/xml.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-javascript', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/javascript.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-css', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/css.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-php', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/php.min.js', array( 'codemirror' ), '6.65.7', true);

		// Addons
		wp_enqueue_script('codemirror-matchbrackets', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/matchbrackets.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-closebrackets', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/closebrackets.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-show-hint', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/show-hint.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-anyword-hint', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/anyword-hint.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-javascript-hint', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/javascript-hint.min.js', array( 'codemirror' ), '6.65.7', true);

		wp_enqueue_script('codemirror-search', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/search.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-searchcursor', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/searchcursor.min.js', array( 'codemirror' ), '6.65.7', true);
		wp_enqueue_script('codemirror-dialog', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/dialog.min.js', array( 'codemirror' ), '6.65.7', true);

		wp_enqueue_style('codemirror-show-hint', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/show-hint.min.css', array( 'codemirror' ), '6.65.7');
		wp_enqueue_style('codemirror-dialog', EASY_ERROR_LOG_DIR_URL . 'assets/codemirror/dialog.min.css', array( 'codemirror' ), '6.65.7');
	}
}
